<?php
session_start();
include_once "../config/db_connect.php";
include_once "../includes/functions.php";

// Check if user is logged in and is an admin
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'admin') {
    header("Location: ../login.php");
    exit;
}

$admin_id = $_SESSION['user_id'];
$admin = get_admin($conn, $admin_id);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Assign Students to Teachers - Admin Dashboard - Panadite Academy</title>
    
    <!-- Stylesheets -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/@fortawesome/fontawesome-free@6.4.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/toastify-js@1.12.0/src/toastify.min.css" rel="stylesheet">
    <link href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <link href="css/admin-styles.css" rel="stylesheet">
    
    <style>
        /* Additional Custom Styles for Assignment Page */
        .avatar-placeholder {
            width: 40px;
            height: 40px;
            font-size: 16px;
            font-weight: bold;
        }
        
        .student-avatar img, .teacher-avatar img {
            width: 40px;
            height: 40px;
            object-fit: cover;
        }
        
        .assignment-card {
            transition: all 0.3s ease;
        }
        
        .assignment-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 20px rgba(0,0,0,0.1);
        }
        
        .teacher-select-container .select2-container {
            width: 100% !important;
        }
        
        .teacher-select-container .select2-selection {
            height: 38px;
            padding: 5px;
            border: 1px solid #ced4da;
        }
        
        .teacher-select-container .select2-selection__arrow {
            height: 36px;
        }
        
        .assignment-count {
            min-width: 24px;
            height: 24px;
            border-radius: 50%;
            background-color: var(--bs-primary);
            color: white;
            font-size: 12px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            margin-left: 8px;
        }
        
        .filter-card {
            border-left: 4px solid var(--bs-primary);
        }
    </style>
</head>
<body>
    <!-- Main Dashboard Container -->
    <div class="dashboard-container">
        <!-- Include sidebar -->
        <?php include_once "includes/admin-sidebar.php"; ?>
        
        <!-- Main Content Area -->
        <div class="content-wrapper">
            <!-- Top Navigation Bar -->
            <?php include_once "includes/admin-navbar.php"; ?>
            
            <!-- Page Content -->
            <div class="container-fluid main-content py-4">
                <div class="row mb-4">
                    <div class="col-12">
                        <div class="d-flex justify-content-between align-items-center">
                            <h4 class="mb-0">Assign Students to Teachers</h4>
                            <nav aria-label="breadcrumb">
                                <ol class="breadcrumb m-0">
                                    <li class="breadcrumb-item"><a href="index.php">Dashboard</a></li>
                                    <li class="breadcrumb-item active" aria-current="page">Manage Assignments</li>
                                </ol>
                            </nav>
                        </div>
                    </div>
                </div>
                
                <!-- Filter and Statistics Section -->
                <div class="row mb-4">
                    <div class="col-12 col-lg-8">
                        <div class="card filter-card shadow-sm border-0">
                            <div class="card-body">
                                <h5 class="card-title">Filter Assignments</h5>
                                <div class="row g-3">
                                    <div class="col-md-4">
                                        <label for="teacherFilter" class="form-label">Teacher</label>
                                        <select id="teacherFilter" class="form-select teacher-select">
                                            <option value="">All Teachers</option>
                                            <!-- Teacher options will be loaded by AJAX -->
                                        </select>
                                    </div>
                                    <div class="col-md-4">
                                        <label for="statusFilter" class="form-label">Status</label>
                                        <select id="statusFilter" class="form-select">
                                            <option value="">All Statuses</option>
                                            <option value="active">Active</option>
                                            <option value="inactive">Inactive</option>
                                        </select>
                                    </div>
                                    <div class="col-md-4 d-flex align-items-end">
                                        <button id="applyFilters" class="btn btn-primary w-100">
                                            <i class="fas fa-filter me-2"></i>Apply Filters
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-12 col-lg-4">
                        <div class="row g-3 h-100">
                            <div class="col-6 col-lg-12">
                                <div class="card h-100 border-0 shadow-sm">
                                    <div class="card-body d-flex align-items-center">
                                        <div class="stat-icon bg-primary-light rounded me-3">
                                            <i class="fas fa-link text-primary"></i>
                                        </div>
                                        <div>
                                            <h3 class="mb-0" id="totalAssignmentsCount">0</h3>
                                            <p class="text-muted mb-0">Total Assignments</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-6 col-lg-12">
                                <div class="card h-100 border-0 shadow-sm">
                                    <div class="card-body d-flex justify-content-center">
                                        <button class="btn btn-success w-100" data-bs-toggle="modal" data-bs-target="#newAssignmentModal">
                                            <i class="fas fa-plus-circle me-2"></i>New Assignment
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Assignments Section -->
                <div class="row mb-3">
                    <div class="col-12">
                        <h5>Current Assignments</h5>
                    </div>
                </div>
                
                <!-- Tab Navigation -->
                <ul class="nav nav-tabs mb-4" id="assignmentTabs">
                    <li class="nav-item">
                        <a class="nav-link active" id="all-tab" data-bs-toggle="tab" href="#all">
                            All <span class="assignment-count" id="allCount">0</span>
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" id="active-tab" data-bs-toggle="tab" href="#active">
                            Active <span class="assignment-count" id="activeCount">0</span>
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" id="inactive-tab" data-bs-toggle="tab" href="#inactive">
                            Inactive <span class="assignment-count" id="inactiveCount">0</span>
                        </a>
                    </li>
                </ul>
                
                <!-- Tab Content -->
                <div class="tab-content">
                    <div class="tab-pane fade show active" id="all">
                        <div class="row assignment-container" id="allAssignments">
                            <!-- Assignment cards will be loaded here -->
                            <div class="col-12 text-center py-5" id="loadingAssignments">
                                <div class="spinner-border text-primary" role="status">
                                    <span class="visually-hidden">Loading...</span>
                                </div>
                                <p class="mt-2">Loading assignments...</p>
                            </div>
                        </div>
                    </div>
                    <div class="tab-pane fade" id="active">
                        <div class="row assignment-container" id="activeAssignments">
                            <!-- Active assignments will be loaded here -->
                        </div>
                    </div>
                    <div class="tab-pane fade" id="inactive">
                        <div class="row assignment-container" id="inactiveAssignments">
                            <!-- Inactive assignments will be loaded here -->
                        </div>
                    </div>
                </div>
                
                <div id="noAssignmentsMessage" class="alert alert-info text-center d-none">
                    <i class="fas fa-info-circle me-2"></i>
                    No student-teacher assignments found. Use the "New Assignment" button to create your first assignment.
                </div>
                
            </div><!-- End main-content -->
        </div><!-- End content-wrapper -->
    </div><!-- End dashboard-container -->
    
    <!-- New Assignment Modal -->
    <div class="modal fade" id="newAssignmentModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Create New Assignment</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form id="assignmentForm">
                        <div class="mb-3">
                            <label for="teacherSelect" class="form-label">Select Teacher</label>
                            <select id="teacherSelect" class="form-select teacher-select" required>
                                <option value="">Choose a teacher</option>
                                <!-- Teacher options will be loaded via AJAX -->
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Select Students</label>
                            <div class="input-group mb-2">
                                <input type="text" id="studentSearch" class="form-control" placeholder="Search for students...">
                                <button class="btn btn-outline-secondary" type="button" id="clearSearch">
                                    <i class="fas fa-times"></i>
                                </button>
                            </div>
                            <div class="card">
                                <div class="card-header bg-light d-flex justify-content-between align-items-center py-2">
                                    <span>Available Students</span>
                                    <div>
                                        <button type="button" class="btn btn-sm btn-link" id="selectAllStudents">Select All</button>
                                        <button type="button" class="btn btn-sm btn-link" id="deselectAllStudents">Deselect All</button>
                                    </div>
                                </div>
                                <div class="card-body p-0" style="max-height: 250px; overflow-y: auto;">
                                    <ul class="list-group list-group-flush" id="studentsList">
                                        <!-- Students will be loaded via AJAX -->
                                        <li class="list-group-item text-center py-4" id="studentsLoading">
                                            <div class="spinner-border spinner-border-sm text-primary" role="status">
                                                <span class="visually-hidden">Loading...</span>
                                            </div>
                                            <span class="ms-2">Loading students...</span>
                                        </li>
                                        <li class="list-group-item text-center py-3 d-none" id="noStudentsFound">
                                            <i class="fas fa-info-circle text-muted me-2"></i>
                                            No unassigned students found
                                        </li>
                                    </ul>
                                </div>
                            </div>
                            <div id="selectedStudentsCount" class="mt-2 text-muted small">
                                0 students selected
                            </div>
                        </div>
                        <div class="mb-3">
                            <label for="assignmentNotes" class="form-label">Notes (Optional)</label>
                            <textarea class="form-control" id="assignmentNotes" rows="3" placeholder="Add any notes about this assignment..."></textarea>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" id="saveAssignment" disabled>
                        <i class="fas fa-save me-1"></i> Create Assignment
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- View Assignment Modal -->
    <div class="modal fade" id="viewAssignmentModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Assignment Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="d-flex align-items-center mb-4">
                        <div id="viewTeacherAvatar" class="me-3 teacher-avatar">
                            <!-- Teacher avatar will be inserted here -->
                        </div>
                        <div>
                            <h5 class="mb-0" id="viewTeacherName">Teacher Name</h5>
                            <p class="text-muted mb-0" id="viewTeacherEmail">teacher@example.com</p>
                        </div>
                    </div>
                    
                    <div class="card mb-3">
                        <div class="card-header">
                            <h6 class="mb-0">Assigned Student</h6>
                        </div>
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div id="viewStudentAvatar" class="me-3 student-avatar">
                                    <!-- Student avatar will be inserted here -->
                                </div>
                                <div>
                                    <h6 class="mb-0" id="viewStudentName">Student Name</h6>
                                    <p class="text-muted mb-0" id="viewStudentEmail">student@example.com</p>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row g-2 mb-3">
                        <div class="col-6">
                            <div class="card h-100">
                                <div class="card-body p-3">
                                    <div class="small text-muted mb-1">Date Assigned</div>
                                    <div id="viewAssignedDate">June 15, 2023</div>
                                </div>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="card h-100">
                                <div class="card-body p-3">
                                    <div class="small text-muted mb-1">Status</div>
                                    <div id="viewStatus"><span class="badge bg-success">Active</span></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="card mb-0" id="viewNotesCard">
                        <div class="card-header">
                            <h6 class="mb-0">Notes</h6>
                        </div>
                        <div class="card-body">
                            <p id="viewNotes" class="mb-0">No notes available.</p>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-danger" id="toggleAssignmentStatus">
                        <i class="fas fa-ban me-1"></i> Deactivate
                    </button>
                    <button type="button" class="btn btn-primary" id="editAssignment">
                        <i class="fas fa-edit me-1"></i> Edit
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- JavaScript Libraries -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/toastify-js"></script>
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.0/dist/jquery.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    
    <script>
        // Basic sidebar toggle functionality
        document.getElementById('sidebarToggle').addEventListener('click', function() {
            document.querySelector('.sidebar').classList.toggle('show');
        });
        
        // Initialize tooltips and Select2
        $(document).ready(function() {
            // Initialize tooltips
            const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
            
            // Initialize Select2 for teacher dropdowns
            $('.teacher-select').select2({
                placeholder: "Select a teacher",
                allowClear: true,
                width: '100%'
            });
            
            // Load teachers for dropdowns
            loadTeachers();
            
            // Load all assignments
            loadAssignments();
            
            // Event handlers
            $("#applyFilters").on("click", function() {
                loadAssignments();
            });
            
            // Teacher selection changes
            $("#teacherSelect").on("change", function() {
                if ($(this).val()) {
                    loadUnassignedStudents($(this).val());
                } else {
                    // Clear student list if no teacher selected
                    $("#studentsList").empty();
                    $("#studentsLoading").removeClass("d-none");
                    $("#noStudentsFound").addClass("d-none");
                    updateSelectedStudentsCount();
                }
            });
            
            // Student search
            $("#studentSearch").on("keyup", function() {
                const searchTerm = $(this).val().toLowerCase();
                $("#studentsList li.student-item").each(function() {
                    const studentName = $(this).find(".student-name").text().toLowerCase();
                    const studentEmail = $(this).find(".student-email").text().toLowerCase();
                    
                    if (studentName.includes(searchTerm) || studentEmail.includes(searchTerm)) {
                        $(this).show();
                    } else {
                        $(this).hide();
                    }
                });
            });
            
            // Clear search
            $("#clearSearch").on("click", function() {
                $("#studentSearch").val("");
                $("#studentsList li.student-item").show();
            });
            
            // Select/deselect all students
            $("#selectAllStudents").on("click", function() {
                $("#studentsList li.student-item:visible input[type='checkbox']").prop("checked", true).trigger("change");
            });
            
            $("#deselectAllStudents").on("click", function() {
                $("#studentsList li.student-item input[type='checkbox']").prop("checked", false).trigger("change");
            });
            
            // Save new assignment
            $("#saveAssignment").on("click", function() {
                saveAssignment();
            });
        });
        
        // Counter to track selected students
        function updateSelectedStudentsCount() {
            const count = $("#studentsList input[type='checkbox']:checked").length;
            $("#selectedStudentsCount").text(count + " students selected");
            
            // Enable save button only if at least one student is selected
            if (count > 0 && $("#teacherSelect").val()) {
                $("#saveAssignment").prop("disabled", false);
            } else {
                $("#saveAssignment").prop("disabled", true);
            }
        }
        
        // Load teachers for select dropdowns
        function loadTeachers() {
            $.ajax({
                url: 'ajax/get_teachers.php',
                type: 'GET',
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        const teachers = response.data;
                        
                        // Clear and populate teacher dropdowns
                        $("#teacherFilter, #teacherSelect").empty();
                        $("#teacherFilter").append('<option value="">All Teachers</option>');
                        $("#teacherSelect").append('<option value="">Choose a teacher</option>');
                        
                        teachers.forEach(function(teacher) {
                            const option = `<option value="${teacher.user_id}">${teacher.first_name} ${teacher.last_name}</option>`;
                            $("#teacherFilter, #teacherSelect").append(option);
                        });
                        
                        // Refresh Select2
                        $('.teacher-select').trigger('change');
                    } else {
                        showToast('Error loading teachers', 'error');
                    }
                },
                error: function() {
                    showToast('Server error while loading teachers', 'error');
                }
            });
        }
        
        // Load unassigned students (not yet assigned to this specific teacher)
        function loadUnassignedStudents(teacherId) {
            $("#studentsLoading").removeClass("d-none");
            $("#noStudentsFound").addClass("d-none");
            $("#studentsList .student-item").remove();
            
            $.ajax({
                url: 'ajax/get_unassigned_students.php',
                type: 'GET',
                data: { teacher_id: teacherId },
                dataType: 'json',
                success: function(response) {
                    $("#studentsLoading").addClass("d-none");
                    
                    if (response.success) {
                        const students = response.data;
                        
                        if (students.length === 0) {
                            $("#noStudentsFound").removeClass("d-none");
                            return;
                        }
                        
                        students.forEach(function(student) {
                            // Create avatar content
                            let avatarContent = '';
                            if (student.profile_image) {
                                avatarContent = `<img src="../${student.profile_image}" alt="${student.first_name} ${student.last_name}" class="rounded-circle">`;
                            } else {
                                const initials = student.first_name.charAt(0) + student.last_name.charAt(0);
                                avatarContent = `<div class="avatar-placeholder rounded-circle d-flex align-items-center justify-content-center bg-primary-light text-primary">${initials}</div>`;
                            }
                            
                            const studentItem = `
                                <li class="list-group-item py-2 student-item">
                                    <div class="d-flex align-items-center">
                                        <div class="form-check">
                                            <input class="form-check-input student-checkbox" type="checkbox" value="${student.user_id}" id="student-${student.user_id}" data-student-id="${student.user_id}">
                                        </div>
                                        <label class="ms-2 d-flex align-items-center w-100" for="student-${student.user_id}">
                                            <div class="student-avatar me-2">
                                                ${avatarContent}
                                            </div>
                                            <div>
                                                <div class="student-name">${student.first_name} ${student.last_name}</div>
                                                <div class="small text-muted student-email">${student.email}</div>
                                            </div>
                                        </label>
                                    </div>
                                </li>
                            `;
                            
                            $("#studentsList").append(studentItem);
                        });
                        
                        // Attach change event to checkboxes
                        $(".student-checkbox").on("change", function() {
                            updateSelectedStudentsCount();
                        });
                    } else {
                        showToast('Error loading students', 'error');
                        $("#noStudentsFound").removeClass("d-none");
                    }
                },
                error: function() {
                    $("#studentsLoading").addClass("d-none");
                    $("#noStudentsFound").removeClass("d-none");
                    showToast('Server error while loading students', 'error');
                }
            });
        }
        
        // Load all assignments based on filters
        function loadAssignments() {
            const teacherId = $("#teacherFilter").val();
            const status = $("#statusFilter").val();
            
            $("#loadingAssignments").removeClass("d-none");
            $("#noAssignmentsMessage").addClass("d-none");
            $("#allAssignments .assignment-card").remove();
            $("#activeAssignments .assignment-card").remove();
            $("#inactiveAssignments .assignment-card").remove();
            
            $.ajax({
                url: 'ajax/get_teacher_student_assignments.php',
                type: 'GET',
                data: {
                    teacher_id: teacherId,
                    status: status
                },
                dataType: 'json',
                success: function(response) {
                    $("#loadingAssignments").addClass("d-none");
                    
                    if (response.success) {
                        const assignments = response.data;
                        
                        // Set counters
                        $("#totalAssignmentsCount").text(assignments.length);
                        
                        // Filter assignments by status
                        const activeAssignments = assignments.filter(a => a.status === 'active');
                        const inactiveAssignments = assignments.filter(a => a.status === 'inactive');
                        
                        // Update tab counts
                        $("#allCount").text(assignments.length);
                        $("#activeCount").text(activeAssignments.length);
                        $("#inactiveCount").text(inactiveAssignments.length);
                        
                        if (assignments.length === 0) {
                            $("#noAssignmentsMessage").removeClass("d-none");
                        } else {
                            renderAssignments(assignments, "#allAssignments");
                            renderAssignments(activeAssignments, "#activeAssignments");
                            renderAssignments(inactiveAssignments, "#inactiveAssignments");
                        }
                    } else {
                        showToast('Error loading assignments', 'error');
                        $("#noAssignmentsMessage").removeClass("d-none");
                    }
                },
                error: function() {
                    $("#loadingAssignments").addClass("d-none");
                    $("#noAssignmentsMessage").removeClass("d-none");
                    showToast('Server error while loading assignments', 'error');
                }
            });
        }
        
        // Render assignment cards
        function renderAssignments(assignments, containerId) {
            assignments.forEach(function(assignment) {
                // Create teacher avatar
                let teacherAvatar = '';
                if (assignment.teacher_profile_image) {
                    teacherAvatar = `<img src="../${assignment.teacher_profile_image}" alt="${assignment.teacher_name}" class="rounded-circle">`;
                } else {
                    const teacherInitials = assignment.teacher_first_name.charAt(0) + assignment.teacher_last_name.charAt(0);
                    teacherAvatar = `<div class="avatar-placeholder rounded-circle d-flex align-items-center justify-content-center bg-primary-light text-primary">${teacherInitials}</div>`;
                }
                
                // Create student avatar
                let studentAvatar = '';
                if (assignment.student_profile_image) {
                    studentAvatar = `<img src="../${assignment.student_profile_image}" alt="${assignment.student_name}" class="rounded-circle">`;
                } else {
                    const studentInitials = assignment.student_first_name.charAt(0) + assignment.student_last_name.charAt(0);
                    studentAvatar = `<div class="avatar-placeholder rounded-circle d-flex align-items-center justify-content-center bg-primary-light text-primary">${studentInitials}</div>`;
                }
                
                // Format date
                const assignedDate = new Date(assignment.assigned_date);
                const formattedDate = assignedDate.toLocaleDateString('en-US', {
                    year: 'numeric',
                    month: 'short',
                    day: 'numeric'
                });
                
                // Create assignment card
                const assignmentCard = `
                    <div class="col-md-6 col-lg-4 mb-4">
                        <div class="card assignment-card h-100 shadow-sm border-0" data-assignment-id="${assignment.assignment_id}">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <div class="teacher-avatar">
                                        ${teacherAvatar}
                                    </div>
                                    <div>
                                        ${assignment.status === 'active' ? 
                                            '<span class="badge bg-success">Active</span>' : 
                                            '<span class="badge bg-secondary">Inactive</span>'}
                                    </div>
                                </div>
                                <h5 class="card-title mb-1">${assignment.teacher_first_name} ${assignment.teacher_last_name}</h5>
                                <p class="text-muted small mb-3">${assignment.teacher_email}</p>
                                
                                <div class="d-flex align-items-center mb-3">
                                    <div class="student-avatar me-2">
                                        ${studentAvatar}
                                    </div>
                                    <div>
                                        <div class="fw-medium">${assignment.student_first_name} ${assignment.student_last_name}</div>
                                        <div class="text-muted small">${assignment.student_email}</div>
                                    </div>
                                </div>
                                
                                <div class="d-flex justify-content-between align-items-center">
                                    <div class="text-muted small">
                                        <i class="fas fa-calendar-alt me-1"></i> ${formattedDate}
                                    </div>
                                    <button class="btn btn-sm btn-outline-primary view-assignment" data-assignment-id="${assignment.assignment_id}">
                                        <i class="fas fa-eye me-1"></i> View
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                `;
                
                $(containerId).append(assignmentCard);
            });
            
            // Attach click events to view buttons
            $(".view-assignment").on("click", function() {
                const assignmentId = $(this).data("assignment-id");
                viewAssignment(assignmentId);
            });
        }
        
        // Show toast notification
        function showToast(message, type = 'success') {
            Toastify({
                text: message,
                duration: 3000,
                close: true,
                gravity: "top",
                position: "right",
                backgroundColor: type === 'success' ? "#28a745" : "#dc3545",
            }).showToast();
        }
        
        // Save new assignment
        function saveAssignment() {
            const teacherId = $("#teacherSelect").val();
            const notes = $("#assignmentNotes").val();
            const selectedStudents = [];
            
            // Collect all selected student IDs
            $("#studentsList input[type='checkbox']:checked").each(function() {
                selectedStudents.push($(this).data("student-id"));
            });
            
            // Check required fields
            if (!teacherId || selectedStudents.length === 0) {
                showToast("Please select a teacher and at least one student", "error");
                return;
            }
            
            // Disable save button and show loading state
            $("#saveAssignment").prop("disabled", true).html(
                `<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Saving...`
            );
            
            $.ajax({
                url: 'ajax/create_teacher_student_assignment.php',
                type: 'POST',
                data: {
                    teacher_id: teacherId,
                    student_ids: selectedStudents,
                    notes: notes
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        // Close modal and reset form
                        $("#newAssignmentModal").modal("hide");
                        $("#assignmentForm")[0].reset();
                        $("#teacherSelect").val("").trigger("change");
                        $("#studentsList").empty();
                        $("#studentsLoading").removeClass("d-none");
                        $("#saveAssignment").html(`<i class="fas fa-save me-1"></i> Create Assignment`);
                        
                        // Reload assignments
                        loadAssignments();
                        
                        // Show success message
                        showToast(response.message || "Assignment created successfully");
                    } else {
                        showToast(response.message || "Error creating assignment", "error");
                        $("#saveAssignment").prop("disabled", false).html(`<i class="fas fa-save me-1"></i> Create Assignment`);
                    }
                },
                error: function() {
                    showToast("Server error while creating assignment", "error");
                    $("#saveAssignment").prop("disabled", false).html(`<i class="fas fa-save me-1"></i> Create Assignment`);
                }
            });
        }
        
        // View assignment details
        function viewAssignment(assignmentId) {
            $.ajax({
                url: 'ajax/get_assignment_details.php',
                type: 'GET',
                data: { assignment_id: assignmentId },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        const assignment = response.data;
                        
                        // Populate teacher information
                        if (assignment.teacher_profile_image) {
                            $("#viewTeacherAvatar").html(`<img src="../${assignment.teacher_profile_image}" alt="${assignment.teacher_name}" class="rounded-circle">`);
                        } else {
                            const teacherInitials = assignment.teacher_first_name.charAt(0) + assignment.teacher_last_name.charAt(0);
                            $("#viewTeacherAvatar").html(`<div class="avatar-placeholder rounded-circle d-flex align-items-center justify-content-center bg-primary-light text-primary">${teacherInitials}</div>`);
                        }
                        $("#viewTeacherName").text(`${assignment.teacher_first_name} ${assignment.teacher_last_name}`);
                        $("#viewTeacherEmail").text(assignment.teacher_email);
                        
                        // Populate student information
                        if (assignment.student_profile_image) {
                            $("#viewStudentAvatar").html(`<img src="../${assignment.student_profile_image}" alt="${assignment.student_name}" class="rounded-circle">`);
                        } else {
                            const studentInitials = assignment.student_first_name.charAt(0) + assignment.student_last_name.charAt(0);
                            $("#viewStudentAvatar").html(`<div class="avatar-placeholder rounded-circle d-flex align-items-center justify-content-center bg-primary-light text-primary">${studentInitials}</div>`);
                        }
                        $("#viewStudentName").text(`${assignment.student_first_name} ${assignment.student_last_name}`);
                        $("#viewStudentEmail").text(assignment.student_email);
                        
                        // Populate assignment details
                        const assignedDate = new Date(assignment.assigned_date);
                        const formattedDate = assignedDate.toLocaleDateString('en-US', {
                            year: 'numeric',
                            month: 'long',
                            day: 'numeric'
                        });
                        $("#viewAssignedDate").text(formattedDate);
                        
                        // Set status badge
                        if (assignment.status === 'active') {
                            $("#viewStatus").html('<span class="badge bg-success">Active</span>');
                            $("#toggleAssignmentStatus").html('<i class="fas fa-ban me-1"></i> Deactivate');
                            $("#toggleAssignmentStatus").removeClass("btn-success").addClass("btn-danger");
                        } else {
                            $("#viewStatus").html('<span class="badge bg-secondary">Inactive</span>');
                            $("#toggleAssignmentStatus").html('<i class="fas fa-check me-1"></i> Activate');
                            $("#toggleAssignmentStatus").removeClass("btn-danger").addClass("btn-success");
                        }
                        
                        // Set notes
                        if (assignment.notes) {
                            $("#viewNotes").text(assignment.notes);
                            $("#viewNotesCard").removeClass("d-none");
                        } else {
                            $("#viewNotes").text("No notes available.");
                            $("#viewNotesCard").removeClass("d-none");
                        }
                        
                        // Set assignment ID for action buttons
                        $("#toggleAssignmentStatus").data("assignment-id", assignment.assignment_id);
                        $("#editAssignment").data("assignment-id", assignment.assignment_id);
                        
                        // Setup toggle action
                        $("#toggleAssignmentStatus").off("click").on("click", function() {
                            const assignmentId = $(this).data("assignment-id");
                            const newStatus = assignment.status === 'active' ? 'inactive' : 'active';
                            toggleAssignmentStatus(assignmentId, newStatus);
                        });
                        
                        // Setup edit action
                        $("#editAssignment").off("click").on("click", function() {
                            // Handle edit in future implementation
                            showToast("Edit functionality will be implemented in future updates", "info");
                        });
                        
                        // Show modal
                        $("#viewAssignmentModal").modal("show");
                    } else {
                        showToast(response.message || "Error loading assignment details", "error");
                    }
                },
                error: function() {
                    showToast("Server error while loading assignment details", "error");
                }
            });
        }
        
        // Toggle assignment status (activate/deactivate)
        function toggleAssignmentStatus(assignmentId, newStatus) {
            // Confirm action
            if (!confirm(newStatus === 'active' ? "Activate this assignment?" : "Deactivate this assignment?")) {
                return;
            }
            
            // Show loading state
            $("#toggleAssignmentStatus").prop("disabled", true).html(
                `<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> ${newStatus === 'active' ? 'Activating...' : 'Deactivating...'}`
            );
            
            $.ajax({
                url: 'ajax/update_assignment_status.php',
                type: 'POST',
                data: {
                    assignment_id: assignmentId,
                    status: newStatus
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        // Close modal
                        $("#viewAssignmentModal").modal("hide");
                        
                        // Reload assignments
                        loadAssignments();
                        
                        // Show success message
                        showToast(response.message || `Assignment ${newStatus === 'active' ? 'activated' : 'deactivated'} successfully`);
                    } else {
                        showToast(response.message || "Error updating assignment status", "error");
                        
                        // Reset button state
                        if (newStatus === 'active') {
                            $("#toggleAssignmentStatus").prop("disabled", false).html('<i class="fas fa-check me-1"></i> Activate');
                        } else {
                            $("#toggleAssignmentStatus").prop("disabled", false).html('<i class="fas fa-ban me-1"></i> Deactivate');
                        }
                    }
                },
                error: function() {
                    showToast("Server error while updating assignment status", "error");
                    
                    // Reset button state
                    if (newStatus === 'active') {
                        $("#toggleAssignmentStatus").prop("disabled", false).html('<i class="fas fa-check me-1"></i> Activate');
                    } else {
                        $("#toggleAssignmentStatus").prop("disabled", false).html('<i class="fas fa-ban me-1"></i> Deactivate');
                    }
                }
            });
        }
    </script>
</body>
</html>
