<?php
require_once '../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Process category actions
$message = '';

// Update category
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_category'])) {
    $category_id = $_POST['category_id'];
    $category_name = trim($_POST['category_name']);
    $description = trim($_POST['description']);
    
    if (empty($category_name)) {
        $message = '<div class="alert alert-danger">Category name is required.</div>';
    } else {
        $stmt = $conn->prepare("UPDATE categories SET category_name = ?, description = ? WHERE category_id = ?");
        $stmt->bind_param("ssi", $category_name, $description, $category_id);
        
        if ($stmt->execute()) {
            $message = '<div class="alert alert-success">Category updated successfully.</div>';
        } else {
            $message = '<div class="alert alert-danger">Error updating category: ' . $conn->error . '</div>';
        }
    }
}

// Delete category
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    $category_id = $_GET['delete'];
    
    // Check if category has courses
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM courses WHERE category_id = ?");
    $stmt->bind_param("i", $category_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    
    if ($row['count'] > 0) {
        $message = '<div class="alert alert-danger">Cannot delete category because it has associated courses.</div>';
    } else {
        // Delete the category
        $stmt = $conn->prepare("DELETE FROM categories WHERE category_id = ?");
        $stmt->bind_param("i", $category_id);
        
        if ($stmt->execute()) {
            $message = '<div class="alert alert-success">Category deleted successfully.</div>';
        } else {
            $message = '<div class="alert alert-danger">Error deleting category: ' . $conn->error . '</div>';
        }
    }
}

// Add new category
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_category'])) {
    $category_name = trim($_POST['category_name']);
    $description = trim($_POST['description']);
    
    if (empty($category_name)) {
        $message = '<div class="alert alert-danger">Category name is required.</div>';
    } else {
        $stmt = $conn->prepare("INSERT INTO categories (category_name, description) VALUES (?, ?)");
        $stmt->bind_param("ss", $category_name, $description);
        
        if ($stmt->execute()) {
            $message = '<div class="alert alert-success">Category added successfully.</div>';
        } else {
            $message = '<div class="alert alert-danger">Error adding category: ' . $conn->error . '</div>';
        }
    }
}

// Fetch all categories
$result = $conn->query("SELECT category_id, category_name, description, created_at FROM categories ORDER BY category_name ASC");
$categories = [];
while ($row = $result->fetch_assoc()) {
    $categories[] = $row;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Categories - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        /* Modern styling for category management */
        .dashboard-content {
            padding: 1.5rem;
        }
        
        .category-management {
            background: #fff;
            border-radius: 1rem;
            box-shadow: 0 0.125rem 0.25rem rgba(0,0,0,0.075);
            padding: 1.5rem;
        }
        
        .category-card {
            border: none;
            border-radius: 0.75rem;
            box-shadow: 0 0.125rem 0.375rem rgba(0,0,0,0.04);
            transition: all 0.3s ease;
            margin-bottom: 1rem;
        }
        
        .category-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 0.25rem 0.75rem rgba(0,0,0,0.08);
        }
        
        .table-container {
            border-radius: 0.75rem;
            overflow: hidden;
            box-shadow: 0 0.125rem 0.375rem rgba(0,0,0,0.04);
            background: #fff;
        }
        
        .table-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1.25rem 1.5rem;
            background: linear-gradient(to right, #f8f9fa, #ffffff);
            border-bottom: 1px solid #f1f1f1;
        }
        
        .table-header h5 {
            margin: 0;
            font-weight: 600;
            color: #333;
        }
        
        .table > :not(caption) > * > * {
            padding: 1rem 1.5rem;
            vertical-align: middle;
        }
        
        .table tbody tr {
            transition: all 0.2s ease;
        }
        
        .table tbody tr:hover {
            background-color: rgba(78, 115, 223, 0.05);
        }
        
        .btn-action-group {
            display: flex;
            gap: 0.5rem;
        }
        
        .btn-action {
            width: 2.2rem;
            height: 2.2rem;
            padding: 0;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            border-radius: 0.5rem;
            transition: all 0.2s;
        }
        
        .btn-action:hover {
            transform: translateY(-2px);
        }
        
        .btn-add-category {
            padding: 0.5rem 1rem;
            border-radius: 0.5rem;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            background: linear-gradient(45deg, #4e73df, #6f86e0);
            border: none;
            box-shadow: 0 0.25rem 0.5rem rgba(78, 115, 223, 0.1);
            transition: all 0.3s ease;
        }
        
        .btn-add-category:hover {
            transform: translateY(-2px);
            box-shadow: 0 0.5rem 1rem rgba(78, 115, 223, 0.2);
        }
        
        .category-badge {
            padding: 0.375rem 0.75rem;
            border-radius: 2rem;
            font-size: 0.75rem;
            font-weight: 600;
            color: #fff;
            background: linear-gradient(45deg, #4e73df, #6f86e0);
            display: inline-flex;
            align-items: center;
            gap: 0.25rem;
        }
        
        .modal-content {
            border-radius: 1rem;
            border: none;
            box-shadow: 0 1rem 3rem rgba(0, 0, 0, 0.175);
        }
        
        .modal-header {
            border-bottom: 1px solid rgba(0,0,0,0.05);
            background: linear-gradient(to right, #f8f9fa, #ffffff);
            padding: 1.25rem 1.5rem;
        }
        
        .modal-footer {
            border-top: 1px solid rgba(0,0,0,0.05);
            padding: 1.25rem 1.5rem;
        }
        
        .modal-body {
            padding: 1.5rem;
        }
        
        .form-control, .form-select {
            padding: 0.75rem 1rem;
            border-radius: 0.5rem;
            border: 1px solid #e0e0e0;
        }
        
        .form-control:focus, .form-select:focus {
            box-shadow: 0 0 0 0.25rem rgba(78, 115, 223, 0.15);
            border-color: #4e73df;
        }
        
        .empty-state {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 3rem;
            text-align: center;
        }
        
        .empty-state i {
            font-size: 3rem;
            color: #6c757d;
            margin-bottom: 1rem;
        }
        
        .category-count {
            background-color: rgba(78, 115, 223, 0.1);
            color: #4e73df;
            padding: 0.25rem 0.75rem;
            border-radius: 1rem;
            font-size: 0.75rem;
            font-weight: 600;
        }
        
        .table-actions-column {
            width: 120px;
            text-align: right;
        }
        
        /* Animation for new items */
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .fade-in-item {
            animation: fadeIn 0.5s ease forwards;
        }
        
        /* Tooltip styles */
        .tooltip-inner {
            box-shadow: 0 0.25rem 0.5rem rgba(0,0,0,0.1);
            border-radius: 0.5rem;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid p-0">
                <div class="d-flex align-items-center justify-content-between mb-4">
                    <h1 class="h3 mb-0 text-gray-800">Category Management</h1>
                    <button type="button" class="btn btn-add-category" data-bs-toggle="modal" data-bs-target="#addCategoryModal">
                        <i class='bx bx-plus'></i> Add New Category
                    </button>
                </div>
            
            <?php if (!empty($message)): ?>
                <div class="alert-container animate-fade-in">
                    <?php echo $message; ?>
                </div>
            <?php endif; ?>
            
            <!-- Categories Table -->
            <div class="card shadow mb-4 animate-fade-in">
                <div class="table-header">
                    <div class="d-flex align-items-center gap-2">
                        <h5>Course Categories</h5>
                        <span class="category-count"><?php echo count($categories); ?></span>
                    </div>
                    <div class="d-flex align-items-center gap-2">
                        <div class="form-check form-switch d-none">
                            <input class="form-check-input" type="checkbox" id="gridView">
                            <label class="form-check-label" for="gridView">Grid View</label>
                        </div>
                    </div>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                            <table class="table table-hover align-middle">
                                <thead class="bg-light">
                                    <tr>
                                        <th class="ps-3" width="60">ID</th>
                                        <th>Name</th>
                                        <th>Description</th>
                                        <th>Created At</th>
                                        <th class="text-end pe-3">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (count($categories) > 0): ?>
                                        <?php foreach ($categories as $category): ?>
                                            <tr class="fade-in-item">
                                                <td class="ps-3 fw-medium"><?php echo $category['category_id']; ?></td>
                                                <td>
                                                    <div class="d-flex align-items-center">
                                                        <div class="category-icon bg-light rounded-circle p-2 me-3 d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;">
                                                            <i class='bx bx-category text-primary'></i>
                                                        </div>
                                                        <div>
                                                            <div class="fw-semibold"><?php echo htmlspecialchars($category['category_name']); ?></div>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td class="text-muted"><?php echo htmlspecialchars($category['description'] ?: 'No description provided'); ?></td>
                                                <td>
                                                    <span class="text-muted small">
                                                        <i class='bx bx-calendar me-1'></i>
                                                        <?php echo date('M d, Y', strtotime($category['created_at'])); ?>
                                                    </span>
                                                </td>
                                                <td class="text-end pe-3">
                                                    <div class="d-flex justify-content-end gap-2">
                                                        <button type="button" class="btn-action btn btn-outline-primary" 
                                                                data-bs-toggle="modal" 
                                                                data-bs-target="#editCategoryModal" 
                                                                data-id="<?php echo $category['category_id']; ?>"
                                                                data-name="<?php echo htmlspecialchars($category['category_name']); ?>"
                                                                data-description="<?php echo htmlspecialchars($category['description'] ?? ''); ?>"
                                                                data-bs-toggle="tooltip"
                                                                data-bs-placement="top"
                                                                title="Edit Category">
                                                            <i class='bx bx-edit'></i>
                                                        </button>
                                                        <a href="categories.php?delete=<?php echo $category['category_id']; ?>" class="btn-action btn btn-outline-danger" 
                                                           onclick="return confirm('Are you sure you want to delete this category? This action cannot be undone.');"
                                                           data-bs-toggle="tooltip"
                                                           data-bs-placement="top"
                                                           title="Delete Category">
                                                            <i class='bx bx-trash'></i>
                                                        </a>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="5">
                                                <div class="empty-state py-5">
                                                    <i class='bx bx-category bx-tada-hover'></i>
                                                    <h6 class="mt-3">No Categories Found</h6>
                                                    <p class="text-muted small">Add your first course category to organize your content</p>
                                                    <button type="button" class="btn btn-add-category btn-sm mt-2" data-bs-toggle="modal" data-bs-target="#addCategoryModal">
                                                        <i class='bx bx-plus'></i> Add New Category
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Add Category Modal -->
    <div class="modal fade" id="addCategoryModal" tabindex="-1" aria-labelledby="addCategoryModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="addCategoryModalLabel">
                        <i class='bx bx-plus-circle me-2 text-primary'></i>Add New Category
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form action="categories.php" method="post">
                    <div class="modal-body p-4">
                        <p class="text-muted mb-4 small">Create a new course category to organize your content. Categories help students find relevant courses.</p>
                        
                        <div class="mb-4">
                            <label for="category_name" class="form-label fw-medium">Category Name <span class="text-danger">*</span></label>
                            <div class="input-group">
                                <span class="input-group-text bg-light">
                                    <i class='bx bx-category text-primary'></i>
                                </span>
                                <input type="text" class="form-control" id="category_name" name="category_name" placeholder="e.g. Programming, Design, Business" required>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="description" class="form-label fw-medium">Description</label>
                            <textarea class="form-control" id="description" name="description" rows="4" placeholder="Describe what kind of courses will be included in this category..."></textarea>
                            <div class="form-text">A good description helps students understand what to expect from courses in this category.</div>
                        </div>
                    </div>
                    <div class="modal-footer justify-content-between">
                        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">
                            <i class='bx bx-x me-1'></i>Cancel
                        </button>
                        <button type="submit" name="add_category" class="btn btn-primary px-4">
                            <i class='bx bx-check me-1'></i>Create Category
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Edit Category Modal -->
    <div class="modal fade" id="editCategoryModal" tabindex="-1" aria-labelledby="editCategoryModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="editCategoryModalLabel">
                        <i class='bx bx-edit-alt me-2 text-primary'></i>Edit Category
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form action="categories.php" method="post">
                    <div class="modal-body p-4">
                        <p class="text-muted mb-4 small">Update the details of this course category to better organize your content.</p>
                        
                        <input type="hidden" id="edit_category_id" name="category_id">
                        
                        <div class="mb-4">
                            <label for="edit_category_name" class="form-label fw-medium">Category Name <span class="text-danger">*</span></label>
                            <div class="input-group">
                                <span class="input-group-text bg-light">
                                    <i class='bx bx-category text-primary'></i>
                                </span>
                                <input type="text" class="form-control" id="edit_category_name" name="category_name" required>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="edit_description" class="form-label fw-medium">Description</label>
                            <textarea class="form-control" id="edit_description" name="description" rows="4"></textarea>
                            <div class="form-text">A good description helps students understand what to expect from courses in this category.</div>
                        </div>
                    </div>
                    <div class="modal-footer justify-content-between">
                        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">
                            <i class='bx bx-x me-1'></i>Cancel
                        </button>
                        <button type="submit" name="update_category" class="btn btn-primary px-4">
                            <i class='bx bx-save me-1'></i>Save Changes
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Initialize tooltips
            var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'))
            var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl, {
                    boundary: document.body
                });
            });
            
            // Fade out alerts after 5 seconds
            setTimeout(function() {
                const alerts = document.querySelectorAll('.alert-container .alert');
                alerts.forEach(function(alert) {
                    alert.classList.add('fade');
                    setTimeout(function() {
                        alert.style.display = 'none';
                    }, 500);
                });
            }, 5000);
            
            // Populate edit category modal
            var editCategoryModal = document.getElementById('editCategoryModal');
            if (editCategoryModal) {
                editCategoryModal.addEventListener('show.bs.modal', function (event) {
                    var button = event.relatedTarget;
                    var id = button.getAttribute('data-id');
                    var name = button.getAttribute('data-name');
                    var description = button.getAttribute('data-description');
                    
                    var idInput = editCategoryModal.querySelector('#edit_category_id');
                    var nameInput = editCategoryModal.querySelector('#edit_category_name');
                    var descriptionInput = editCategoryModal.querySelector('#edit_description');
                    
                    idInput.value = id;
                    nameInput.value = name;
                    descriptionInput.value = description || '';
                });
            }
            
            // Focus on category name field when add modal is shown
            var addCategoryModal = document.getElementById('addCategoryModal');
            if (addCategoryModal) {
                addCategoryModal.addEventListener('shown.bs.modal', function () {
                    document.getElementById('category_name').focus();
                });
            }
        });
    </script>
    
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Sidebar toggle functionality
        const sidebarToggle = document.querySelector('.sidebar-toggle');
        const sidebarToggleMobile = document.querySelector('.sidebar-toggle-mobile');
        const sidebar = document.querySelector('.sidebar');
        
        if (sidebarToggle) {
            sidebarToggle.addEventListener('click', () => {
                sidebar.classList.toggle('close');
            });
        }
        
        if (sidebarToggleMobile) {
            sidebarToggleMobile.addEventListener('click', () => {
                sidebar.classList.toggle('show');
            });
        }
        
        // Responsive sidebar behavior
        function checkScreenSize() {
            if (window.innerWidth < 768) {
                sidebar.classList.remove('close');
                sidebar.classList.remove('show');
            } else if (window.innerWidth < 992) {
                sidebar.classList.add('close');
                sidebar.classList.remove('show');
            }
        }
        
        // Initial check and listen for window resize
        checkScreenSize();
        window.addEventListener('resize', checkScreenSize);
        
        // Dropdown functionality
        document.querySelectorAll('.dropdown-toggle').forEach(item => {
            item.addEventListener('click', event => {
                event.preventDefault();
                const dropdownMenu = item.nextElementSibling;
                if (dropdownMenu && dropdownMenu.classList.contains('dropdown-menu')) {
                    dropdownMenu.classList.toggle('show');
                }
            });
        });
        
        // Close dropdowns when clicking outside
        document.addEventListener('click', event => {
            const isDropdownButton = event.target.classList.contains('dropdown-toggle');
            if (!isDropdownButton) {
                document.querySelectorAll('.dropdown-menu.show').forEach(dropdown => {
                    dropdown.classList.remove('show');
                });
            }
        });
    });
    </script>
</body>
</html>
