<?php
/**
 * Panadite Academy Dynamic Certificate System
 * Beautiful, branded certificates generated programmatically
 */

require_once __DIR__ . '/../auth/functions.php';
requireRole('admin');

// Database connection for online server
require_once __DIR__ . '/../config/db_connect.php';
require_once 'auto_certificate_generator.php';

// Initialize the automated certificate generator
$certGenerator = new AutoCertificateGenerator($conn);

// Get certificate statistics and recent certificates
$stats = $certGenerator->getStatistics();
$certificates = $certGenerator->getAllCertificates(20);

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Certificate Generator - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link rel="stylesheet" href="/assets/css/user-cards.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        :root {
            --panadite-primary: #4169E1;      /* Royal Blue from logo */
            --panadite-secondary: #20B2AA;    /* Teal from branding */
            --panadite-accent: #FFD700;       /* Gold accent */
            --panadite-dark: #1e3a8a;         /* Dark Blue */
            --panadite-light: #f0f8ff;        /* Alice Blue */
            --panadite-gradient: linear-gradient(135deg, var(--panadite-primary), var(--panadite-secondary));
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .main-header {
            background: var(--panadite-gradient);
            color: white;
            padding: 2rem 0;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
        }
        
        .card {
            border-radius: 10px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            border: none;
            overflow: hidden;
        }
        
        .card-header-custom {
            background: var(--panadite-gradient);
            color: white;
            padding: 1.5rem;
            border: none;
        }
        
        .btn-panadite {
            background: var(--panadite-gradient);
            color: white;
            border: none;
            border-radius: 10px;
            padding: 12px 24px;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        
        .btn-panadite:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(65, 105, 225, 0.3);
            color: white;
        }
        
        .form-control, .form-select {
            border-radius: 10px;
            border: 2px solid #e9ecef;
            padding: 12px;
            transition: all 0.3s ease;
        }
        
        .form-control:focus, .form-select:focus {
            border-color: var(--panadite-primary);
            box-shadow: 0 0 0 0.2rem rgba(65, 105, 225, 0.25);
        }
        
        /* Modern Geometric Certificate Preview Styles */
        .certificate-container {
            perspective: 1000px;
            margin: 2rem 0;
        }
        
        .certificate-preview {
            width: 320px;
            height: 240px;
            background: #ffffff;
            margin: 15px auto;
            position: relative;
            overflow: hidden;
            font-family: 'Poppins', sans-serif;
            box-shadow: 0 8px 25px rgba(0,0,0,0.15);
            transform: scale(0.85);
            transition: transform 0.3s ease;
        }
        
        .certificate-preview:hover {
            transform: scale(0.9);
        }
        
        /* Clean geometric corner elements for preview */
        .preview-geo-tl {
            position: absolute;
            top: 0;
            left: 0;
            width: 45px;
            height: 45px;
            background: linear-gradient(135deg, #4169E1 0%, #5B7CFA 100%);
            clip-path: polygon(0 0, 100% 0, 0 100%);
            opacity: 0.9;
            z-index: 1;
        }
        
        .preview-geo-tr {
            position: absolute;
            top: 0;
            right: 0;
            width: 36px;
            height: 36px;
            background: linear-gradient(225deg, #FFD700 0%, #F4C430 100%);
            clip-path: polygon(100% 0, 100% 100%, 0 0);
            opacity: 0.9;
            z-index: 1;
        }
        
        .preview-geo-bl {
            position: absolute;
            bottom: 0;
            left: 0;
            width: 30px;
            height: 30px;
            background: linear-gradient(45deg, #20B2AA 0%, #17a2b8 100%);
            clip-path: polygon(0 0, 0 100%, 100% 100%);
            opacity: 0.85;
            z-index: 1;
        }
        
        .preview-geo-br {
            position: absolute;
            bottom: 0;
            right: 0;
            width: 24px;
            height: 24px;
            background: linear-gradient(315deg, #4169E1 0%, #20B2AA 100%);
            clip-path: polygon(100% 0, 100% 100%, 0 100%);
            opacity: 0.8;
            z-index: 1;
        }
        
        /* Preview content */
        .preview-content {
            position: relative;
            z-index: 2;
            padding: 15px;
            height: 100%;
            display: flex;
            flex-direction: column;
            justify-content: space-between;
            text-align: center;
        }
        
        .preview-header {
            margin-bottom: 8px;
        }
        
        .preview-border {
            position: absolute;
            top: 6px;
            left: 6px;
            right: 6px;
            bottom: 6px;
            border: 0.5px solid #F3F4F6;
            border-radius: 2px;
            z-index: 2;
        }
        
        .preview-watermark {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%) rotate(-45deg);
            font-size: 1.8rem;
            color: rgba(65, 105, 225, 0.02);
            font-weight: 300;
            z-index: 0;
            font-family: 'Poppins', sans-serif;
            letter-spacing: 2px;
        }
        
        .preview-logo {
            width: 24px;
            height: 24px;
            margin: 0 auto 4px;
            background: url('/assets/images/logo.jpeg') center/contain no-repeat;
            border-radius: 50%;
            border: 1px solid #E5E7EB;
            filter: drop-shadow(0 1px 3px rgba(0,0,0,0.08));
        }
        
        .preview-academy {
            font-size: 0.42rem;
            font-weight: 600;
            color: #1F2937;
            letter-spacing: 0.8px;
            margin: 0;
            text-transform: uppercase;
        }
        
        .preview-tagline {
            font-size: 0.28rem;
            color: #6B7280;
            font-weight: 400;
            margin: 2px 0 0 0;
            letter-spacing: 0.2px;
        }
        
        .preview-title-section {
            margin: 8px 0;
        }
        
        .preview-title {
            font-size: 0.85rem;
            font-weight: 700;
            color: #111827;
            letter-spacing: 1px;
            margin: 0;
        }
        
        .preview-subtitle {
            font-size: 0.38rem;
            color: #6B7280;
            margin: 2px 0 0 0;
            font-weight: 400;
            letter-spacing: 0.3px;
        }
        
        .preview-body {
            flex-grow: 1;
            display: flex;
            flex-direction: column;
            justify-content: center;
            margin: 8px 0;
            padding: 0 12px;
        }
        
        .preview-text {
            font-size: 0.32rem;
            color: #4B5563;
            margin: 2px 0;
            line-height: 1.3;
            font-family: 'Poppins', sans-serif;
        }
        
        .preview-student {
            font-size: 0.6rem;
            font-weight: 600;
            color: #4169E1;
            margin: 4px 0;
            position: relative;
            display: inline-block;
        }
        
        .preview-student::after {
            content: '';
            position: absolute;
            bottom: -2px;
            left: 50%;
            transform: translateX(-50%);
            width: 18px;
            height: 0.8px;
            background: #FFD700;
        }
        
        .preview-course {
            font-size: 0.45rem;
            color: #20B2AA;
            font-weight: 500;
            margin: 4px 0;
            font-style: italic;
        }
        
        .preview-footer {
            margin-top: 8px;
            padding-top: 4px;
            border-top: 0.5px solid #E5E7EB;
        }
        
        .preview-signatures {
            display: flex;
            justify-content: space-between;
            align-items: end;
        }
        
        .preview-sig {
            font-size: 0.26rem;
            color: #6B7280;
            font-weight: 400;
            line-height: 1.2;
            text-align: center;
            flex: 1;
        }
        
        .preview-sig::before {
            content: '';
            display: block;
            width: 30px;
            height: 0.4px;
            background: #D1D5DB;
            margin: 0 auto 2px;
        }
        
        .preview-info {
            text-align: center;
            flex: 1;
        }
        
        .preview-number {
            font-size: 0.3rem;
            color: #374151;
            font-weight: 500;
            margin-bottom: 1px;
            font-family: 'Poppins', sans-serif;
        }
        
        .preview-date {
            font-size: 0.26rem;
            color: #9CA3AF;
            font-family: 'Poppins', sans-serif;
        }
        
        /* Clean Achievement Badge for preview */
        .preview-badge {
            position: absolute;
            right: 18px;
            bottom: 30px;
            width: 21px;
            height: 21px;
            background: linear-gradient(135deg, #FFD700 0%, #F4C430 100%);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 3;
            box-shadow: 0 1px 4px rgba(255, 215, 0, 0.25);
            border: 1px solid #fff;
        }
        
        .badge-content {
            text-align: center;
            color: white;
        }
        
        .badge-year {
            font-size: 0.32rem;
            font-weight: 600;
            margin-bottom: 0;
        }
        
        .badge-text {
            font-size: 0.18rem;
            font-weight: 500;
            text-transform: uppercase;
            letter-spacing: 0.2px;
        }
        
        /* Certificate preview hover effect only */
        .certificate-preview:hover {
            transform: scale(0.95);
        }
            font-size: 1.8rem;
            font-weight: bold;
            font-style: italic;
            margin: 15px 0;
        }
        
        .certificate-footer {
            display: flex;
            justify-content: space-between;
            align-items: end;
            margin-top: 40px;
            padding-top: 20px;
            border-top: 2px solid var(--panadite-light);
        }
        
        .signature-line {
            text-align: center;
            width: 200px;
        }
        
        .signature-line hr {
            border: none;
            border-top: 2px solid var(--panadite-primary);
            margin: 10px 0 5px 0;
        }
        
        .watermark {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%) rotate(-45deg);
            font-size: 8rem;
            color: var(--panadite-light);
            font-weight: bold;
            z-index: 0;
            opacity: 0.05;
        }
        
        .certificate-content {
            position: relative;
            z-index: 1;
        }
        
        .stats-card {
            background: var(--panadite-gradient);
            color: white;
            border-radius: 15px;
            padding: 1.5rem;
            text-align: center;
        }
        
        .stats-number {
            font-size: 2.5rem;
            font-weight: bold;
            margin-bottom: 0.5rem;
        }
        
        .loading-spinner {
            display: none;
            text-align: center;
            color: var(--panadite-primary);
        }
        
        .alert-custom {
            border-radius: 15px;
            border: none;
            padding: 1rem 1.5rem;
        }
        
        .table-custom {
            border-radius: 15px;
            overflow: hidden;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .table-custom thead {
            background: var(--panadite-gradient);
            color: white;
        }
        
        .btn-sm-custom {
            padding: 5px 12px;
            font-size: 0.875rem;
            border-radius: 8px;
        }
        
        @media print {
            .certificate-preview {
                transform: none;
                box-shadow: none;
                page-break-inside: avoid;
            }
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <!-- Page Header -->
            <div class="page-header">
                <h1><i class='bx bx-award'></i> Automated Certificate System</h1>
                <p class="text-muted">View and manage automatically generated certificates - No manual generation needed!</p>
            </div>

            <!-- Statistics Cards -->
            <div class="row mb-4">
                <div class="col-md-3 mb-4">
                    <div class="card h-100 text-center">
                        <div class="card-body">
                            <h2 class="display-4 fw-bold text-primary mb-2"><?php echo $stats['total_certificates']; ?></h2>
                            <p class="card-text text-muted mb-0">Total Certificates</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-3 mb-4">
                    <div class="card h-100 text-center">
                        <div class="card-body">
                            <h2 class="display-4 fw-bold text-success mb-2"><?php echo $stats['emails_sent']; ?></h2>
                            <p class="card-text text-muted mb-0">Emails Sent</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-3 mb-4">
                    <div class="card h-100 text-center">
                        <div class="card-body">
                            <h2 class="display-4 fw-bold text-warning mb-2"><?php echo $stats['emails_pending']; ?></h2>
                            <p class="card-text text-muted mb-0">Emails Pending</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-3 mb-4">
                    <div class="card h-100 text-center">
                        <div class="card-body">
                            <h2 class="display-4 fw-bold text-info mb-2"><?php echo $stats['this_month']; ?></h2>
                            <p class="card-text text-muted mb-0">This Month</p>
                        </div>
                    </div>
                </div>
            </div>
            <!-- Automation Information -->
            <div class="row">
                <div class="col-lg-8">
                    <div class="certificate-card mb-4">
                        <div class="card-header">
                            <h5 class="mb-0"><i class="bx bx-cog me-2"></i>🤖 Fully Automated Certificate System</h5>
                        </div>
                        <div class="card-body p-4">
                            <div class="alert alert-success border-0" style="background: linear-gradient(135deg, #d4edda, #c3e6cb);">
                                <div class="d-flex align-items-center mb-3">
                                    <i class="bx bx-check-circle text-success me-2" style="font-size: 2rem;"></i>
                                    <h6 class="mb-0 fw-bold">System is Fully Automated!</h6>
                                </div>
                                <p class="mb-0">Certificates are automatically generated and emailed to students when they complete their courses. No manual intervention required!</p>
                            </div>
                            
                            <div class="certificate-preview">
                                <!-- Clean geometric corner elements -->
                                <div class="preview-geo-tl"></div>
                                <div class="preview-geo-tr"></div>
                                <div class="preview-geo-bl"></div>
                                <div class="preview-geo-br"></div>
                                
                                <!-- Inner border -->
                                <div class="preview-border"></div>
                                
                                <!-- Subtle watermark -->
                                <div class="preview-watermark">PANADITE</div>
                                
                                <!-- Content -->
                                <div class="preview-content">
                                    <div class="preview-header">
                                        <div class="preview-logo"></div>
                                        <div class="preview-academy">PANADITE ACADEMY</div>
                                        <div class="preview-tagline">Get ahead of the curve</div>
                                    </div>
                                    <div class="preview-title-section">
                                        <div class="preview-title">CERTIFICATE</div>
                                        <div class="preview-subtitle">of Completion</div>
                                    </div>
                                    <div class="preview-body">
                                        <div class="preview-text">This certificate is presented to</div>
                                        <div class="preview-student">Student Name</div>
                                        <div class="preview-text">in recognition of successful completion of</div>
                                        <div class="preview-course">Course Title</div>
                                        <div class="preview-text">demonstrating dedication and achievement</div>
                                    </div>
                                    <div class="preview-footer">
                                        <div class="preview-signatures">
                                            <div class="preview-sig">Director<br>Panadite Academy</div>
                                            <div class="preview-info">
                                                <div class="preview-number">PANADITE-2024-XXXX</div>
                                                <div class="preview-date">Issued on Date</div>
                                            </div>
                                            <div class="preview-sig">Academic Registrar<br>Panadite Academy</div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Clean Achievement Badge -->
                                <div class="preview-badge">
                                    <div class="badge-content">
                                        <div class="badge-year">2024</div>
                                        <div class="badge-text">Certified</div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row text-center mt-4">
                                <div class="col-md-4 mb-4">
                                    <div class="automation-step">
                                        <div class="step-icon-wrapper">
                                            <div class="step-icon bg-primary text-white rounded-circle mx-auto mb-3">
                                                <i class='bx bx-user-check'></i>
                                            </div>
                                            <div class="step-connector"></div>
                                        </div>
                                        <h5 class="fw-bold text-primary mb-2">1. Student Completes Course</h5>
                                        <p class="text-muted small mb-0">When a student finishes all course requirements</p>
                                    </div>
                                </div>
                                
                                <div class="col-md-4 mb-4">
                                    <div class="automation-step">
                                        <div class="step-icon-wrapper">
                                            <div class="step-icon bg-success text-white rounded-circle mx-auto mb-3">
                                                <i class='bx bx-certification'></i>
                                            </div>
                                            <div class="step-connector"></div>
                                        </div>
                                        <h5 class="fw-bold text-success mb-2">2. Certificate Auto-Generated</h5>
                                        <p class="text-muted small mb-0">System creates beautiful certificate instantly</p>
                                    </div>
                                </div>
                                
                                <div class="col-md-4 mb-4">
                                    <div class="automation-step">
                                        <div class="step-icon-wrapper">
                                            <div class="step-icon bg-info text-white rounded-circle mx-auto mb-3">
                                                <i class='bx bx-mail-send'></i>
                                            </div>
                                        </div>
                                        <h5 class="fw-bold text-info mb-2">3. Email Sent to Student</h5>
                                        <p class="text-muted small mb-0">Certificate automatically emailed to student</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Certificate Receiving Process Management -->
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0"><i class="bx bx-cog me-2"></i>Certificate Receiving Process</h5>
                        </div>
                        <div class="card-body">
                            <!-- Process Status Indicators -->
                            <div class="row mb-4">
                                <div class="col-md-6 mb-3">
                                    <div class="d-flex align-items-center p-3 bg-light rounded">
                                        <div class="flex-shrink-0">
                                            <i class="bx bx-check-circle text-success fs-4"></i>
                                        </div>
                                        <div class="flex-grow-1 ms-3">
                                            <h6 class="mb-0">Auto-Generation</h6>
                                            <small class="text-muted">Active & Working</small>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="d-flex align-items-center p-3 bg-light rounded">
                                        <div class="flex-shrink-0">
                                            <i class="bx bx-envelope text-primary fs-4"></i>
                                        </div>
                                        <div class="flex-grow-1 ms-3">
                                            <h6 class="mb-0">Email Delivery</h6>
                                            <small class="text-muted">PHP Mail Active</small>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Process Management Tabs -->
                            <ul class="nav nav-pills mb-3" id="process-tabs" role="tablist">
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link active" id="triggers-tab" data-bs-toggle="pill" data-bs-target="#triggers" type="button" role="tab">
                                        <i class="bx bx-play-circle me-1"></i>Triggers
                                    </button>
                                </li>
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link" id="delivery-tab" data-bs-toggle="pill" data-bs-target="#delivery" type="button" role="tab">
                                        <i class="bx bx-send me-1"></i>Delivery
                                    </button>
                                </li>
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link" id="student-portal-tab" data-bs-toggle="pill" data-bs-target="#student-portal" type="button" role="tab">
                                        <i class="bx bx-user me-1"></i>Student Portal
                                    </button>
                                </li>
                            </ul>

                            <div class="tab-content" id="process-content">
                                <!-- Course Completion Triggers Tab -->
                                <div class="tab-pane fade show active" id="triggers" role="tabpanel">
                                    <h6 class="mb-3">Course Completion Detection</h6>
                                    <div class="mb-3">
                                        <label class="form-label">Completion Threshold (%)</label>
                                        <input type="number" class="form-control" value="100" min="1" max="100">
                                        <small class="text-muted">Minimum completion percentage to trigger certificate</small>
                                    </div>
                                    <div class="mb-3">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" checked>
                                            <label class="form-check-label">Require all lessons completed</label>
                                        </div>
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" checked>
                                            <label class="form-check-label">Require quiz/assessment pass</label>
                                        </div>
                                    </div>
                                    <button class="btn btn-outline-primary btn-sm">
                                        <i class="bx bx-test-tube me-1"></i>Test Trigger System
                                    </button>
                                </div>

                                <!-- Email Delivery Tab -->
                                <div class="tab-pane fade" id="delivery" role="tabpanel">
                                    <h6 class="mb-3">Email Configuration</h6>
                                    <form id="emailConfigForm">
                                        <div class="mb-3">
                                            <label class="form-label">From Email</label>
                                            <input type="email" id="fromEmail" class="form-control" value="admin@panaditeacademy.com">
                                        </div>
                                        <div class="mb-3">
                                            <label class="form-label">Subject Template</label>
                                            <input type="text" id="subjectTemplate" class="form-control" value="🎓 Your Certificate from Panadite Academy - {course_title}">
                                        </div>
                                        <div class="mb-3">
                                            <label class="form-label">Email Template</label>
                                            <select id="emailTemplate" class="form-select">
                                                <option value="professional" selected>Professional Template</option>
                                                <option value="simple">Simple Template</option>
                                                <option value="custom">Custom HTML</option>
                                            </select>
                                        </div>
                                        
                                    </form>
                                    
                                    <div class="d-flex gap-2 flex-wrap">
                                        <button type="button" class="btn btn-outline-info btn-sm">
                                            <i class="bx bx-edit me-1"></i>Edit Template
                                        </button>
                                        <button type="button" id="saveEmailConfigBtn" class="btn btn-outline-primary btn-sm">
                                            <i class="bx bx-save me-1"></i>Save Configuration
                                        </button>
                                    </div>
                                </div>

                                <!-- Student Portal Tab -->
                                <div class="tab-pane fade" id="student-portal" role="tabpanel">
                                    <h6 class="mb-3">Student Access & Notifications</h6>
                                    <div class="mb-3">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" checked>
                                            <label class="form-check-label">Show certificates in student dashboard</label>
                                        </div>
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" checked>
                                            <label class="form-check-label">Allow PDF download from portal</label>
                                        </div>
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" checked>
                                            <label class="form-check-label">Send browser notifications</label>
                                        </div>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">Certificate Verification URL</label>
                                        <div class="input-group">
                                            <input type="text" class="form-control" value="https://panaditeacademy.com/verify/" readonly>
                                            <button class="btn btn-outline-secondary" type="button">
                                                <i class="bx bx-copy"></i>
                                            </button>
                                        </div>
                                    </div>
                                    <button class="btn btn-outline-primary btn-sm me-2">
                                        <i class="bx bx-show me-1"></i>Preview Student View
                                    </button>
                                    <button class="btn btn-outline-warning btn-sm">
                                        <i class="bx bx-shield-check me-1"></i>Test Verification
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

            </div>
        
        <!-- Certificate Preview -->
        <div id="certificatePreview" class="certificate-preview" style="display: none;">
            <div class="certificate-border"></div>
            <div class="certificate-inner-border"></div>
            <div class="watermark">PANADITE</div>
            
            <div class="certificate-content text-center">
                <div class="certificate-logo">PA</div>
                
                <h1 class="certificate-title">CERTIFICATE OF COMPLETION</h1>
                
                <div class="certificate-text">
                    <p>This is to certify that</p>
                    <h2 id="previewStudentName" style="color: var(--primary-blue); font-size: 2.5rem; margin: 1rem 0;">Student Name</h2>
                    <p>has successfully completed the course</p>
                    <h3 id="previewCourseTitle" style="color: var(--secondary-teal); font-size: 1.8rem; margin: 1rem 0;">Course Title</h3>
                    <p>at Panadite Academy</p>
                </div>
                
                <div class="certificate-details">
                    <div>
                        <strong>Certificate No:</strong><br>
                        <span id="previewCertNumber">PANADITE-2024-XXXX</span>
                    </div>
                    <div>
                        <strong>Date Issued:</strong><br>
                        <span id="previewIssueDate"><?= date('F j, Y') ?></span>
                    </div>
                    <div>
                        <strong>Director:</strong><br>
                        <div class="signature-line">
                            <hr>
                            <small>Panadite Academy</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="text-center mt-3" id="certificateActions" style="display: none;">
            <button type="button" class="btn btn-outline-primary" onclick="window.print()">
                <i class="bx bx-printer me-2"></i>Print Certificate
            </button>
            <button type="button" class="btn btn-outline-success" id="downloadPDF">
                <i class="bx bx-download me-2"></i>Download PDF
            </button>
        </div>

        <!-- Recent Certificates -->
        <?php if ($recentResult && $recentResult->num_rows > 0): ?>
        <div class="certificate-card mt-4">
            <div class="card-header">
                <h5 class="mb-0"><i class="bx bx-history me-2"></i>Recent Certificates</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table mb-0">
                        <thead>
                            <tr>
                                <th>Certificate #</th>
                                <th>Student</th>
                                <th>Course</th>
                                <th>Issue Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while ($cert = $recentResult->fetch_assoc()): ?>
                            <tr>
                                <td><strong><?= htmlspecialchars($cert['certificate_number']) ?></strong></td>
                                <td><?= htmlspecialchars($cert['student_name']) ?></td>
                                <td><?= htmlspecialchars($cert['course_title']) ?></td>
                                <td><?= date('M j, Y', strtotime($cert['issue_date'])) ?></td>
                                <td>
                                    <button class="btn btn-outline-primary btn-sm me-1" onclick="viewCertificate(<?= $cert['certificate_id'] ?>)">
                                        <i class="bx bx-show"></i> View
                                    </button>
                                </td>
                            </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        <?php else: ?>
            <div class="text-center py-4">
                <i class="bx bx-inbox" style="font-size: 3rem; color: #ddd;"></i>
                <p class="text-muted mt-2">No certificates generated yet. Generate your first certificate above!</p>
            </div>
        <?php endif; ?>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <script>
        $(document).ready(function() {
            // Course selection handler
            $('#courseSelect').on('change', function() {
                const courseId = $(this).val();
                const studentSelect = $('#studentSelect');
                const generateBtn = $('#generateCertificate');
                
                if (courseId) {
                    studentSelect.prop('disabled', true).html('<option>Loading students...</option>');
                    generateBtn.prop('disabled', true);
                    
                    $.ajax({
                        url: 'certificate_templates.php',
                        type: 'POST',
                        data: {
                            action: 'get_students',
                            course_id: courseId
                        },
                        dataType: 'json',
                        success: function(response) {
                            if (response.success && response.students.length > 0) {
                                let options = '<option value="">Select a student...</option>';
                                response.students.forEach(function(student) {
                                    options += `<option value="${student.user_id}">
                                              ${student.first_name} ${student.last_name}
                                            </option>`;
                                });
                                studentSelect.html(options).prop('disabled', false);
                                showAlert(`Found ${response.students.length} completed student(s)`, 'success');
                            } else {
                                studentSelect.html('<option value="">No completed students found</option>');
                                showAlert('No students have completed this course yet', 'info');
                            }
                        },
                        error: function(xhr, status, error) {
                            console.error('AJAX Error:', error);
                            studentSelect.html('<option value="">Error loading students</option>');
                            showAlert('Error loading students. Please try again.', 'danger');
                        }
                    });
                } else {
                    studentSelect.prop('disabled', true).html('<option value="">First select a course...</option>');
                    generateBtn.prop('disabled', true);
                }
            });
            
            // Student selection handler
            $('#studentSelect').on('change', function() {
                const studentId = $(this).val();
                const courseId = $('#courseSelect').val();
                const generateBtn = $('#generateCertificate');
                
                if (studentId && courseId) {
                    generateBtn.prop('disabled', false);
                } else {
                    generateBtn.prop('disabled', true);
                }
            });
            
            // Certificate generation handler
            $('#generateCertificate').on('click', function() {
                const courseId = $('#courseSelect').val();
                const studentId = $('#studentSelect').val();
                
                if (!courseId || !studentId) {
                    showAlert('Please select both course and student', 'warning');
                    return;
                }
                
                const loadingSpinner = $('#loadingSpinner');
                const generateBtn = $(this);
                
                loadingSpinner.show();
                generateBtn.prop('disabled', true);
                
                $.ajax({
                    url: 'certificate_templates.php',
                    type: 'POST',
                    data: {
                        action: 'generate_certificate',
                        course_id: courseId,
                        student_id: studentId
                    },
                    dataType: 'json',
                    success: function(response) {
                        loadingSpinner.hide();
                        generateBtn.prop('disabled', false);
                        
                        if (response.success) {
                            showCertificatePreview(response.certificate);
                            showAlert('Certificate generated successfully!', 'success');
                            
                            // Refresh the page to show the new certificate in recent list
                            setTimeout(function() {
                                location.reload();
                            }, 2000);
                        } else {
                            showAlert(response.message || 'Failed to generate certificate', 'danger');
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('AJAX Error:', error);
                        loadingSpinner.hide();
                        generateBtn.prop('disabled', false);
                        showAlert('Error generating certificate. Please try again.', 'danger');
                    }
                });
            });
        });
        
        function showCertificatePreview(data) {
            $('#previewStudentName').text(data.student_data.first_name + ' ' + data.student_data.last_name);
            $('#previewCourseTitle').text(data.student_data.course_title);
            $('#previewCertNumber').text(data.certificate_number);
            $('#previewIssueDate').text(data.issue_date);
            
            $('#certificatePreview').fadeIn();
            
            // Scroll to certificate
            $('html, body').animate({
                scrollTop: $('#certificatePreview').offset().top - 50
            }, 800);
        }
        
        function showAlert(message, type) {
            const alertHtml = `
                <div class="alert alert-${type} alert-custom alert-dismissible fade show mt-3" role="alert">
                    <i class="bx bx-${type === 'success' ? 'check-circle' : type === 'warning' ? 'error-circle' : 'info-circle'} me-2"></i>
                    ${message}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            `;
            
            // Remove existing alerts
            $('.alert').remove();
            
            // Add new alert after the form
            $('#certificateForm').after(alertHtml);
            
            // Auto-remove after 5 seconds
            setTimeout(() => {
                $('.alert').fadeOut();
            }, 5000);
        }
        
        function printCertificate() {
            window.print();
        }
        
        function downloadCertificate() {
            // This would integrate with a PDF generation library
            showAlert('PDF download functionality will be implemented with jsPDF integration', 'info');
        }
        
        function viewCertificate(certificateId) {
            showAlert('Certificate viewing functionality will be implemented', 'info');
        }
        
        function revokeCertificate(certificateId) {
            if (confirm('Are you sure you want to revoke this certificate?')) {
                showAlert('Certificate revocation functionality will be implemented', 'info');
            }
        }
    </script>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Make existing Send Test Email button functional
        document.addEventListener('DOMContentLoaded', function() {
            // Find existing Send Test Email button and add functionality
            const sendTestButtons = document.querySelectorAll('button, a');
            sendTestButtons.forEach(button => {
                if (button.textContent.includes('Send Test Email')) {
                    button.addEventListener('click', function(e) {
                        e.preventDefault();
                        sendTestEmail();
                    });
                }
            });
        });
        
        function sendTestEmail() {
            // Get recipient email via prompt
            const recipientEmail = prompt('Enter email address to send test certificate to:');
            
            if (!recipientEmail) {
                return; // User cancelled
            }
            
            if (!validateEmail(recipientEmail)) {
                alert('Please enter a valid email address');
                return;
            }
            
            // Get other details via prompts or use defaults
            const studentName = prompt('Enter test student name:', 'John Doe') || 'John Doe';
            const courseTitle = prompt('Enter test course title:', 'Sample Course') || 'Sample Course';
            
            // Show loading state
            showTestEmailStatus('Sending test email to ' + recipientEmail + '...', 'info');
            
            // Send test email via API
            fetch('api/send_test_certificate.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    recipient_email: recipientEmail,
                    student_name: studentName,
                    course_title: courseTitle,
                    from_email: 'admin@panaditeacademy.com',
                    subject_template: '🎓 Your Certificate from Panadite Academy - {course_title}',
                    test_mode: true
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showTestEmailStatus('✅ Test email sent successfully to ' + recipientEmail + '!', 'success');
                } else {
                    showTestEmailStatus('❌ Failed to send test email: ' + (data.message || 'Unknown error'), 'danger');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showTestEmailStatus('❌ Error sending test email: ' + error.message, 'danger');
            });
        }
        
        // Email validation function
        function validateEmail(email) {
            const emailPattern = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailPattern.test(email);
        }
        
        // Show status message function
        function showTestEmailStatus(message, type) {
            // Try to find existing status div, or create a temporary alert
            let statusDiv = document.getElementById('emailTestStatus');
            if (!statusDiv) {
                // Create temporary alert at top of page
                const alert = document.createElement('div');
                alert.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
                alert.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
                alert.innerHTML = `
                    <i class="bx bx-info-circle me-1"></i>
                    ${message}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                `;
                document.body.appendChild(alert);
                
                // Auto-remove success messages after 5 seconds
                if (type === 'success') {
                    setTimeout(() => {
                        alert.remove();
                    }, 5000);
                }
            } else {
                // Use existing status div
                const messageSpan = document.getElementById('emailStatusMessage');
                const alertDiv = statusDiv.querySelector('.alert');
                
                alertDiv.className = `alert alert-${type}`;
                messageSpan.textContent = message;
                statusDiv.style.display = 'block';
                
                // Auto-hide success messages after 5 seconds
                if (type === 'success') {
                    setTimeout(() => {
                        statusDiv.style.display = 'none';
                    }, 5000);
                }
            }
        }

        // Test Email Functionality removed - using automated certificate system instead
        
        // Email validation function
        function validateEmail(email) {
            const emailPattern = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailPattern.test(email);
        }
        
        // Show status message function
        function showStatus(message, type) {
            const statusDiv = document.getElementById('emailTestStatus');
            const messageSpan = document.getElementById('emailStatusMessage');
            const alertDiv = statusDiv.querySelector('.alert');
            
            alertDiv.className = `alert alert-${type}`;
            messageSpan.textContent = message;
            statusDiv.style.display = 'block';
            
            // Auto-hide success messages after 5 seconds
            if (type === 'success') {
                setTimeout(() => {
                    statusDiv.style.display = 'none';
                }, 5000);
            }
        }

        function viewCertificate(certificateId) {
            // Certificate viewing functionality
            console.log('Viewing certificate:', certificateId);
        }
        
        document.addEventListener('DOMContentLoaded', function() {
            // Sidebar toggle functionality
            const sidebarToggle = document.querySelector('.sidebar-toggle');
            const sidebarToggleMobile = document.querySelector('.sidebar-toggle-mobile');
            const sidebar = document.querySelector('.sidebar');
        
        if (sidebarToggle) {
            sidebarToggle.addEventListener('click', () => {
                sidebar.classList.toggle('close');
            });
        }
        
        if (sidebarToggleMobile) {
            sidebarToggleMobile.addEventListener('click', () => {
                sidebar.classList.toggle('show');
            });
        }
        
        // Responsive sidebar behavior
        function checkScreenSize() {
            if (window.innerWidth < 768) {
                sidebar.classList.remove('close');
                sidebar.classList.remove('show');
            } else if (window.innerWidth < 992) {
                sidebar.classList.add('close');
                sidebar.classList.remove('show');
            }
        }
        
        // Initial check and listen for window resize
        checkScreenSize();
        window.addEventListener('resize', checkScreenSize);
        
        // Dropdown functionality (if not using Bootstrap's dropdown)
        document.querySelectorAll('.dropdown-toggle').forEach(item => {
            item.addEventListener('click', event => {
                event.preventDefault();
                const dropdownMenu = item.nextElementSibling;
                if (dropdownMenu && dropdownMenu.classList.contains('dropdown-menu')) {
                    dropdownMenu.classList.toggle('show');
                }
            });
        });
        
        // Close dropdowns when clicking outside
        document.addEventListener('click', event => {
            const isDropdownButton = event.target.classList.contains('dropdown-toggle');
            if (!isDropdownButton) {
                document.querySelectorAll('.dropdown-menu.show').forEach(dropdown => {
                    dropdown.classList.remove('show');
                });
            }
        });
    });
    </script>
</body>
</html>
