<?php
require_once __DIR__ . '/../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Initialize variables for certificate management
$message = '';
$messageType = '';
$errorFields = [];
$formData = [
    'id' => '',
    'student_id' => '',
    'course_id' => '',
    'title' => '',
    'description' => '',
    'issue_date' => date('Y-m-d'),
    'expiry_date' => '',
    'certificate_number' => 'CERT-' . strtoupper(substr(md5(uniqid()), 0, 8)),
    'certificate_url' => ''
];

// Process form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'create') {
        // Validate form data
        $formData = [
            'student_id' => trim($_POST['student_id'] ?? ''),
            'course_id' => trim($_POST['course_id'] ?? ''),
            'title' => trim($_POST['title'] ?? ''),
            'description' => trim($_POST['description'] ?? ''),
            'issue_date' => trim($_POST['issue_date'] ?? date('Y-m-d')),
            'expiry_date' => trim($_POST['expiry_date'] ?? ''),
            'certificate_number' => trim($_POST['certificate_number'] ?? 'CERT-' . strtoupper(substr(md5(uniqid()), 0, 8))),
            'certificate_url' => trim($_POST['certificate_url'] ?? '')
        ];
        
        // Basic validation
        if (empty($formData['student_id'])) {
            $errorFields[] = 'student_id';
        }
        
        if (empty($formData['course_id'])) {
            $errorFields[] = 'course_id';
        }
        
        if (empty($formData['title'])) {
            $errorFields[] = 'title';
        }
        
        if (empty($formData['description'])) {
            $errorFields[] = 'description';
        }
        
        // If no errors, proceed with certificate creation
        if (empty($errorFields)) {
            $stmt = $conn->prepare("INSERT INTO certificates 
                (student_id, course_id, title, description, issue_date, expiry_date, certificate_number, certificate_url) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                
            $stmt->bind_param("iissssss", 
                $formData['student_id'], 
                $formData['course_id'], 
                $formData['title'], 
                $formData['description'], 
                $formData['issue_date'], 
                $formData['expiry_date'], 
                $formData['certificate_number'], 
                $formData['certificate_url']
            );
            
            if ($stmt->execute()) {
                $message = 'Certificate created successfully';
                $messageType = 'success';
                
                // Reset form data
                $formData = [
                    'id' => '',
                    'student_id' => '',
                    'course_id' => '',
                    'title' => '',
                    'description' => '',
                    'issue_date' => date('Y-m-d'),
                    'expiry_date' => '',
                    'certificate_number' => 'CERT-' . strtoupper(substr(md5(uniqid()), 0, 8)),
                    'certificate_url' => ''
                ];
            } else {
                $message = 'Error creating certificate: ' . $stmt->error;
                $messageType = 'danger';
            }
        } else {
            $message = 'Please fill in all required fields';
            $messageType = 'warning';
        }
    } elseif (isset($_POST['action']) && $_POST['action'] === 'delete' && isset($_POST['certificate_id'])) {
        // Delete certificate
        $stmt = $conn->prepare("DELETE FROM certificates WHERE id = ?");
        $stmt->bind_param("i", $_POST['certificate_id']);
        
        if ($stmt->execute()) {
            $message = 'Certificate deleted successfully';
            $messageType = 'success';
        } else {
            $message = 'Error deleting certificate: ' . $stmt->error;
            $messageType = 'danger';
        }
    }
}

// Get all certificates with student and course information
$certificatesQuery = "SELECT c.*, 
    u.first_name, u.last_name, u.email,
    courses.title as course_title, courses.course_code
    FROM certificates c
    JOIN users u ON c.student_id = u.user_id
    JOIN courses ON c.course_id = courses.id
    ORDER BY c.issue_date DESC";
    
$certificatesResult = $conn->query($certificatesQuery);
$certificates = [];

if ($certificatesResult) {
    while ($row = $certificatesResult->fetch_assoc()) {
        $certificates[] = $row;
    }
}

// Get certificate statistics
$statsQuery = "SELECT 
    COUNT(*) as total_certificates,
    SUM(CASE WHEN expiry_date IS NULL OR expiry_date >= CURDATE() THEN 1 ELSE 0 END) as valid_certificates,
    SUM(CASE WHEN expiry_date < CURDATE() THEN 1 ELSE 0 END) as expired_certificates,
    SUM(CASE WHEN issue_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as recent_certificates
    FROM certificates";
    
$statsResult = $conn->query($statsQuery);
$stats = [
    'total' => 0,
    'valid' => 0,
    'expired' => 0,
    'recent' => 0
];

if ($statsResult && $row = $statsResult->fetch_assoc()) {
    $stats['total'] = $row['total_certificates'];
    $stats['valid'] = $row['valid_certificates'];
    $stats['expired'] = $row['expired_certificates'];
    $stats['recent'] = $row['recent_certificates'];
}

// Get students for dropdown
$studentsQuery = "SELECT user_id, first_name, last_name, email FROM users WHERE user_role = 'student' ORDER BY first_name, last_name";
$studentsResult = $conn->query($studentsQuery);
$students = [];

if ($studentsResult) {
    while ($row = $studentsResult->fetch_assoc()) {
        $students[] = $row;
    }
}

// Get courses for dropdown
$coursesQuery = "SELECT id, title, course_code FROM courses ORDER BY title";
$coursesResult = $conn->query($coursesQuery);
$courses = [];

if ($coursesResult) {
    while ($row = $coursesResult->fetch_assoc()) {
        $courses[] = $row;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Certificates Management - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link rel="stylesheet" href="/assets/css/pages/certificates.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.4/css/dataTables.bootstrap5.min.css">
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Certificate Management Content -->
        <div class="dashboard-content">
            <!-- Page Header -->
            <div class="page-header">
                <h1>Certificate Management</h1>
                <div>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addCertificateModal">
                        <i class='bx bx-plus-circle'></i> Issue New Certificate
                    </button>
                </div>
            </div>
            
            <!-- Alert for messages -->
            <?php if (!empty($message)): ?>
                <div class="alert alert-<?= $messageType ?> alert-dismissible fade show" role="alert">
                    <?= $message ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>
            
            <!-- Stats Cards -->
            <div class="row g-3 mb-4">
                <div class="col-md-3">
                    <div class="card stats-card bg-primary-light text-primary">
                        <div class="card-body">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h6 class="mb-0">Total Certificates</h6>
                                    <h2 class="mt-2 mb-0"><?= $stats['total'] ?></h2>
                                </div>
                                <div class="stats-icon">
                                    <i class="bx bx-certification"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-md-3">
                    <div class="card stats-card bg-success-light text-success">
                        <div class="card-body">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h6 class="mb-0">Valid Certificates</h6>
                                    <h2 class="mt-2 mb-0"><?= $stats['valid'] ?></h2>
                                </div>
                                <div class="stats-icon">
                                    <i class="bx bx-check-shield"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-md-3">
                    <div class="card stats-card bg-warning-light text-warning">
                        <div class="card-body">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h6 class="mb-0">Recent Certificates</h6>
                                    <h2 class="mt-2 mb-0"><?= $stats['recent'] ?></h2>
                                </div>
                                <div class="stats-icon">
                                    <i class="bx bx-time"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-md-3">
                    <div class="card stats-card bg-danger-light text-danger">
                        <div class="card-body">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h6 class="mb-0">Expired Certificates</h6>
                                    <h2 class="mt-2 mb-0"><?= $stats['expired'] ?></h2>
                                </div>
                                <div class="stats-icon">
                                    <i class="bx bx-x-circle"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Certificates Table -->
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">All Certificates</h5>
                </div>
                <div class="card-body">
                    <?php if (count($certificates) > 0): ?>
                        <div class="table-responsive">
                            <table id="certificatesTable" class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Certificate #</th>
                                        <th>Student</th>
                                        <th>Course</th>
                                        <th>Title</th>
                                        <th>Issue Date</th>
                                        <th>Expiry Date</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($certificates as $cert): 
                                        $isExpired = !empty($cert['expiry_date']) && strtotime($cert['expiry_date']) < time();
                                        $status = $isExpired ? 'Expired' : 'Valid';
                                        $statusClass = $isExpired ? 'expired' : 'valid';
                                    ?>
                                    <tr>
                                        <td><?= htmlspecialchars($cert['certificate_number']) ?></td>
                                        <td><?= htmlspecialchars($cert['first_name'] . ' ' . $cert['last_name']) ?><br>
                                            <small class="text-muted"><?= htmlspecialchars($cert['email']) ?></small>
                                        </td>
                                        <td><?= htmlspecialchars($cert['course_title']) ?><br>
                                            <small class="text-muted"><?= htmlspecialchars($cert['course_code']) ?></small>
                                        </td>
                                        <td><?= htmlspecialchars($cert['title']) ?></td>
                                        <td><?= date('M j, Y', strtotime($cert['issue_date'])) ?></td>
                                        <td>
                                            <?= !empty($cert['expiry_date']) ? date('M j, Y', strtotime($cert['expiry_date'])) : 'Never' ?>
                                        </td>
                                        <td>
                                            <span class="certificate-badge <?= $statusClass ?>">
                                                <i class="bx <?= $isExpired ? 'bx-x-circle' : 'bx-check-circle' ?>"></i>
                                                <?= $status ?>
                                            </span>
                                        </td>
                                        <td>
                                            <div class="btn-group">
                                                <a href="<?= $cert['certificate_url'] ?>" class="btn btn-sm btn-outline-primary" target="_blank">
                                                    <i class="bx bx-show"></i>
                                                </a>
                                                <a href="<?= $cert['certificate_url'] ?>" class="btn btn-sm btn-outline-success" download>
                                                    <i class="bx bx-download"></i>
                                                </a>
                                                <button type="button" class="btn btn-sm btn-outline-danger" 
                                                        data-bs-toggle="modal" 
                                                        data-bs-target="#deleteCertificateModal"
                                                        data-cert-id="<?= $cert['id'] ?>"
                                                        data-cert-number="<?= $cert['certificate_number'] ?>">
                                                    <i class="bx bx-trash"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php else: ?>
                        <div class="empty-state">
                            <i class="empty-state-icon bx bx-certification"></i>
                            <h4>No Certificates Yet</h4>
                            <p>No certificates have been issued yet. Click the button below to issue your first certificate.</p>
                            <button class="btn btn-primary btn-lg" data-bs-toggle="modal" data-bs-target="#addCertificateModal">
                                <i class="bx bx-plus-circle me-2"></i> Issue New Certificate
                            </button>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </section>
    
    <!-- Add Certificate Modal -->
    <div class="modal fade" id="addCertificateModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Issue New Certificate</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="POST" action="">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="create">
                        
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="student_id" class="form-label">Student</label>
                                <select name="student_id" id="student_id" class="form-select <?= in_array('student_id', $errorFields) ? 'is-invalid' : '' ?>" required>
                                    <option value="">Select Student</option>
                                    <?php foreach ($students as $student): ?>
                                        <option value="<?= $student['user_id'] ?>" <?= $formData['student_id'] == $student['user_id'] ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($student['first_name'] . ' ' . $student['last_name'] . ' (' . $student['email'] . ')') ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <?php if (in_array('student_id', $errorFields)): ?>
                                    <div class="invalid-feedback">Please select a student</div>
                                <?php endif; ?>
                            </div>
                            <div class="col-md-6">
                                <label for="course_id" class="form-label">Course</label>
                                <select name="course_id" id="course_id" class="form-select <?= in_array('course_id', $errorFields) ? 'is-invalid' : '' ?>" required>
                                    <option value="">Select Course</option>
                                    <?php foreach ($courses as $course): ?>
                                        <option value="<?= $course['id'] ?>" <?= $formData['course_id'] == $course['id'] ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($course['title'] . ' (' . $course['course_code'] . ')') ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <?php if (in_array('course_id', $errorFields)): ?>
                                    <div class="invalid-feedback">Please select a course</div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="title" class="form-label">Certificate Title</label>
                            <input type="text" name="title" id="title" class="form-control <?= in_array('title', $errorFields) ? 'is-invalid' : '' ?>" 
                                placeholder="Certificate of Completion" value="<?= htmlspecialchars($formData['title']) ?>" required>
                            <?php if (in_array('title', $errorFields)): ?>
                                <div class="invalid-feedback">Please enter a certificate title</div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="mb-3">
                            <label for="description" class="form-label">Description</label>
                            <textarea name="description" id="description" class="form-control <?= in_array('description', $errorFields) ? 'is-invalid' : '' ?>" 
                                placeholder="Certificate description" rows="3" required><?= htmlspecialchars($formData['description']) ?></textarea>
                            <?php if (in_array('description', $errorFields)): ?>
                                <div class="invalid-feedback">Please enter a description</div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="issue_date" class="form-label">Issue Date</label>
                                <input type="date" name="issue_date" id="issue_date" class="form-control" 
                                    value="<?= $formData['issue_date'] ?>" required>
                            </div>
                            <div class="col-md-6">
                                <label for="expiry_date" class="form-label">Expiry Date (optional)</label>
                                <input type="date" name="expiry_date" id="expiry_date" class="form-control" 
                                    value="<?= $formData['expiry_date'] ?>">
                                <small class="text-muted">Leave blank if certificate does not expire</small>
                            </div>
                        </div>
                        
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="certificate_number" class="form-label">Certificate Number</label>
                                <input type="text" name="certificate_number" id="certificate_number" class="form-control" 
                                    value="<?= htmlspecialchars($formData['certificate_number']) ?>" required>
                                <small class="text-muted">Auto-generated, but can be modified</small>
                            </div>
                            <div class="col-md-6">
                                <label for="certificate_url" class="form-label">Certificate URL</label>
                                <input type="url" name="certificate_url" id="certificate_url" class="form-control" 
                                    placeholder="https://example.com/certificates/cert.pdf" value="<?= htmlspecialchars($formData['certificate_url']) ?>">
                                <small class="text-muted">URL to the PDF certificate file</small>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Issue Certificate</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Delete Certificate Modal -->
    <div class="modal fade" id="deleteCertificateModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Delete Certificate</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <p>Are you sure you want to delete certificate <strong id="certificateToDelete"></strong>?</p>
                    <p class="text-danger">This action cannot be undone.</p>
                </div>
                <form method="POST" action="">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="certificate_id" id="certificate_id" value="">
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-danger">Delete Certificate</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/dataTables.bootstrap5.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // DataTable initialization
        $('#certificatesTable').DataTable({
            order: [[4, 'desc']], // Sort by issue date by default (newest first)
            language: {
                search: "Search certificates:",
                lengthMenu: "Show _MENU_ certificates per page",
                info: "Showing _START_ to _END_ of _TOTAL_ certificates",
                emptyTable: "No certificates found"
            }
        });
        
        // Sidebar toggle functionality
        const sidebarToggle = document.querySelector('.sidebar-toggle');
        const sidebarToggleMobile = document.querySelector('.sidebar-toggle-mobile');
        const sidebar = document.querySelector('.sidebar');
        
        if (sidebarToggle) {
            sidebarToggle.addEventListener('click', () => {
                sidebar.classList.toggle('close');
            });
        }
        
        if (sidebarToggleMobile) {
            sidebarToggleMobile.addEventListener('click', () => {
                sidebar.classList.toggle('show');
            });
        }
        
        // Handle certificate deletion modal
        const deleteCertificateModal = document.getElementById('deleteCertificateModal');
        if (deleteCertificateModal) {
            deleteCertificateModal.addEventListener('show.bs.modal', function(event) {
                const button = event.relatedTarget;
                const certId = button.getAttribute('data-cert-id');
                const certNumber = button.getAttribute('data-cert-number');
                
                document.getElementById('certificate_id').value = certId;
                document.getElementById('certificateToDelete').textContent = certNumber;
            });
        }
        
        // Auto-generate certificate number when adding a new certificate
        document.querySelector('#addCertificateModal button[type="submit"]').addEventListener('click', function() {
            if (document.getElementById('certificate_number').value === '') {
                const randomPart = Math.random().toString(36).substring(2, 10).toUpperCase();
                document.getElementById('certificate_number').value = 'CERT-' + randomPart;
            }
        });
    });
    </script>
</body>
</html>
