<?php
require_once '../auth/functions.php';
require_once '../config/fast_email_service.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Bridge database connections - Email service needs PDO, but admin uses MySQLi
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for email service: " . $e->getMessage());
        $pdo = null;
    }
}

/**
 * Send course approval email notification to teacher
 */
function sendCourseApprovalEmailNotification($course_id) {
    global $conn;
    
    try {
        // Get course and teacher details
        $stmt = $conn->prepare("
            SELECT c.title, c.description, c.price, c.created_at, 
                   u.first_name, u.last_name, u.email, u.user_id,
                   cat.category_name
            FROM courses c 
            JOIN users u ON c.teacher_id = u.user_id 
            LEFT JOIN categories cat ON c.category_id = cat.category_id 
            WHERE c.course_id = ?
        ");
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $course = $result->fetch_assoc();
        
        if (!$course) {
            error_log("Course not found for approval email: $course_id");
            return false;
        }
        
        $teacher_name = $course['first_name'] . ' ' . $course['last_name'];
        $teacher_email = $course['email'];
        $course_title = $course['title'];
        $category = $course['category_name'] ?? 'General';
        $price = floatval($course['price']);
        $created_date = date('M j, Y', strtotime($course['created_at']));
        
        // Create email content
        $subject = "🎉 Course Approved: \"$course_title\" is Now Live!";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#28a745; color:white; padding:20px; text-align:center;'>
                <h1>🎓 Course Approved!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Congratulations, " . htmlspecialchars($teacher_name) . "! 🎉</h2>
                <p>Your course <strong>\"" . htmlspecialchars($course_title) . "\"</strong> has been approved and is now live!</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Course Details:</h3>
                    <p><strong>Category:</strong> " . htmlspecialchars($category) . "</p>
                    <p><strong>Price:</strong> R" . number_format($price, 2) . "</p>
                    <p><strong>Created:</strong> " . $created_date . "</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/teacher/courses.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        View Dashboard
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($teacher_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($teacher_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending course approval email: " . $e->getMessage());
        return false;
    }
}

// Process course actions
$message = '';

// Handle course deletion
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    $course_id = $_GET['delete'];
    
    // Start transaction for cascading deletion
    $conn->autocommit(FALSE);
    
    try {
        // Get enrollment count for feedback
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM enrollments WHERE course_id = ?");
        if (!$stmt) throw new Exception("Failed to prepare enrollment count query: " . $conn->error);
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $enrollment_count = $result->fetch_assoc()['count'];
        
        // Helper function to safely execute deletion queries
        $executeDelete = function($query, $description) use ($conn, $course_id) {
            $stmt = $conn->prepare($query);
            if (!$stmt) {
                error_log("Failed to prepare $description query: " . $conn->error);
                return; // Skip this deletion if table doesn't exist
            }
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $stmt->close();
        };
        
        // 1. Delete quiz submissions (if quiz tables exist)
        $executeDelete("DELETE FROM quiz_submissions WHERE quiz_id IN (SELECT quiz_id FROM quizzes WHERE course_id = ?)", "quiz submissions");
        
        // 2. Delete quizzes associated with this course
        $executeDelete("DELETE FROM quizzes WHERE course_id = ?", "quizzes");
        
        // 3. Delete course progress records (if table exists)
        $executeDelete("DELETE FROM course_progress WHERE course_id = ?", "course progress");
        
        // 4. Delete video progress records (if table exists)
        $video_ids_result = $conn->query("SELECT video_id FROM videos WHERE course_id = $course_id");
        if ($video_ids_result && $video_ids_result->num_rows > 0) {
            while ($video_row = $video_ids_result->fetch_assoc()) {
                $executeDelete("DELETE FROM video_progress WHERE video_id = " . $video_row['video_id'], "video progress");
            }
        }
        
        // 5. Delete enrollments
        $executeDelete("DELETE FROM enrollments WHERE course_id = ?", "enrollments");
        
        // 6. Delete course materials (if table exists)
        $executeDelete("DELETE FROM course_materials WHERE course_id = ?", "course materials");
        
        // 7. Delete videos associated with this course
        $executeDelete("DELETE FROM videos WHERE course_id = ?", "videos");
        
        // 8. Delete any cart items with this course (if table exists)
        $executeDelete("DELETE FROM cart_items WHERE course_id = ?", "cart items");
        
        // 9. Delete any wishlist items with this course (if table exists)
        $executeDelete("DELETE FROM wishlist WHERE course_id = ?", "wishlist items");
        
        // 10. Delete any reviews for this course (if table exists)
        $executeDelete("DELETE FROM course_reviews WHERE course_id = ?", "course reviews");
        
        // 11. Finally delete the course itself
        $stmt = $conn->prepare("DELETE FROM courses WHERE course_id = ?");
        if (!$stmt) throw new Exception("Failed to prepare course deletion query: " . $conn->error);
        $stmt->bind_param("i", $course_id);
        
        if ($stmt->execute()) {
            // Commit the transaction
            $conn->commit();
            
            $details = $enrollment_count > 0 ? " (including {$enrollment_count} enrollments and all associated data)" : " and all associated data";
            $message = '<div class="alert alert-success">
                <strong>Course deleted successfully!</strong><br>
                All related records have been removed' . $details . '.
            </div>';
        } else {
            throw new Exception("Failed to delete course: " . $conn->error);
        }
        
    } catch (Exception $e) {
        // Rollback the transaction on error
        $conn->rollback();
        $message = '<div class="alert alert-danger">Error deleting course: ' . $e->getMessage() . '</div>';
    }
    
    // Re-enable autocommit
    $conn->autocommit(TRUE);
}

// Toggle course publication status
if (isset($_GET['toggle_publish']) && is_numeric($_GET['toggle_publish'])) {
    $course_id = $_GET['toggle_publish'];
    
    // First get current status
    $stmt = $conn->prepare("SELECT is_published FROM courses WHERE course_id = ?");
    $stmt->bind_param("i", $course_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    
    if ($result->num_rows > 0) {
        // Toggle the status
        $new_status = $row['is_published'] ? 0 : 1;
        $status_text = $new_status ? "published" : "unpublished";
        
        $stmt = $conn->prepare("UPDATE courses SET is_published = ? WHERE course_id = ?");
        $stmt->bind_param("ii", $new_status, $course_id);
        
        if ($stmt->execute()) {
            // If course is being published (approved), send email notification
            if ($new_status == 1) {
                global $debug_info;
                $email_sent = sendCourseApprovalEmailNotification($course_id);
                
                // Build debug output for display
                $debug_output = '';
                if (isset($debug_info) && is_array($debug_info)) {
                    $debug_output .= '<div class="mt-3"><h6>🔍 Email Debug Information:</h6>';
                    
                    // Show steps
                    if (!empty($debug_info['steps'])) {
                        $debug_output .= '<div class="mb-2"><strong>Steps:</strong><ul class="mb-0">';
                        foreach ($debug_info['steps'] as $step) {
                            $debug_output .= '<li>' . htmlspecialchars($step) . '</li>';
                        }
                        $debug_output .= '</ul></div>';
                    }
                    
                    // Show errors
                    if (!empty($debug_info['errors'])) {
                        $debug_output .= '<div class="mb-2"><strong>Errors:</strong><ul class="mb-0 text-danger">';
                        foreach ($debug_info['errors'] as $error) {
                            $debug_output .= '<li>' . htmlspecialchars($error) . '</li>';
                        }
                        $debug_output .= '</ul></div>';
                    }
                    
                    $debug_output .= '<div><strong>Final Result:</strong> ' . ($debug_info['success'] ? '✅ Success' : '❌ Failed') . '</div>';
                    $debug_output .= '</div>';
                }
                
                if ($email_sent) {
                    $message = '<div class="alert alert-success">Course ' . $status_text . ' successfully! ✉️ Approval email sent to teacher.' . $debug_output . '</div>';
                } else {
                    $message = '<div class="alert alert-warning">Course ' . $status_text . ' successfully, but failed to send notification email to teacher.' . $debug_output . '</div>';
                }
            } else {
                $message = '<div class="alert alert-success">Course ' . $status_text . ' successfully.</div>';
            }
        } else {
            $message = '<div class="alert alert-danger">Error updating course status: ' . $conn->error . '</div>';
        }
    }
}

// Fetch all courses with teacher and category names
$query = "SELECT c.*, u.first_name, u.last_name, cat.category_name, 
         (SELECT COUNT(*) FROM videos WHERE course_id = c.course_id) as video_count,
         (SELECT COUNT(*) FROM enrollments WHERE course_id = c.course_id) as enrollment_count,
         DATEDIFF(NOW(), c.created_at) as days_old
         FROM courses c 
         JOIN users u ON c.teacher_id = u.user_id
         JOIN categories cat ON c.category_id = cat.category_id
         ORDER BY c.created_at DESC";

$result = $conn->query($query);
$courses = [];
$course_stats = [
    'new' => 0,      // Past 7 days
    'weekly' => 0,   // Past 30 days
    'monthly' => 0,  // Past 365 days
    'yearly' => 0,   // Older than 1 year
    'published' => 0,
    'draft' => 0
];

if ($result) {
    while ($row = $result->fetch_assoc()) {
        $courses[] = $row;
        
        // Categorize by age
        if ($row['days_old'] <= 7) {
            $course_stats['new']++;
        } elseif ($row['days_old'] <= 30) {
            $course_stats['weekly']++;
        } elseif ($row['days_old'] <= 365) {
            $course_stats['monthly']++;
        } else {
            $course_stats['yearly']++;
        }
        
        // Categorize by status
        if ($row['is_published']) {
            $course_stats['published']++;
        } else {
            $course_stats['draft']++;
        }
    }
}

// Get all teachers for the add/edit form
$teachers = [];
$result = $conn->query("SELECT user_id, username, first_name, last_name FROM users WHERE user_role = 'teacher' ORDER BY first_name, last_name");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $teachers[] = $row;
    }
}

// Get all categories for the add/edit form
$categories = [];
$result = $conn->query("SELECT category_id, category_name FROM categories ORDER BY category_name");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $categories[] = $row;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Course Management - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href="https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <style>
        /* Override for courses page */
        .main-content {
            padding-top: 60px !important; /* Exact navbar height */
            margin-top: -10px !important;
        }
        
        .dashboard-content {
            padding: 0 15px !important;
            margin-top: 3rem !important;
        }
        
        .page-header {
            margin-top: 0 !important;
            margin-bottom: 10px !important;
            padding-top: 0 !important;
        }
        
        h1.h2 {
            margin-top: 0 !important;
            margin-bottom: 5px !important;
        }
        
        .filter-container {
            margin-bottom: 10px !important;
            margin-top: 0 !important;
        }
        
        /* Course-specific styles */
        .course-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
            padding: 5px 0;
            margin-top: 15px;
        }
        
        .course-card {
            background-color: #fff;
            border-radius: 10px;
            overflow: hidden;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            transition: all 0.3s ease;
            display: flex;
            flex-direction: column;
        }
        
        .course-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
        }
        
        .course-thumbnail {
            position: relative;
            height: 150px;
            background-color: #f8f9fa;
            overflow: hidden;
        }
        
        .course-thumbnail img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.3s ease;
        }
        
        .course-card:hover .course-thumbnail img {
            transform: scale(1.05);
        }
        
        .course-thumbnail-placeholder {
            display: flex;
            align-items: center;
            justify-content: center;
            width: 100%;
            height: 100%;
            background-image: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
            color: white;
        }
        
        .course-thumbnail-placeholder i {
            font-size: 50px;
            opacity: 0.7;
        }
        
        .course-badge {
            position: absolute;
            top: 10px;
            right: 10px;
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            color: white;
            display: flex;
            align-items: center;
            z-index: 1;
        }
        
        .course-badge.published {
            background-color: #28a745;
        }
        
        .course-badge.unpublished {
            background-color: #6c757d;
        }
        
        .course-badge i {
            margin-right: 5px;
            font-size: 14px;
        }
        
        /* Age badges for NEW and RECENT courses */
        .course-age-badge {
            position: absolute;
            top: 10px;
            left: 10px;
            color: white;
            font-size: 10px;
            font-weight: 700;
            text-transform: uppercase;
            padding: 4px 8px;
            border-radius: 12px;
            z-index: 2;
            display: flex;
            align-items: center;
        }
        
        .course-age-badge.new {
            background: linear-gradient(45deg, #28a745, #20c997);
            animation: pulse 2s infinite;
        }
        
        .course-age-badge.recent {
            background: linear-gradient(45deg, #007bff, #17a2b8);
        }
        
        .course-age-badge i {
            margin-right: 4px;
            font-size: 12px;
        }
        
        @keyframes pulse {
            0% {
                box-shadow: 0 0 0 0 rgba(40, 167, 69, 0.7);
            }
            70% {
                box-shadow: 0 0 0 10px rgba(40, 167, 69, 0);
            }
            100% {
                box-shadow: 0 0 0 0 rgba(40, 167, 69, 0);
            }
        }
        
        /* Filter button states */
        .filter-btn.active {
            background-color: #0d6efd !important;
            border-color: #0d6efd !important;
            color: white !important;
        }
        
        /* Hide filtered courses */
        .course-item.hidden {
            display: none !important;
        }
        
        .course-price {
            position: absolute;
            bottom: 10px;
            right: 10px;
            background-color: rgba(0, 0, 0, 0.7);
            color: white;
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 14px;
            font-weight: 600;
            z-index: 1;
        }
        
        .course-details {
            padding: 15px;
            flex-grow: 1;
            display: flex;
            flex-direction: column;
        }
        
        .course-category {
            color: #4e73df;
            font-size: 12px;
            font-weight: 500;
            margin-bottom: 5px;
            display: flex;
            align-items: center;
        }
        
        .course-category i {
            margin-right: 5px;
        }
        
        .course-title {
            font-size: 16px;
            font-weight: 600;
            margin-bottom: 10px;
            color: #333;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
            height: 42px;
        }
        
        .course-meta {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 8px;
            margin-bottom: 15px;
        }
        
        .course-meta > div {
            display: flex;
            align-items: center;
            color: #6c757d;
            font-size: 12px;
        }
        
        .course-meta i {
            margin-right: 5px;
            font-size: 14px;
        }
        
        .course-stats {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 10px;
            margin-top: auto;
        }
        
        .course-stat {
            display: flex;
            align-items: center;
            background-color: #f8f9fa;
            padding: 8px;
            border-radius: 5px;
        }
        
        .course-stat i {
            margin-right: 5px;
            font-size: 14px;
            color: #4e73df;
        }
        
        .course-stat span {
            font-size: 12px;
            font-weight: 600;
        }
        
        .course-actions {
            display: flex;
            justify-content: space-between;
            border-top: 1px solid #f1f1f1;
            padding: 12px 15px;
            background: #fafafa;
        }
        
        .course-actions .btn {
            width: 34px;
            height: 34px;
            padding: 0;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 6px;
        }
        
        .course-actions .btn i {
            font-size: 16px;
        }
        
        .animate-fade-in {
            animation: fadeIn 0.5s ease forwards;
            opacity: 0;
            animation-delay: calc(var(--delay) * 1s);
        }
        
        @keyframes fadeIn {
            0% { opacity: 0; transform: translateY(10px); }
            100% { opacity: 1; transform: translateY(0); }
        }
        
        @media (max-width: 768px) {
            .course-grid {
                grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            }
        }
        
        @media (max-width: 576px) {
            .course-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
        
     
</head>
<body>
    <div class="wrapper">
        <!-- Sidebar -->
        <?php include('components/sidebar.php'); ?>
        <!-- Main Content -->
        <div class="main-content">
            <!-- Navbar -->
            <?php include('components/navbar.php'); ?>
            
            <!-- Dashboard Content -->
            <div class="dashboard-content" style="padding-top: 0; margin-top: -15px;">
                <div class="page-header">
                    <h1 class="h2">Course Management</h1>
                </div>
                
                <?php echo $message; ?>
                
                <!-- Course Statistics Cards -->
                <div class="row mb-4">
                    <div class="col-md-3 col-sm-6 mb-3">
                        <div class="card bg-success text-white">
                            <div class="card-body">
                                <div class="d-flex align-items-center">
                                    <i class='bx bx-time-five' style="font-size: 2rem; margin-right: 15px;"></i>
                                    <div>
                                        <h4 class="mb-0"><?php echo $course_stats['new']; ?></h4>
                                        <small>New (Past 7 days)</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 col-sm-6 mb-3">
                        <div class="card bg-info text-white">
                            <div class="card-body">
                                <div class="d-flex align-items-center">
                                    <i class='bx bx-calendar-week' style="font-size: 2rem; margin-right: 15px;"></i>
                                    <div>
                                        <h4 class="mb-0"><?php echo $course_stats['weekly']; ?></h4>
                                        <small>Weekly (Past 30 days)</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 col-sm-6 mb-3">
                        <div class="card bg-warning text-white">
                            <div class="card-body">
                                <div class="d-flex align-items-center">
                                    <i class='bx bx-calendar' style="font-size: 2rem; margin-right: 15px;"></i>
                                    <div>
                                        <h4 class="mb-0"><?php echo $course_stats['monthly']; ?></h4>
                                        <small>Monthly (Past year)</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 col-sm-6 mb-3">
                        <div class="card bg-secondary text-white">
                            <div class="card-body">
                                <div class="d-flex align-items-center">
                                    <i class='bx bx-archive' style="font-size: 2rem; margin-right: 15px;"></i>
                                    <div>
                                        <h4 class="mb-0"><?php echo $course_stats['yearly']; ?></h4>
                                        <small>Older (1+ years)</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Status Filter Cards -->
                <div class="row mb-4">
                    <div class="col-md-6 mb-3">
                        <div class="card bg-primary text-white">
                            <div class="card-body">
                                <div class="d-flex align-items-center justify-content-between">
                                    <div class="d-flex align-items-center">
                                        <i class='bx bx-show' style="font-size: 2rem; margin-right: 15px;"></i>
                                        <div>
                                            <h4 class="mb-0"><?php echo $course_stats['published']; ?></h4>
                                            <small>Published Courses</small>
                                        </div>
                                    </div>
                                    <button class="btn btn-outline-light btn-sm filter-btn" data-filter="published">Filter</button>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6 mb-3">
                        <div class="card bg-dark text-white">
                            <div class="card-body">
                                <div class="d-flex align-items-center justify-content-between">
                                    <div class="d-flex align-items-center">
                                        <i class='bx bx-hide' style="font-size: 2rem; margin-right: 15px;"></i>
                                        <div>
                                            <h4 class="mb-0"><?php echo $course_stats['draft']; ?></h4>
                                            <small>Draft Courses</small>
                                        </div>
                                    </div>
                                    <button class="btn btn-outline-light btn-sm filter-btn" data-filter="draft">Filter</button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Filter Controls -->
                <div class="card mb-4">
                    <div class="card-body">
                        <div class="row align-items-center">
                            <div class="col-md-8">
                                <div class="btn-group" role="group">
                                    <button type="button" class="btn btn-outline-primary active filter-btn" data-filter="all">All Courses</button>
                                    <button type="button" class="btn btn-outline-success filter-btn" data-filter="new">New (7 days)</button>
                                    <button type="button" class="btn btn-outline-info filter-btn" data-filter="recent">Recent (30 days)</button>
                                    <button type="button" class="btn btn-outline-warning filter-btn" data-filter="older">Older</button>
                                </div>
                            </div>
                            <div class="col-md-4 text-end">
                                <div class="btn-group" role="group">
                                    <button type="button" class="btn btn-outline-primary filter-btn" data-filter="published">Published</button>
                                    <button type="button" class="btn btn-outline-secondary filter-btn" data-filter="draft">Draft</button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Courses Section -->
                <div class="table-container animate-fade-in" style="--delay: 0.1s">
                    <div class="table-header d-flex justify-content-between align-items-center mb-3">
                        <h5>
                            <?php if (count($courses) > 0): ?>
                                <?php echo count($courses); ?> Courses
                            <?php else: ?>
                                No Courses Found
                            <?php endif; ?>
                        </h5>
                    </div>
                </div>
                
                <!-- Course Grid Layout -->
                <?php if (count($courses) > 0): ?>
                <div class="course-grid">
                    <?php $loop = 0; foreach ($courses as $course): $loop++; 
                        // Calculate age classification for filtering
                        $days_old = $course['days_old'];
                        $age_class = '';
                        $age_label = '';
                        if ($days_old <= 7) {
                            $age_class = 'new';
                            $age_label = 'NEW';
                        } elseif ($days_old <= 30) {
                            $age_class = 'recent';
                            $age_label = 'RECENT';
                        } else {
                            $age_class = 'older';
                            $age_label = '';
                        }
                        $status_class = $course['is_published'] ? 'published' : 'draft';
                    ?>
                    <div class="course-card animate-fade-in course-item" 
                         data-age="<?php echo $age_class; ?>" 
                         data-status="<?php echo $status_class; ?>"
                         data-days="<?php echo $days_old; ?>"
                         style="--delay: <?php echo 0.1 + 0.05 * $loop; ?>s">
                        <div class="course-thumbnail">
                            <?php if (!empty($course['thumbnail'])): ?>
                                <img src="<?php echo $course['thumbnail']; ?>" alt="<?php echo htmlspecialchars($course['title']); ?>" loading="lazy" onerror="this.onerror=null;this.parentNode.innerHTML='<div class=\'course-thumbnail-placeholder\'><i class=\'bx bx-book\'></i></div>';">
                            <?php else: ?>
                                <div class="course-thumbnail-placeholder">
                                    <i class='bx bx-book'></i>
                                </div>
                            <?php endif; ?>
                            
                            <?php if (!empty($age_label)): ?>
                            <div class="course-age-badge <?php echo $age_class; ?>">
                                <i class='bx bx-<?php echo $age_class === 'new' ? 'star' : 'time-five'; ?>'></i>
                                <?php echo $age_label; ?>
                            </div>
                            <?php endif; ?>
                            
                            <div class="course-badge <?php echo $course['is_published'] ? 'published' : 'unpublished'; ?>">
                                <i class='bx bx-<?php echo $course['is_published'] ? 'check-circle' : 'hide'; ?>'></i>
                                <?php echo $course['is_published'] ? 'Published' : 'Draft'; ?>
                            </div>
                            <div class="course-price">
                                R<?php echo number_format($course['price'], 2); ?>
                            </div>
                        </div>
                        
                        <div class="course-details">
                            <div class="course-category">
                                <i class='bx bx-category'></i> <?php echo htmlspecialchars($course['category_name']); ?>
                            </div>
                            
                            <h3 class="course-title"><?php echo htmlspecialchars($course['title']); ?></h3>
                            
                            <div class="course-meta">
                                <div>
                                    <i class='bx bx-user'></i> <?php echo htmlspecialchars($course['first_name'] . ' ' . $course['last_name']); ?>
                                </div>
                                <div>
                                    <i class='bx bx-calendar'></i> <?php echo date('M d, Y', strtotime($course['created_at'])); ?>
                                </div>
                            </div>
                            
                            <div class="course-stats">
                                <div class="course-stat">
                                    <i class='bx bx-video'></i> <span><?php echo $course['video_count']; ?> Videos</span>
                                </div>
                                <div class="course-stat">
                                    <i class='bx bx-user-plus'></i> <span><?php echo $course['enrollment_count']; ?> Enrollments</span>
                                </div>
                            </div>
                        </div>
                        
                        <div class="course-actions">
                            <a href="edit_course.php?id=<?php echo $course['course_id']; ?>" class="btn btn-outline-primary" title="Edit Course">
                                <i class='bx bx-edit-alt'></i>
                            </a>
                            <a href="videos.php?course_id=<?php echo $course['course_id']; ?>" class="btn btn-outline-info" title="Manage Videos">
                                <i class='bx bx-video'></i>
                            </a>
                            <a href="courses.php?toggle_publish=<?php echo $course['course_id']; ?>" class="btn btn-outline-secondary" title="<?php echo $course['is_published'] ? 'Unpublish' : 'Publish'; ?>">
                                <i class='bx bx-<?php echo $course['is_published'] ? 'hide' : 'show'; ?>'></i>
                            </a>
                            <a href="#" class="btn btn-outline-danger delete-course" data-id="<?php echo $course['course_id']; ?>" title="Delete Course" <?php echo $course['enrollment_count'] > 0 ? 'disabled' : ''; ?>>
                                <i class='bx bx-trash'></i>
                            </a>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php else: ?>
                <div class="alert alert-info">
                    <i class='bx bx-info-circle'></i> No courses available. <a href="add_course.php">Add your first course</a>.
                </div>
                <?php endif; ?>
            <div class="pt-2 pb-4">
                <ul class="nav flex-column">
                    <li class="nav-item">
                        <a class="nav-link text-white" href="dashboard.php">
                            <i class="fas fa-tachometer-alt"></i> Dashboard
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link text-white" href="users.php">
                            <i class="fas fa-users"></i> Users
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link active text-white" href="courses.php">
                            <i class="fas fa-graduation-cap"></i> Courses
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link text-white" href="videos.php">
                            <i class="fas fa-video"></i> Videos
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link text-white" href="categories.php">
                            <i class="fas fa-tags"></i> Categories
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link text-white" href="payments.php">
                            <i class="fas fa-money-bill"></i> Payments
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link text-white" href="payfast.php">
                            <i class="fas fa-credit-card"></i> PayFast Settings
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link text-white" href="/Panadite academy/auth/logout.php">
                            <i class="fas fa-sign-out-alt"></i> Logout
                        </a>
                    </li>
                </ul>
                            <a class="nav-link text-white" href="/Panadite academy/auth/logout.php">
                                <i class="fas fa-sign-out-alt"></i> Logout
                            </a>
                        </li>
                    </ul>
                </div>
            </div>
            
           
                
               
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
    $(document).ready(function() {
        // Filter functionality
        $('.filter-btn').click(function() {
            const filter = $(this).data('filter');
            
            // Update active button state
            $('.filter-btn').removeClass('active');
            $(this).addClass('active');
            
            // Show/hide courses based on filter
            filterCourses(filter);
        });
        
        function filterCourses(filter) {
            $('.course-item').removeClass('hidden').show();
            
            if (filter === 'all') {
                // Show all courses
                return;
            }
            
            $('.course-item').each(function() {
                const $course = $(this);
                const age = $course.data('age');
                const status = $course.data('status');
                const days = parseInt($course.data('days'));
                let shouldShow = false;
                
                switch(filter) {
                    case 'new':
                        shouldShow = (days <= 7);
                        break;
                    case 'recent':
                        shouldShow = (days <= 30);
                        break;
                    case 'older':
                        shouldShow = (days > 30);
                        break;
                    case 'published':
                        shouldShow = (status === 'published');
                        break;
                    case 'draft':
                        shouldShow = (status === 'draft');
                        break;
                }
                
                if (!shouldShow) {
                    $course.addClass('hidden').hide();
                }
            });
            
            // Update count display
            updateCourseCount();
        }
        
        function updateCourseCount() {
            const visibleCourses = $('.course-item:visible').length;
            const totalCourses = $('.course-item').length;
            
            // Update any count displays if they exist
            if (visibleCourses !== totalCourses) {
                console.log(`Showing ${visibleCourses} of ${totalCourses} courses`);
            }
        }
        
        // Delete course functionality
        $('.delete-course').click(function(e) {
            e.preventDefault();
            const courseId = $(this).data('id');
            const courseTitle = $(this).closest('.course-card').find('.course-title').text();
            
            if (confirm(`Are you sure you want to delete "${courseTitle}"?`)) {
                window.location.href = `courses.php?delete=${courseId}`;
            }
        });
        
        // Add smooth transitions for filtering
        $('.course-item').css({
            'transition': 'all 0.3s ease',
            'opacity': '1'
        });
        
        // Animate course cards on load
        $('.course-item').each(function(index) {
            $(this).css({
                'animation-delay': (index * 0.1) + 's'
            });
        });
        
        // Add tooltips to age badges
        $('.course-age-badge').each(function() {
            const days = $(this).closest('.course-item').data('days');
            let tooltip = '';
            
            if (days <= 7) {
                tooltip = `Created ${days} day${days !== 1 ? 's' : ''} ago`;
            } else if (days <= 30) {
                tooltip = `Created ${days} days ago`;
            }
            
            if (tooltip) {
                $(this).attr('title', tooltip);
            }
        });
    });
    </script>
</body>
</html>
