<?php
require_once __DIR__ . '/../config/db_connect.php';

echo "<h2>🔍 Assessment Management Query Debug</h2>";
echo "<p><strong>Timestamp:</strong> " . date('Y-m-d H:i:s') . "</p><hr>";

// Test the exact query used in assessment_management.php
$assessments_query = "
    SELECT 
        COALESCE(sa.assessment_id, CONCAT('pending_', ar.request_id)) as assessment_id,
        COALESCE(sa.title, CONCAT('SITA Assignment - ', c.title)) as title,
        COALESCE(sa.status, CASE WHEN ar.status = 'assignment_created' THEN 'awaiting_completion' ELSE ar.status END) as status,
        COALESCE(sa.created_date, ar.assignment_created_date) as created_date,
        COALESCE(sa.student_id, ar.student_id) as student_id,
        COALESCE(sa.course_id, ar.course_id) as course_id,
        ar.request_id,
        ar.status as request_status,
        u.first_name, u.last_name, u.email,
        c.title as course_title,
        ar.request_date, ar.assignment_instructions,
        ar.assignment_document_path, ar.documents_submitted_date,
        ar.final_comments, ar.rejection_reason,
        grader.first_name as grader_first_name, grader.last_name as grader_last_name,
        sa.score, sa.feedback, sa.graded_date,
        COUNT(DISTINCT ad.document_id) as document_count,
        GROUP_CONCAT(CONCAT(ad.document_type, ':', ad.verification_status) SEPARATOR ',') as document_statuses
    FROM accreditation_requests ar
    JOIN users u ON ar.student_id = u.user_id
    JOIN courses c ON ar.course_id = c.course_id
    LEFT JOIN sita_assessments sa ON ar.request_id = sa.accreditation_request_id
    LEFT JOIN users grader ON sa.graded_by = grader.user_id
    LEFT JOIN accreditation_documents ad ON ar.request_id = ad.request_id
    WHERE ar.status IN ('assignment_created', 'documents_submitted', 'assessment_passed', 'assessment_failed', 'approved_final', 'certificate_issued', 'rejected')
       OR sa.assessment_id IS NOT NULL
    GROUP BY ar.request_id, sa.assessment_id
    ORDER BY 
        CASE ar.status 
            WHEN 'documents_submitted' THEN 1 
            WHEN 'assignment_created' THEN 2 
            WHEN 'approved_final' THEN 3 
            ELSE 4 
        END,
        COALESCE(ar.documents_submitted_date, sa.created_date, ar.assignment_created_date) DESC
";

echo "<h3>🧪 Step 1: Testing Assessment Query</h3>";

$result = $conn->query($assessments_query);
if ($result) {
    echo "<p><strong>Query executed successfully!</strong></p>";
    echo "<p><strong>Number of rows returned:</strong> " . $result->num_rows . "</p>";
    
    if ($result->num_rows > 0) {
        echo "<h4>📋 Results:</h4>";
        echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
        echo "<tr style='background: #e3f2fd;'>";
        echo "<th>Request ID</th><th>Student</th><th>Course</th><th>Request Status</th><th>Documents</th><th>Created Date</th>";
        echo "</tr>";
        
        while ($row = $result->fetch_assoc()) {
            echo "<tr>";
            echo "<td><strong>" . $row['request_id'] . "</strong></td>";
            echo "<td>" . htmlspecialchars($row['first_name'] . ' ' . $row['last_name']) . "</td>";
            echo "<td>" . htmlspecialchars($row['course_title']) . "</td>";
            echo "<td><span style='padding:0.25rem 0.5rem; border-radius:0.25rem; background: " . 
                 ($row['request_status'] === 'documents_submitted' ? '#d1edff' : '#fff3cd') . ";'>" . 
                 $row['request_status'] . "</span></td>";
            echo "<td>" . $row['document_count'] . "/4</td>";
            echo "<td>" . ($row['created_date'] ? date('M j, Y H:i', strtotime($row['created_date'])) : 'N/A') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<div style='background: #ffebee; padding: 15px; border-left: 4px solid #f44336;'>";
        echo "<h4 style='color: #c62828;'>❌ No Results Found</h4>";
        echo "<p>The query returned 0 rows. Let's check what's in the database...</p>";
        echo "</div>";
    }
} else {
    echo "<div style='background: #ffebee; padding: 15px; border-left: 4px solid #f44336;'>";
    echo "<h4 style='color: #c62828;'>❌ Query Error</h4>";
    echo "<p><strong>Error:</strong> " . $conn->error . "</p>";
    echo "</div>";
}

echo "<h3>🔍 Step 2: Raw Data Check</h3>";

// Check what's actually in accreditation_requests
$raw_query = "SELECT request_id, student_id, course_id, status, documents_submitted_date, assignment_created_date FROM accreditation_requests ORDER BY request_id DESC LIMIT 10";
$raw_result = $conn->query($raw_query);

echo "<h4>📊 Accreditation Requests (Last 10):</h4>";
if ($raw_result && $raw_result->num_rows > 0) {
    echo "<table border='1' style='border-collapse: collapse;'>";
    echo "<tr style='background: #fff3e0;'><th>ID</th><th>Student ID</th><th>Course ID</th><th>Status</th><th>Docs Date</th><th>Assignment Date</th></tr>";
    while ($row = $raw_result->fetch_assoc()) {
        echo "<tr>";
        echo "<td>" . $row['request_id'] . "</td>";
        echo "<td>" . $row['student_id'] . "</td>";
        echo "<td>" . $row['course_id'] . "</td>";
        echo "<td><strong>" . $row['status'] . "</strong></td>";
        echo "<td>" . ($row['documents_submitted_date'] ?? 'NULL') . "</td>";
        echo "<td>" . ($row['assignment_created_date'] ?? 'NULL') . "</td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "<p style='color: red;'>No accreditation requests found!</p>";
}

// Check users table join
echo "<h4>👥 User Join Test:</h4>";
$user_test = "SELECT ar.request_id, ar.status, u.first_name, u.last_name FROM accreditation_requests ar LEFT JOIN users u ON ar.student_id = u.user_id ORDER BY ar.request_id DESC LIMIT 5";
$user_result = $conn->query($user_test);

if ($user_result && $user_result->num_rows > 0) {
    echo "<table border='1' style='border-collapse: collapse;'>";
    echo "<tr style='background: #f3e5f5;'><th>Request ID</th><th>Status</th><th>Student Name</th></tr>";
    while ($row = $user_result->fetch_assoc()) {
        echo "<tr>";
        echo "<td>" . $row['request_id'] . "</td>";
        echo "<td>" . $row['status'] . "</td>";
        echo "<td>" . ($row['first_name'] ? htmlspecialchars($row['first_name'] . ' ' . $row['last_name']) : '<span style="color:red;">USER NOT FOUND</span>') . "</td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "<p style='color: red;'>User join test failed!</p>";
}

// Check courses table join
echo "<h4>📚 Course Join Test:</h4>";
$course_test = "SELECT ar.request_id, ar.status, c.title FROM accreditation_requests ar LEFT JOIN courses c ON ar.course_id = c.course_id ORDER BY ar.request_id DESC LIMIT 5";
$course_result = $conn->query($course_test);

if ($course_result && $course_result->num_rows > 0) {
    echo "<table border='1' style='border-collapse: collapse;'>";
    echo "<tr style='background: #e8f5e8;'><th>Request ID</th><th>Status</th><th>Course Title</th></tr>";
    while ($row = $course_result->fetch_assoc()) {
        echo "<tr>";
        echo "<td>" . $row['request_id'] . "</td>";
        echo "<td>" . $row['status'] . "</td>";
        echo "<td>" . ($row['title'] ? htmlspecialchars($row['title']) : '<span style="color:red;">COURSE NOT FOUND</span>') . "</td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "<p style='color: red;'>Course join test failed!</p>";
}

echo "<h3>🎯 Step 3: Recommendation</h3>";
echo "<div style='background: #e3f2fd; padding: 15px; border-left: 4px solid #2196f3;'>";
echo "<p><strong>If you see Request #6 with status 'assignment_created' above:</strong></p>";
echo "<ol>";
echo "<li><strong>Run the status update script:</strong> <a href='/update_request_status.php' target='_blank'>update_request_status.php</a></li>";
echo "<li><strong>This will change status to 'documents_submitted'</strong></li>";
echo "<li><strong>Then refresh the assessment management page</strong></li>";
echo "</ol>";
echo "</div>";

echo "<hr><p><strong>Generated:</strong> " . date('Y-m-d H:i:s') . "</p>";
?>
