<?php
require_once '../auth/functions.php';
require_once '../config/db_connect.php';
require_once '../config/fast_email_service.php';

// Check if user is admin
requireRole('admin');

$debug_info = [];
$test_results = [];

function addDebugInfo($message, $type = 'info') {
    global $debug_info;
    $debug_info[] = [
        'message' => $message,
        'type' => $type,
        'timestamp' => date('Y-m-d H:i:s')
    ];
}

// Handle email test
if ($_POST && isset($_POST['test_email'])) {
    $testEmail = filter_var($_POST['test_email'], FILTER_VALIDATE_EMAIL);
    
    if (!$testEmail) {
        addDebugInfo('Invalid email address provided', 'error');
    } else {
        addDebugInfo('Starting email debugging for: ' . $testEmail, 'info');
        
        // Test 1: Check email settings
        addDebugInfo('=== TEST 1: Email Settings ===', 'header');
        $settings = getEmailSettings();
        addDebugInfo('SMTP Enabled: ' . ($settings['smtp_enabled'] ? 'YES' : 'NO'), 'info');
        addDebugInfo('SMTP Host: ' . ($settings['smtp_host'] ?: 'Not configured'), 'info');
        addDebugInfo('SMTP Port: ' . $settings['smtp_port'], 'info');
        addDebugInfo('SMTP Username: ' . ($settings['smtp_username'] ?: 'Not configured'), 'info');
        addDebugInfo('From Email: ' . ($settings['from_email'] ?: 'Not configured'), 'info');
        
        // Test 2: Try native PHP mail first
        addDebugInfo('=== TEST 2: Native PHP mail() Function ===', 'header');
        $subject = 'TEST 1 - Native PHP Mail - ' . date('H:i:s');
        $body = 'This is a test using native PHP mail() function. Time: ' . date('Y-m-d H:i:s');
        
        $nativeResult = @mail($testEmail, $subject, $body, 
            "From: test@panaditeacademy.com\r\n" .
            "Content-Type: text/html; charset=UTF-8\r\n" .
            "X-Mailer: PHP/" . phpversion()
        );
        
        addDebugInfo('Native mail() result: ' . ($nativeResult ? 'SUCCESS' : 'FAILED'), 
                     $nativeResult ? 'success' : 'error');
        
        // Test 3: Try your fast email service
        addDebugInfo('=== TEST 3: Fast Email Service ===', 'header');
        $subject2 = 'TEST 2 - Fast Email Service - ' . date('H:i:s');
        $body2 = '<h2>Fast Email Service Test</h2><p>Time: ' . date('Y-m-d H:i:s') . '</p>';
        
        $fastEmailResult = sendEmailNow($testEmail, $subject2, $body2);
        addDebugInfo('Fast Email Service result: ' . ($fastEmailResult ? 'SUCCESS' : 'FAILED'), 
                     $fastEmailResult ? 'success' : 'error');
        
        // Test 4: Try individual SMTP methods if SMTP is enabled
        if ($settings['smtp_enabled']) {
            addDebugInfo('=== TEST 4: Individual SMTP Methods ===', 'header');
            
            // Test SMTPS (port 465)
            $subject3 = 'TEST 3 - SMTPS Method - ' . date('H:i:s');
            $body3 = '<h2>SMTPS Test</h2><p>Testing SMTPS (port 465). Time: ' . date('Y-m-d H:i:s') . '</p>';
            
            try {
                $smtpsResult = sendEmailSMTPS($testEmail, $subject3, $body3, $settings);
                addDebugInfo('SMTPS (port 465) result: ' . ($smtpsResult ? 'SUCCESS' : 'FAILED'), 
                             $smtpsResult ? 'success' : 'error');
            } catch (Exception $e) {
                addDebugInfo('SMTPS Error: ' . $e->getMessage(), 'error');
            }
            
            // Test STARTTLS (port 587)
            $subject4 = 'TEST 4 - STARTTLS Method - ' . date('H:i:s');
            $body4 = '<h2>STARTTLS Test</h2><p>Testing STARTTLS (port 587). Time: ' . date('Y-m-d H:i:s') . '</p>';
            
            try {
                $starttlsResult = sendEmailSMTP($testEmail, $subject4, $body4, $settings);
                addDebugInfo('STARTTLS (port 587) result: ' . ($starttlsResult ? 'SUCCESS' : 'FAILED'), 
                             $starttlsResult ? 'success' : 'error');
            } catch (Exception $e) {
                addDebugInfo('STARTTLS Error: ' . $e->getMessage(), 'error');
            }
        }
        
        // Test 5: Check server capabilities
        addDebugInfo('=== TEST 5: Server Information ===', 'header');
        addDebugInfo('PHP Version: ' . PHP_VERSION, 'info');
        addDebugInfo('Mail function available: ' . (function_exists('mail') ? 'YES' : 'NO'), 'info');
        addDebugInfo('OpenSSL loaded: ' . (extension_loaded('openssl') ? 'YES' : 'NO'), 'info');
        addDebugInfo('Socket functions: ' . (function_exists('fsockopen') ? 'YES' : 'NO'), 'info');
        
        // Test 6: Check email logs
        addDebugInfo('=== TEST 6: Email Logs ===', 'header');
        if (isset($pdo)) {
            try {
                $stmt = $pdo->prepare("SELECT * FROM email_logs ORDER BY created_at DESC LIMIT 5");
                $stmt->execute();
                $logs = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if ($logs) {
                    foreach ($logs as $log) {
                        addDebugInfo("Log: {$log['to_email']} - {$log['status']} - {$log['created_at']}", 'info');
                    }
                } else {
                    addDebugInfo('No email logs found', 'warning');
                }
            } catch (Exception $e) {
                addDebugInfo('Email logs table not accessible: ' . $e->getMessage(), 'warning');
            }
        }
        
        addDebugInfo('=== EMAIL DEBUGGING COMPLETE ===', 'header');
        addDebugInfo('Check your email inbox (including spam/junk folder) for test messages', 'warning');
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Email Debugging - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .debug-container {
            max-width: 900px;
            margin: 30px auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.2);
            overflow: hidden;
        }
        .header {
            background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
            color: white;
            text-align: center;
            padding: 30px;
        }
        .debug-log {
            max-height: 600px;
            overflow-y: auto;
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
        }
        .log-entry {
            padding: 10px 15px;
            border-bottom: 1px solid #e9ecef;
            font-family: 'Courier New', monospace;
            font-size: 0.9rem;
        }
        .log-entry:last-child {
            border-bottom: none;
        }
        .log-header {
            background: #007bff;
            color: white;
            font-weight: bold;
        }
        .log-info {
            background: #ffffff;
            color: #333;
        }
        .log-success {
            background: #d1edff;
            color: #0c5460;
            border-left: 4px solid #17a2b8;
        }
        .log-error {
            background: #f8d7da;
            color: #721c24;
            border-left: 4px solid #dc3545;
        }
        .log-warning {
            background: #fff3cd;
            color: #856404;
            border-left: 4px solid #ffc107;
        }
        .form-control {
            border-radius: 10px;
            border: 2px solid #e9ecef;
            padding: 12px 15px;
        }
        .btn-debug {
            background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
            border: none;
            border-radius: 10px;
            padding: 12px 30px;
            font-weight: 600;
            color: white;
        }
    </style>
</head>
<body>
    <div class="debug-container">
        <div class="header">
            <i class="bx bx-bug bx-lg mb-2"></i>
            <h1>Email Debugging Tool</h1>
            <p class="mb-0">Test and debug your email sending functionality</p>
        </div>
        
        <div class="p-4">
            <div class="alert alert-info">
                <i class="bx bx-info-circle me-2"></i>
                <strong>This tool will:</strong>
                <ul class="mb-0 mt-2">
                    <li>Test multiple email sending methods</li>
                    <li>Check your SMTP configuration</li>
                    <li>Provide detailed debugging information</li>
                    <li>Send actual test emails to verify delivery</li>
                </ul>
            </div>
            
            <form method="POST" class="mb-4">
                <div class="row align-items-end">
                    <div class="col-md-8">
                        <label for="test_email" class="form-label fw-bold">
                            <i class="bx bx-envelope me-2"></i>Test Email Address
                        </label>
                        <input type="email" class="form-control" id="test_email" name="test_email" 
                               placeholder="Enter email to test delivery" required 
                               value="<?= htmlspecialchars($_POST['test_email'] ?? '') ?>">
                    </div>
                    <div class="col-md-4">
                        <button type="submit" class="btn btn-debug btn-lg w-100">
                            <i class="bx bx-play me-2"></i>Start Debug Test
                        </button>
                    </div>
                </div>
            </form>
            
            <?php if (!empty($debug_info)): ?>
            <div class="debug-log">
                <?php foreach ($debug_info as $entry): ?>
                <div class="log-entry log-<?= $entry['type'] ?>">
                    <span class="text-muted">[<?= $entry['timestamp'] ?>]</span> <?= htmlspecialchars($entry['message']) ?>
                </div>
                <?php endforeach; ?>
            </div>
            
            <div class="mt-4 alert alert-warning">
                <i class="bx bx-time me-2"></i>
                <strong>Important:</strong> Check your email inbox (and spam/junk folder) for test messages. 
                Email delivery can take 1-5 minutes depending on your email provider.
            </div>
            <?php endif; ?>
            
            <div class="text-center mt-4">
                <a href="test_certificate_email.php" class="btn btn-outline-primary me-2">
                    <i class="bx bx-envelope me-2"></i>Certificate Email Test
                </a>
                <a href="certificate_templates.php" class="btn btn-outline-secondary">
                    <i class="bx bx-arrow-back me-2"></i>Back to Templates
                </a>
            </div>
        </div>
    </div>
</body>
</html>
