<?php
require_once __DIR__ . '/../auth/functions.php';
requireRole('admin');

echo "<!DOCTYPE html>
<html lang='en'>
<head>
    <meta charset='UTF-8'>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <title>Payment Tables Debug - Panadite Academy</title>
    <link href='https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css' rel='stylesheet'>
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <style>
        body { background: #f8f9fa; font-family: Arial, sans-serif; }
        .debug-container { max-width: 1400px; margin: 2rem auto; padding: 0 1rem; }
        .debug-card { background: white; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); margin-bottom: 2rem; padding: 1.5rem; }
        .table-name { color: #007bff; font-weight: bold; font-size: 1.1rem; }
        .revenue-highlight { background: #e8f5e9; padding: 1rem; border-radius: 8px; margin: 1rem 0; }
        .error-highlight { background: #ffebee; padding: 1rem; border-radius: 8px; margin: 1rem 0; }
        .success-highlight { background: #e8f5e9; padding: 1rem; border-radius: 8px; margin: 1rem 0; }
        .warning-highlight { background: #fff3e0; padding: 1rem; border-radius: 8px; margin: 1rem 0; }
        pre { background: #f5f5f5; padding: 1rem; border-radius: 5px; overflow-x: auto; }
        .summary-card { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; }
    </style>
</head>
<body>
<div class='debug-container'>
    <div class='debug-card summary-card'>
        <h1><i class='bx bx-bug'></i> Payment Tables Debug Report</h1>
        <p class='mb-0'>Comprehensive analysis of all payment-related tables and revenue calculations</p>
        <small>Generated: " . date('Y-m-d H:i:s') . "</small>
    </div>";

// Debug all payment-related tables
$tables_to_debug = [
    'course_purchases',
    'video_purchases', 
    'enrollments',
    'payments',
    'transactions',
    'withdrawal_requests',
    'teacher_earnings',
    'teacher_payments',
    'student_payments'
];

$total_revenue_sources = [];

foreach ($tables_to_debug as $table) {
    echo "<div class='debug-card'>";
    echo "<h3 class='table-name'><i class='bx bx-table'></i> Table: {$table}</h3>";
    
    // Check if table exists
    $check_query = "SHOW TABLES LIKE '{$table}'";
    $result = $conn->query($check_query);
    
    if (!$result || $result->num_rows == 0) {
        echo "<div class='warning-highlight'>";
        echo "<i class='bx bx-info-circle'></i> <strong>Table does not exist</strong>";
        echo "</div>";
        continue;
    }
    
    // Show table structure
    echo "<h5>Table Structure:</h5>";
    $structure_query = "DESCRIBE {$table}";
    $structure_result = $conn->query($structure_query);
    
    if ($structure_result) {
        echo "<div class='table-responsive'>";
        echo "<table class='table table-sm table-bordered'>";
        echo "<thead><tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr></thead><tbody>";
        
        while ($column = $structure_result->fetch_assoc()) {
            echo "<tr>";
            echo "<td><code>{$column['Field']}</code></td>";
            echo "<td>{$column['Type']}</td>";
            echo "<td>{$column['Null']}</td>";
            echo "<td>{$column['Key']}</td>";
            echo "<td>{$column['Default']}</td>";
            echo "<td>{$column['Extra']}</td>";
            echo "</tr>";
        }
        echo "</tbody></table></div>";
    }
    
    // Show record count
    $count_query = "SELECT COUNT(*) as total FROM {$table}";
    $count_result = $conn->query($count_query);
    $total_records = $count_result ? $count_result->fetch_assoc()['total'] : 0;
    
    echo "<div class='success-highlight'>";
    echo "<i class='bx bx-list-ul'></i> <strong>Total Records:</strong> {$total_records}";
    echo "</div>";
    
    if ($total_records > 0) {
        // Show sample data (first 5 records)
        echo "<h5>Sample Data (First 5 Records):</h5>";
        $sample_query = "SELECT * FROM {$table} LIMIT 5";
        $sample_result = $conn->query($sample_query);
        
        if ($sample_result && $sample_result->num_rows > 0) {
            echo "<div class='table-responsive'>";
            echo "<table class='table table-sm table-striped'>";
            
            // Table headers
            $fields = $sample_result->fetch_fields();
            echo "<thead><tr>";
            foreach ($fields as $field) {
                echo "<th>{$field->name}</th>";
            }
            echo "</tr></thead><tbody>";
            
            // Reset result pointer
            $sample_result->data_seek(0);
            while ($row = $sample_result->fetch_assoc()) {
                echo "<tr>";
                foreach ($row as $value) {
                    echo "<td>" . htmlspecialchars(substr(strval($value), 0, 50)) . (strlen(strval($value)) > 50 ? '...' : '') . "</td>";
                }
                echo "</tr>";
            }
            echo "</tbody></table></div>";
        }
        
        // Calculate revenue if amount column exists
        $amount_columns = ['amount', 'price', 'total', 'cost'];
        foreach ($amount_columns as $amount_col) {
            $check_col_query = "SHOW COLUMNS FROM {$table} LIKE '{$amount_col}'";
            $col_result = $conn->query($check_col_query);
            
            if ($col_result && $col_result->num_rows > 0) {
                // Calculate total revenue from this table
                $revenue_query = "SELECT SUM({$amount_col}) as total_revenue FROM {$table}";
                
                // Add status filter if status column exists
                $status_check = $conn->query("SHOW COLUMNS FROM {$table} LIKE 'status'");
                if ($status_check && $status_check->num_rows > 0) {
                    $revenue_query = "SELECT 
                        SUM({$amount_col}) as total_revenue,
                        SUM(CASE WHEN status = 'completed' THEN {$amount_col} ELSE 0 END) as completed_revenue,
                        COUNT(*) as total_records,
                        COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_records
                    FROM {$table}";
                }
                
                $revenue_result = $conn->query($revenue_query);
                if ($revenue_result) {
                    $revenue_data = $revenue_result->fetch_assoc();
                    
                    echo "<div class='revenue-highlight'>";
                    echo "<h6><i class='bx bx-money'></i> Revenue Calculation from '{$amount_col}' column:</h6>";
                    
                    if (isset($revenue_data['completed_revenue'])) {
                        echo "<strong>Total Revenue (All):</strong> R" . number_format($revenue_data['total_revenue'] ?: 0, 2) . "<br>";
                        echo "<strong>Completed Revenue:</strong> R" . number_format($revenue_data['completed_revenue'] ?: 0, 2) . "<br>";
                        echo "<strong>Total Records:</strong> {$revenue_data['total_records']}<br>";
                        echo "<strong>Completed Records:</strong> {$revenue_data['completed_records']}";
                        
                        $total_revenue_sources["{$table}_{$amount_col}_completed"] = $revenue_data['completed_revenue'] ?: 0;
                        $total_revenue_sources["{$table}_{$amount_col}_all"] = $revenue_data['total_revenue'] ?: 0;
                    } else {
                        echo "<strong>Total Revenue:</strong> R" . number_format($revenue_data['total_revenue'] ?: 0, 2);
                        $total_revenue_sources["{$table}_{$amount_col}"] = $revenue_data['total_revenue'] ?: 0;
                    }
                    echo "</div>";
                }
                break; // Only use first amount column found
            }
        }
    }
    
    echo "</div>"; // End debug-card
}

// Summary of all revenue sources
echo "<div class='debug-card'>";
echo "<h3><i class='bx bx-calculator'></i> Revenue Sources Summary</h3>";

if (empty($total_revenue_sources)) {
    echo "<div class='error-highlight'>";
    echo "<i class='bx bx-error-circle'></i> <strong>No revenue sources found!</strong>";
    echo "</div>";
} else {
    echo "<div class='table-responsive'>";
    echo "<table class='table table-striped'>";
    echo "<thead><tr><th>Source</th><th>Amount (R)</th></tr></thead><tbody>";
    
    $grand_total = 0;
    foreach ($total_revenue_sources as $source => $amount) {
        echo "<tr>";
        echo "<td><code>{$source}</code></td>";
        echo "<td class='text-end'><strong>R" . number_format($amount, 2) . "</strong></td>";
        echo "</tr>";
        
        // Add to grand total if it's a 'completed' source or doesn't have status distinction
        if (strpos($source, '_completed') !== false || strpos($source, '_all') === false) {
            $grand_total += $amount;
        }
    }
    
    echo "</tbody></table></div>";
    
    echo "<div class='revenue-highlight'>";
    echo "<h5><i class='bx bx-trending-up'></i> Recommended Total Revenue: R" . number_format($grand_total, 2) . "</h5>";
    echo "<small>This combines completed transactions from all payment tables</small>";
    echo "</div>";
}

// Check dashboard calculation method
echo "<h4><i class='bx bx-search'></i> Dashboard vs Payments Page Analysis</h4>";
echo "<div class='warning-highlight'>";
echo "<strong>Dashboard shows:</strong> R158,900.00<br>";
echo "<strong>Payments page shows:</strong> R15,400.00<br>";
echo "<strong>Difference:</strong> R" . number_format(158900 - 15400, 2);
echo "</div>";

// Provide recommendations
echo "<div class='success-highlight'>";
echo "<h5><i class='bx bx-lightbulb'></i> Recommendations:</h5>";
echo "<ol>";
echo "<li>Check if the dashboard is reading from a different table or using different filters</li>";
echo "<li>Verify that both systems are using the same status conditions (e.g., 'completed' vs 'approved')</li>";
echo "<li>Ensure decimal/float precision isn't causing rounding issues</li>";
echo "<li>Check for any cached revenue values that might be outdated</li>";
echo "<li>Verify the date range filters being applied in both calculations</li>";
echo "</ol>";
echo "</div>";

echo "</div>"; // End summary debug-card

// Generate SQL fixes if needed
echo "<div class='debug-card'>";
echo "<h3><i class='bx bx-wrench'></i> Debugging Queries</h3>";
echo "<p>Run these queries manually to investigate further:</p>";

echo "<h5>1. Check Course Purchases Details:</h5>";
echo "<pre>SELECT status, COUNT(*) as count, SUM(amount) as total FROM course_purchases GROUP BY status;</pre>";

echo "<h5>2. Check Video Purchases Details:</h5>";
echo "<pre>SELECT status, COUNT(*) as count, SUM(amount) as total FROM video_purchases GROUP BY status;</pre>";

echo "<h5>3. Find All Tables with 'amount' columns:</h5>";
echo "<pre>SELECT TABLE_NAME, COLUMN_NAME FROM information_schema.COLUMNS 
WHERE TABLE_SCHEMA = DATABASE() 
AND COLUMN_NAME IN ('amount', 'price', 'total', 'cost');</pre>";

echo "<h5>4. Check for any 'transactions' or 'payments' tables:</h5>";
echo "<pre>SHOW TABLES LIKE '%payment%';
SHOW TABLES LIKE '%transaction%';</pre>";

echo "</div>";

echo "</div>"; // End debug-container
echo "</body></html>";
?>
