<?php
require_once '../auth/functions.php';
require_once '../config/db_connect.php';

// Check if user is admin
requireRole('admin');

$debug_results = [];
$quiz_stats = [];
$marking_issues = [];

try {
    // 1. Check quiz marking tables structure
    $debug_results['table_structure'] = [];
    
    $tables_to_check = ['quizzes', 'quiz_questions', 'quiz_attempts', 'quiz_answers', 'question_options'];
    
    foreach ($tables_to_check as $table) {
        $result = $conn->query("DESCRIBE {$table}");
        if ($result) {
            $debug_results['table_structure'][$table] = $result->fetch_all(MYSQLI_ASSOC);
        } else {
            $marking_issues[] = "Table '{$table}' not found or inaccessible";
        }
    }
    
    // 2. Analyze quiz marking statistics
    $quiz_stats_query = "
        SELECT 
            q.quiz_id,
            q.title as quiz_title,
            c.title as course_title,
            COUNT(DISTINCT qa.attempt_id) as total_attempts,
            COUNT(DISTINCT qa.student_id) as unique_students,
            AVG(qa.score) as avg_score,
            MAX(qa.score) as max_score,
            MIN(qa.score) as min_score,
            COUNT(CASE WHEN qa.score >= q.passing_score THEN 1 END) as passed_attempts,
            q.passing_score,
            COUNT(DISTINCT qq.question_id) as total_questions
        FROM quizzes q
        LEFT JOIN courses c ON q.course_id = c.course_id
        LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id
        LEFT JOIN quiz_questions qq ON q.quiz_id = qq.quiz_id
        GROUP BY q.quiz_id
        ORDER BY total_attempts DESC
        LIMIT 20
    ";
    
    $quiz_stats_result = $conn->query($quiz_stats_query);
    if ($quiz_stats_result) {
        $quiz_stats = $quiz_stats_result->fetch_all(MYSQLI_ASSOC);
    }
    
    // 3. Check for marking inconsistencies
    $inconsistency_query = "
        SELECT 
            qa.attempt_id,
            qa.quiz_id,
            qa.score as attempt_score,
            SUM(qan.points_earned) as calculated_score,
            (qa.score - SUM(qan.points_earned)) as score_difference,
            COUNT(qan.question_id) as answered_questions,
            u.first_name,
            u.last_name,
            q.title as quiz_title
        FROM quiz_attempts qa
        LEFT JOIN quiz_answers qan ON qa.attempt_id = qan.attempt_id
        LEFT JOIN users u ON qa.student_id = u.user_id
        LEFT JOIN quizzes q ON qa.quiz_id = q.quiz_id
        GROUP BY qa.attempt_id
        HAVING ABS(score_difference) > 0.1
        ORDER BY ABS(score_difference) DESC
        LIMIT 10
    ";
    
    $inconsistency_result = $conn->query($inconsistency_query);
    $marking_inconsistencies = [];
    if ($inconsistency_result) {
        $marking_inconsistencies = $inconsistency_result->fetch_all(MYSQLI_ASSOC);
    }
    
    // 4. Analyze question types and marking accuracy
    $question_types_query = "
        SELECT 
            qq.question_type,
            COUNT(qq.question_id) as question_count,
            COUNT(DISTINCT qq.quiz_id) as quizzes_using,
            AVG(qan.points_earned) as avg_points_earned,
            AVG(qq.points) as avg_points_possible,
            (AVG(qan.points_earned) / AVG(qq.points)) * 100 as avg_success_rate
        FROM quiz_questions qq
        LEFT JOIN quiz_answers qan ON qq.question_id = qan.question_id
        GROUP BY qq.question_type
        ORDER BY question_count DESC
    ";
    
    $question_types_result = $conn->query($question_types_query);
    $question_type_stats = [];
    if ($question_types_result) {
        $question_type_stats = $question_types_result->fetch_all(MYSQLI_ASSOC);
    }
    
    // 5. Check for recent quiz attempts with detailed marking
    $recent_attempts_query = "
        SELECT 
            qa.attempt_id,
            qa.quiz_id,
            qa.student_id,
            qa.score,
            qa.attempt_date,
            u.first_name,
            u.last_name,
            q.title as quiz_title,
            c.title as course_title,
            COUNT(qan.question_id) as questions_answered,
            SUM(qan.points_earned) as total_points_earned,
            SUM(qq.points) as total_points_possible
        FROM quiz_attempts qa
        LEFT JOIN users u ON qa.student_id = u.user_id
        LEFT JOIN quizzes q ON qa.quiz_id = q.quiz_id
        LEFT JOIN courses c ON q.course_id = c.course_id
        LEFT JOIN quiz_answers qan ON qa.attempt_id = qan.attempt_id
        LEFT JOIN quiz_questions qq ON qan.question_id = qq.question_id
        WHERE qa.attempt_date >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY qa.attempt_id
        ORDER BY qa.attempt_date DESC
        LIMIT 15
    ";
    
    $recent_attempts_result = $conn->query($recent_attempts_query);
    $recent_attempts = [];
    if ($recent_attempts_result) {
        $recent_attempts = $recent_attempts_result->fetch_all(MYSQLI_ASSOC);
    }
    
    // 6. Check for ungraded or partially graded quizzes
    $ungraded_query = "
        SELECT 
            qa.attempt_id,
            qa.quiz_id,
            qa.student_id,
            qa.score,
            q.title as quiz_title,
            u.first_name,
            u.last_name,
            COUNT(qq.question_id) as total_questions,
            COUNT(qan.question_id) as answered_questions,
            SUM(CASE WHEN qan.is_correct IS NULL THEN 1 ELSE 0 END) as ungraded_questions
        FROM quiz_attempts qa
        LEFT JOIN quizzes q ON qa.quiz_id = q.quiz_id
        LEFT JOIN users u ON qa.student_id = u.user_id
        LEFT JOIN quiz_questions qq ON q.quiz_id = qq.quiz_id
        LEFT JOIN quiz_answers qan ON qa.attempt_id = qan.attempt_id AND qq.question_id = qan.question_id
        GROUP BY qa.attempt_id
        HAVING ungraded_questions > 0 OR answered_questions < total_questions
        ORDER BY qa.attempt_date DESC
        LIMIT 10
    ";
    
    $ungraded_result = $conn->query($ungraded_query);
    $ungraded_attempts = [];
    if ($ungraded_result) {
        $ungraded_attempts = $ungraded_result->fetch_all(MYSQLI_ASSOC);
    }

} catch (Exception $e) {
    $marking_issues[] = "Database error: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Quiz Marking System Debug - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .debug-container {
            max-width: 1400px;
            margin: 20px auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.2);
            overflow: hidden;
        }
        .header {
            background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
            color: white;
            text-align: center;
            padding: 30px;
        }
        .debug-section {
            margin: 20px;
            padding: 20px;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            background: #f8f9fa;
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin: 20px 0;
        }
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 10px;
            border-left: 4px solid #28a745;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .issue-card {
            background: #fff5f5;
            border-left-color: #dc3545;
            border: 1px solid #fed7d7;
        }
        .success-card {
            background: #f0fff4;
            border-left-color: #28a745;
        }
        .warning-card {
            background: #fffbf0;
            border-left-color: #ffc107;
        }
        .table-responsive {
            max-height: 400px;
            overflow-y: auto;
        }
        .json-display {
            background: #282c34;
            color: #abb2bf;
            padding: 15px;
            border-radius: 8px;
            font-family: 'Courier New', monospace;
            font-size: 12px;
            overflow-x: auto;
            max-height: 300px;
            overflow-y: auto;
        }
    </style>
</head>
<body>
    <div class="debug-container">
        <div class="header">
            <i class="bx bx-bug bx-lg mb-2"></i>
            <h1>Quiz Marking System Debug</h1>
            <p class="mb-0">Comprehensive analysis of quiz grading and marking system</p>
            <small>Generated: <?= date('Y-m-d H:i:s') ?></small>
        </div>
        
        <div class="p-4">
            <!-- Issues Summary -->
            <?php if (!empty($marking_issues)): ?>
            <div class="debug-section issue-card">
                <h4><i class="bx bx-error text-danger me-2"></i>Critical Issues Found</h4>
                <?php foreach ($marking_issues as $issue): ?>
                <div class="alert alert-danger mb-2">
                    <i class="bx bx-x-circle me-2"></i><?= htmlspecialchars($issue) ?>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
            
            <!-- Quiz Statistics -->
            <div class="debug-section success-card">
                <h4><i class="bx bx-chart text-success me-2"></i>Quiz Marking Statistics</h4>
                <div class="stats-grid">
                    <div class="stat-card">
                        <h5>Total Quizzes</h5>
                        <div class="h2 text-primary"><?= count($quiz_stats) ?></div>
                    </div>
                    <div class="stat-card">
                        <h5>Active Attempts</h5>
                        <div class="h2 text-info"><?= array_sum(array_column($quiz_stats, 'total_attempts')) ?></div>
                    </div>
                    <div class="stat-card">
                        <h5>Unique Students</h5>
                        <div class="h2 text-warning"><?= array_sum(array_column($quiz_stats, 'unique_students')) ?></div>
                    </div>
                    <div class="stat-card">
                        <h5>Average Success</h5>
                        <div class="h2 text-success">
                            <?php
                            $overall_avg = count($quiz_stats) > 0 ? array_sum(array_column($quiz_stats, 'avg_score')) / count($quiz_stats) : 0;
                            echo number_format($overall_avg, 1) . '%';
                            ?>
                        </div>
                    </div>
                </div>
                
                <?php if (!empty($quiz_stats)): ?>
                <div class="table-responsive">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Quiz</th>
                                <th>Course</th>
                                <th>Attempts</th>
                                <th>Students</th>
                                <th>Avg Score</th>
                                <th>Pass Rate</th>
                                <th>Questions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($quiz_stats as $quiz): ?>
                            <tr>
                                <td><?= htmlspecialchars($quiz['quiz_title']) ?></td>
                                <td><?= htmlspecialchars($quiz['course_title']) ?></td>
                                <td><?= $quiz['total_attempts'] ?></td>
                                <td><?= $quiz['unique_students'] ?></td>
                                <td><?= number_format($quiz['avg_score'], 1) ?>%</td>
                                <td>
                                    <?php 
                                    $pass_rate = $quiz['total_attempts'] > 0 ? ($quiz['passed_attempts'] / $quiz['total_attempts']) * 100 : 0;
                                    echo number_format($pass_rate, 1) . '%';
                                    ?>
                                </td>
                                <td><?= $quiz['total_questions'] ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Question Type Analysis -->
            <?php if (!empty($question_type_stats)): ?>
            <div class="debug-section">
                <h4><i class="bx bx-list-check text-info me-2"></i>Question Type Analysis</h4>
                <div class="table-responsive">
                    <table class="table table-bordered">
                        <thead>
                            <tr>
                                <th>Question Type</th>
                                <th>Count</th>
                                <th>Used in Quizzes</th>
                                <th>Avg Points Earned</th>
                                <th>Avg Points Possible</th>
                                <th>Success Rate</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($question_type_stats as $type): ?>
                            <tr>
                                <td>
                                    <span class="badge bg-primary"><?= htmlspecialchars($type['question_type']) ?></span>
                                </td>
                                <td><?= $type['question_count'] ?></td>
                                <td><?= $type['quizzes_using'] ?></td>
                                <td><?= number_format($type['avg_points_earned'], 2) ?></td>
                                <td><?= number_format($type['avg_points_possible'], 2) ?></td>
                                <td>
                                    <?php 
                                    $success_rate = $type['avg_success_rate'];
                                    $badge_class = $success_rate >= 80 ? 'success' : ($success_rate >= 60 ? 'warning' : 'danger');
                                    ?>
                                    <span class="badge bg-<?= $badge_class ?>"><?= number_format($success_rate, 1) ?>%</span>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Marking Inconsistencies -->
            <?php if (!empty($marking_inconsistencies)): ?>
            <div class="debug-section warning-card">
                <h4><i class="bx bx-error-alt text-warning me-2"></i>Marking Inconsistencies Found</h4>
                <div class="table-responsive">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Student</th>
                                <th>Quiz</th>
                                <th>Recorded Score</th>
                                <th>Calculated Score</th>
                                <th>Difference</th>
                                <th>Questions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($marking_inconsistencies as $inconsistency): ?>
                            <tr>
                                <td><?= htmlspecialchars($inconsistency['first_name'] . ' ' . $inconsistency['last_name']) ?></td>
                                <td><?= htmlspecialchars($inconsistency['quiz_title']) ?></td>
                                <td><?= number_format($inconsistency['attempt_score'], 2) ?></td>
                                <td><?= number_format($inconsistency['calculated_score'], 2) ?></td>
                                <td>
                                    <span class="badge bg-warning"><?= number_format($inconsistency['score_difference'], 2) ?></span>
                                </td>
                                <td><?= $inconsistency['answered_questions'] ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Recent Attempts -->
            <?php if (!empty($recent_attempts)): ?>
            <div class="debug-section">
                <h4><i class="bx bx-time text-primary me-2"></i>Recent Quiz Attempts (Last 30 Days)</h4>
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Student</th>
                                <th>Quiz</th>
                                <th>Course</th>
                                <th>Score</th>
                                <th>Questions</th>
                                <th>Points</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_attempts as $attempt): ?>
                            <tr>
                                <td><?= date('M d, Y H:i', strtotime($attempt['attempt_date'])) ?></td>
                                <td><?= htmlspecialchars($attempt['first_name'] . ' ' . $attempt['last_name']) ?></td>
                                <td><?= htmlspecialchars($attempt['quiz_title']) ?></td>
                                <td><?= htmlspecialchars($attempt['course_title']) ?></td>
                                <td>
                                    <?php 
                                    $score = $attempt['score'];
                                    $badge_class = $score >= 80 ? 'success' : ($score >= 60 ? 'warning' : 'danger');
                                    ?>
                                    <span class="badge bg-<?= $badge_class ?>"><?= number_format($score, 1) ?>%</span>
                                </td>
                                <td><?= $attempt['questions_answered'] ?></td>
                                <td><?= number_format($attempt['total_points_earned'], 1) ?> / <?= number_format($attempt['total_points_possible'], 1) ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Ungraded Attempts -->
            <?php if (!empty($ungraded_attempts)): ?>
            <div class="debug-section issue-card">
                <h4><i class="bx bx-help-circle text-danger me-2"></i>Ungraded or Incomplete Attempts</h4>
                <div class="alert alert-warning">
                    <strong>Action Required:</strong> These attempts have ungraded questions or missing answers.
                </div>
                <div class="table-responsive">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Student</th>
                                <th>Quiz</th>
                                <th>Score</th>
                                <th>Total Questions</th>
                                <th>Answered</th>
                                <th>Ungraded</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($ungraded_attempts as $ungraded): ?>
                            <tr>
                                <td><?= htmlspecialchars($ungraded['first_name'] . ' ' . $ungraded['last_name']) ?></td>
                                <td><?= htmlspecialchars($ungraded['quiz_title']) ?></td>
                                <td><?= number_format($ungraded['score'], 1) ?>%</td>
                                <td><?= $ungraded['total_questions'] ?></td>
                                <td><?= $ungraded['answered_questions'] ?></td>
                                <td>
                                    <span class="badge bg-danger"><?= $ungraded['ungraded_questions'] ?></span>
                                </td>
                                <td>
                                    <button class="btn btn-sm btn-outline-primary" onclick="reviewAttempt(<?= $ungraded['attempt_id'] ?>)">
                                        Review
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Database Structure -->
            <div class="debug-section">
                <h4><i class="bx bx-data text-secondary me-2"></i>Database Structure Analysis</h4>
                <button class="btn btn-outline-secondary mb-3" onclick="toggleStructure()">
                    <i class="bx bx-show me-1"></i>Toggle Database Structure
                </button>
                <div id="dbStructure" style="display: none;">
                    <div class="json-display">
                        <?= json_encode($debug_results['table_structure'], JSON_PRETTY_PRINT) ?>
                    </div>
                </div>
            </div>
            
            <div class="text-center mt-4">
                <a href="certificate_templates.php" class="btn btn-outline-secondary me-2">
                    <i class="bx bx-arrow-back me-1"></i>Back to Admin
                </a>
                <button onclick="location.reload()" class="btn btn-primary">
                    <i class="bx bx-refresh me-1"></i>Refresh Debug
                </button>
            </div>
        </div>
    </div>

    <script>
        function toggleStructure() {
            const element = document.getElementById('dbStructure');
            element.style.display = element.style.display === 'none' ? 'block' : 'none';
        }
        
        function reviewAttempt(attemptId) {
            // Open attempt for manual review
            window.open(`../teacher/teacher_quiz_results.php?attempt_id=${attemptId}`, '_blank');
        }
        
        // Auto-refresh every 5 minutes
        setTimeout(() => {
            location.reload();
        }, 300000);
    </script>
</body>
</html>
