<?php
require_once '../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Check if course ID is provided
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header('Location: courses.php');
    exit();
}

$course_id = $_GET['id'];
$message = '';

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Collect and sanitize input
    $title = trim($_POST['title']);
    $description = trim($_POST['description']);
    $teacher_id = $_POST['teacher_id'];
    $category_id = $_POST['category_id'];
    $price = floatval($_POST['price']);
    $is_published = isset($_POST['is_published']) ? 1 : 0;
    
    // Validate input
    if (empty($title)) {
        $message = '<div class="alert alert-danger">Course title is required.</div>';
    } else {
        // Handle thumbnail upload if provided
        $thumbnail_update = '';
        if (!empty($_FILES['thumbnail']['name'])) {
            $upload_dir = '../uploads/thumbnails/';
            
            // Create directory if it doesn't exist
            if (!file_exists($upload_dir)) {
                mkdir($upload_dir, 0777, true);
            }
            
            $file_ext = strtolower(pathinfo($_FILES['thumbnail']['name'], PATHINFO_EXTENSION));
            $file_name = uniqid() . '_' . time() . '.' . $file_ext;
            $target_file = $upload_dir . $file_name;
            
            // Check file type
            $allowed_types = array('jpg', 'jpeg', 'png', 'gif');
            if (!in_array($file_ext, $allowed_types)) {
                $message = '<div class="alert alert-danger">Only JPG, JPEG, PNG and GIF files are allowed for thumbnails.</div>';
            } 
            // Check file size (max 2MB)
            elseif ($_FILES['thumbnail']['size'] > 2097152) {
                $message = '<div class="alert alert-danger">File size should be less than 2MB.</div>';
            }
            // Upload file
            elseif (move_uploaded_file($_FILES['thumbnail']['tmp_name'], $target_file)) {
                $thumbnail_update = ", thumbnail = '/Panadite academy/uploads/thumbnails/" . $file_name . "'";
                
                // Get old thumbnail to delete later
                $stmt = $conn->prepare("SELECT thumbnail FROM courses WHERE course_id = ?");
                $stmt->bind_param("i", $course_id);
                $stmt->execute();
                $result = $stmt->get_result();
                if ($result->num_rows > 0) {
                    $row = $result->fetch_assoc();
                    $old_thumbnail = $row['thumbnail'];
                    // We'll delete the old file after successful update
                }
            } else {
                $message = '<div class="alert alert-danger">Failed to upload thumbnail.</div>';
            }
        }
        
        // If no error message, proceed with database update
        if (empty($message)) {
            $sql = "UPDATE courses SET 
                    title = ?, 
                    description = ?, 
                    teacher_id = ?, 
                    category_id = ?, 
                    price = ?, 
                    is_published = ? 
                    {$thumbnail_update}
                    WHERE course_id = ?";
            
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("ssiidii", $title, $description, $teacher_id, $category_id, $price, $is_published, $course_id);
            
            if ($stmt->execute()) {
                $message = '<div class="alert alert-success">Course updated successfully.</div>';
                
                // Delete old thumbnail if a new one was uploaded
                if (!empty($thumbnail_update) && !empty($old_thumbnail)) {
                    $old_file = '../' . ltrim($old_thumbnail, '/Panadite academy/');
                    if (file_exists($old_file)) {
                        unlink($old_file);
                    }
                }
            } else {
                $message = '<div class="alert alert-danger">Error updating course: ' . $conn->error . '</div>';
            }
        }
    }
}

// Fetch course data
$stmt = $conn->prepare("SELECT * FROM courses WHERE course_id = ?");
$stmt->bind_param("i", $course_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    header('Location: courses.php');
    exit();
}

$course = $result->fetch_assoc();

// Get all teachers for the form
$teachers = [];
$result = $conn->query("SELECT user_id, username, first_name, last_name FROM users WHERE user_role = 'teacher' ORDER BY first_name, last_name");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $teachers[] = $row;
    }
}

// Get all categories for the form
$categories = [];
$result = $conn->query("SELECT category_id, category_name FROM categories ORDER BY category_name");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $categories[] = $row;
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Course - Admin Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <style>
        /* Override for edit course page */
        .main-content {
            padding-top: 60px !important; /* Exact navbar height */
            margin-top: -10px !important;
        }
        
        .dashboard-content {
            padding: 0 15px !important;
            margin-top: 3rem !important;
        }
        
        .page-header {
            margin-top: 0 !important;
            margin-bottom: 10px !important;
            padding-top: 0 !important;
        }
        
        h1.h2 {
            margin-top: 0 !important;
            margin-bottom: 5px !important;
        }
        
        .edit-course-form {
            background-color: #fff;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            padding: 20px;
            margin-bottom: 30px;
            animation: fadeIn 0.6s ease-in-out;
        }
        
        .form-section {
            margin-bottom: 25px;
        }
        
        .form-section-header {
            border-bottom: 1px solid #eee;
            padding-bottom: 10px;
            margin-bottom: 20px;
            font-weight: 500;
            color: var(--primary-color);
            display: flex;
            align-items: center;
        }
        
        .form-section-header i {
            margin-right: 8px;
            font-size: 1.1rem;
        }
        
        .course-thumbnail-preview {
            width: 100%;
            height: 200px;
            border-radius: 8px;
            overflow: hidden;
            margin-bottom: 15px;
            position: relative;
            background-color: #f8f9fa;
            border: 1px solid #eee;
        }
        
        .course-thumbnail-preview img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.3s ease;
        }
        
        .course-thumbnail-preview:hover img {
            transform: scale(1.05);
        }
        
        .current-thumbnail-text {
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            background-color: rgba(0,0,0,0.6);
            color: white;
            padding: 5px 10px;
            font-size: 12px;
            text-align: center;
        }
        
        .thumbnail-placeholder {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 100%;
            width: 100%;
            background-image: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
            color: white;
        }
        
        .thumbnail-placeholder i {
            font-size: 40px;
            margin-bottom: 10px;
        }
        
        .form-buttons {
            display: flex;
            gap: 10px;
            padding-top: 15px;
            border-top: 1px solid #eee;
        }
        
        .form-buttons .btn {
            padding: 10px 20px;
            border-radius: 6px;
            display: flex;
            align-items: center;
        }
        
        .form-buttons .btn i {
            margin-right: 8px;
        }
        
        /* Form control styling */
        .form-control:focus, .form-select:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 0.2rem rgba(var(--primary-rgb), 0.15);
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .course-thumbnail-preview {
                height: 180px;
            }
        }
        
        /* Animation */
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
    </style>
</head>
<body>
    <div class="wrapper">
        <!-- Sidebar -->
        <?php include 'components/sidebar.php'; ?>

        <!-- Main Content -->
        <div class="main-content">
            <?php include 'components/navbar.php'; ?>
            
            <div class="dashboard-content">
                <div class="page-header d-flex justify-content-between flex-wrap align-items-center">
                    <h1 class="h2">Edit Course</h1>
                    <div class="action-buttons">
                        <a href="courses.php" class="btn btn-outline-secondary d-flex align-items-center">
                            <i class='bx bx-arrow-back me-1'></i> Back to Courses
                        </a>
                    </div>
                </div>
                
                <?php echo $message; ?>
                
                <div class="edit-course-form">
                    <form action="edit_course.php?id=<?php echo $course_id; ?>" method="post" enctype="multipart/form-data">
                        <!-- Basic Information Section -->
                        <div class="form-section">
                            <div class="form-section-header">
                                <i class='bx bxs-info-circle'></i> Basic Information
                            </div>
                            <div class="row">
                                <div class="col-md-8">
                                    <div class="mb-3">
                                        <label for="title" class="form-label">Course Title</label>
                                        <input type="text" class="form-control" id="title" name="title" value="<?php echo htmlspecialchars($course['title']); ?>" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="description" class="form-label">Description</label>
                                        <textarea class="form-control" id="description" name="description" rows="5" placeholder="Enter a detailed description of the course..."><?php echo htmlspecialchars($course['description']); ?></textarea>
                                    </div>
                                </div>
                                
                                <div class="col-md-4">
                                    <!-- Current Thumbnail Preview -->
                                    <div class="course-thumbnail-preview">
                                        <?php if (!empty($course['thumbnail'])): ?>
                                            <img src="<?php echo $course['thumbnail']; ?>" alt="<?php echo htmlspecialchars($course['title']); ?>">
                                            <div class="current-thumbnail-text">Current Thumbnail</div>
                                        <?php else: ?>
                                            <div class="thumbnail-placeholder">
                                                <i class='bx bx-image'></i>
                                                <span>No Thumbnail</span>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="thumbnail" class="form-label">Update Thumbnail</label>
                                        <input type="file" class="form-control" id="thumbnail" name="thumbnail" accept="image/*">
                                        <div class="form-text">Recommended size: 1280x720px (16:9 ratio)</div>
                                        <div class="form-text">Leave blank to keep current thumbnail</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Classification Section -->
                        <div class="form-section">
                            <div class="form-section-header">
                                <i class='bx bx-category'></i> Classification & Organization
                            </div>
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="teacher_id" class="form-label">Teacher</label>
                                        <select class="form-select" id="teacher_id" name="teacher_id" required>
                                            <option value="">Select a Teacher</option>
                                            <?php foreach ($teachers as $teacher): ?>
                                                <option value="<?php echo $teacher['user_id']; ?>" <?php echo ($teacher['user_id'] == $course['teacher_id']) ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($teacher['first_name'] . ' ' . $teacher['last_name'] . ' (' . $teacher['username'] . ')'); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="category_id" class="form-label">Category</label>
                                        <select class="form-select" id="category_id" name="category_id" required>
                                            <option value="">Select a Category</option>
                                            <?php foreach ($categories as $category): ?>
                                                <option value="<?php echo $category['category_id']; ?>" <?php echo ($category['category_id'] == $course['category_id']) ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($category['category_name']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="price" class="form-label">Price (ZAR)</label>
                                        <div class="input-group">
                                            <span class="input-group-text">R</span>
                                            <input type="number" class="form-control" id="price" name="price" min="0" step="0.01" value="<?php echo $course['price']; ?>">
                                        </div>
                                        <div class="form-text">Set to 0.00 for free courses</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Publishing Section -->
                        <div class="form-section">
                            <div class="form-section-header">
                                <i class='bx bx-publish'></i> Publishing Options
                            </div>
                            <div class="mb-3 form-check">
                                <input type="checkbox" class="form-check-input" id="is_published" name="is_published" <?php echo $course['is_published'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="is_published">Course is published</label>
                                <div class="form-text">When checked, this course will be visible to all users. Unpublished courses are only visible to admins and the assigned teacher.</div>
                            </div>
                        </div>
                        
                        <div class="form-buttons">
                            <button type="submit" class="btn btn-primary">
                                <i class='bx bx-save'></i> Update Course
                            </button>
                            <a href="courses.php" class="btn btn-outline-secondary">
                                <i class='bx bx-x'></i> Cancel
                            </a>
                            <a href="videos.php?course_id=<?php echo $course_id; ?>" class="btn btn-info ms-auto">
                                <i class='bx bxs-video'></i> Manage Videos
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Initialize tooltips
        $(function () {
            var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'))
            var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl)
            })
        });
        
        // Sidebar toggle functionality
        document.addEventListener('DOMContentLoaded', function() {
            // Desktop sidebar toggle
            const sidebarToggle = document.querySelector('.sidebar-toggle');
            if (sidebarToggle) {
                sidebarToggle.addEventListener('click', function() {
                    document.querySelector('.wrapper').classList.toggle('sidebar-collapsed');
                });
            }
            
            // Mobile sidebar toggle
            const mobileSidebarToggle = document.querySelector('.sidebar-toggle-mobile');
            if (mobileSidebarToggle) {
                mobileSidebarToggle.addEventListener('click', function() {
                    document.querySelector('.wrapper').classList.toggle('sidebar-mobile-open');
                });
            }
        });
    </script>
</body>
</html>
