<?php
require_once '../auth/functions.php';
require_once '../includes/db_connect.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Check if user ID is provided
if (!isset($_GET['id']) || empty($_GET['id'])) {
    header("Location: users.php?error=No user ID provided");
    exit;
}

$user_id = intval($_GET['id']);
$successMessage = '';
$errorMessage = '';

// Get comprehensive user data (same query as teacher profile page for complete data)
$stmt = $conn->prepare("SELECT u.user_id, u.first_name, u.last_name, u.email, u.bio, u.profile_picture,
                       u.cell_number as phone, u.user_role,
                       u.title, u.id_number, u.suburb, u.province, u.communication_method,
                       u.owns_laptop, u.training_preference, u.education_status, u.employment_status,
                       u.heard_about_us, u.referral_person_name, u.job_field_other as experience,
                       u.job_field_interest, u.password, u.email_verified_at, u.created_at,
                       t.teacher_id, t.website, t.social_linkedin, t.social_twitter, t.social_facebook, t.social_youtube,
                       t.expertise as course_title,
                       tp.bio as course_description, tp.qualifications
                       FROM users u 
                       LEFT JOIN teachers t ON u.user_id = t.user_id
                       LEFT JOIN teacher_profiles tp ON t.teacher_id = tp.teacher_id
                       WHERE u.user_id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    // No user found with this ID
    header("Location: users.php?error=User not found");
    exit;
}

$user = $result->fetch_assoc();

// Debug output for page display
$debugOutput = [];

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $debugOutput[] = "🔄 Form submitted for user $user_id";
    $debugOutput[] = "📝 POST data received: " . json_encode($_POST);
    
    // Check all admin users in database (using correct column names)
    $adminCheck = $conn->query("SELECT user_id, email, first_name, last_name FROM users WHERE email LIKE '%admin%' OR user_id = 1");
    if ($adminCheck) {
        $debugOutput[] = "👥 Admin users in database:";
        while ($admin = $adminCheck->fetch_assoc()) {
            $debugOutput[] = "   - ID: {$admin['user_id']}, Email: {$admin['email']}, Name: {$admin['first_name']} {$admin['last_name']}";
        }
    } else {
        $debugOutput[] = "❌ Failed to query admin users: " . $conn->error;
    }
    
    error_log("DEBUG: Form submitted for user $user_id");
    error_log("DEBUG: POST data: " . print_r($_POST, true));
    
    // Validate and sanitize inputs
    $firstName = trim($_POST['first_name']);
    $lastName = trim($_POST['last_name']);
    $email = trim($_POST['email']);
    $userRole = trim($_POST['role']);
    
    $debugOutput[] = "✅ Processed inputs - Name: $firstName $lastName, Email: $email, Role: $userRole";
    error_log("DEBUG: Processed inputs - Name: $firstName $lastName, Email: $email, Role: $userRole");
    
    // Additional user information fields
    $title = trim($_POST['title'] ?? '');
    $cellNumber = trim($_POST['cell_number'] ?? '');
    $idNumber = trim($_POST['id_number'] ?? '');
    $suburb = trim($_POST['suburb'] ?? '');
    $province = trim($_POST['province'] ?? '');
    $communicationMethod = trim($_POST['communication_method'] ?? '');
    $ownsLaptop = trim($_POST['owns_laptop'] ?? '');
    $trainingPreference = trim($_POST['training_preference'] ?? '');
    $learningGoals = trim($_POST['learning_goals'] ?? '');
    
    // Background information fields
    $educationStatus = trim($_POST['education_status'] ?? '');
    $employmentStatus = trim($_POST['employment_status'] ?? '');
    $heardAboutUs = trim($_POST['heard_about_us'] ?? '');
    $referralPersonName = trim($_POST['referral_person_name'] ?? '');
    
    // Teacher-specific fields
    $courseTitle = trim($_POST['course_title'] ?? '');
    $experience = trim($_POST['experience'] ?? '');
    $courseDescription = trim($_POST['course_description'] ?? '');
    $qualifications = trim($_POST['qualifications'] ?? '');
    $website = trim($_POST['website'] ?? '');
    $socialLinkedin = trim($_POST['social_linkedin'] ?? '');
    $socialTwitter = trim($_POST['social_twitter'] ?? '');
    
    $isActive = isset($_POST['is_active']) ? 1 : 0;
    
    // Check if email is already taken by another user
    $checkEmailStmt = $conn->prepare("SELECT user_id FROM users WHERE email = ? AND user_id != ?");
    $checkEmailStmt->bind_param("si", $email, $user_id);
    $checkEmailStmt->execute();
    $emailResult = $checkEmailStmt->get_result();
    
    if ($emailResult->num_rows > 0) {
        $errorMessage = "Email address is already in use by another account.";
    } else {
        // FIXED: Update user with correct schema (removed non-existent is_active column)
        $updateStmt = $conn->prepare("
            UPDATE users SET 
                first_name = ?, last_name = ?, email = ?, user_role = ?, title = ?, 
                cell_number = ?, id_number = ?, suburb = ?, province = ?, 
                communication_method = ?, owns_laptop = ?, 
                education_status = ?, employment_status = ?, 
                heard_about_us = ?, referral_person_name = ? 
            WHERE user_id = ?
        ");
        
        if (!$updateStmt) {
            $errorMessage = "Database error: " . $conn->error;
        } else {
            $updateStmt->bind_param(
                "sssssssssssssssi", 
                $firstName, $lastName, $email, $userRole, $title, 
                $cellNumber, $idNumber, $suburb, $province, 
                $communicationMethod, $ownsLaptop, 
                $educationStatus, $employmentStatus, 
                $heardAboutUs, $referralPersonName, $user_id
            );
            
            if ($updateStmt->execute()) {
                $successMessage = "User information updated successfully!";
                $debugOutput[] = "✅ Main user update executed successfully";
                $debugOutput[] = "📊 Affected rows: " . $updateStmt->affected_rows;
                
                // Debug: Log successful user update
                error_log("DEBUG: User $user_id main update successful");
                
                // If this is a teacher, update teacher-specific tables
            if ($userRole === 'teacher') {
                // Check if teacher record exists
                $checkTeacher = "SELECT teacher_id FROM teachers WHERE user_id = ?";
                $checkStmt = $conn->prepare($checkTeacher);
                $checkStmt->bind_param("i", $user_id);
                $checkStmt->execute();
                $teacherExists = $checkStmt->get_result()->num_rows > 0;
                
                if (!$teacherExists) {
                    // Create basic teacher record if it doesn't exist (admin can edit details later)
                    $createTeacher = "INSERT INTO teachers (user_id, first_name, last_name, bio, expertise) VALUES (?, ?, ?, ?, ?)";
                    $createStmt = $conn->prepare($createTeacher);
                    $teacherBio = 'Welcome to my teaching profile! I am passionate about education and helping students achieve their learning goals.';
                    $teacherExpertise = 'Education Specialist';
                    
                    $createStmt->bind_param("issss", $user_id, $firstName, $lastName, $teacherBio, $teacherExpertise);
                    if ($createStmt->execute()) {
                        $successMessage .= " Teacher profile created!";
                    } else {
                        $errorMessage .= " Error creating teacher profile: " . $createStmt->error;
                    }
                } else {
                    // Update existing teacher record
                    $updateTeacher = "UPDATE teachers SET first_name = ?, last_name = ?, bio = ?, expertise = ?, website = ?, social_linkedin = ?, social_twitter = ?, experience = ?, updated_at = NOW() WHERE user_id = ?";
                    $teacherStmt = $conn->prepare($updateTeacher);
                    $teacherBio = $courseDescription ?: ($user['bio'] ?? 'Welcome to my teaching profile!');
                    $teacherExpertise = $courseTitle ?: 'Education Specialist';
                    
                    $teacherStmt->bind_param("ssssssssi", $firstName, $lastName, $teacherBio, $teacherExpertise, $website, $socialLinkedin, $socialTwitter, $experience, $user_id);
                    if ($teacherStmt->execute()) {
                        $successMessage .= " Teacher profile updated!";
                    } else {
                        $errorMessage .= " Error updating teacher profile: " . $teacherStmt->error;
                    }
                }
                
                // Update/Create teacher_profiles record
                $updateProfile = "INSERT INTO teacher_profiles (user_id, expertise, bio, qualifications, created_at, updated_at) VALUES (?, ?, ?, ?, NOW(), NOW()) ON DUPLICATE KEY UPDATE expertise = VALUES(expertise), bio = VALUES(bio), qualifications = VALUES(qualifications), updated_at = NOW()";
                $profileStmt = $conn->prepare($updateProfile);
                $teacherBio = $courseDescription ?: ($user['bio'] ?? 'Welcome to my teaching profile!');
                $teacherExpertise = $courseTitle ?: 'Education Specialist';
                $profileStmt->bind_param("isss", $user_id, $teacherExpertise, $teacherBio, $qualifications);
                
                if ($profileStmt->execute()) {
                    $successMessage .= " Complete teacher information saved!";
                } else {
                    $errorMessage .= " Error updating teacher profile details: " . $profileStmt->error;
                }
            }
            
                // Refresh user data after update
                $stmt->execute();
                $result = $stmt->get_result();
                $user = $result->fetch_assoc();
            } else {
                $errorMessage = "Error updating user: " . $conn->error;
            }
        }
    }
    
    // Handle password change if provided
    if (!empty($_POST['new_password'])) {
        $debugOutput[] = "🔒 Password update requested for user $user_id";
        $debugOutput[] = "📏 New password length: " . strlen($_POST['new_password']);
        $debugOutput[] = "🔍 Passwords match: " . ($_POST['new_password'] === $_POST['confirm_password'] ? 'YES' : 'NO');
        
        error_log("DEBUG: Password update requested for user $user_id");
        error_log("DEBUG: New password length: " . strlen($_POST['new_password']));
        error_log("DEBUG: Passwords match: " . ($_POST['new_password'] === $_POST['confirm_password'] ? 'YES' : 'NO'));
        
        if ($_POST['new_password'] === $_POST['confirm_password']) {
            $hashedPassword = password_hash($_POST['new_password'], PASSWORD_DEFAULT);
            error_log("DEBUG: Generated password hash: " . $hashedPassword);
            
            $passwordStmt = $conn->prepare("UPDATE users SET password = ? WHERE user_id = ?");
            if (!$passwordStmt) {
                $errorMessage .= " Password update prepare failed: " . $conn->error;
                error_log("DEBUG: Password prepare failed: " . $conn->error);
            } else {
                $debugOutput[] = "🔧 SQL Query: UPDATE users SET password = '[HASH]' WHERE user_id = $user_id";
                $passwordStmt->bind_param("si", $hashedPassword, $user_id);
                
                if ($passwordStmt->execute()) {
                    $affectedRows = $passwordStmt->affected_rows;
                    $debugOutput[] = "📊 Password update affected rows: " . $affectedRows;
                    error_log("DEBUG: Password update affected rows: " . $affectedRows);
                    
                    if ($affectedRows > 0) {
                        $successMessage .= " Password updated successfully!";
                        $debugOutput[] = "✅ Password updated successfully for user $user_id";
                        
                        // Verify the update by checking the new password hash in database
                        $verifyStmt = $conn->prepare("SELECT password FROM users WHERE user_id = ?");
                        $verifyStmt->bind_param("i", $user_id);
                        $verifyStmt->execute();
                        $verifyResult = $verifyStmt->get_result();
                        if ($verifyRow = $verifyResult->fetch_assoc()) {
                            $debugOutput[] = "🔍 Verified: New password hash in DB: " . substr($verifyRow['password'], 0, 20) . "...";
                        }
                        
                        error_log("DEBUG: Password updated successfully for user $user_id");
                    } else {
                        $errorMessage .= " Password update failed - no rows affected";
                        $debugOutput[] = "❌ Password update failed - no rows affected";
                        error_log("DEBUG: Password update failed - no rows affected");
                    }
                } else {
                    $errorMessage .= " Error updating password: " . $conn->error;
                    error_log("DEBUG: Password update execute failed: " . $conn->error);
                }
                $passwordStmt->close();
            }
        } else {
            $errorMessage .= " Passwords do not match.";
            error_log("DEBUG: Passwords do not match");
        }
    } else {
        $debugOutput[] = "ℹ️ No password update requested (empty new_password field)";
        error_log("DEBUG: No password update requested (empty new_password)");
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit User - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        /* Custom styles for Edit User page */
        .dashboard-content {
            background-color: #f8f9fc;
            background-image: url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%234e73df' fill-opacity='0.03'%3E%3Cpath d='M36 34v-4h-2v4h-4v2h4v4h2v-4h4v-2h-4zm0-30V0h-2v4h-4v2h4v4h2V6h4V4h-4zM6 34v-4H4v4H0v2h4v4h2v-4h4v-2H6zM6 4V0H4v4H0v2h4v4h2V6h4V4H6z'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E");
            min-height: calc(100vh - 70px);
            padding: 1.5rem 0;
        }
        
        .user-edit-container {
            background-color: white;
            border-radius: 0.75rem;
            box-shadow: 0 0.2rem 1.5rem rgba(46, 54, 80, 0.08);
            margin: 0.5rem 0.5rem 1.5rem 0.5rem;
            overflow: hidden;
            transition: all 0.3s ease;
            border: 1px solid rgba(225, 230, 240, 0.6);
        }
        
        .user-edit-container:hover {
            box-shadow: 0 0.3rem 2rem rgba(46, 54, 80, 0.1);
            transform: translateY(-2px);
        }
        
        .user-edit-header {
            position: relative;
            padding: 1.75rem 2rem;
            background: linear-gradient(135deg, #4e73df, #6f86e0, #7e95f5);
            color: white;
            display: flex;
            align-items: center;
            gap: 1.5rem;
        }
        
        .user-edit-header .user-avatar {
            display: flex;
            align-items: center;
            justify-content: center;
            width: 68px;
            height: 68px;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.2);
            box-shadow: 0 0.25rem 1rem rgba(0, 0, 0, 0.1);
            border: 3px solid rgba(255, 255, 255, 0.3);
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .user-edit-header .user-avatar::after {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: rgba(255, 255, 255, 0.1);
            transform: rotate(45deg);
            animation: shimmer 3s infinite linear;
            pointer-events: none;
        }
        
        @keyframes shimmer {
            0% { transform: translateX(-50%) rotate(45deg); }
            100% { transform: translateX(50%) rotate(45deg); }
        }
        
        .user-edit-header .user-avatar i {
            font-size: 2.8rem;
            color: white;
            text-shadow: 0 2px 5px rgba(0, 0, 0, 0.2);
        }
        
        .user-edit-header h2 {
            margin: 0;
            font-size: 1.75rem;
            font-weight: 600;
            position: relative;
            z-index: 1;
            text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }
        
        .user-edit-header p {
            margin: 0;
            opacity: 0.85;
            font-size: 0.9rem;
            margin-top: 0.3rem;
            letter-spacing: 0.01rem;
        }
        
        .user-edit-body {
            padding: 1.5rem;
        }
        
        .form-label {
            font-weight: 600;
            color: #4c5267;
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
            letter-spacing: 0.01rem;
        }
        
        .form-control, .form-select {
            border-left: none;
            border-color: #e3e6f0;
            color: #3a3b45;
            padding: 0.6rem 1rem;
            font-size: 0.95rem;
            border-top-right-radius: 8px;
            border-bottom-right-radius: 8px;
            transition: all 0.2s ease;
        }
        
        .form-control:focus, .form-select:focus {
            box-shadow: 0 0 0 0.15rem rgba(78, 115, 223, 0.1);
            border-color: rgba(78, 115, 223, 0.3);
        }
        
        .input-group:focus-within .input-group-text {
            border-color: rgba(78, 115, 223, 0.3);
            background-color: rgba(78, 115, 223, 0.02);
        }
        
        .input-group-text {
            background-color: #f8f9fc;
            border-right: none;
            color: #4e73df;
            transition: all 0.2s ease;
            border-top-left-radius: 8px;
            border-bottom-left-radius: 8px;
            padding-left: 15px;
            padding-right: 15px;
            border-color: #e3e6f0;
        }
        
        .form-switch .form-check-input {
            width: 2.5em;
            height: 1.25em;
        }
        
        .form-check-input:checked {
            background-color: #4e73df;
            border-color: #4e73df;
        }
        
        .form-section-header {
            display: flex;
            align-items: center;
            background: linear-gradient(to right, rgba(78, 115, 223, 0.05), rgba(255,255,255,0));
            padding: 0.9rem 1.5rem;
            border-bottom: 1px solid #e3e6f0;
            border-top-left-radius: 0.75rem;
            border-top-right-radius: 0.75rem;
            gap: 0.75rem;
        }
        
        .form-section-header i {
            font-size: 1.4rem;
            color: #4e73df;
            background: rgba(78, 115, 223, 0.1);
            width: 36px;
            height: 36px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 8px;
            box-shadow: 0 2px 6px rgba(78, 115, 223, 0.08);
        }
        
        .form-section-header h3 {
            margin: 0;
            font-size: 1.15rem;
            font-weight: 600;
            color: #3a3b45;
            letter-spacing: 0.01rem;
        }
        
        .form-section-body {
            padding: 1.75rem 1.5rem;
        }
        
        .action-bar {
            padding: 1rem 1.5rem;
            background-color: #f8f9fc;
            border-top: 1px solid #e9ecef;
            display: flex;
            align-items: center;
            justify-content: flex-end;
            gap: 1rem;
        }
        
        .btn {
            padding: 0.55rem 1.25rem;
            border-radius: 0.5rem;
            font-weight: 500;
            font-size: 0.95rem;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #4e73df, #6f86e0);
            border: none;
            box-shadow: 0 0.25rem 0.75rem rgba(78, 115, 223, 0.18);
            position: relative;
            z-index: 1;
            overflow: hidden;
        }
        
        .btn-primary::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, rgba(255,255,255,0), rgba(255,255,255,0.2), rgba(255,255,255,0));
            transform: skewX(-25deg);
            transition: all 0.75s ease;
            z-index: -1;
        }
        
        .btn-primary:hover::before {
            left: 100%;
        }
        
        .btn-primary:hover {
            transform: translateY(-3px);
            box-shadow: 0 0.5rem 1rem rgba(78, 115, 223, 0.25);
        }
        
        .btn-outline-secondary {
            border-color: #e2e5f3;
            color: #5a5c69;
            background-color: white;
        }
        
        .btn-outline-secondary:hover {
            background-color: #f8f9fc;
            color: #4e73df;
            transform: translateY(-3px);
            border-color: #d8ddf2;
        }
        
        .alert {
            border-radius: 0.75rem;
            border-left-width: 4px;
            font-size: 0.95rem;
            padding: 1rem 1.25rem;
            margin-bottom: 1.5rem;
            box-shadow: 0 0.15rem 1rem rgba(0, 0, 0, 0.05);
            position: relative;
            display: flex;
            align-items: center;
            animation: fadeIn 0.4s ease-out forwards;
        }
        
        .alert:before {
            font-family: 'boxicons';
            font-size: 1.2rem;
            display: inline-block;
            margin-right: 0.75rem;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .alert-success {
            border-left-color: #1cc88a;
            background-color: #f0fdf9;
            color: #117855;
        }
        
        .alert-success:before {
            content: "\ec2b";
            color: #1cc88a;
        }
        
        .alert-danger {
            border-left-color: #e74a3b;
            background-color: #fdf0f0;
            color: #a72a1d;
        }
        
        .alert-danger:before {
            content: "\ed98";
            color: #e74a3b;
        }
        
        .form-text {
            font-size: 0.85rem;
            color: #8e91a0;
            margin-top: 0.5rem;
        }
        
        .required-field::after {
            content: '*';
            color: #e74a3b;
            margin-left: 0.25rem;
            font-size: 0.9rem;
            font-weight: 600;
        }
        
        .password-strength {
            height: 6px;
            border-radius: 6px;
            margin-top: 0.75rem;
            background-color: #e9ecef;
            overflow: hidden;
            box-shadow: inset 0 1px 2px rgba(0,0,0,0.05);
        }
        
        .password-strength-meter {
            height: 100%;
            width: 0%;
            border-radius: 6px;
            transition: width 0.4s cubic-bezier(0.2, 0.8, 0.2, 1.2);
        }
        
        .password-weak {
            background: linear-gradient(90deg, #e74a3b, #e9615a);
            width: 30%;
            box-shadow: 0 0 6px rgba(231, 74, 59, 0.5);
        }
        
        .password-medium {
            background: linear-gradient(90deg, #f6c23e, #ffd54f);
            width: 65%;
            box-shadow: 0 0 6px rgba(246, 194, 62, 0.5);
        }
        
        .password-strong {
            background: linear-gradient(90deg, #1cc88a, #36e9aa);
            width: 100%;
            box-shadow: 0 0 6px rgba(28, 200, 138, 0.5);
        }
        
        /* Page Header Styles */
        .page-header {
            position: relative;
            margin-bottom: 2rem;
            padding: 0.5rem 0.5rem 1.25rem;
            border-bottom: 1px solid rgba(230, 234, 241, 0.7);
        }
        
        .page-header::after {
            content: '';
            position: absolute;
            bottom: -2px;
            left: 0;
            width: 60px;
            height: 4px;
            background: linear-gradient(90deg, #4e73df, #6f86e0);
            border-radius: 2px;
        }
        
        .page-title-container {
            display: flex;
            flex-direction: column;
        }
        
        .page-title {
            font-size: 1.65rem;
            font-weight: 600;
            color: #333;
            margin-bottom: 0.3rem;
            display: flex;
            align-items: center;
        }
        
        .title-icon {
            font-size: 1.75rem;
            margin-right: 0.75rem;
            color: #4e73df;
            background: rgba(78, 115, 223, 0.1);
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(78, 115, 223, 0.12);
        }
        
        .page-subtitle {
            font-size: 0.95rem;
            color: #6c757d;
            margin: 0;
            padding-left: 3.5rem;
        }
        
        .back-btn {
            transition: all 0.2s ease;
            border-radius: 8px;
            padding: 0.5rem 1rem;
        }
        
        .back-btn:hover {
            transform: translateX(-3px);
        }
        
        /* Responsive adjustments */
        @media (max-width: 768px) {
            .page-header {
                padding: 0.5rem 0 0.75rem;
            }
            
            .page-title {
                font-size: 1.35rem;
            }
            
            .page-subtitle {
                padding-left: 3rem;
                font-size: 0.85rem;
            }
            
            .title-icon {
                width: 35px;
                height: 35px;
                font-size: 1.5rem;
                margin-right: 0.5rem;
            }
            
            .back-btn {
                padding: 0.4rem 0.75rem;
                font-size: 0.8rem;
            }
            
            .user-edit-header {
                flex-direction: column;
                text-align: center;
                padding: 1.25rem 1rem;
            }
            
            .user-edit-header .user-avatar {
                margin: 0 auto 1rem;
            }
            
            .action-bar {
                flex-direction: column;
                gap: 0.5rem;
            }
            
            .action-bar .btn {
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid p-0">
                <div class="page-header mb-4">
                    <div class="d-flex align-items-center justify-content-between">
                        <div class="page-title-container">
                            <h1 class="h3 mb-0 page-title"><i class='bx bx-user-pin title-icon'></i> Edit User</h1>
                            <p class="page-subtitle">Modify user details and permissions</p>
                        </div>
                        <a href="users.php" class="btn btn-outline-secondary btn-sm back-btn">
                            <i class='bx bx-arrow-back'></i> Back to Users
                        </a>
                    </div>
                </div>
                
                <?php if (!empty($successMessage)): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="bx bx-check-circle me-2"></i><?php echo $successMessage; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                <?php if (!empty($errorMessage)): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="bx bx-error me-2"></i><?php echo $errorMessage; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                <?php if (!empty($debugOutput)): ?>
                <div class="alert alert-info alert-dismissible fade show" role="alert">
                    <h6><i class="bx bx-bug me-2"></i>Debug Information:</h6>
                    <?php foreach ($debugOutput as $debug): ?>
                    <div class="small mb-1"><?php echo htmlspecialchars($debug); ?></div>
                    <?php endforeach; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <!-- User Edit Content -->
                <div class="user-edit-container">
                    <div class="user-edit-header d-flex align-items-center">
                        <div class="user-avatar">
                            <i class='bx bx-user'></i>
                        </div>
                        <div>
                            <h2><?php echo htmlspecialchars($user['first_name'].' '.$user['last_name']); ?></h2>
                            <p>User ID: <?php echo htmlspecialchars($user['user_id']); ?> &middot; Member since <?php echo date('M d, Y', strtotime($user['created_at'])); ?></p>
                        </div>
                    </div>
                    
                    <div class="user-edit-body">
                        <form action="edit_user.php?id=<?php echo htmlspecialchars($user_id); ?>" method="POST">
                            <div class="form-section mb-4 shadow-sm">
                                <div class="form-section-header">
                                    <i class="bx bx-user-circle"></i>
                                    <h3>Personal Information</h3>
                                </div>
                                <div class="form-section-body">
                                    <div class="row g-4">
                                        <div class="col-md-6">
                                            <label for="first_name" class="form-label required-field">First Name</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-user"></i></span>
                                                <input type="text" class="form-control" id="first_name" name="first_name" value="<?php echo htmlspecialchars($user['first_name']); ?>" required>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="last_name" class="form-label required-field">Last Name</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-user"></i></span>
                                                <input type="text" class="form-control" id="last_name" name="last_name" value="<?php echo htmlspecialchars($user['last_name']); ?>" required>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="email" class="form-label required-field">Email Address</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-envelope"></i></span>
                                                <input type="email" class="form-control" id="email" name="email" value="<?php echo htmlspecialchars($user['email']); ?>" required>
                                            </div>
                                            <div class="form-text">Email address will be used for login and notifications.</div>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="role" class="form-label required-field">User Role</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class='bx bx-shield-quarter'></i></span>
                                                <select class="form-select" id="role" name="role" required>
                                                    <option value="student" <?php echo $user['user_role'] === 'student' ? 'selected' : ''; ?>>Student</option>
                                                    <option value="teacher" <?php echo $user['user_role'] === 'teacher' ? 'selected' : ''; ?>>Teacher</option>
                                                    <option value="admin" <?php echo $user['user_role'] === 'admin' ? 'selected' : ''; ?>>Administrator</option>
                                                </select>
                                            </div>
                                            <div class="form-text">Different roles have different permissions in the system.</div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Contact Information Section -->
                            <div class="form-section mb-4 shadow-sm">
                                <div class="form-section-header">
                                    <i class="bx bx-phone"></i>
                                    <h3>Contact Information</h3>
                                </div>
                                <div class="form-section-body">
                                    <div class="row g-4">
                                        <div class="col-md-6">
                                            <label for="title" class="form-label">Title</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-crown"></i></span>
                                                <input type="text" class="form-control" id="title" name="title" 
                                                       value="<?php echo htmlspecialchars($user['title'] ?? ''); ?>"
                                                       placeholder="e.g., Mr, Ms, Dr">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                             <label for="cell_number" class="form-label">Cell Number</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-phone"></i></span>
                                                <input type="tel" class="form-control" id="cell_number" name="cell_number" 
                                                       value="<?php echo htmlspecialchars($user['phone'] ?? ''); ?>"
                                                       placeholder="e.g., 0823456789">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="id_number" class="form-label">ID Number</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-id-card"></i></span>
                                                <input type="text" class="form-control" id="id_number" name="id_number" 
                                                       value="<?php echo htmlspecialchars($user['id_number'] ?? ''); ?>"
                                                       placeholder="South African ID Number">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="communication_method" class="form-label">Preferred Communication</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-message-detail"></i></span>
                                                <select class="form-select" id="communication_method" name="communication_method">
                                                    <option value="">Select Method</option>
                                                    <option value="Email" <?php echo ($user['communication_method'] ?? '') === 'Email' ? 'selected' : ''; ?>>Email</option>
                                                    <option value="WhatsApp" <?php echo ($user['communication_method'] ?? '') === 'WhatsApp' ? 'selected' : ''; ?>>WhatsApp</option>
                                                    <option value="SMS" <?php echo ($user['communication_method'] ?? '') === 'SMS' ? 'selected' : ''; ?>>SMS</option>
                                                    <option value="Phone Call" <?php echo ($user['communication_method'] ?? '') === 'Phone Call' ? 'selected' : ''; ?>>Phone Call</option>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Location Information Section -->
                            <div class="form-section mb-4 shadow-sm">
                                <div class="form-section-header">
                                    <i class="bx bx-map"></i>
                                    <h3>Location Information</h3>
                                </div>
                                <div class="form-section-body">
                                    <div class="row g-4">
                                        <div class="col-md-6">
                                            <label for="suburb" class="form-label">Suburb/Area</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-map-pin"></i></span>
                                                <input type="text" class="form-control" id="suburb" name="suburb" 
                                                       value="<?php echo htmlspecialchars($user['suburb'] ?? ''); ?>"
                                                       placeholder="e.g., Pretoria Central">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="province" class="form-label">Province</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-world"></i></span>
                                                <select class="form-select" id="province" name="province">
                                                    <option value="">Select Province</option>
                                                    <option value="Eastern Cape" <?php echo ($user['province'] ?? '') === 'Eastern Cape' ? 'selected' : ''; ?>>Eastern Cape</option>
                                                    <option value="Free State" <?php echo ($user['province'] ?? '') === 'Free State' ? 'selected' : ''; ?>>Free State</option>
                                                    <option value="Gauteng" <?php echo ($user['province'] ?? '') === 'Gauteng' ? 'selected' : ''; ?>>Gauteng</option>
                                                    <option value="KwaZulu-Natal" <?php echo ($user['province'] ?? '') === 'KwaZulu-Natal' ? 'selected' : ''; ?>>KwaZulu-Natal</option>
                                                    <option value="Limpopo" <?php echo ($user['province'] ?? '') === 'Limpopo' ? 'selected' : ''; ?>>Limpopo</option>
                                                    <option value="Mpumalanga" <?php echo ($user['province'] ?? '') === 'Mpumalanga' ? 'selected' : ''; ?>>Mpumalanga</option>
                                                    <option value="Northern Cape" <?php echo ($user['province'] ?? '') === 'Northern Cape' ? 'selected' : ''; ?>>Northern Cape</option>
                                                    <option value="North West" <?php echo ($user['province'] ?? '') === 'North West' ? 'selected' : ''; ?>>North West</option>
                                                    <option value="Western Cape" <?php echo ($user['province'] ?? '') === 'Western Cape' ? 'selected' : ''; ?>>Western Cape</option>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Learning Preferences Section -->
                            <div class="form-section mb-4 shadow-sm">
                                <div class="form-section-header">
                                    <i class="bx bx-laptop"></i>
                                    <h3>Learning Preferences</h3>
                                </div>
                                <div class="form-section-body">
                                    <div class="row g-4">
                                        <div class="col-md-6">
                                            <label for="owns_laptop" class="form-label">Owns Laptop/PC</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-desktop"></i></span>
                                                <select class="form-select" id="owns_laptop" name="owns_laptop">
                                                    <option value="">Select Option</option>
                                                    <option value="Yes" <?php echo ($user['owns_laptop'] ?? '') === 'Yes' ? 'selected' : ''; ?>>Yes</option>
                                                    <option value="No" <?php echo ($user['owns_laptop'] ?? '') === 'No' ? 'selected' : ''; ?>>No</option>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="training_preference" class="form-label">Training Preference</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-book-reader"></i></span>
                                                <select class="form-select" id="training_preference" name="training_preference">
                                                    <option value="">Select Preference</option>
                                                    <option value="Online" <?php echo ($user['training_preference'] ?? '') === 'Online' ? 'selected' : ''; ?>>Online</option>
                                                    <option value="Onsite" <?php echo ($user['training_preference'] ?? '') === 'Onsite' ? 'selected' : ''; ?>>Onsite</option>
                                                    <option value="Hybrid" <?php echo ($user['training_preference'] ?? '') === 'Hybrid' ? 'selected' : ''; ?>>Hybrid</option>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="job_field_interest" class="form-label">Job Field Interest</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-briefcase-alt"></i></span>
                                                <select class="form-select" id="job_field_interest" name="job_field_interest">
                                                    <option value="">Select Job Field</option>
                                                    <option value="Data Capturing" <?php echo ($user['job_field_interest'] ?? '') === 'Data Capturing' ? 'selected' : ''; ?>>Data Capturing</option>
                                                    <option value="Administrative Assistant" <?php echo ($user['job_field_interest'] ?? '') === 'Administrative Assistant' ? 'selected' : ''; ?>>Administrative Assistant</option>
                                                    <option value="Customer Service" <?php echo ($user['job_field_interest'] ?? '') === 'Customer Service' ? 'selected' : ''; ?>>Customer Service</option>
                                                    <option value="IT Support" <?php echo ($user['job_field_interest'] ?? '') === 'IT Support' ? 'selected' : ''; ?>>IT Support</option>
                                                    <option value="Digital Marketing" <?php echo ($user['job_field_interest'] ?? '') === 'Digital Marketing' ? 'selected' : ''; ?>>Digital Marketing</option>
                                                    <option value="Web Development" <?php echo ($user['job_field_interest'] ?? '') === 'Web Development' ? 'selected' : ''; ?>>Web Development</option>
                                                    <option value="Graphic Design" <?php echo ($user['job_field_interest'] ?? '') === 'Graphic Design' ? 'selected' : ''; ?>>Graphic Design</option>
                                                    <option value="Project Management" <?php echo ($user['job_field_interest'] ?? '') === 'Project Management' ? 'selected' : ''; ?>>Project Management</option>
                                                    <option value="Other" <?php echo ($user['job_field_interest'] ?? '') === 'Other' ? 'selected' : ''; ?>>Other</option>
                                                </select>
                                            </div>
                                        </div>
                                        <?php if (!empty($user['job_field_other'])): ?>
                                        <div class="col-md-6">
                                            <label for="job_field_other" class="form-label">Other Job Field</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-info-circle"></i></span>
                                                <input type="text" class="form-control" id="job_field_other" name="job_field_other" 
                                                       value="<?php echo htmlspecialchars($user['job_field_other'] ?? ''); ?>" 
                                                       placeholder="Specify other job field interest">
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                        <div class="col-12">
                                            <label for="learning_goals" class="form-label">Learning Goals</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-target-lock"></i></span>
                                                <textarea class="form-control" id="learning_goals" name="learning_goals" 
                                                          rows="3" placeholder="Describe learning goals and objectives..."><?php echo htmlspecialchars($user['learning_goals'] ?? ''); ?></textarea>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Background Information Section -->
                            <div class="form-section mb-4 shadow-sm">
                                <div class="form-section-header">
                                    <i class="bx bx-book"></i>
                                    <h3>Background Information</h3>
                                </div>
                                <div class="form-section-body">
                                    <div class="row g-4">
                                        <div class="col-md-6">
                                            <label for="education_status" class="form-label">Education Status</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-graduation"></i></span>
                                                <select class="form-select" id="education_status" name="education_status">
                                                    <option value="">Select Education Level</option>
                                                    <!-- Teacher/Coded Values -->
                                                    <option value="matric" <?php echo ($user['education_status'] ?? '') === 'matric' ? 'selected' : ''; ?>>Matric/Grade 12</option>
                                                    <option value="diploma" <?php echo ($user['education_status'] ?? '') === 'diploma' ? 'selected' : ''; ?>>Diploma</option>
                                                    <option value="degree" <?php echo ($user['education_status'] ?? '') === 'degree' ? 'selected' : ''; ?>>Bachelor's Degree</option>
                                                    <option value="honours" <?php echo ($user['education_status'] ?? '') === 'honours' ? 'selected' : ''; ?>>Honours Degree</option>
                                                    <option value="masters" <?php echo ($user['education_status'] ?? '') === 'masters' ? 'selected' : ''; ?>>Master's Degree</option>
                                                    <option value="phd" <?php echo ($user['education_status'] ?? '') === 'phd' ? 'selected' : ''; ?>>PhD</option>
                                                    <!-- Student Profile Values (for compatibility) -->
                                                    <option value="High School" <?php echo ($user['education_status'] ?? '') === 'High School' ? 'selected' : ''; ?>>High School (Student)</option>
                                                    <option value="University Degree/Diploma" <?php echo ($user['education_status'] ?? '') === 'University Degree/Diploma' ? 'selected' : ''; ?>>University Degree/Diploma (Student)</option>
                                                    <option value="Postgraduate" <?php echo ($user['education_status'] ?? '') === 'Postgraduate' ? 'selected' : ''; ?>>Postgraduate (Student)</option>
                                                    <option value="Professional Certification" <?php echo ($user['education_status'] ?? '') === 'Professional Certification' ? 'selected' : ''; ?>>Professional Certification (Student)</option>
                                                    <option value="Other" <?php echo ($user['education_status'] ?? '') === 'Other' ? 'selected' : ''; ?>>Other</option>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="employment_status" class="form-label">Employment Status</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-briefcase"></i></span>
                                                <select class="form-select" id="employment_status" name="employment_status">
                                                    <option value="">Select Employment Status</option>
                                                    <option value="employed_fulltime" <?php echo ($user['employment_status'] ?? '') === 'employed_fulltime' ? 'selected' : ''; ?>>Employed Full-time</option>
                                                    <option value="employed_parttime" <?php echo ($user['employment_status'] ?? '') === 'employed_parttime' ? 'selected' : ''; ?>>Employed Part-time</option>
                                                    <option value="self_employed" <?php echo ($user['employment_status'] ?? '') === 'self_employed' ? 'selected' : ''; ?>>Self-employed</option>
                                                    <option value="unemployed" <?php echo ($user['employment_status'] ?? '') === 'unemployed' ? 'selected' : ''; ?>>Unemployed</option>
                                                    <option value="student" <?php echo ($user['employment_status'] ?? '') === 'student' ? 'selected' : ''; ?>>Student</option>
                                                    <option value="retired" <?php echo ($user['employment_status'] ?? '') === 'retired' ? 'selected' : ''; ?>>Retired</option>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="heard_about_us" class="form-label">How did you hear about us?</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-search"></i></span>
                                                <select class="form-select" id="heard_about_us" name="heard_about_us">
                                                    <option value="">Select Option</option>
                                                    <option value="google" <?php echo ($user['heard_about_us'] ?? '') === 'google' ? 'selected' : ''; ?>>Google Search</option>
                                                    <option value="facebook" <?php echo ($user['heard_about_us'] ?? '') === 'facebook' ? 'selected' : ''; ?>>Facebook</option>
                                                    <option value="instagram" <?php echo ($user['heard_about_us'] ?? '') === 'instagram' ? 'selected' : ''; ?>>Instagram</option>
                                                    <option value="linkedin" <?php echo ($user['heard_about_us'] ?? '') === 'linkedin' ? 'selected' : ''; ?>>LinkedIn</option>
                                                    <option value="referral" <?php echo ($user['heard_about_us'] ?? '') === 'referral' ? 'selected' : ''; ?>>Referral from friend/colleague</option>
                                                    <option value="advertisement" <?php echo ($user['heard_about_us'] ?? '') === 'advertisement' ? 'selected' : ''; ?>>Online Advertisement</option>
                                                    <option value="other" <?php echo ($user['heard_about_us'] ?? '') === 'other' ? 'selected' : ''; ?>>Other</option>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="referral_person_name" class="form-label">Referred by (Optional)</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-user-plus"></i></span>
                                                <input type="text" class="form-control" id="referral_person_name" name="referral_person_name" 
                                                       value="<?php echo htmlspecialchars($user['referral_person_name'] ?? ''); ?>" 
                                                       placeholder="Name of person who referred you">
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <?php if ($user['user_role'] === 'teacher'): ?>
                            <!-- Teacher/Facilitator Information Section - Only shown for teachers -->
                            <div class="form-section mb-4 shadow-sm">
                                <div class="form-section-header">
                                    <i class="bx bx-chalkboard"></i>
                                    <h3>Teacher/Facilitator Information</h3>
                                </div>
                                <div class="form-section-body">
                                    <?php
                                    // Teacher data is now available in main $user array from comprehensive query
                                    // No need for separate query since we have LEFT JOINs in main query?>
                                    <div class="row g-4">
                                        <div class="col-md-8">
                                            <label for="course_title" class="form-label">Course/Expertise Title</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-book-open"></i></span>
                                                <input type="text" class="form-control" id="course_title" name="course_title" 
                                                       value="<?php echo htmlspecialchars($user['course_title'] ?? ''); ?>" 
                                                       placeholder="What course does this teacher specialize in?">
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <label for="experience" class="form-label">Years of Experience</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-time"></i></span>
                                                <select class="form-select" id="experience" name="experience">
                                                    <option value="">Select experience level</option>
                                                    <option value="0-1" <?php echo ($user['experience'] ?? '') === '0-1' ? 'selected' : ''; ?>>0-1 years</option>
                                                    <option value="2-5" <?php echo ($user['experience'] ?? '') === '2-5' ? 'selected' : ''; ?>>2-5 years</option>
                                                    <option value="6-10" <?php echo ($user['experience'] ?? '') === '6-10' ? 'selected' : ''; ?>>6-10 years</option>
                                                    <option value="11+" <?php echo ($user['experience'] ?? '') === '11+' ? 'selected' : ''; ?>>11+ years</option>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-12">
                                            <label for="course_description" class="form-label">Course/Bio Description</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-text"></i></span>
                                                <textarea class="form-control" id="course_description" name="course_description" rows="3" 
                                                          placeholder="Course description and teaching approach..."><?php echo htmlspecialchars($user['course_description'] ?? ''); ?></textarea>
                                            </div>
                                        </div>
                                        <div class="col-12">
                                            <label for="qualifications" class="form-label">Qualifications & Certifications</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-medal"></i></span>
                                                <textarea class="form-control" id="qualifications" name="qualifications" rows="3" 
                                                          placeholder="List relevant qualifications, certifications, and expertise..."><?php echo htmlspecialchars($user['qualifications'] ?? ''); ?></textarea>
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <label for="website" class="form-label">Website</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-globe"></i></span>
                                                <input type="url" class="form-control" id="website" name="website" 
                                                       value="<?php echo htmlspecialchars($user['website'] ?? ''); ?>" 
                                                       placeholder="https://example.com">
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <label for="social_linkedin" class="form-label">LinkedIn Profile</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bxl-linkedin"></i></span>
                                                <input type="url" class="form-control" id="social_linkedin" name="social_linkedin" 
                                                       value="<?php echo htmlspecialchars($user['social_linkedin'] ?? ''); ?>" 
                                                       placeholder="https://linkedin.com/in/username">
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <label for="social_twitter" class="form-label">Twitter Profile</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bxl-twitter"></i></span>
                                                <input type="url" class="form-control" id="social_twitter" name="social_twitter" 
                                                       value="<?php echo htmlspecialchars($user['social_twitter'] ?? ''); ?>" 
                                                       placeholder="https://twitter.com/username">
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Account Settings Section -->
                            <div class="form-section mb-4 shadow-sm">
                                <div class="form-section-header">
                                    <i class="bx bx-cog"></i>
                                    <h3>Account Settings</h3>
                                </div>
                                <div class="form-section-body">
                                    <div class="row g-4">
                                        <div class="col-md-6">
                                            <div class="form-check form-switch">
                                                <input class="form-check-input" type="checkbox" id="is_active" name="is_active" 
                                                       <?php echo ($user['is_active'] ?? 1) ? 'checked' : ''; ?>>
                                                <label class="form-check-label" for="is_active">
                                                    <strong>Account Status</strong>
                                                </label>
                                            </div>
                                            <div class="form-text">Inactive accounts cannot login to the system.</div>
                                        </div>
                                        <div class="col-md-6">
                                            <label class="form-label">Account Created</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class="bx bx-calendar"></i></span>
                                                <input type="text" class="form-control" readonly
                                                       value="<?php echo date('F j, Y g:i A', strtotime($user['created_at'] ?? 'now')); ?>">
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="form-section mb-4 shadow-sm">
                                <div class="form-section-header">
                                    <i class='bx bx-lock-alt'></i>
                                    <h3>Password Settings</h3>
                                </div>
                                <div class="form-section-body">
                                    <div class="row g-4">
                                        <div class="col-md-6">
                                            <label for="new_password" class="form-label">New Password</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class='bx bx-lock'></i></span>
                                                <input type="password" class="form-control" id="new_password" name="new_password" placeholder="Enter new password">
                                            </div>
                                            <div class="form-text mb-2">Leave blank to keep current password.</div>
                                            <div class="password-strength">
                                                <div class="password-strength-meter" id="password-strength-meter"></div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="confirm_password" class="form-label">Confirm Password</label>
                                            <div class="input-group mb-1">
                                                <span class="input-group-text"><i class='bx bx-check-shield'></i></span>
                                                <input type="password" class="form-control" id="confirm_password" name="confirm_password" placeholder="Confirm new password">
                                            </div>
                                            <div id="password-match-message" class="form-text">Passwords must match.</div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="action-bar">
                                <a href="users.php" class="btn btn-outline-secondary">
                                    <i class="bx bx-arrow-back"></i> Cancel
                                </a>
                                <button type="submit" class="btn btn-primary">
                                    <i class="bx bx-save"></i> Save Changes
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Debug info
            console.log('User data loaded:', <?php echo json_encode($user); ?>);
            
            // Form submission tracking
            const userForm = document.querySelector('form');
            if (userForm) {
                userForm.addEventListener('submit', function(e) {
                    console.log('Form submitted', {
                        firstName: document.getElementById('first_name').value,
                        lastName: document.getElementById('last_name').value,
                        email: document.getElementById('email').value,
                        role: document.getElementById('role').value
                    });
                });
            }
            // Sidebar toggle functionality
            const sidebarToggle = document.querySelector('.sidebar-toggle');
            const sidebarToggleMobile = document.querySelector('.sidebar-toggle-mobile');
            const sidebar = document.querySelector('.sidebar');
            
            // Add fade-in animation to the user edit container
            const userEditContainer = document.querySelector('.user-edit-container');
            if (userEditContainer) {
                setTimeout(() => {
                    userEditContainer.style.opacity = '0';
                    userEditContainer.style.transition = 'opacity 0.5s ease-in-out';
                    setTimeout(() => {
                        userEditContainer.style.opacity = '1';
                    }, 50);
                }, 50);
            }
            
            // Password strength meter functionality
            const passwordInput = document.getElementById('new_password');
            const strengthMeter = document.getElementById('password-strength-meter');
            const confirmInput = document.getElementById('confirm_password');
            const matchMessage = document.getElementById('password-match-message');
            
            if (passwordInput && strengthMeter) {
                passwordInput.addEventListener('input', function() {
                    const val = passwordInput.value;
                    let strength = 0;
                    
                    if (val.length >= 8) strength += 1;
                    if (val.match(/[a-z]+/)) strength += 1;
                    if (val.match(/[A-Z]+/)) strength += 1;
                    if (val.match(/[0-9]+/)) strength += 1;
                    if (val.match(/[^a-zA-Z0-9]+/)) strength += 1;
                    
                    // Reset classes
                    strengthMeter.className = 'password-strength-meter';
                    
                    // Add appropriate class
                    if (val.length === 0) {
                        strengthMeter.style.width = '0%';
                    } else if (strength < 3) {
                        strengthMeter.classList.add('password-weak');
                        strengthMeter.style.width = '30%';
                    } else if (strength < 5) {
                        strengthMeter.classList.add('password-medium');
                        strengthMeter.style.width = '65%';
                    } else {
                        strengthMeter.classList.add('password-strong');
                        strengthMeter.style.width = '100%';
                    }
                });
            }
            
            if (confirmInput && matchMessage) {
                const validatePasswordMatch = function() {
                    if (passwordInput.value === '' && confirmInput.value === '') {
                        matchMessage.textContent = 'Passwords must match.';
                        matchMessage.className = 'form-text';
                    } else if (passwordInput.value === confirmInput.value) {
                        matchMessage.innerHTML = '<i class="bx bx-check-circle"></i> Passwords match!';
                        matchMessage.className = 'form-text text-success';
                    } else {
                        matchMessage.innerHTML = '<i class="bx bx-error-circle"></i> Passwords do not match!';
                        matchMessage.className = 'form-text text-danger';
                    }
                };
                
                passwordInput.addEventListener('input', validatePasswordMatch);
                confirmInput.addEventListener('input', validatePasswordMatch);
            }
            
            // Auto-dismiss alerts after 5 seconds
            const alerts = document.querySelectorAll('.alert');
            if (alerts.length) {
                setTimeout(function() {
                    alerts.forEach(alert => {
                        const bsAlert = new bootstrap.Alert(alert);
                        bsAlert.close();
                    });
                }, 5000);
            }
            
            if (sidebarToggle) {
                sidebarToggle.addEventListener('click', () => {
                    sidebar.classList.toggle('close');
                });
            }
            
            if (sidebarToggleMobile) {
                sidebarToggleMobile.addEventListener('click', () => {
                    sidebar.classList.toggle('show');
                });
            }
            
            // Responsive sidebar behavior
            function checkScreenSize() {
                if (window.innerWidth < 768) {
                    sidebar.classList.remove('close');
                    sidebar.classList.remove('show');
                } else if (window.innerWidth < 992) {
                    sidebar.classList.add('close');
                    sidebar.classList.remove('show');
                }
            }
            
            // Initial check and listen for window resize
            checkScreenSize();
            window.addEventListener('resize', checkScreenSize);
        });
    </script>
</body>
</html>
