<?php
/**
 * Admin Email Compose - Panadite Academy
 * Send emails using secure SMTP system
 */

require_once __DIR__ . '/../auth/functions.php';

require_once '../config/fast_email_service.php';
require_once '../config/email_inbox_service.php';
require_once '../config/email_drafts_service.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Get current user
$currentUser = getCurrentUser();

// Handle loading existing draft
$loadedDraft = null;
if (isset($_GET['draft_id']) && is_numeric($_GET['draft_id'])) {
    $draftId = intval($_GET['draft_id']);
    $userId = $currentUser['user_id'] ?? $currentUser['id'];
    $loadedDraft = getDraft($draftId, $userId);
}
$emailSettings = getEmailSettings();

// Handle reply
$replyTo = '';
$replySubject = '';
$replyBody = '';

if (isset($_GET['reply']) && !empty($_GET['reply'])) {
    $replyEmailId = intval($_GET['reply']);
    $emailResult = getEmailDetails($replyEmailId);
    
    if ($emailResult['success']) {
        $originalEmail = $emailResult['email'];
        $replyTo = $originalEmail['from'];
        $replySubject = 'Re: ' . preg_replace('/^Re:\s*/i', '', $originalEmail['subject']);
        
        // Format original email for reply
        $replyBody = "\n\n--- Original Message ---\n";
        $replyBody .= "From: " . $originalEmail['from'] . "\n";
        $replyBody .= "Date: " . $originalEmail['date'] . "\n";
        $replyBody .= "Subject: " . $originalEmail['subject'] . "\n\n";
        $replyBody .= strip_tags($originalEmail['body']);
    }
}

// Handle AJAX form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['ajax'])) {
    header('Content-Type: application/json');
    
    $to = trim($_POST['to']);
    $cc = trim($_POST['cc'] ?? '');
    $bcc = trim($_POST['bcc'] ?? '');
    $subject = trim($_POST['subject']);
    $body = trim($_POST['body']);
    $priority = $_POST['priority'] ?? 'normal';
    
    $errors = [];
    
    // Validation
    if (empty($to)) {
        $errors[] = 'Recipient email is required';
    } elseif (!filter_var($to, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Invalid recipient email address';
    }
    
    if (empty($subject)) {
        $errors[] = 'Subject is required';
    }
    
    if (empty($body)) {
        $errors[] = 'Message body is required';
    }
    
    // Validate CC emails if provided
    if (!empty($cc)) {
        $ccEmails = array_map('trim', explode(',', $cc));
        foreach ($ccEmails as $ccEmail) {
            if (!filter_var($ccEmail, FILTER_VALIDATE_EMAIL)) {
                $errors[] = "Invalid CC email address: $ccEmail";
            }
        }
    }
    
    // Validate BCC emails if provided
    if (!empty($bcc)) {
        $bccEmails = array_map('trim', explode(',', $bcc));
        foreach ($bccEmails as $bccEmail) {
            if (!filter_var($bccEmail, FILTER_VALIDATE_EMAIL)) {
                $errors[] = "Invalid BCC email address: $bccEmail";
            }
        }
    }
    
    if (!empty($errors)) {
        echo json_encode(['success' => false, 'errors' => $errors]);
        exit();
    }
    
    // Prepare email headers and body
    $headers = "From: {$emailSettings['from_name']} <{$emailSettings['from_email']}>\r\n";
    $headers .= "Reply-To: {$emailSettings['from_email']}\r\n";
    $headers .= "X-Mailer: Panadite Academy Email System\r\n";
    $headers .= "MIME-Version: 1.0\r\n";
    $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    
    // Add priority header
    if ($priority == 'high') {
        $headers .= "X-Priority: 1\r\n";
        $headers .= "X-MSMail-Priority: High\r\n";
    } elseif ($priority == 'low') {
        $headers .= "X-Priority: 5\r\n";
        $headers .= "X-MSMail-Priority: Low\r\n";
    }
    
    // Convert plain text to HTML
    $htmlBody = nl2br(htmlspecialchars($body));
    $htmlBody = "
    <html>
    <head>
        <meta charset='UTF-8'>
    </head>
    <body style='font-family: Arial, sans-serif; line-height: 1.6; color: #333;'>
        $htmlBody
        <br><br>
        <hr style='border: none; border-top: 1px solid #ccc; margin: 20px 0;'>
        <p style='font-size: 12px; color: #666;'>
            Sent from Panadite Academy Email System<br>
            <a href='https://panaditeacademy.com'>www.panaditeacademy.com</a>
        </p>
    </body>
    </html>";
    
    // Send email
    $result = sendEmailNow($to, $subject, $htmlBody);
    
    if ($result) {
        // Log the sent email
        try {
            $stmt = $pdo->prepare("INSERT INTO email_logs (email_type, recipient, subject, status, created_at, admin_user_id) VALUES (?, ?, ?, ?, NOW(), ?)");
            $stmt->execute(['admin_compose', $to, $subject, 'sent', $currentUser['id']]);
        } catch (Exception $e) {
            // Log error but don't fail the email sending
            error_log("Failed to log sent email: " . $e->getMessage());
        }
        
        $sentCount = 1;
        
        // Handle CC and BCC
        if (!empty($cc)) {
            foreach ($ccEmails as $ccEmail) {
                if (sendEmailNow($ccEmail, $subject, $htmlBody)) {
                    $sentCount++;
                }
            }
        }
        
        if (!empty($bcc)) {
            foreach ($bccEmails as $bccEmail) {
                if (sendEmailNow($bccEmail, $subject, $htmlBody)) {
                    $sentCount++;
                }
            }
        }
        
        echo json_encode([
            'success' => true, 
            'message' => "Email sent successfully to $to" . ($sentCount > 1 ? " and " . ($sentCount - 1) . " other recipients" : ""),
            'sent_count' => $sentCount
        ]);
    } else {
        echo json_encode(['success' => false, 'errors' => ['Failed to send email. Please check your email configuration.']]);
    }
    exit();
}

// Handle draft operations
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['draft_action'])) {
    header('Content-Type: application/json');
    
    $action = $_POST['draft_action'];
    
    // Debug: Check if user is properly authenticated
    if (!$currentUser || (!isset($currentUser['id']) && !isset($currentUser['user_id']))) {
        echo json_encode(['success' => false, 'error' => 'User not authenticated - currentUser: ' . json_encode($currentUser)]);
        exit();
    }
    
    // Handle both possible user ID key names
    $userId = $currentUser['user_id'] ?? $currentUser['id'];
    
    // Debug: Check database connection
    global $pdo;
    if (!$pdo) {
        echo json_encode(['success' => false, 'error' => 'Database connection failed']);
        exit();
    }
    
    // Debug: Test database query
    try {
        $stmt = $pdo->query("SELECT 1");
        if (!$stmt) {
            echo json_encode(['success' => false, 'error' => 'Database query test failed']);
            exit();
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => 'Database error: ' . $e->getMessage()]);
        exit();
    }
    
    switch ($action) {
        case 'save':
            $to = trim($_POST['to'] ?? '');
            $cc = trim($_POST['cc'] ?? '');
            $bcc = trim($_POST['bcc'] ?? '');
            $subject = trim($_POST['subject'] ?? '');
            $body = trim($_POST['body'] ?? '');
            $priority = $_POST['priority'] ?? 'normal';
            $draftId = !empty($_POST['draft_id']) ? intval($_POST['draft_id']) : null;
            
            // Validation - at least one field should have content
            if (empty($to) && empty($subject) && empty($body)) {
                echo json_encode(['success' => false, 'error' => 'Cannot save empty draft']);
                exit();
            }
            
            // Debug logging for PHP backend
            error_log("Draft Save Debug - User ID: " . $userId);
            error_log("Draft Save Debug - To: '$to'");
            error_log("Draft Save Debug - Subject: '$subject'");
            error_log("Draft Save Debug - Body: '$body'");
            error_log("Draft Save Debug - Priority: '$priority'");
            error_log("Draft Save Debug - Draft ID: " . ($draftId ?: 'null'));
            
            $savedDraftId = saveDraft($userId, $to, $cc, $bcc, $subject, $body, $priority, $draftId);
            
            error_log("Draft Save Debug - Result: " . ($savedDraftId ?: 'false'));
            
            if ($savedDraftId) {
                echo json_encode([
                    'success' => true, 
                    'message' => 'Draft saved successfully',
                    'draft_id' => $savedDraftId
                ]);
            } else {
                echo json_encode(['success' => false, 'error' => 'Failed to save draft - check server error log for details']);
            }
            break;
            
        case 'load':
            $draftId = intval($_POST['draft_id']);
            $draft = getDraft($draftId, $userId);
            
            if ($draft) {
                echo json_encode(['success' => true, 'draft' => $draft]);
            } else {
                echo json_encode(['success' => false, 'error' => 'Draft not found']);
            }
            break;
            
        case 'delete':
            $draftId = intval($_POST['draft_id']);
            $result = deleteDraft($draftId, $userId);
            
            if ($result) {
                echo json_encode(['success' => true, 'message' => 'Draft deleted successfully']);
            } else {
                echo json_encode(['success' => false, 'error' => 'Failed to delete draft']);
            }
            break;
            
        case 'list':
            $drafts = getUserDrafts($userId, 20);
            echo json_encode(['success' => true, 'drafts' => $drafts]);
            break;
            
        case 'auto_save':
            $to = trim($_POST['to'] ?? '');
            $cc = trim($_POST['cc'] ?? '');
            $bcc = trim($_POST['bcc'] ?? '');
            $subject = trim($_POST['subject'] ?? '');
            $body = trim($_POST['body'] ?? '');
            $priority = $_POST['priority'] ?? 'normal';
            
            $savedDraftId = autoSaveDraft($userId, $to, $cc, $bcc, $subject, $body, $priority);
            
            if ($savedDraftId) {
                echo json_encode(['success' => true, 'draft_id' => $savedDraftId]);
            } else {
                echo json_encode(['success' => false, 'error' => 'Auto-save failed']);
            }
            break;
            
        default:
            echo json_encode(['success' => false, 'error' => 'Invalid action']);
    }
    exit();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Compose Email - Panadite Academy Admin</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link rel="stylesheet" href="/assets/css/user-cards.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        .compose-header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border-radius: 10px 10px 0 0; }
        .email-editor { min-height: 300px; }
        .priority-badge { padding: 3px 8px; border-radius: 12px; font-size: 11px; }
        .priority-high { background: #dc3545; color: white; }
        .priority-normal { background: #6c757d; color: white; }
        .priority-low { background: #28a745; color: white; }
        
        /* Beautiful Notification Styles */
        .notification {
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 9999;
            max-width: 400px;
            padding: 16px 24px;
            border-radius: 12px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.12);
            backdrop-filter: blur(10px);
            transform: translateX(100%);
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 12px;
        }
        .notification.show {
            transform: translateX(0);
        }
        .notification.success {
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
            color: white;
            border-left: 4px solid #34d399;
        }
        .notification.error {
            background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
            color: white;
            border-left: 4px solid #f87171;
        }
        .notification i {
            font-size: 24px;
            opacity: 0.9;
        }
        .notification .close-btn {
            margin-left: auto;
            background: none;
            border: none;
            color: inherit;
            font-size: 18px;
            cursor: pointer;
            opacity: 0.8;
            transition: opacity 0.2s;
        }
        .notification .close-btn:hover {
            opacity: 1;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
        
        <div class="container-fluid">
            <!-- Page Header -->
            <div class="d-flex justify-content-between align-items-center mb-4">
                <div>
                    <h2><i class="bx bx-edit"></i> Compose Email</h2>
                    <p class="text-muted">Send secure emails using your SMTP system</p>
                </div>
                <div class="d-flex gap-2">
                    <a href="email_inbox.php" class="btn btn-outline-secondary">
                        <i class="bx bx-arrow-back"></i> Back to Inbox
                    </a>
                    <a href="email_settings.php" class="btn btn-outline-primary">
                        <i class="bx bx-cog"></i> Email Settings
                    </a>
                </div>
            </div>

            <?php if (!empty($errors)): ?>
            <div class="alert alert-danger">
                <h6><i class="bx bx-error"></i> Please fix the following errors:</h6>
                <ul class="mb-0">
                    <?php foreach ($errors as $error): ?>
                    <li><?php echo htmlspecialchars($error); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>

            <!-- Compose Form -->
            <div class="card shadow-lg">
                <div class="card-header compose-header">
                    <h5 class="mb-0"><i class="bx bx-envelope"></i> New Message</h5>
                </div>
                <div class="card-body">
                    <form method="POST" id="composeForm">
                        <!-- From Field (Read-only) -->
                        <div class="row mb-3">
                            <label class="col-sm-2 col-form-label"><strong>From:</strong></label>
                            <div class="col-sm-10">
                                <input type="text" class="form-control-plaintext" 
                                       value="<?php echo htmlspecialchars($emailSettings['from_name'] . ' <' . $emailSettings['from_email'] . '>'); ?>" 
                                       readonly>
                            </div>
                        </div>

                        <!-- To Field -->
                        <div class="row mb-3">
                            <label for="to" class="col-sm-2 col-form-label"><strong>To:</strong></label>
                            <div class="col-sm-10">
                                <input type="email" class="form-control" id="to" name="to" 
                                       value="<?php echo htmlspecialchars($loadedDraft['to_email'] ?? $replyTo); ?>" 
                                       placeholder="recipient@example.com" required>
                                <div class="form-text">Enter recipient email address</div>
                            </div>
                        </div>

                        <!-- CC Field -->
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="cc" class="form-label">CC</label>
                                    <input type="email" class="form-control" id="cc" name="cc" value="<?= htmlspecialchars($loadedDraft['cc_email'] ?? '') ?>">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="bcc" class="form-label">BCC</label>
                                    <input type="email" class="form-control" id="bcc" name="bcc" value="<?= htmlspecialchars($loadedDraft['bcc_email'] ?? '') ?>">
                                </div>
                            </div>
                        </div>

                        <!-- Priority -->
                        <div class="mb-3">
                            <label for="priority" class="form-label">Priority</label>
                            <select class="form-select" id="priority" name="priority">
                                <option value="low" <?= isset($loadedDraft) && $loadedDraft['priority'] == 'low' ? 'selected' : '' ?>>Low</option>
                                <option value="normal" <?= !isset($loadedDraft) || $loadedDraft['priority'] == 'normal' ? 'selected' : '' ?>>Normal</option>
                                <option value="high" <?= isset($loadedDraft) && $loadedDraft['priority'] == 'high' ? 'selected' : '' ?>>High</option>
                            </select>
                        </div>

                        <!-- Subject -->
                        <div class="row mb-3">
                            <label for="subject" class="col-sm-2 col-form-label"><strong>Subject:</strong></label>
                            <div class="col-sm-10">
                                <input type="text" class="form-control" id="subject" name="subject" 
                                       value="<?php echo htmlspecialchars($loadedDraft['subject'] ?? $replySubject); ?>" 
                                       placeholder="Enter email subject" required>
                            </div>
                        </div>

                        <!-- Message Body -->
                        <div class="row mb-4">
                            <label for="body" class="col-sm-2 col-form-label"><strong>Message:</strong></label>
                            <div class="col-sm-10">
                                <textarea class="form-control email-editor" id="body" name="body" rows="12" 
                                          placeholder="Type your message here..." required><?php echo htmlspecialchars($loadedDraft['body'] ?? $replyBody); ?></textarea>
                                <div class="form-text">
                                    <i class="bx bx-info-circle"></i> HTML formatting will be applied automatically. 
                                    Line breaks will be preserved.
                                </div>
                            </div>
                        </div>

                        <!-- Action Buttons -->
                        <div class="row">
                            <div class="col-sm-10 offset-sm-2">
                                <div class="d-flex gap-2">
                                    <button type="submit" class="btn btn-primary btn-lg">
                                        <i class="bx bx-send"></i> Send Email
                                    </button>
                                    <button type="button" class="btn btn-outline-secondary" onclick="saveDraft()">
                                        <i class="bx bx-save"></i> Save Draft
                                    </button>
                                    <a href="email_inbox.php" class="btn btn-outline-danger">
                                        <i class="bx bx-x"></i> Cancel
                                    </a>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Email Preview -->
            <div class="card mt-4">
                <div class="card-header">
                    <h6><i class="bx bx-show"></i> Email Preview</h6>
                </div>
                <div class="card-body">
                    <div id="emailPreview" class="border rounded p-3" style="background-color: #f8f9fa;">
                        <div class="email-preview-content">
                            <p><strong>To:</strong> <span id="previewTo">-</span></p>
                            <p><strong>Subject:</strong> <span id="previewSubject">-</span></p>
                            <p><strong>Priority:</strong> <span id="previewPriority" class="priority-badge priority-normal">Normal</span></p>
                            <hr>
                            <div id="previewBody" style="background: white; padding: 15px; border-radius: 5px; min-height: 100px;">
                                <em class="text-muted">Message preview will appear here...</em>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        </div>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Live preview functionality
        function updatePreview() {
            const to = document.getElementById('to').value || '-';
            const subject = document.getElementById('subject').value || '-';
            const body = document.getElementById('body').value || 'Message preview will appear here...';
            const priority = document.getElementById('priority').value;
            
            document.getElementById('previewTo').textContent = to;
            document.getElementById('previewSubject').textContent = subject;
            document.getElementById('previewBody').innerHTML = body.replace(/\n/g, '<br>');
            
            // Update priority badge
            const priorityBadge = document.getElementById('previewPriority');
            priorityBadge.textContent = priority.charAt(0).toUpperCase() + priority.slice(1) + ' Priority';
            priorityBadge.className = 'priority-badge priority-' + priority;
        }

        // Attach event listeners
        document.getElementById('to').addEventListener('input', updatePreview);
        document.getElementById('subject').addEventListener('input', updatePreview);
        document.getElementById('body').addEventListener('input', updatePreview);
        document.getElementById('priority').addEventListener('change', updatePreview);

        // AJAX Form submission
        document.getElementById('composeForm').addEventListener('submit', function(e) {
            e.preventDefault(); // Prevent default form submission
            
            const to = document.getElementById('to').value.trim();
            const subject = document.getElementById('subject').value.trim();
            const body = document.getElementById('body').value.trim();
            
            // Validation
            if (!to || !subject || !body) {
                showNotification('Please fill in all required fields (To, Subject, and Message).', 'error');
                return false;
            }
            
            // Email validation
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(to)) {
                showNotification('Please enter a valid email address in the To field.', 'error');
                return false;
            }
            
            // Confirm send
            if (!confirm('Are you sure you want to send this email?')) {
                return false;
            }
            
            // Show sending state
            const submitBtn = document.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            submitBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Sending...';
            submitBtn.disabled = true;
            
            // Prepare form data
            const formData = new FormData(this);
            formData.append('ajax', '1');
            
            // Send AJAX request
            fetch('email_compose.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                // Reset button
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
                
                if (data.success) {
                    // Show success notification
                    showNotification(data.message, 'success');
                    
                    // Clear form after 2 seconds
                    setTimeout(() => {
                        if (confirm('Email sent successfully! Would you like to compose another email?')) {
                            clearForm();
                        } else {
                            window.location.href = 'email_inbox.php';
                        }
                    }, 2000);
                    
                } else {
                    // Show error notification
                    const errorMsg = data.errors ? data.errors.join('<br>') : 'Failed to send email';
                    showNotification(errorMsg, 'error');
                }
            })
            .catch(error => {
                // Reset button
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
                
                showNotification('Network error. Please try again.', 'error');
                console.error('Error:', error);
            });
        });

        // Beautiful notification system
        function showNotification(message, type = 'success') {
            // Remove any existing notifications
            const existingNotifications = document.querySelectorAll('.notification');
            existingNotifications.forEach(notification => notification.remove());
            
            // Create notification element
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            
            const icon = type === 'success' ? 'bx-check-circle' : 'bx-error-circle';
            
            notification.innerHTML = `
                <i class="bx ${icon}"></i>
                <div class="notification-content">
                    <div class="notification-message">${message}</div>
                </div>
                <button class="close-btn" onclick="this.parentElement.remove()">
                    <i class="bx bx-x"></i>
                </button>
            `;
            
            // Add to page
            document.body.appendChild(notification);
            
            // Show notification with animation
            setTimeout(() => {
                notification.classList.add('show');
            }, 100);
            
            // Auto-remove after 5 seconds
            setTimeout(() => {
                if (notification.parentElement) {
                    notification.classList.remove('show');
                    setTimeout(() => {
                        if (notification.parentElement) {
                            notification.remove();
                        }
                    }, 300);
                }
            }, 5000);
        }

        // Clear form function
        function clearForm() {
            document.getElementById('to').value = '';
            document.getElementById('cc').value = '';
            document.getElementById('bcc').value = '';
            document.getElementById('subject').value = '';
            document.getElementById('body').value = '';
            document.getElementById('priority').value = 'normal';
            updatePreview();
        }

        // Save draft functionality
        function saveDraft() {
            const to = document.getElementById('to').value.trim();
            const cc = document.getElementById('cc').value.trim();
            const bcc = document.getElementById('bcc').value.trim();
            const subject = document.getElementById('subject').value.trim();
            const body = document.getElementById('body').value.trim();
            const priority = document.getElementById('priority').value;
            
            // Debug logging
            console.log('Save Draft Debug:', {
                to: to,
                cc: cc, 
                bcc: bcc,
                subject: subject,
                body: body,
                priority: priority
            });
            
            // Check if there's content to save
            if (!to && !subject && !body) {
                showNotification('Cannot save empty draft', 'error');
                return;
            }
            
            // Show saving state
            const saveBtn = document.querySelector('button[onclick="saveDraft()"]');
            const originalText = saveBtn.innerHTML;
            saveBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Saving...';
            saveBtn.disabled = true;
            
            // Prepare form data
            const formData = new FormData();
            formData.append('draft_action', 'save');
            formData.append('to', to);
            formData.append('cc', cc);
            formData.append('bcc', bcc);
            formData.append('subject', subject);
            formData.append('body', body);
            formData.append('priority', priority);
            
            // Add draft ID if editing existing draft
            if (window.currentDraftId) {
                formData.append('draft_id', window.currentDraftId);
            }
            
            // Send AJAX request
            fetch('email_compose.php', {
                method: 'POST',
                body: formData
            })
            .then(response => {
                console.log('Response status:', response.status);
                console.log('Response headers:', response.headers);
                return response.text(); // Get as text first to see raw response
            })
            .then(responseText => {
                console.log('Raw response:', responseText);
                
                // Try to parse as JSON
                let data;
                try {
                    data = JSON.parse(responseText);
                    console.log('Parsed JSON:', data);
                } catch (e) {
                    console.error('JSON parse error:', e);
                    console.error('Response was not valid JSON:', responseText);
                    throw new Error('Invalid JSON response: ' + responseText.substring(0, 200));
                }
                
                // Reset button
                saveBtn.innerHTML = originalText;
                saveBtn.disabled = false;
                
                if (data.success) {
                    // Store draft ID for future updates
                    window.currentDraftId = data.draft_id;
                    showNotification(data.message, 'success');
                    
                    // Update page title to show draft status
                    document.title = 'Draft Saved - Compose Email - Panadite Academy Admin';
                } else {
                    console.error('Save draft failed:', data);
                    showNotification(data.error || 'Failed to save draft', 'error');
                }
            })
            .catch(error => {
                // Reset button
                saveBtn.innerHTML = originalText;
                saveBtn.disabled = false;
                
                showNotification('Network error while saving draft: ' + error.message, 'error');
                console.error('Detailed error:', error);
            });
        }

        // Auto-save draft every 2 minutes
        let autoSaveTimeout;
        function scheduleAutoSave() {
            clearTimeout(autoSaveTimeout);
            autoSaveTimeout = setTimeout(function() {
                const to = document.getElementById('to').value.trim();
                const cc = document.getElementById('cc').value.trim();
                const bcc = document.getElementById('bcc').value.trim();
                const subject = document.getElementById('subject').value.trim();
                const body = document.getElementById('body').value.trim();
                const priority = document.getElementById('priority').value;
                
                // Only auto-save if there's meaningful content
                if (to || subject || body) {
                    console.log('Auto-saving draft...');
                    
                    // Prepare form data
                    const formData = new FormData();
                    formData.append('draft_action', 'auto_save');
                    formData.append('to', to);
                    formData.append('cc', cc);
                    formData.append('bcc', bcc);
                    formData.append('subject', subject);
                    formData.append('body', body);
                    formData.append('priority', priority);
                    
                    // Send auto-save request
                    fetch('email_compose.php', {
                        method: 'POST',
                        body: formData
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            // Update draft ID and show subtle indication
                            window.currentDraftId = data.draft_id;
                            
                            // Show subtle auto-save indicator
                            const indicator = document.createElement('div');
                            indicator.style.cssText = `
                                position: fixed; top: 10px; right: 10px; z-index: 10000;
                                background: rgba(16, 185, 129, 0.9); color: white;
                                padding: 5px 10px; border-radius: 5px; font-size: 12px;
                                opacity: 0; transition: opacity 0.3s;
                            `;
                            indicator.textContent = 'Auto-saved';
                            document.body.appendChild(indicator);
                            
                            setTimeout(() => indicator.style.opacity = '1', 10);
                            setTimeout(() => {
                                indicator.style.opacity = '0';
                                setTimeout(() => indicator.remove(), 300);
                            }, 2000);
                        }
                    })
                    .catch(error => {
                        console.error('Auto-save failed:', error);
                    });
                }
                
                // Schedule next auto-save
                scheduleAutoSave();
            }, 120000); // 2 minutes
        }
        
        // Start auto-save scheduling
        scheduleAutoSave();
        
        // Reset auto-save timer on user input
        ['to', 'cc', 'bcc', 'subject', 'body', 'priority'].forEach(fieldId => {
            const element = document.getElementById(fieldId);
            if (element) {
                element.addEventListener('input', scheduleAutoSave);
                element.addEventListener('change', scheduleAutoSave);
            }
        });

        // Initialize preview
        updatePreview();
    </script>
</body>
</html>
