<?php
session_start();

// Include necessary files
require_once __DIR__ . '/../auth/functions.php';
require_once '../config/email_drafts_service.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Get current user
$currentUser = getCurrentUser();

// Handle draft actions via AJAX
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    $action = $_POST['action'];
    $userId = $currentUser['user_id'] ?? $currentUser['id'];
    
    switch ($action) {
        case 'delete':
            $draftId = intval($_POST['draft_id']);
            $result = deleteDraft($draftId, $userId);
            
            if ($result) {
                echo json_encode(['success' => true, 'message' => 'Draft deleted successfully']);
            } else {
                echo json_encode(['success' => false, 'error' => 'Failed to delete draft']);
            }
            break;
            
        case 'load':
            $draftId = intval($_POST['draft_id']);
            $draft = getDraft($draftId, $userId);
            
            if ($draft) {
                echo json_encode(['success' => true, 'draft' => $draft]);
            } else {
                echo json_encode(['success' => false, 'error' => 'Draft not found']);
            }
            break;
            
        default:
            echo json_encode(['success' => false, 'error' => 'Invalid action']);
    }
    exit();
}

// Get user's drafts
$userId = $currentUser['user_id'] ?? $currentUser['id'];
$drafts = getUserDrafts($userId, 50);
$draftsCount = getDraftsCount($userId);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Email Drafts - Panadite Academy Admin</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link rel="stylesheet" href="/assets/css/user-cards.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        .drafts-header { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); 
            color: white; 
            border-radius: 10px 10px 0 0; 
        }
        .draft-item {
            border-left: 4px solid #667eea;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        .draft-item:hover {
            background-color: #f8f9fa;
            border-left-color: #764ba2;
            transform: translateX(5px);
        }
        .draft-preview {
            color: #6c757d;
            font-size: 0.9em;
            max-height: 3em;
            overflow: hidden;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
        }
        .draft-meta {
            font-size: 0.8em;
            color: #6c757d;
        }
        .draft-actions {
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        .draft-item:hover .draft-actions {
            opacity: 1;
        }
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #6c757d;
        }
        .empty-state i {
            font-size: 4rem;
            margin-bottom: 20px;
            opacity: 0.5;
        }
        .priority-badge {
            padding: 2px 6px;
            border-radius: 10px;
            font-size: 10px;
            font-weight: 500;
        }
        .priority-high { background: #dc3545; color: white; }
        .priority-normal { background: #6c757d; color: white; }
        .priority-low { background: #28a745; color: white; }
        
        /* Beautiful Notification Styles */
        .notification {
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 9999;
            max-width: 400px;
            padding: 16px 24px;
            border-radius: 12px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.12);
            backdrop-filter: blur(10px);
            transform: translateX(100%);
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 12px;
        }
        .notification.show {
            transform: translateX(0);
        }
        .notification.success {
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
            color: white;
            border-left: 4px solid #34d399;
        }
        .notification.error {
            background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
            color: white;
            border-left: 4px solid #f87171;
        }
        .notification i {
            font-size: 24px;
            opacity: 0.9;
        }
        .notification .close-btn {
            margin-left: auto;
            background: none;
            border: none;
            color: inherit;
            font-size: 18px;
            cursor: pointer;
            opacity: 0.8;
            transition: opacity 0.2s;
        }
        .notification .close-btn:hover {
            opacity: 1;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="row">
                <div class="col-12">
                    <div class="card shadow-sm">
                        <!-- Header -->
                        <div class="card-header drafts-header">
                            <div class="d-flex justify-content-between align-items-center">
                                <h4 class="mb-0">
                                    <i class="bx bx-edit me-2"></i>
                                    Email Drafts
                                    <span class="badge bg-light text-dark ms-2"><?= $draftsCount ?></span>
                                </h4>
                                <div>
                                    <a href="email_compose.php" class="btn btn-light btn-sm me-2">
                                        <i class="bx bx-plus"></i> New Email
                                    </a>
                                    <button class="btn btn-light btn-sm" onclick="refreshDrafts()">
                                        <i class="bx bx-refresh"></i> Refresh
                                    </button>
                                </div>
                            </div>
                        </div>

                        <!-- Drafts List -->
                        <div class="card-body p-0">
                            <?php if (empty($drafts)): ?>
                                <div class="empty-state">
                                    <i class="bx bx-edit-alt"></i>
                                    <h5>No Drafts Found</h5>
                                    <p class="mb-3">You haven't saved any email drafts yet.</p>
                                    <a href="email_compose.php" class="btn btn-primary">
                                        <i class="bx bx-plus"></i> Compose Email
                                    </a>
                                </div>
                            <?php else: ?>
                                <div id="drafts-container">
                                    <?php foreach ($drafts as $draft): ?>
                                        <div class="draft-item p-3 border-bottom" data-draft-id="<?= $draft['id'] ?>">
                                            <div class="row align-items-center">
                                                <div class="col-md-8">
                                                    <div class="d-flex align-items-center mb-2">
                                                        <strong class="me-2">To: <?= htmlspecialchars($draft['to_email']) ?></strong>
                                                        <span class="priority-badge priority-<?= $draft['priority'] ?>">
                                                            <?= ucfirst($draft['priority']) ?>
                                                        </span>
                                                    </div>
                                                    <h6 class="mb-1">
                                                        <?= htmlspecialchars($draft['subject'] ?: '[No Subject]') ?>
                                                    </h6>
                                                    <div class="draft-preview">
                                                        <?= htmlspecialchars(formatDraftPreview($draft['body'], 150)) ?>
                                                    </div>
                                                </div>
                                                <div class="col-md-3">
                                                    <div class="draft-meta">
                                                        <div><i class="bx bx-time"></i> Created: <?= date('M j, Y g:i A', strtotime($draft['created_at'])) ?></div>
                                                        <?php if ($draft['updated_at'] != $draft['created_at']): ?>
                                                            <div><i class="bx bx-edit"></i> Updated: <?= date('M j, Y g:i A', strtotime($draft['updated_at'])) ?></div>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                                <div class="col-md-1">
                                                    <div class="draft-actions d-flex gap-1">
                                                        <button class="btn btn-sm btn-primary" onclick="loadDraft(<?= $draft['id'] ?>)" title="Edit Draft">
                                                            <i class="bx bx-edit"></i>
                                                        </button>
                                                        <button class="btn btn-sm btn-danger" onclick="deleteDraft(<?= $draft['id'] ?>)" title="Delete Draft">
                                                            <i class="bx bx-trash"></i>
                                                        </button>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>

    <script>
        // Beautiful notification system
        function showNotification(message, type = 'success') {
            // Remove any existing notifications
            const existingNotifications = document.querySelectorAll('.notification');
            existingNotifications.forEach(notification => notification.remove());
            
            // Create notification element
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            
            const icon = type === 'success' ? 'bx-check-circle' : 'bx-error-circle';
            
            notification.innerHTML = `
                <i class="bx ${icon}"></i>
                <div class="notification-content">
                    <div class="notification-message">${message}</div>
                </div>
                <button class="close-btn" onclick="this.parentElement.remove()">
                    <i class="bx bx-x"></i>
                </button>
            `;
            
            // Add to page
            document.body.appendChild(notification);
            
            // Show notification with animation
            setTimeout(() => {
                notification.classList.add('show');
            }, 100);
            
            // Auto-remove after 5 seconds
            setTimeout(() => {
                if (notification.parentElement) {
                    notification.classList.remove('show');
                    setTimeout(() => {
                        if (notification.parentElement) {
                            notification.remove();
                        }
                    }, 300);
                }
            }, 5000);
        }

        // Load draft for editing
        function loadDraft(draftId) {
            // Redirect to compose page with draft ID
            window.location.href = `email_compose.php?draft_id=${draftId}`;
        }

        // Delete draft
        function deleteDraft(draftId) {
            if (!confirm('Are you sure you want to delete this draft? This action cannot be undone.')) {
                return;
            }

            const formData = new FormData();
            formData.append('action', 'delete');
            formData.append('draft_id', draftId);

            fetch('email_drafts.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Remove the draft item from the page
                    const draftElement = document.querySelector(`[data-draft-id="${draftId}"]`);
                    if (draftElement) {
                        draftElement.style.opacity = '0';
                        draftElement.style.transform = 'translateX(-100%)';
                        setTimeout(() => {
                            draftElement.remove();
                            
                            // Check if no more drafts
                            const remainingDrafts = document.querySelectorAll('.draft-item');
                            if (remainingDrafts.length === 0) {
                                location.reload(); // Reload to show empty state
                            } else {
                                // Update count
                                const badge = document.querySelector('.badge');
                                if (badge) {
                                    badge.textContent = remainingDrafts.length;
                                }
                            }
                        }, 300);
                    }
                    
                    showNotification(data.message, 'success');
                } else {
                    showNotification(data.error || 'Failed to delete draft', 'error');
                }
            })
            .catch(error => {
                showNotification('Network error while deleting draft', 'error');
                console.error('Error:', error);
            });
        }

        // Refresh drafts
        function refreshDrafts() {
            location.reload();
        }

        // Add click handlers for draft items (excluding action buttons)
        document.addEventListener('DOMContentLoaded', function() {
            const draftItems = document.querySelectorAll('.draft-item');
            draftItems.forEach(item => {
                item.addEventListener('click', function(e) {
                    // Don't trigger if clicking on action buttons
                    if (e.target.closest('.draft-actions')) {
                        return;
                    }
                    
                    const draftId = this.dataset.draftId;
                    loadDraft(draftId);
                });
            });
        });
    </script>
</body>
</html>
