<?php
require_once '../auth/functions.php';
require_once '../includes/config.php';

// Ensure user is logged in as an admin
requireRole('admin');

$userId = $_SESSION['user_id'];
$user = getCurrentUser();

$message = '';
$messageType = '';

// Get current PayFast settings
$pfSettings = [];
try {
    // First check the structure of the payfast_settings table
    $idExists = false;
    $tableStructureQuery = "SHOW COLUMNS FROM payfast_settings";
    $structureResult = $conn->query($tableStructureQuery);
    
    if ($structureResult) {
        while ($column = $structureResult->fetch_assoc()) {
            if ($column['Field'] === 'id') {
                $idExists = true;
                break;
            }
        }
    }
    
    // Fetch settings based on table structure
    if ($idExists) {
        $stmt = $conn->prepare("SELECT * FROM payfast_settings WHERE id = 1");
    } else {
        // If no ID column, just get the first row
        $stmt = $conn->prepare("SELECT * FROM payfast_settings LIMIT 1");
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    $pfSettings = $result->fetch_assoc();
    
    // If no settings exist, initialize with empty values
    if (!$pfSettings) {
        $pfSettings = [
            'merchant_id' => '',
            'merchant_key' => '',
            'passphrase' => '',
            'sandbox_mode' => 0
        ];
    }
} catch (Exception $e) {
    $message = "Error retrieving PayFast settings: " . $e->getMessage();
    $messageType = 'danger';
}

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $merchant_id = trim($_POST['merchant_id']);
    $merchant_key = trim($_POST['merchant_key']);
    $passphrase = trim($_POST['passphrase']);
    $test_mode = isset($_POST['test_mode']) ? 1 : 0;
    
    try {
        // Check if id column exists
        $idExists = false;
        $tableStructureQuery = "SHOW COLUMNS FROM payfast_settings";
        $structureResult = $conn->query($tableStructureQuery);
        
        if ($structureResult) {
            while ($column = $structureResult->fetch_assoc()) {
                if ($column['Field'] === 'id') {
                    $idExists = true;
                    break;
                }
            }
        }
        
        // Check if any rows exist
        $checkQuery = "SELECT COUNT(*) as count FROM payfast_settings";
        $checkResult = $conn->query($checkQuery);
        $rowCount = 0;
        if ($checkResult && $row = $checkResult->fetch_assoc()) {
            $rowCount = (int)$row['count'];
        }
        
        if ($rowCount === 0) {
            // Insert new settings if table is empty
            $stmt = $conn->prepare("INSERT INTO payfast_settings (merchant_id, merchant_key, passphrase, sandbox_mode) VALUES (?, ?, ?, ?)");
            $stmt->bind_param("sssi", $merchant_id, $merchant_key, $passphrase, $test_mode);
        } else {
            // Update existing settings
            if ($idExists) {
                $stmt = $conn->prepare("
                    UPDATE payfast_settings 
                    SET merchant_id = ?, merchant_key = ?, passphrase = ?, sandbox_mode = ?
                    WHERE id = 1
                ");
            } else {
                // If no id column, update the first row
                $stmt = $conn->prepare("
                    UPDATE payfast_settings 
                    SET merchant_id = ?, merchant_key = ?, passphrase = ?, sandbox_mode = ?
                    LIMIT 1
                ");
            }
            $stmt->bind_param("sssi", $merchant_id, $merchant_key, $passphrase, $test_mode);
        }
        $stmt->execute();
        
        if ($stmt->affected_rows > 0) {
            $message = "PayFast settings updated successfully!";
            $messageType = 'success';
            
            // Refresh settings
            if ($idExists) {
                $stmt = $conn->prepare("SELECT * FROM payfast_settings WHERE id = 1");
            } else {
                $stmt = $conn->prepare("SELECT * FROM payfast_settings LIMIT 1");
            }
            $stmt->execute();
            $result = $stmt->get_result();
            $pfSettings = $result->fetch_assoc();
        } else {
            $message = "No changes were made to the PayFast settings.";
            $messageType = 'info';
        }
    } catch (Exception $e) {
        $message = "Error updating PayFast settings: " . $e->getMessage();
        $messageType = 'danger';
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PayFast Settings - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="page-header">
                    <h1 class="h2">PayFast Settings</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="dashboard.php" class="btn btn-sm btn-outline-secondary">
                            <i class="fas fa-arrow-left me-1"></i> Back to Dashboard
                        </a>
                    </div>
            </div>
            
            <?php if (!empty($message)): ?>
                <div class="alert-container animate-fade-in">
                    <div class="alert alert-<?php echo $messageType; ?> alert-dismissible fade show" role="alert">
                        <?php echo $message; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                </div>
            <?php endif; ?>
            
            <div class="table-container animate-fade-in">
                <div class="table-header">
                    <h5>PayFast Integration Settings</h5>
                </div>
                    <div class="card-body p-4">
                        <form action="" method="post">
                            <!-- Form with tips on the right side -->
                            <div class="row mb-4">
                                <div class="col-md-7">
                                    <div class="form-floating mb-0">
                                        <input type="text" class="form-control" id="merchant_id" name="merchant_id" value="<?php echo htmlspecialchars($pfSettings['merchant_id'] ?? ''); ?>" required>
                                        <label for="merchant_id">Merchant ID <span class="text-danger">*</span></label>
                                    </div>
                                </div>
                                <div class="col-md-5">
                                    <div class="form-tip">
                                        <div class="tip-icon"><i class='bx bx-info-circle'></i></div>
                                        <div class="tip-text">Your PayFast Merchant ID from your PayFast account.</div>
                                    </div>
                                </div>
                            </div>

                            <div class="row mb-4">
                                <div class="col-md-7">
                                    <div class="form-floating mb-0">
                                        <input type="text" class="form-control" id="merchant_key" name="merchant_key" value="<?php echo htmlspecialchars($pfSettings['merchant_key'] ?? ''); ?>" required>
                                        <label for="merchant_key">Merchant Key <span class="text-danger">*</span></label>
                                    </div>
                                </div>
                                <div class="col-md-5">
                                    <div class="form-tip">
                                        <div class="tip-icon"><i class='bx bx-info-circle'></i></div>
                                        <div class="tip-text">Your PayFast Merchant Key from your PayFast account.</div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row mb-4">
                                <div class="col-md-7">
                                    <div class="form-floating mb-0">
                                        <input type="text" class="form-control" id="passphrase" name="passphrase" value="<?php echo htmlspecialchars($pfSettings['passphrase'] ?? ''); ?>">
                                        <label for="passphrase">Passphrase</label>
                                    </div>
                                </div>
                                <div class="col-md-5">
                                    <div class="form-tip">
                                        <div class="tip-icon"><i class='bx bx-info-circle'></i></div>
                                        <div class="tip-text">Optional security passphrase set in your PayFast account.</div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row mb-4">
                                <div class="col-md-7">
                                    <div class="form-check form-switch ps-0">
                                        <div class="d-flex align-items-center">
                                            <div class="form-check form-switch">
                                                <input type="checkbox" class="form-check-input" id="test_mode" name="test_mode" <?php echo ($pfSettings['sandbox_mode'] ?? 0) ? 'checked' : ''; ?>>
                                                <label class="form-check-label ps-2" for="test_mode">Test Mode</label>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-5">
                                    <div class="form-tip">
                                        <div class="tip-icon"><i class='bx bx-info-circle'></i></div>
                                        <div class="tip-text">If checked, the system will use the PayFast sandbox environment for testing.</div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mt-4 mb-4">
                                <button type="submit" class="btn btn-primary px-4 py-2"><i class='bx bx-save me-2'></i>Save Settings</button>
                            </div>
                        </form>
                        
                        <hr class="my-4">
                        
                        <!-- Integration Guide Card -->
                        <div class="integration-guide p-4 bg-light rounded-3 border animate-fade-in" style="--delay: 0.2s">
                            <h5 class="d-flex align-items-center mb-3"><i class='bx bx-book-open me-2 text-primary'></i> PayFast Integration Guide</h5>
                            <p>Follow these steps to set up your PayFast integration:</p>
                            <ol class="ps-3">
                                <li class="mb-2">Sign up for a <a href="https://www.payfast.co.za/" target="_blank" class="text-decoration-none">PayFast account</a> if you don't have one.</li>
                                <li class="mb-2">Get your Merchant ID and Merchant Key from your PayFast dashboard.</li>
                                <li class="mb-2">Optionally set a Passphrase in your PayFast account for added security.</li>
                                <li class="mb-2">Configure your Return, Cancel, and Notify URLs in your PayFast account.</li>
                                <li class="mb-2">Test your integration using the sandbox environment before going live.</li>
                            </ol>
                            
                            <div class="alert alert-info mt-3 mb-0">
                                <strong><i class='bx bx-test-tube me-1'></i> Test Credentials:</strong>
                                <div class="row mt-2">
                                    <div class="col-md-6">
                                        <div class="mb-1"><strong>Sandbox Merchant ID:</strong> 10000100</div>
                                    </div>
                                    <div class="col-md-6">
                                        <div><strong>Sandbox Merchant Key:</strong> 46f0cd694581a</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Sidebar toggle functionality
        const sidebarToggle = document.querySelector('.sidebar-toggle');
        const sidebarToggleMobile = document.querySelector('.sidebar-toggle-mobile');
        const sidebar = document.querySelector('.sidebar');
        
        if (sidebarToggle) {
            sidebarToggle.addEventListener('click', () => {
                sidebar.classList.toggle('close');
            });
        }
        
        if (sidebarToggleMobile) {
            sidebarToggleMobile.addEventListener('click', () => {
                sidebar.classList.toggle('show');
            });
        }
        
        // Responsive sidebar behavior
        function checkScreenSize() {
            if (window.innerWidth < 768) {
                sidebar.classList.remove('close');
                sidebar.classList.remove('show');
            } else if (window.innerWidth < 992) {
                sidebar.classList.add('close');
                sidebar.classList.remove('show');
            }
        }
        
        // Initial check and listen for window resize
        checkScreenSize();
        window.addEventListener('resize', checkScreenSize);
        
        // Dropdown functionality
        document.querySelectorAll('.dropdown-toggle').forEach(item => {
            item.addEventListener('click', event => {
                event.preventDefault();
                const dropdownMenu = item.nextElementSibling;
                if (dropdownMenu && dropdownMenu.classList.contains('dropdown-menu')) {
                    dropdownMenu.classList.toggle('show');
                }
            });
        });
        
        // Close dropdowns when clicking outside
        document.addEventListener('click', event => {
            const isDropdownButton = event.target.classList.contains('dropdown-toggle');
            if (!isDropdownButton) {
                document.querySelectorAll('.dropdown-menu.show').forEach(dropdown => {
                    dropdown.classList.remove('show');
                });
            }
        });
    });
    </script>
</body>
</html>
