<?php
require_once __DIR__ . '/../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

$message = '';

// Handle delete operations
if ($_POST && isset($_POST['action'])) {
    try {
        $conn->autocommit(FALSE);
        
        switch ($_POST['action']) {
            case 'delete_course_payment':
                $payment_id = intval($_POST['payment_id']);
                $stmt = $conn->prepare("DELETE FROM course_purchases WHERE purchase_id = ?");
                $stmt->bind_param("i", $payment_id);
                $stmt->execute();
                $message = '<div class="alert alert-success"><i class="bx bx-check-circle"></i> Course payment deleted successfully!</div>';
                break;
                
            case 'delete_video_payment':
                $payment_id = intval($_POST['payment_id']);
                $stmt = $conn->prepare("DELETE FROM video_purchases WHERE purchase_id = ?");
                $stmt->bind_param("i", $payment_id);
                $stmt->execute();
                $message = '<div class="alert alert-success"><i class="bx bx-check-circle"></i> Video payment deleted successfully!</div>';
                break;
                
            case 'delete_withdrawal':
                $request_id = intval($_POST['request_id']);
                $stmt = $conn->prepare("DELETE FROM withdrawal_requests WHERE id = ?");
                $stmt->bind_param("i", $request_id);
                $stmt->execute();
                $message = '<div class="alert alert-success"><i class="bx bx-check-circle"></i> Withdrawal request deleted successfully!</div>';
                break;
                
            case 'delete_teacher_earning':
                $earning_id = intval($_POST['earning_id']);
                $stmt = $conn->prepare("DELETE FROM teacher_earnings WHERE earning_id = ?");
                $stmt->bind_param("i", $earning_id);
                $stmt->execute();
                $message = '<div class="alert alert-success"><i class="bx bx-check-circle"></i> Teacher earning deleted successfully!</div>';
                break;
                
            case 'delete_referral_commission':
                $commission_id = intval($_POST['commission_id']);
                $stmt = $conn->prepare("DELETE FROM teacher_referral_commissions WHERE id = ?");
                $stmt->bind_param("i", $commission_id);
                $stmt->execute();
                $message = '<div class="alert alert-success"><i class="bx bx-check-circle"></i> Referral commission deleted successfully!</div>';
                break;
                
            case 'delete_all_payments':
                // Delete all payment records from all tables
                $conn->query("DELETE FROM course_purchases");
                $conn->query("DELETE FROM video_purchases");
                $conn->query("DELETE FROM withdrawal_requests");
                $conn->query("DELETE FROM teacher_earnings");
                
                // Reset auto-increment IDs
                $conn->query("ALTER TABLE course_purchases AUTO_INCREMENT = 1");
                $conn->query("ALTER TABLE video_purchases AUTO_INCREMENT = 1");
                $conn->query("ALTER TABLE withdrawal_requests AUTO_INCREMENT = 1");
                $conn->query("ALTER TABLE teacher_earnings AUTO_INCREMENT = 1");
                
                $message = '<div class="alert alert-success"><i class="bx bx-check-circle"></i> All payment records deleted successfully! System cleaned.</div>';
                break;
        }
        
        $conn->commit();
        $conn->autocommit(TRUE);
        
    } catch (Exception $e) {
        $conn->rollback();
        $conn->autocommit(TRUE);
        $message = '<div class="alert alert-danger"><i class="bx bx-error-circle"></i> Error: ' . $e->getMessage() . '</div>';
    }
}

// Get payments data

$course_payments = [];
$withdrawal_requests = [];
$teacher_earnings = [];
$referral_payments = [];

// Video payments removed - table has 0 records

// Get referral payments (teacher commission data)
try {
    // Check if teacher_referral_commissions table exists
    $table_check = $conn->query("SHOW TABLES LIKE 'teacher_referral_commissions'");
    
    if ($table_check && $table_check->num_rows > 0) {
        $referral_query = "
            SELECT trc.id, trc.teacher_id as referrer_id, trc.referred_student_id, trc.course_id, 
                   trc.commission_amount, trc.status, trc.created_at, trc.purchase_amount,
                   trc.commission_percentage, trc.purchase_id,
                   COALESCE(t.first_name, 'Deleted') as referrer_first_name,
                   COALESCE(t.last_name, 'Teacher') as referrer_last_name,
                   COALESCE(u1.username, 'Deleted User') as referrer_name,
                   COALESCE(u1.email, 'No Email') as referrer_email,
                   COALESCE(u2.username, 'Deleted User') as referred_name,
                   COALESCE(u2.first_name, 'Deleted') as referred_first_name,
                   COALESCE(u2.last_name, 'User') as referred_last_name,
                   COALESCE(u2.email, 'No Email') as referred_email,
                   COALESCE(c.title, 'Deleted Course') as course_title
            FROM teacher_referral_commissions trc
            LEFT JOIN teachers t ON trc.teacher_id = t.teacher_id
            LEFT JOIN users u1 ON trc.teacher_id = u1.user_id
            LEFT JOIN users u2 ON trc.referred_student_id = u2.user_id
            LEFT JOIN courses c ON trc.course_id = c.course_id
            ORDER BY trc.created_at DESC
        ";
        
        $stmt = $conn->prepare($referral_query);
        if ($stmt) {
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $referral_payments[] = $row;
            }
            $stmt->close();
        }
    }
} catch (Exception $e) {
    $message .= '<div class="alert alert-info alert-dismissible fade show" role="alert">
        <i class="bx bx-info-circle me-2"></i>Teacher referral commissions table not found. This is normal if not using teacher referral system.
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>';
    $referral_payments = [];
}

// Get course payments
try {
    // Check if payment_date column exists in course_purchases table
    $checkDateColumn = "SHOW COLUMNS FROM course_purchases LIKE 'payment_date'"; 
    $dateColumnResult = $conn->query($checkDateColumn);
    $hasDateColumn = ($dateColumnResult && $dateColumnResult->num_rows > 0);
    
    // Check if created_at column exists as an alternative
    $checkCreatedColumn = "SHOW COLUMNS FROM course_purchases LIKE 'created_at'"; 
    $createdColumnResult = $conn->query($checkCreatedColumn);
    $hasCreatedColumn = ($createdColumnResult && $createdColumnResult->num_rows > 0);
    
    // Check if payment_method column exists
    $checkMethodColumn = "SHOW COLUMNS FROM course_purchases LIKE 'payment_method'"; 
    $methodColumnResult = $conn->query($checkMethodColumn);
    $hasMethodColumn = ($methodColumnResult && $methodColumnResult->num_rows > 0);
    
    // Check if status column exists
    $checkStatusColumn = "SHOW COLUMNS FROM course_purchases LIKE 'status'"; 
    $statusColumnResult = $conn->query($checkStatusColumn);
    $hasStatusColumn = ($statusColumnResult && $statusColumnResult->num_rows > 0);
    
    // Build query based on available columns
    $dateField = '';
    $methodField = '';
    $statusField = '';
    $orderBy = '';
    
    if ($hasDateColumn) {
        $dateField = 'cp.payment_date';
        $orderBy = 'cp.payment_date DESC';
    } elseif ($hasCreatedColumn) {
        $dateField = 'cp.created_at as payment_date';
        $orderBy = 'cp.created_at DESC';
    } else {
        // Fallback if neither column exists
        $dateField = "'No Date Available' as payment_date";
        $orderBy = 'cp.purchase_id DESC';
    }
    
    // Handle payment method column
    if ($hasMethodColumn) {
        $methodField = 'cp.payment_method';
    } else {
        $methodField = "'Unknown' as payment_method";
    }
    
    // Handle status column
    if ($hasStatusColumn) {
        $statusField = 'cp.status';
    } else {
        // Check for payment_status column (more common)
        $checkPaymentStatusColumn = "SHOW COLUMNS FROM course_purchases LIKE 'payment_status'"; 
        $paymentStatusColumnResult = $conn->query($checkPaymentStatusColumn);
        $hasPaymentStatusColumn = ($paymentStatusColumnResult && $paymentStatusColumnResult->num_rows > 0);
        
        if ($hasPaymentStatusColumn) {
            $statusField = 'cp.payment_status as status';
        } else {
            $statusField = "'completed' as status";
        }
    }
    
    $query = "
        SELECT cp.purchase_id, cp.user_id, cp.course_id, cp.amount, {$dateField}, {$methodField}, {$statusField},
               COALESCE(u.username, 'Deleted User') as username, 
               COALESCE(u.email, 'No Email') as email, 
               COALESCE(u.first_name, 'Deleted') as first_name, 
               COALESCE(u.last_name, 'User') as last_name,
               COALESCE(c.title, 'Deleted Course') as course_title
        FROM course_purchases cp
        LEFT JOIN users u ON cp.user_id = u.user_id
        LEFT JOIN courses c ON cp.course_id = c.course_id
        ORDER BY {$orderBy}
    ";
    
    $stmt = $conn->prepare($query);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $course_payments[] = $row;
    }
} catch (Exception $e) {
    $message .= '<div class="alert alert-danger alert-dismissible fade show" role="alert">
        <i class="bx bx-error-circle me-2"></i>Error retrieving course payments: ' . $e->getMessage() . '
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>';
}

// Get withdrawal requests (with table existence check)
try {
    // First check if withdrawal_requests table exists
    $table_check = $conn->query("SHOW TABLES LIKE 'withdrawal_requests'");
    
    if ($table_check && $table_check->num_rows > 0) {
        $withdrawal_query = "
            SELECT wr.*, u.username, u.email, u.first_name, u.last_name
            FROM withdrawal_requests wr
            JOIN users u ON wr.teacher_id = u.user_id
            ORDER BY wr.requested_at DESC
        ";
        
        $stmt = $conn->prepare($withdrawal_query);
        if ($stmt) {
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $withdrawal_requests[] = $row;
            }
            $stmt->close();
        } else {
            throw new Exception("Failed to prepare withdrawal requests query: " . $conn->error);
        }
    } else {
        // Table doesn't exist, leave withdrawal_requests empty
        $withdrawal_requests = [];
    }
} catch (Exception $e) {
    $message .= '<div class="alert alert-warning alert-dismissible fade show" role="alert">
        <i class="bx bx-info-circle me-2"></i>Withdrawal requests table not found or accessible. This is normal if not using withdrawal system.
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>';
    $withdrawal_requests = [];
}

// Get teacher earnings (with table existence check)
try {
    // First check if teacher_earnings table exists
    $table_check = $conn->query("SHOW TABLES LIKE 'teacher_earnings'");
    
    if ($table_check && $table_check->num_rows > 0) {
        $earnings_query = "
            SELECT te.earning_id, te.teacher_id, te.purchase_id, te.course_id, te.amount, 
                   te.status, te.created_at, te.updated_at, te.transaction_type, te.withdrawal_request_id,
                   COALESCE(u.username, 'Deleted User') as username, 
                   COALESCE(u.email, 'No Email') as email, 
                   COALESCE(u.first_name, 'Deleted') as first_name, 
                   COALESCE(u.last_name, 'Teacher') as last_name,
                   COALESCE(c.title, 'Deleted Course') as course_title
            FROM teacher_earnings te
            LEFT JOIN users u ON te.teacher_id = u.user_id
            LEFT JOIN courses c ON te.course_id = c.course_id
            ORDER BY te.created_at DESC
        ";
        
        $stmt = $conn->prepare($earnings_query);
        if ($stmt) {
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $teacher_earnings[] = $row;
            }
            $stmt->close();
        } else {
            throw new Exception("Failed to prepare teacher earnings query: " . $conn->error);
        }
    } else {
        // Table doesn't exist, leave teacher_earnings empty
        $teacher_earnings = [];
    }
} catch (Exception $e) {
    $message .= '<div class="alert alert-warning alert-dismissible fade show" role="alert">
        <i class="bx bx-info-circle me-2"></i>Teacher earnings table not found or accessible. This is normal if not using teacher earnings system.
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>';
    $teacher_earnings = [];
}

// Calculate comprehensive revenue statistics
$total_revenue = 0;
$course_revenue = 0;
$referral_revenue = 0;
$monthly_revenue = 0;
$total_withdrawals = 0;
$total_teacher_earnings = 0;

// Revenue calculation mode - set this to match your dashboard behavior
$count_all_transactions = true; // Set to false to only count 'completed' transactions

// Revenue calculation for production


foreach ($course_payments as $payment) {
    // Since we want all transactions, just include all payments
    if ($count_all_transactions) {
        $total_revenue += $payment['amount'];
        $course_revenue += $payment['amount'];
        
        // Calculate monthly revenue (current month)
        try {
            $payment_date = new DateTime($payment['payment_date']);
            $current_month = new DateTime('first day of this month');
            if ($payment_date >= $current_month) {
                $monthly_revenue += $payment['amount'];
            }
        } catch (Exception $e) {
            // Skip this payment for monthly calculation if date is invalid
        }
    } else {
        // Only completed transactions
        if ($payment['status'] === 'completed') {
            $total_revenue += $payment['amount'];
            $course_revenue += $payment['amount'];
            
            // Calculate monthly revenue (current month)
            try {
                $payment_date = new DateTime($payment['payment_date']);
                $current_month = new DateTime('first day of this month');
                if ($payment_date >= $current_month) {
                    $monthly_revenue += $payment['amount'];
                }
            } catch (Exception $e) {
                // Skip this payment for monthly calculation if date is invalid
            }
        }
    }
}

// Calculate referral commission revenue
foreach ($referral_payments as $referral) {
    // Count all transactions or only approved/paid ones
    if ($count_all_transactions || in_array($referral['status'], ['approved', 'paid'])) {
        $total_revenue += $referral['commission_amount'];
        $referral_revenue += $referral['commission_amount'];
        
        // Calculate monthly revenue (current month)
        try {
            $referral_date = new DateTime($referral['created_at']);
            $current_month = new DateTime('first day of this month');
            if ($referral_date >= $current_month) {
                $monthly_revenue += $referral['commission_amount'];
            }
        } catch (Exception $e) {
            // Skip this referral for monthly calculation if date is invalid
        }
    }
}

// Debug output removed for production

// Calculate withdrawal amounts
foreach ($withdrawal_requests as $withdrawal) {
    if ($withdrawal['status'] === 'approved' || $withdrawal['status'] === 'completed') {
        $total_withdrawals += $withdrawal['amount'];
    }
}

// Calculate teacher earnings (positive values are earnings, negative are deductions)
foreach ($teacher_earnings as $earning) {
    $total_teacher_earnings += $earning['amount'];
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Management - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        .bg-purple { background-color: #8b5cf6 !important; }
        .avatar-text.bg-purple { background-color: #8b5cf6 !important; color: white; }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <!-- Page Header -->
            <div class="page-header">
                <h1>Payment Management</h1>
            </div>
            
            <?php if (!empty($message)): ?>
                <div class="alert-container animate-fade-in">
                    <?php echo $message; ?>
                </div>
            <?php endif; ?>
            
            <!-- Revenue Stats Cards -->
            <div class="row g-4 mb-4">
                <div class="col-12 col-md-6 col-lg-3">
                    <div class="stat-card animate-fade-in">
                        <div class="stat-card-info">
                            <span class="stat-card-title">Total Revenue</span>
                            <span class="stat-card-value">R<?= number_format($total_revenue, 2) ?></span>
                            <span class="stat-card-description">From all purchases</span>
                        </div>
                        <div class="stat-card-icon bg-success">
                            <i class='bx bx-money'></i>
                        </div>
                    </div>
                </div>
                
                <div class="col-12 col-md-6 col-lg-3">
                    <div class="stat-card animate-fade-in" style="--delay: 0.1s">
                        <div class="stat-card-info">
                            <span class="stat-card-title">Course Revenue</span>
                            <span class="stat-card-value">R<?= number_format($course_revenue, 2) ?></span>
                            <span class="stat-card-description">From course enrollments</span>
                        </div>
                        <div class="stat-card-icon bg-warning">
                            <i class='bx bx-book-open'></i>
                        </div>
                    </div>
                </div>
                
                <div class="col-12 col-md-6 col-lg-3">
                    <div class="stat-card animate-fade-in" style="--delay: 0.15s">
                        <div class="stat-card-info">
                            <span class="stat-card-title">Referral Revenue</span>
                            <span class="stat-card-value">R<?= number_format($referral_revenue, 2) ?></span>
                            <span class="stat-card-description">From referral commissions</span>
                        </div>
                        <div class="stat-card-icon bg-purple">
                            <i class='bx bx-group'></i>
                        </div>
                    </div>
                </div>
                
                <div class="col-12 col-md-6 col-lg-3">
                    <div class="stat-card animate-fade-in" style="--delay: 0.25s">
                        <div class="stat-card-info">
                            <span class="stat-card-title">Total Withdrawals</span>
                            <span class="stat-card-value">R<?= number_format($total_withdrawals, 2) ?></span>
                            <span class="stat-card-description">Approved withdrawals</span>
                        </div>
                        <div class="stat-card-icon bg-danger">
                            <i class='bx bx-money-withdraw'></i>
                        </div>
                    </div>
                </div>
                
                <div class="col-12 col-md-6 col-lg-3">
                    <div class="stat-card animate-fade-in" style="--delay: 0.35s">
                        <div class="stat-card-info">
                            <span class="stat-card-title">Teacher Earnings</span>
                            <span class="stat-card-value">R<?= number_format($total_teacher_earnings, 2) ?></span>
                            <span class="stat-card-description">Total teacher earnings</span>
                        </div>
                        <div class="stat-card-icon bg-info">
                            <i class='bx bx-user-voice'></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Delete All Payments Button -->
            <div class="d-flex justify-content-end mb-4">
                <button type="button" class="btn btn-danger" onclick="confirmDeleteAll()">
                    <i class="bx bx-trash"></i> Delete All Payment Records
                </button>
            </div>
            
            <!-- Payment Tabs -->
            <div class="nav nav-pills mb-4" id="payment-tabs" role="tablist">
                <button class="nav-link active" id="course-payments-tab" data-bs-toggle="pill" data-bs-target="#course-payments" type="button" role="tab">
                    <i class="bx bx-book"></i>
                    Course Payments (<?= count($course_payments) ?>)
                </button>
                <button class="nav-link" id="referral-payments-tab" data-bs-toggle="pill" data-bs-target="#referral-payments" type="button" role="tab">
                    <i class="bx bx-group"></i>
                    Referral Commissions (<?= count($referral_payments) ?>)
                </button>
                <button class="nav-link" id="withdrawal-requests-tab" data-bs-toggle="pill" data-bs-target="#withdrawal-requests" type="button" role="tab">
                    <i class="bx bx-money-withdraw"></i>
                    Withdrawal Requests (<?= count($withdrawal_requests) ?>)
                </button>
                <button class="nav-link" id="teacher-earnings-tab" data-bs-toggle="pill" data-bs-target="#teacher-earnings" type="button" role="tab">
                    <i class="bx bx-user-voice"></i>
                    Teacher Earnings (<?= count($teacher_earnings) ?>)
                </button>
            </div>

            <div class="tab-content">
                <!-- Course Payments Tab -->
                <div class="tab-pane fade show active" id="course-payments" role="tabpanel">
                    <div class="table-header d-flex justify-content-between align-items-center mb-3">
                        <h5>Course Payments</h5>
                        <div class="search-container">
                            <input type="text" id="courseSearch" class="form-control form-control-sm" placeholder="Search payments..." style="width: 250px;">
                        </div>
                    </div>
                    
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>User</th>
                                    <th>Course</th>
                                    <th>Amount</th>
                                    <th>Payment Date</th>
                                    <th>Method</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($course_payments)): ?>
                                    <tr>
                                        <td colspan="8" class="text-center text-muted">No course payments found</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($course_payments as $payment): ?>
                                        <tr>
                                            <td>#<?= $payment['purchase_id'] ?></td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="avatar me-2">
                                                        <div class="avatar-text rounded-circle bg-primary">
                                                            <?= strtoupper(substr($payment['first_name'], 0, 1) . substr($payment['last_name'], 0, 1)) ?>
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <h6 class="mb-0"><?= htmlspecialchars($payment['first_name'] . ' ' . $payment['last_name']) ?></h6>
                                                        <small class="text-muted"><?= htmlspecialchars($payment['email']) ?></small>
                                                    </div>
                                                </div>
                                            </td>
                                            <td><?= htmlspecialchars($payment['course_title']) ?></td>
                                            <td><strong>R<?= number_format($payment['amount'], 2) ?></strong></td>
                                            <td><?= date('M d, Y H:i', strtotime($payment['payment_date'])) ?></td>
                                            <td><?= ucfirst($payment['payment_method']) ?></td>
                                            <td>
                                                <span class="badge bg-<?= $payment['status'] === 'completed' ? 'success' : 'warning' ?>">
                                                    <?= ucfirst($payment['status']) ?>
                                                </span>
                                            </td>
                                            <td>
                                                <div class="btn-group">
                                                    <button class="btn btn-sm btn-outline-secondary" data-bs-toggle="modal" data-bs-target="#receiptModal" 
                                                            data-payment-id="<?= $payment['purchase_id'] ?>" data-payment-type="course" title="View Receipt">
                                                        <i class='bx bx-receipt'></i>
                                                    </button>
                                                    <button class="btn btn-sm btn-outline-danger" onclick="deleteCoursePayment(<?= $payment['purchase_id'] ?>)" title="Delete Payment">
                                                        <i class='bx bx-trash'></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                    
                <!-- Referral Commissions Tab -->
                <div class="tab-pane fade" id="referral-payments" role="tabpanel">
                    <div class="table-header d-flex justify-content-between align-items-center mb-3">
                        <h5>Referral Commissions</h5>
                        <div class="search-container">
                            <input type="text" id="referralSearch" class="form-control form-control-sm" placeholder="Search referrals..." style="width: 250px;">
                        </div>
                    </div>
                    
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Referrer</th>
                                    <th>Referred Student</th>
                                    <th>Course</th>
                                    <th>Purchase Amount</th>
                                    <th>Commission</th>
                                    <th>Status</th>
                                    <th>Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($referral_payments)): ?>
                                    <tr>
                                        <td colspan="9" class="text-center text-muted">No referral commissions found</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($referral_payments as $referral): ?>
                                        <tr>
                                            <td>#<?= $referral['id'] ?></td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="avatar me-2">
                                                        <div class="avatar-text rounded-circle bg-purple">
                                                            <?= strtoupper(substr($referral['referrer_first_name'], 0, 1) . substr($referral['referrer_last_name'], 0, 1)) ?>
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <h6 class="mb-0"><?= htmlspecialchars($referral['referrer_first_name'] . ' ' . $referral['referrer_last_name']) ?></h6>
                                                        <small class="text-muted"><?= htmlspecialchars($referral['referrer_email']) ?></small>
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="avatar me-2">
                                                        <div class="avatar-text rounded-circle bg-success">
                                                            <?= strtoupper(substr($referral['referred_first_name'], 0, 1) . substr($referral['referred_last_name'], 0, 1)) ?>
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <h6 class="mb-0"><?= htmlspecialchars($referral['referred_first_name'] . ' ' . $referral['referred_last_name']) ?></h6>
                                                        <small class="text-muted"><?= htmlspecialchars($referral['referred_name']) ?></small>
                                                    </div>
                                                </div>
                                            </td>
                                            <td><?= htmlspecialchars($referral['course_title']) ?></td>
                                            <td><strong>R<?= number_format($referral['purchase_amount'], 2) ?></strong></td>
                                            <td><strong class="text-success">R<?= number_format($referral['commission_amount'], 2) ?></strong></td>
                                            <td>
                                                <span class="badge bg-<?= 
                                                    $referral['status'] === 'paid' ? 'success' : 
                                                    ($referral['status'] === 'pending' ? 'success' : 
                                                    ($referral['status'] === 'approved' ? 'success' : 'secondary')) 
                                                ?>">
                                                    <?= $referral['status'] === 'pending' ? 'Active' : ucfirst($referral['status']) ?>
                                                </span>
                                            </td>
                                            <td><?= date('M d, Y H:i', strtotime($referral['created_at'])) ?></td>
                                            <td>
                                                <button class="btn btn-sm btn-outline-danger" onclick="deleteReferralCommission(<?= $referral['id'] ?>)" title="Delete Commission">
                                                    <i class='bx bx-trash'></i>
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- Withdrawal Requests Tab -->
                <div class="tab-pane fade" id="withdrawal-requests" role="tabpanel">
                    <div class="table-header d-flex justify-content-between align-items-center mb-3">
                        <h5>Withdrawal Requests</h5>
                        <div class="search-container">
                            <input type="text" id="withdrawalSearch" class="form-control form-control-sm" placeholder="Search withdrawals..." style="width: 250px;">
                        </div>
                    </div>
                    
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Teacher</th>
                                    <th>Amount</th>
                                    <th>Payment Method</th>
                                    <th>Request Date</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($withdrawal_requests)): ?>
                                    <tr>
                                        <td colspan="7" class="text-center text-muted">No withdrawal requests found</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($withdrawal_requests as $request): ?>
                                        <tr>
                                            <td>#<?= $request['id'] ?></td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="avatar me-2">
                                                        <div class="avatar-text rounded-circle bg-warning">
                                                            <?= strtoupper(substr($request['first_name'], 0, 1) . substr($request['last_name'], 0, 1)) ?>
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <h6 class="mb-0"><?= htmlspecialchars($request['first_name'] . ' ' . $request['last_name']) ?></h6>
                                                        <small class="text-muted"><?= htmlspecialchars($request['email']) ?></small>
                                                    </div>
                                                </div>
                                            </td>
                                            <td><strong>R<?= number_format($request['amount'], 2) ?></strong></td>
                                            <td><?= htmlspecialchars($request['payment_method']) ?></td>
                                            <td><?= date('M d, Y H:i', strtotime($request['requested_at'])) ?></td>
                                            <td>
                                                <span class="badge bg-<?= 
                                                    $request['status'] === 'approved' ? 'success' : 
                                                    ($request['status'] === 'pending' ? 'warning' : 'danger') 
                                                ?>">
                                                    <?= ucfirst($request['status']) ?>
                                                </span>
                                            </td>
                                            <td>
                                                <div class="btn-group">
                                                    <button class="btn btn-sm btn-outline-info" onclick="viewWithdrawalDetails(<?= $request['id'] ?>)" title="View Details">
                                                        <i class='bx bx-info-circle'></i>
                                                    </button>
                                                    <button class="btn btn-sm btn-outline-danger" onclick="deleteWithdrawal(<?= $request['id'] ?>)" title="Delete Request">
                                                        <i class='bx bx-trash'></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- Teacher Earnings Tab -->
                <div class="tab-pane fade" id="teacher-earnings" role="tabpanel">
                    <div class="table-header d-flex justify-content-between align-items-center mb-3">
                        <h5>Teacher Earnings</h5>
                        <div class="search-container">
                            <input type="text" id="earningsSearch" class="form-control form-control-sm" placeholder="Search earnings..." style="width: 250px;">
                        </div>
                    </div>
                    
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Teacher</th>
                                    <th>Course</th>
                                    <th>Amount</th>
                                    <th>Status</th>
                                    <th>Type</th>
                                    <th>Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($teacher_earnings)): ?>
                                    <tr>
                                        <td colspan="8" class="text-center text-muted">No teacher earnings found</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($teacher_earnings as $earning): ?>
                                        <tr>
                                            <td>#<?= $earning['earning_id'] ?></td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="avatar me-2">
                                                        <div class="avatar-text rounded-circle bg-success">
                                                            <?= strtoupper(substr($earning['first_name'], 0, 1) . substr($earning['last_name'], 0, 1)) ?>
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <h6 class="mb-0"><?= htmlspecialchars($earning['first_name'] . ' ' . $earning['last_name']) ?></h6>
                                                        <small class="text-muted"><?= htmlspecialchars($earning['email']) ?></small>
                                                    </div>
                                                </div>
                                            </td>
                                            <td><?= htmlspecialchars($earning['course_title'] ?? 'N/A') ?></td>
                                            <td><strong>R<?= number_format($earning['amount'], 2) ?></strong></td>
                                            <td>
                                                <span class="badge bg-<?= $earning['status'] === 'available' ? 'success' : ($earning['status'] === 'withdrawn' ? 'secondary' : 'warning') ?>">
                                                    <?= ucfirst($earning['status']) ?>
                                                </span>
                                            </td>
                                            <td>
                                                <span class="badge bg-info"><?= ucfirst(str_replace('_', ' ', $earning['transaction_type'] ?? 'earning')) ?></span>
                                            </td>
                                            <td><?= date('M d, Y H:i', strtotime($earning['created_at'])) ?></td>
                                            <td>
                                                <button class="btn btn-sm btn-outline-danger" onclick="deleteTeacherEarning(<?= $earning['earning_id'] ?>)" title="Delete Earning">
                                                    <i class='bx bx-trash'></i>
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                </div>
            </div>
        </div>
    </section>
    
    <!-- Export Modal -->
    <div class="modal fade" id="exportModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Export Payment Data</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form action="export_payments.php" method="post">
                        <div class="mb-3">
                            <label for="exportType" class="form-label">Payment Type</label>
                            <select class="form-select" id="exportType" name="type">
                                <option value="all">All Payments</option>
                                <option value="course">Course Payments Only</option>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label for="dateRange" class="form-label">Date Range</label>
                            <select class="form-select" id="dateRange" name="date_range">
                                <option value="all">All Time</option>
                                <option value="this_month">This Month</option>
                                <option value="last_month">Last Month</option>
                                <option value="this_year">This Year</option>
                                <option value="custom">Custom Range</option>
                            </select>
                        </div>
                        <div class="mb-3 date-range-picker d-none">
                            <div class="row">
                                <div class="col-md-6">
                                    <label for="startDate" class="form-label">Start Date</label>
                                    <input type="date" class="form-control" id="startDate" name="start_date">
                                </div>
                                <div class="col-md-6">
                                    <label for="endDate" class="form-label">End Date</label>
                                    <input type="date" class="form-control" id="endDate" name="end_date">
                                </div>
                            </div>
                        </div>
                        <div class="mb-3">
                            <label for="format" class="form-label">Export Format</label>
                            <select class="form-select" id="format" name="format">
                                <option value="csv">CSV</option>
                                <option value="excel">Excel</option>
                                <option value="pdf">PDF</option>
                            </select>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Export</button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Receipt Modal -->
    <div class="modal fade" id="receiptModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Payment Receipt</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="text-center mb-4">
                        <img src="/assets/images/logo.jpeg" alt="Panadite Academy" style="height: 60px;">
                        <h5 class="mt-2">Panadite Academy</h5>
                        <p class="text-muted">Official Receipt</p>
                    </div>
                    <div id="receiptContent">
                        <!-- Content will be loaded dynamically -->
                        <p class="text-center">Loading receipt data...</p>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-primary" id="printReceipt">
                        <i class='bx bx-printer'></i> Print
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Delete Confirmation Forms -->
    <form id="deleteForm" method="POST" style="display: none;">
        <input type="hidden" name="action" id="deleteAction">
        <input type="hidden" name="payment_id" id="deletePaymentId">
        <input type="hidden" name="request_id" id="deleteRequestId">
        <input type="hidden" name="earning_id" id="deleteEarningId">
    </form>
    
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Sidebar toggle functionality
        const sidebarToggle = document.querySelector('.sidebar-toggle');
        const sidebarToggleMobile = document.querySelector('.sidebar-toggle-mobile');
        const sidebar = document.querySelector('.sidebar');
        
        if (sidebarToggle) {
            sidebarToggle.addEventListener('click', () => {
                sidebar.classList.toggle('close');
            });
        }
        
        if (sidebarToggleMobile) {
            sidebarToggleMobile.addEventListener('click', () => {
                sidebar.classList.toggle('show');
            });
        }
        
        // Responsive sidebar behavior
        function checkScreenSize() {
            if (window.innerWidth < 768) {
                sidebar.classList.remove('close');
                sidebar.classList.remove('show');
            } else if (window.innerWidth < 992) {
                sidebar.classList.add('close');
                sidebar.classList.remove('show');
            }
        }
        
        // Initial check and listen for window resize
        checkScreenSize();
        window.addEventListener('resize', checkScreenSize);
        
        // Show/hide date range picker
        document.getElementById('dateRange').addEventListener('change', function() {
            const dateRangePicker = document.querySelector('.date-range-picker');
            if (this.value === 'custom') {
                dateRangePicker.classList.remove('d-none');
            } else {
                dateRangePicker.classList.add('d-none');
            }
        });
        
        // Search functionality for course payments
        document.getElementById('courseSearch').addEventListener('keyup', function() {
            const searchTerm = this.value.toLowerCase();
            const tableRows = document.querySelectorAll('#courses tbody tr');
            
            tableRows.forEach(row => {
                const text = row.textContent.toLowerCase();
                if (text.includes(searchTerm)) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        });
        

        
        // Handle receipt modal
        document.getElementById('receiptModal').addEventListener('show.bs.modal', function(event) {
            const button = event.relatedTarget;
            const paymentId = button.getAttribute('data-payment-id');
            const paymentType = button.getAttribute('data-payment-type');
            
            // In a real implementation, you would fetch the receipt data via AJAX
            // For now, we'll just simulate loading receipt data
            const receiptContent = document.getElementById('receiptContent');
            
            // Simulate loading with setTimeout
            setTimeout(function() {
                if (paymentType === 'course') {
                    // Find the payment data
                    const payment = <?php echo json_encode($course_payments); ?>.find(p => p.purchase_id == paymentId);
                    if (payment) {
                        receiptContent.innerHTML = `
                            <div class="receipt-details">
                                <div class="row mb-2">
                                    <div class="col-5 text-muted">Receipt No:</div>
                                    <div class="col-7">#C${payment.purchase_id}</div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-5 text-muted">Date:</div>
                                    <div class="col-7">${new Date(payment.payment_date).toLocaleDateString()}</div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-5 text-muted">Customer:</div>
                                    <div class="col-7">${payment.first_name} ${payment.last_name}</div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-5 text-muted">Email:</div>
                                    <div class="col-7">${payment.email}</div>
                                </div>
                                <hr>
                                <div class="row mb-2">
                                    <div class="col-7 text-muted">Course:</div>
                                    <div class="col-5 text-end">${payment.course_title}</div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-7 text-muted">Payment Method:</div>
                                    <div class="col-5 text-end">${payment.payment_method.toUpperCase()}</div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-7 text-muted">Status:</div>
                                    <div class="col-5 text-end">${payment.status.toUpperCase()}</div>
                                </div>
                                <hr>
                                <div class="row mb-2">
                                    <div class="col-7"><strong>Total Amount:</strong></div>
                                    <div class="col-5 text-end"><strong>R${Number(payment.amount).toFixed(2)}</strong></div>
                                </div>
                            </div>
                        `;
                    } else {
                        receiptContent.innerHTML = '<p class="text-center text-danger">Receipt data not found.</p>';
                    }
                } else {
                    receiptContent.innerHTML = '<div class="alert alert-warning">Video payments are not supported.</div>';
                    if (false) { // Disabled video payment receipt handling
                        receiptContent.innerHTML = `
                            <div class="receipt-details">
                                <div class="row mb-2">
                                    <div class="col-5 text-muted">Receipt No:</div>
                                    <div class="col-7">#V0</div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-5 text-muted">Date:</div>
                                    <div class="col-7">${new Date(payment.payment_date).toLocaleDateString()}</div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-5 text-muted">Customer:</div>
                                    <div class="col-7">${payment.first_name} ${payment.last_name}</div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-5 text-muted">Email:</div>
                                    <div class="col-7">${payment.email}</div>
                                </div>
                                <hr>
                                <div class="row mb-2">
                                    <div class="col-7 text-muted">Video:</div>
                                    <div class="col-5 text-end">${payment.video_title}</div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-7 text-muted">Payment Method:</div>
                                    <div class="col-5 text-end">${payment.payment_method.toUpperCase()}</div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-7 text-muted">Status:</div>
                                    <div class="col-5 text-end">${payment.status.toUpperCase()}</div>
                                </div>
                                <hr>
                                <div class="row mb-2">
                                    <div class="col-7"><strong>Total Amount:</strong></div>
                                    <div class="col-5 text-end"><strong>R${Number(payment.amount).toFixed(2)}</strong></div>
                                </div>
                            </div>
                        `;
                    } else {
                        receiptContent.innerHTML = '<p class="text-center text-danger">Receipt data not found.</p>';
                    }
                }
            }, 500);
        });
        
        // Print receipt functionality
        document.getElementById('printReceipt').addEventListener('click', function() {
            const printContents = document.querySelector('#receiptModal .modal-body').innerHTML;
            const originalContents = document.body.innerHTML;
            
            document.body.innerHTML = `
                <div style="padding: 20px; max-width: 800px; margin: 0 auto;">
                    ${printContents}
                </div>
            `;
            
            window.print();
            document.body.innerHTML = originalContents;
            location.reload();
        });
    });

    // Delete Functions
    function deleteCoursePayment(paymentId) {
        if (confirm('Are you sure you want to delete this course payment?')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="action" value="delete_course_payment">
                <input type="hidden" name="payment_id" value="${paymentId}">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    function deleteReferralCommission(commissionId) {
        if (confirm('Are you sure you want to delete this referral commission?')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="action" value="delete_referral_commission">
                <input type="hidden" name="commission_id" value="${commissionId}">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    function deleteVideoPayment(paymentId) {
        if (confirm('Are you sure you want to delete this video payment? This action cannot be undone.')) {
            document.getElementById('deleteAction').value = 'delete_video_payment';
            document.getElementById('deletePaymentId').value = paymentId;
            document.getElementById('deleteForm').submit();
        }
    }

    function deleteWithdrawal(requestId) {
        if (confirm('Are you sure you want to delete this withdrawal request? This action cannot be undone.')) {
            document.getElementById('deleteAction').value = 'delete_withdrawal';
            document.getElementById('deleteRequestId').value = requestId;
            document.getElementById('deleteForm').submit();
        }
    }

    function deleteTeacherEarning(earningId) {
        if (confirm('Are you sure you want to delete this teacher earning record? This action cannot be undone.')) {
            document.getElementById('deleteAction').value = 'delete_teacher_earning';
            document.getElementById('deleteEarningId').value = earningId;
            document.getElementById('deleteForm').submit();
        }
    }

    function confirmDeleteAll() {
        if (confirm('⚠️ WARNING: This will delete ALL payment records from the entire platform!\n\nThis includes:\n• All course payments\n• All video payments\n• All withdrawal requests\n• All teacher earnings\n\nThis action CANNOT be undone. Are you absolutely sure?')) {
            if (confirm('This is your final warning. Click OK to permanently delete ALL payment data and clean the system.')) {
                document.getElementById('deleteAction').value = 'delete_all_payments';
                document.getElementById('deleteForm').submit();
            }
        }
    }

    function viewWithdrawalDetails(requestId) {
        // Redirect to withdrawal details page
        window.location.href = 'withdrawal-requests.php?view=' + requestId;
    }

    // Search functionality for each tab
    document.getElementById('courseSearch').addEventListener('input', function() {
        filterTable('course-payments', this.value);
    });

    document.getElementById('videoSearch').addEventListener('input', function() {
        filterTable('video-payments', this.value);
    });

    document.getElementById('withdrawalSearch').addEventListener('input', function() {
        filterTable('withdrawal-requests', this.value);
    });

    document.getElementById('earningsSearch').addEventListener('input', function() {
        filterTable('teacher-earnings', this.value);
    });

    function filterTable(tabId, searchTerm) {
        const tab = document.getElementById(tabId);
        const rows = tab.querySelectorAll('tbody tr');
        const term = searchTerm.toLowerCase();

        rows.forEach(row => {
            const text = row.textContent.toLowerCase();
            if (text.includes(term)) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        });
    }
    </script>
</body>
</html>
