<?php
/**
 * POWERFUL PDF EDITOR SCRIPT
 * Real PDF editing with embedded text fields
 * This script actually modifies PDF files and embeds text properly
 */

require_once '../config/database.php';

// Check if TCPDF is available, if not provide installation instructions
if (!class_exists('TCPDF')) {
    // Download and include TCPDF library
    $tcpdf_path = __DIR__ . '/libraries/tcpdf/tcpdf.php';
    if (file_exists($tcpdf_path)) {
        require_once $tcpdf_path;
    } else {
        die('TCPDF library not found. Please install TCPDF library in /admin/libraries/tcpdf/');
    }
}

class PowerfulPDFEditor {
    private $conn;
    private $template_path;
    private $output_path;
    
    public function __construct($database_connection) {
        $this->conn = $database_connection;
        $this->template_path = __DIR__ . '/../uploads/certificates/templates/';
        $this->output_path = __DIR__ . '/../uploads/certificates/generated/';
        
        // Create output directory if it doesn't exist
        if (!is_dir($this->output_path)) {
            mkdir($this->output_path, 0755, true);
        }
    }
    
    /**
     * Generate a modified PDF with embedded text fields
     */
    public function generateCertificateWithFields($template_id, $student_data = null) {
        try {
            // Get template information
            $template = $this->getTemplate($template_id);
            if (!$template) {
                throw new Exception("Template not found");
            }
            
            // Get template fields
            $fields = $this->getTemplateFields($template_id);
            
            // Original PDF path
            $original_pdf = $this->template_path . $template['template_file'];
            if (!file_exists($original_pdf)) {
                throw new Exception("Template PDF file not found");
            }
            
            // Create new PDF with embedded fields
            $modified_pdf = $this->createModifiedPDF($original_pdf, $fields, $student_data);
            
            return $modified_pdf;
            
        } catch (Exception $e) {
            error_log("PDF Generation Error: " . $e->getMessage());
            throw new Exception("Failed to generate PDF: " . $e->getMessage());
        }
    }
    
    /**
     * Create modified PDF using TCPDF with proper text embedding
     */
    private function createModifiedPDF($original_pdf, $fields, $student_data = null) {
        // Create new PDF instance
        $pdf = new TCPDF('L', 'PT', 'A4', true, 'UTF-8', false);
        
        // Set document properties
        $pdf->SetCreator('Panadite Academy Certificate Generator');
        $pdf->SetAuthor('Panadite Academy');
        $pdf->SetTitle('Certificate of Completion');
        
        // Remove header and footer
        $pdf->setPrintHeader(false);
        $pdf->setPrintFooter(false);
        
        // Set margins
        $pdf->SetMargins(0, 0, 0);
        $pdf->SetAutoPageBreak(false, 0);
        
        // Add a page
        $pdf->AddPage();
        
        // Import original PDF as background
        try {
            // Use setSourceFile and importPage for background
            if (class_exists('FPDI')) {
                $pdf->setSourceFile($original_pdf);
                $template_id = $pdf->importPage(1);
                $pdf->useTemplate($template_id, 0, 0);
            } else {
                // If FPDI not available, create a white background
                $pdf->SetFillColor(255, 255, 255);
                $pdf->Rect(0, 0, $pdf->getPageWidth(), $pdf->getPageHeight(), 'F');
                
                // Add a notice about the original template
                $pdf->SetFont('helvetica', 'B', 12);
                $pdf->SetTextColor(100, 100, 100);
                $pdf->SetXY(50, 50);
                $pdf->Cell(0, 20, 'Original PDF template would appear here if FPDI library is installed', 0, 1, 'L');
            }
        } catch (Exception $e) {
            error_log("Background PDF import failed: " . $e->getMessage());
            // Continue with white background
            $pdf->SetFillColor(255, 255, 255);
            $pdf->Rect(0, 0, $pdf->getPageWidth(), $pdf->getPageHeight(), 'F');
        }
        
        // Add each field as text overlay
        foreach ($fields as $field) {
            $this->addTextToPDF($pdf, $field, $student_data);
        }
        
        // Generate unique filename
        $filename = 'certificate_' . time() . '_' . rand(1000, 9999) . '.pdf';
        $output_file = $this->output_path . $filename;
        
        // Save the PDF
        $pdf->Output($output_file, 'F');
        
        return [
            'file_path' => $output_file,
            'filename' => $filename,
            'url' => '/uploads/certificates/generated/' . $filename
        ];
    }
    
    /**
     * Add text field to PDF with proper positioning and styling
     */
    private function addTextToPDF($pdf, $field, $student_data = null) {
        // Get text content
        $text = $this->getFieldText($field, $student_data);
        
        // Set font
        $font_family = $this->mapFontFamily($field['font_family']);
        $font_size = max(8, min(72, intval($field['font_size']))); // Limit font size
        $font_weight = $field['font_weight'] === 'bold' ? 'B' : '';
        
        $pdf->SetFont($font_family, $font_weight, $font_size);
        
        // Set text color
        $color = $this->hexToRgb($field['font_color']);
        $pdf->SetTextColor($color['r'], $color['g'], $color['b']);
        
        // Position and add text
        $x = floatval($field['x_position']);
        $y = floatval($field['y_position']);
        $width = floatval($field['width']);
        $height = floatval($field['height']);
        
        // Set position
        $pdf->SetXY($x, $y);
        
        // Add text with proper alignment
        $align = strtoupper($field['text_align']);
        if (!in_array($align, ['L', 'C', 'R', 'J'])) {
            $align = 'L';
        }
        
        // Use MultiCell for better text handling
        $pdf->MultiCell($width, $height, $text, 0, $align, false, 1, $x, $y, true, 0, false, true, $height, 'M');
    }
    
    /**
     * Get appropriate text for field
     */
    private function getFieldText($field, $student_data = null) {
        // Use custom text if available
        if (!empty($field['custom_text'])) {
            return $field['custom_text'];
        }
        
        // Use student data if provided
        if ($student_data && isset($student_data[$field['field_type']])) {
            return $student_data[$field['field_type']];
        }
        
        // Default sample data
        $samples = [
            'student_name' => 'John Doe Smith',
            'course_title' => 'Advanced Web Development',
            'completion_date' => date('F j, Y'),
            'certificate_number' => 'CERT-' . date('Y') . '-' . rand(1000, 9999),
            'director_name' => 'Dr. Sarah Wilson',
            'certificate_text' => 'This certifies that the above named student has successfully completed the course requirements.'
        ];
        
        return $samples[$field['field_type']] ?? ucwords(str_replace('_', ' ', $field['field_type']));
    }
    
    /**
     * Map font families to TCPDF supported fonts
     */
    private function mapFontFamily($font_family) {
        $font_map = [
            'Arial' => 'helvetica',
            'Helvetica' => 'helvetica',
            'Times New Roman' => 'times',
            'Times' => 'times',
            'Courier New' => 'courier',
            'Courier' => 'courier',
            'Georgia' => 'times',
            'Verdana' => 'helvetica',
            'Tahoma' => 'helvetica',
        ];
        
        return $font_map[$font_family] ?? 'helvetica';
    }
    
    /**
     * Convert hex color to RGB
     */
    private function hexToRgb($hex) {
        $hex = ltrim($hex, '#');
        
        if (strlen($hex) === 3) {
            $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
        }
        
        return [
            'r' => hexdec(substr($hex, 0, 2)),
            'g' => hexdec(substr($hex, 2, 2)),
            'b' => hexdec(substr($hex, 4, 2))
        ];
    }
    
    /**
     * Get template data from database
     */
    private function getTemplate($template_id) {
        $stmt = $this->conn->prepare("SELECT * FROM certificate_templates WHERE template_id = ?");
        $stmt->bind_param("i", $template_id);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_assoc();
    }
    
    /**
     * Get template fields from database
     */
    private function getTemplateFields($template_id) {
        $stmt = $this->conn->prepare("SELECT * FROM certificate_template_fields WHERE template_id = ? ORDER BY field_id");
        $stmt->bind_param("i", $template_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $fields = [];
        while ($field = $result->fetch_assoc()) {
            $fields[] = $field;
        }
        
        return $fields;
    }
}

// AJAX endpoint for generating preview PDF
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'generate_pdf_preview') {
    header('Content-Type: application/json');
    
    try {
        $template_id = intval($_POST['template_id']);
        
        $pdf_editor = new PowerfulPDFEditor($conn);
        $result = $pdf_editor->generateCertificateWithFields($template_id);
        
        echo json_encode([
            'success' => true,
            'pdf_url' => $result['url'],
            'filename' => $result['filename']
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    
    exit;
}

// AJAX endpoint for generating certificate with student data
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'generate_certificate') {
    header('Content-Type: application/json');
    
    try {
        $template_id = intval($_POST['template_id']);
        $student_data = json_decode($_POST['student_data'], true);
        
        $pdf_editor = new PowerfulPDFEditor($conn);
        $result = $pdf_editor->generateCertificateWithFields($template_id, $student_data);
        
        echo json_encode([
            'success' => true,
            'pdf_url' => $result['url'],
            'filename' => $result['filename']
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    
    exit;
}

?>
