<?php
require_once '../auth/functions.php';
require_once '../includes/config.php';
require_once '../includes/db_connection.php';

// Ensure user is logged in as an admin
requireRole('admin');

$userId = $_SESSION['user_id'];
$user = getCurrentUser();

// Function to check if a column exists in a table
function columnExists($conn, $table, $column) {
    $sql = "SHOW COLUMNS FROM `$table` LIKE '$column'";
    $result = $conn->query($sql);
    return $result->num_rows > 0;
}

// Check for columns in videos table
$hasUserIdColumn = columnExists($conn, 'videos', 'user_id');
$hasApprovedColumn = columnExists($conn, 'videos', 'approved');
$hasViewsColumn = columnExists($conn, 'videos', 'views');
$hasLikesColumn = columnExists($conn, 'videos', 'likes');
$hasCommentsColumn = columnExists($conn, 'videos', 'comments');
$hasDurationColumn = columnExists($conn, 'videos', 'duration');
$hasPriceColumn = columnExists($conn, 'videos', 'price');
$hasThumbnailColumn = columnExists($conn, 'videos', 'thumbnail');

// Check if video ID is provided
if (!isset($_GET['id']) || empty($_GET['id'])) {
    $errorMessage = "No video ID provided";
    $requestedUrl = $_SERVER['REQUEST_URI'];
    $videoExists = false;
} else {
    $videoId = intval($_GET['id']);
    
    // Attempt to fetch the video and related data
    // First check what tables and columns exist
    $videoTableColumns = [];
    $result = $conn->query("SHOW COLUMNS FROM videos");
    while ($row = $result->fetch_assoc()) {
        $videoTableColumns[] = $row['Field'];
    }
    
    // For debugging - output all available columns
    $debugMode = false; // Set to true to enable debugging
    if ($debugMode) {
        echo '<!-- Available columns in videos table: ' . implode(', ', $videoTableColumns) . ' -->';
    }

    // Check if users table exists
    $userTableExists = false;
    $result = $conn->query("SHOW TABLES LIKE 'users'");
    $userTableExists = $result->num_rows > 0;

    // Build query based on available schema
    if ($userTableExists && in_array('user_id', $videoTableColumns)) {
        // If user_id column exists and users table exists, use the join
        $sql = "SELECT v.*, u.username, u.email, u.first_name, u.last_name FROM videos v 
               LEFT JOIN users u ON v.user_id = u.id
               WHERE v.video_id = ?";
    } else {
        // Otherwise just get the video data
        $sql = "SELECT v.* FROM videos v WHERE v.video_id = ?";
    }

    // Also try to check for file path in other possible columns
    $possibleFileColumns = ['video_url', 'file_path', 'file_name', 'video_path', 'url', 'path'];
    
    // Enable direct database access for the specific video_id
    $directSql = "SELECT * FROM videos WHERE video_id = ?";
    $directStmt = $conn->prepare($directSql);
    $directStmt->bind_param("i", $videoId);
    $directStmt->execute();
    $directResult = $directStmt->get_result();
    
    if ($directResult && $directResult->num_rows > 0) {
        // Store direct DB access result for later use
        $directVideoData = $directResult->fetch_assoc();
        
        if ($debugMode) {
            echo '<!-- Direct DB video data: ' . json_encode($directVideoData) . ' -->';
            
            // Check for the most important fields
            if (isset($directVideoData['video_url'])) {
                echo '<!-- Found video_url: ' . $directVideoData['video_url'] . ' -->';
            }
            if (isset($directVideoData['file_path'])) {
                echo '<!-- Found file_path: ' . $directVideoData['file_path'] . ' -->';
            }
        }
    }
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $videoId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    // Direct database debugging for this specific video
    if ($debugMode) {
        // Get raw data from database for debugging
        $rawSql = "SELECT * FROM videos WHERE video_id = $videoId";
        $rawResult = $conn->query($rawSql);
        echo '<!-- Raw database query: ' . $rawSql . ' -->';
        
        if ($rawResult && $rawResult->num_rows > 0) {
            $rawVideo = $rawResult->fetch_assoc();
            echo '<!-- Raw video data from DB: ' . json_encode($rawVideo) . ' -->';
        } else {
            echo '<!-- No raw video data found in database -->';
        }
    }
    
    if ($result->num_rows > 0) {
        $video = $result->fetch_assoc();
        $videoExists = true;
        
        // Set default username if not available
        if (!isset($video['username'])) {
            $video['username'] = 'Unknown User';
        }
    } else {
        $errorMessage = "The requested video does not exist";
        $requestedUrl = $_SERVER['REQUEST_URI'];
        $videoExists = false;
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $videoExists ? "Preview: " . htmlspecialchars($video['title']) : "Video Not Found"; ?> | Panadite Admin</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Boxicons CSS -->
    <link href="https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        /* Modern color variables */
        :root {
            --primary-color: #4e73df;
            --success-color: #1cc88a;
            --info-color: #36b9cc;
            --warning-color: #f6c23e;
            --danger-color: #e74a3b;
            --secondary-color: #858796;
            --light-color: #f8f9fc;
            --dark-color: #5a5c69;
        }
        
        .not-found-container {
            min-height: 70vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .not-found-content {
            text-align: center;
            max-width: 600px;
        }
        
        .error-code {
            font-size: 6rem;
            font-weight: 700;
            color: var(--primary-color);
            line-height: 1;
            margin-bottom: 1rem;
            background: linear-gradient(45deg, #e74a3b, #f6c23e);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            opacity: 0.8;
        }
        
        .error-divider {
            width: 80px;
            height: 5px;
            background: linear-gradient(45deg, #e74a3b, #f6c23e);
            margin: 1.5rem auto;
            border-radius: 5px;
        }
        
        .error-icon {
            font-size: 5rem;
            color: var(--warning-color);
            margin-bottom: 1rem;
            opacity: 0.7;
        }
        
        .error-url {
            display: inline-block;
            padding: 0.5rem 1rem;
            background-color: rgba(0,0,0,0.05);
            border-radius: 0.5rem;
            margin-top: 1rem;
            font-family: monospace;
            word-break: break-all;
        }
        
        .action-buttons {
            margin-top: 2rem;
        }
        
        .action-buttons .btn {
            margin: 0 0.5rem;
            border-radius: 50px;
            padding: 0.5rem 1.5rem;
            box-shadow: 0 3px 6px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
        }
        
        .action-buttons .btn:hover {
            transform: translateY(-3px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.15);
        }
        
        /* Video Player Styling */
        .video-container {
            border-radius: 0.75rem;
            overflow: hidden;
            box-shadow: 0 0.25rem 0.75rem rgba(0, 0, 0, 0.1);
            margin-bottom: 1.5rem;
            min-height: 400px;
        }
        
        .video-player {
            width: 100%;
            height: auto;
            min-height: 450px;
            max-height: 700px;
            background-color: #000;
        }
        
        /* Responsive iframe container for YouTube videos */
        .embed-responsive {
            position: relative;
            display: block;
            width: 100%;
            padding: 0;
            overflow: hidden;
        }
        
        .embed-responsive-16by9 {
            aspect-ratio: 16/9;
            min-height: 450px;
        }
        
        .embed-responsive .embed-responsive-item,
        .embed-responsive iframe,
        .embed-responsive embed,
        .embed-responsive object,
        .embed-responsive video {
            position: absolute;
            top: 0;
            bottom: 0;
            left: 0;
            width: 100%;
            height: 100%;
            border: 0;
        }
        
        /* Larger video on bigger screens */
        @media (min-width: 1200px) {
            .video-player {
                min-height: 500px;
                max-height: 800px;
            }
            
            .embed-responsive-16by9 {
                min-height: 500px;
            }
        }
        
        @media (max-width: 768px) {
            .video-player {
                min-height: 300px;
                max-height: 400px;
            }
            
            .embed-responsive-16by9 {
                min-height: 300px;
            }
        }
        
        .video-info-card {
            border: none;
            border-radius: 0.75rem;
            box-shadow: 0 0.25rem 0.5rem rgba(0, 0, 0, 0.05);
        }
        
        .video-title {
            font-weight: 600;
            margin-bottom: 0.5rem;
        }
        
        .video-meta {
            display: flex;
            align-items: center;
            gap: 1.5rem;
            margin-bottom: 1rem;
        }
        
        .video-meta-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            color: #6c757d;
        }
        
        .video-description {
            margin-top: 1rem;
            color: #495057;
            white-space: pre-line;
        }
        
        .upload-details {
            background-color: #f8f9fa;
            border-radius: 0.5rem;
            padding: 1rem;
            margin-top: 1rem;
            font-size: 0.9rem;
        }
        
        .upload-details-title {
            font-weight: 600;
            margin-bottom: 0.5rem;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid p-4">
                <?php if ($videoExists): ?>
                <!-- Video Preview Content -->
                <div class="d-sm-flex justify-content-between align-items-center mb-4">
                    <div class="page-header-wrapper">
                        <h2 class="mb-0 fw-bold">
                            <div class="d-flex align-items-center">
                                <div class="icon-wrapper bg-info-subtle rounded-circle p-3 me-3" style="width: 48px; height: 48px; display: flex; align-items: center; justify-content: center;">
                                    <i class="bx bx-video fs-3 text-info"></i>
                                </div>
                                Video Preview
                            </div>
                        </h2>
                        <p class="text-muted mt-2">Viewing details for video ID: <?php echo $videoId; ?></p>
                    </div>
                    <div class="d-flex">
                        <a href="videos.php" class="btn btn-sm btn-outline-primary rounded-pill px-3 me-2">
                            <i class="bx bx-arrow-back me-1"></i> Back to Videos
                        </a>
                        <a href="edit_video.php?id=<?php echo $videoId; ?>" class="btn btn-sm btn-primary rounded-pill px-3">
                            <i class="bx bx-edit me-1"></i> Edit Video
                        </a>
                    </div>
                </div>

                <div class="row">
                    <div class="col-lg-8">
                        <div class="video-container mb-4">
                            <?php
                            // Check video type to determine display method
                            $videoType = isset($video['video_type']) ? $video['video_type'] : 'file'; // Default to file for backward compatibility
                            $debugMode = false; // Set to true for debugging
                            
                            if ($debugMode) {
                                echo '<!-- Video Type: ' . $videoType . ' -->';
                                echo '<!-- Available video data: ' . json_encode($video) . ' -->';
                            }
                            
                            if ($videoType === 'url') {
                                // Handle URL/External videos (YouTube, etc.)
                                $externalUrl = isset($video['external_url']) ? $video['external_url'] : '';
                                $platform = isset($video['platform']) ? $video['platform'] : '';
                                $videoIdExternal = isset($video['video_id_external']) ? $video['video_id_external'] : '';
                                $embedCode = isset($video['embed_code']) ? $video['embed_code'] : '';
                                
                                if ($debugMode) {
                                    echo '<!-- External URL: ' . $externalUrl . ' -->';
                                    echo '<!-- Platform: ' . $platform . ' -->';
                                    echo '<!-- External ID: ' . $videoIdExternal . ' -->';
                                }
                                
                                if ($platform === 'youtube' && !empty($videoIdExternal)) {
                                    // YouTube iframe embed
                                    $youtubeEmbedUrl = "https://www.youtube.com/embed/" . $videoIdExternal;
                                    ?>
                                    <div class="embed-responsive embed-responsive-16by9">
                                        <iframe class="embed-responsive-item video-player" 
                                                src="<?= htmlspecialchars($youtubeEmbedUrl) ?>" 
                                                title="<?= htmlspecialchars($video['title']) ?>"
                                                frameborder="0" 
                                                allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" 
                                                allowfullscreen>
                                        </iframe>
                                    </div>
                                    <?php
                                } elseif (!empty($embedCode)) {
                                    // Use custom embed code if available
                                    echo $embedCode;
                                } elseif (!empty($externalUrl)) {
                                    // Generic external URL handling
                                    if (strpos($externalUrl, 'youtube.com') !== false || strpos($externalUrl, 'youtu.be') !== false) {
                                        // Extract YouTube ID from various URL formats
                                        $youtubeId = '';
                                        if (preg_match('/(?:youtube\.com\/watch\?v=|youtu\.be\/|youtube\.com\/embed\/)([^&\n?#]+)/', $externalUrl, $matches)) {
                                            $youtubeId = $matches[1];
                                        }
                                        
                                        if ($youtubeId) {
                                            $youtubeEmbedUrl = "https://www.youtube.com/embed/" . $youtubeId;
                                            ?>
                                            <div class="embed-responsive embed-responsive-16by9">
                                                <iframe class="embed-responsive-item video-player" 
                                                        src="<?= htmlspecialchars($youtubeEmbedUrl) ?>" 
                                                        title="<?= htmlspecialchars($video['title']) ?>"
                                                        frameborder="0" 
                                                        allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" 
                                                        allowfullscreen>
                                                </iframe>
                                            </div>
                                            <?php
                                        } else {
                                            // Fallback: link to external URL
                                            ?>
                                            <div class="p-5 text-center bg-info text-white rounded">
                                                <i class="bx bx-link-external fs-1 mb-3"></i>
                                                <h5>External Video</h5>
                                                <p class="mb-3">This video is hosted externally.</p>
                                                <a href="<?= htmlspecialchars($externalUrl) ?>" target="_blank" class="btn btn-light">
                                                    <i class="bx bx-play me-1"></i> Watch Video
                                                </a>
                                            </div>
                                            <?php
                                        }
                                    } else {
                                        // Other external platforms
                                        ?>
                                        <div class="p-5 text-center bg-info text-white rounded">
                                            <i class="bx bx-link-external fs-1 mb-3"></i>
                                            <h5>External Video</h5>
                                            <p class="mb-3">This video is hosted on <?= htmlspecialchars($platform ?: 'external platform') ?>.</p>
                                            <a href="<?= htmlspecialchars($externalUrl) ?>" target="_blank" class="btn btn-light">
                                                <i class="bx bx-play me-1"></i> Watch Video
                                            </a>
                                        </div>
                                        <?php
                                    }
                                } else {
                                    // No valid external URL found
                                    ?>
                                    <div class="p-5 text-center bg-warning text-dark rounded">
                                        <i class="bx bx-error fs-1 mb-3"></i>
                                        <h5>External Video Configuration Error</h5>
                                        <p class="mb-0">This video is marked as external but no valid URL or embed code was found.</p>
                                    </div>
                                    <?php
                                }
                            } else {
                                // Handle file videos (original logic)
                                $filePathValue = null;
                                $possibleFileColumns = ['video_url', 'file_path', 'file_name', 'video_path'];
                                
                                foreach ($possibleFileColumns as $column) {
                                    if (isset($video[$column]) && !empty($video[$column])) {
                                        $filePathValue = $video[$column];
                                        break;
                                    }
                                }
                                
                                if (!empty($filePathValue)) {
                                    // Set up video URL
                                    $videoUrl = '../' . ltrim($filePathValue, './');
                                    
                                    // Set up thumbnail
                                    $thumbnailUrl = '';
                                    if (isset($video['thumbnail']) && !empty($video['thumbnail'])) {
                                        $thumbnailUrl = $video['thumbnail'];
                                        if (!filter_var($thumbnailUrl, FILTER_VALIDATE_URL)) {
                                            $thumbnailUrl = '../' . ltrim($video['thumbnail'], './');
                                        }
                                    }
                                    ?>
                                    <video class="video-player" controls <?= !empty($thumbnailUrl) ? 'poster="' . htmlspecialchars($thumbnailUrl) . '"' : '' ?>>
                                        <source src="<?= htmlspecialchars($videoUrl) ?>" type="video/mp4">
                                        <source src="<?= htmlspecialchars('../uploads/videos/' . basename($filePathValue)) ?>" type="video/mp4">
                                        Your browser does not support the video tag.
                                    </video>
                                    <?php
                                } else {
                                    // No file path found
                                    ?>
                                    <div class="p-5 text-center bg-dark text-white rounded">
                                        <i class="bx bx-video-off fs-1 mb-3"></i>
                                        <h5>Video file not available</h5>
                                        <p class="mb-0">The video file for this entry could not be found.</p>
                                    </div>
                                    <?php
                                }
                            }
                            ?>
                        </div>
                        
                        <div class="card video-info-card mb-4">
                            <div class="card-body p-4">
                                <h3 class="video-title"><?php echo htmlspecialchars($video['title']); ?></h3>
                                
                                <div class="video-meta">
                                    <div class="video-meta-item">
                                        <i class="bx bx-calendar"></i>
                                        <span><?php echo isset($video['created_at']) ? date('M d, Y', strtotime($video['created_at'])) : 'Unknown date'; ?></span>
                                    </div>


                                </div>
                                
                                <hr>
                                
                                <div class="video-description">
                                    <?php echo isset($video['description']) ? nl2br(htmlspecialchars($video['description'])) : 'No description provided.'; ?>
                                </div>
                                
                                <div class="upload-details mt-4">
                                    <h6 class="upload-details-title">Video Details</h6>
                                    <div class="row">
                                        <div class="col-md-6">
                                            <p class="mb-1"><strong>Video ID:</strong> <?php echo $video['video_id']; ?></p>
                                            
                                            <p class="mb-1"><strong>Status:</strong>
                                            <?php
                                            // Get status from database and display appropriate badge
                                            $statusBadgeClass = 'bg-secondary';
                                            $statusText = 'Status Unknown';
                                            
                                            // First try from direct video data (most accurate source)
                                            if (isset($directVideoData)) {
                                                if (isset($directVideoData['approved'])) {
                                                    $approvedValue = $directVideoData['approved'];
                                                    
                                                    // Handle different values for 'approved'
                                                    if ($approvedValue == 1 || strtolower($approvedValue) === 'true' || 
                                                        strtolower($approvedValue) === 'yes' || strtolower($approvedValue) === 'active') {
                                                        $statusBadgeClass = 'bg-success';
                                                        $statusText = 'Approved';
                                                    } else {
                                                        $statusBadgeClass = 'bg-warning text-dark';
                                                        $statusText = 'Pending';
                                                    }
                                                } 
                                                elseif (isset($directVideoData['status'])) {
                                                    $statusValue = $directVideoData['status'];
                                                    
                                                    if ($statusValue == 1 || strtolower($statusValue) === 'active' || 
                                                        strtolower($statusValue) === 'approved' || strtolower($statusValue) === 'yes') {
                                                        $statusBadgeClass = 'bg-success';
                                                        $statusText = 'Active';
                                                    } else {
                                                        $statusBadgeClass = 'bg-danger';
                                                        $statusText = 'Inactive';
                                                    }
                                                }
                                                else {
                                                    // Default to Active if using direct data but no status found
                                                    $statusBadgeClass = 'bg-success';
                                                    $statusText = 'Active';
                                                }
                                            }
                                            // Fallback to JOIN query data if direct access didn't work
                                            else if (in_array('approved', $videoTableColumns) && isset($video['approved'])) {
                                                if ($video['approved'] == '1' || strtolower($video['approved']) === 'true' || strtolower($video['approved']) === 'yes') {
                                                    $statusBadgeClass = 'bg-success';
                                                    $statusText = 'Approved';
                                                } else {
                                                    $statusBadgeClass = 'bg-warning text-dark';
                                                    $statusText = 'Pending';
                                                }
                                            } elseif (in_array('status', $videoTableColumns) && isset($video['status'])) {
                                                if (strtolower($video['status']) === 'active' || $video['status'] === '1') {
                                                    $statusBadgeClass = 'bg-success';
                                                    $statusText = 'Active';
                                                } else {
                                                    $statusBadgeClass = 'bg-danger';
                                                    $statusText = 'Inactive';
                                                }
                                            } elseif (in_array('state', $videoTableColumns) && isset($video['state'])) {
                                                if (strtolower($video['state']) === 'active' || $video['state'] === '1') {
                                                    $statusBadgeClass = 'bg-success';
                                                    $statusText = 'Active';
                                                } else {
                                                    $statusBadgeClass = 'bg-danger';
                                                    $statusText = 'Inactive';
                                                }
                                            } elseif (in_array('is_active', $videoTableColumns) && isset($video['is_active'])) {
                                                if ($video['is_active'] === '1' || strtolower($video['is_active']) === 'true') {
                                                    $statusBadgeClass = 'bg-success';
                                                    $statusText = 'Active';
                                                } else {
                                                    $statusBadgeClass = 'bg-danger';
                                                    $statusText = 'Inactive';
                                                }
                                            } elseif (in_array('active', $videoTableColumns) && isset($video['active'])) {
                                                if ($video['active'] === '1' || strtolower($video['active']) === 'true') {
                                                    $statusBadgeClass = 'bg-success';
                                                    $statusText = 'Active';
                                                } else {
                                                    $statusBadgeClass = 'bg-danger';
                                                    $statusText = 'Inactive';
                                                }
                                            } else {
                                                // Default to Active if no status column found (better UX than 'unknown')
                                                $statusBadgeClass = 'bg-success';
                                                $statusText = 'Active';
                                            }
                                            
                                            if ($debugMode) {
                                                echo '<!-- Status determined as: ' . $statusText . ' -->';
                                            }
                                            ?>
                                            <span class="badge <?php echo $statusBadgeClass; ?> rounded-pill"><?php echo $statusText; ?></span>
                                            </p>
                                            
                                            <p class="mb-1"><strong>File Name:</strong> <?php 
                                            // Check various possible file path columns and display one if found
                                            $fileName = 'Not available';
                                            foreach ($possibleFileColumns as $column) {
                                                if (isset($video[$column]) && !empty($video[$column])) {
                                                    $fileName = basename($video[$column]);
                                                    break;
                                                }
                                            }
                                            echo htmlspecialchars($fileName);
                                            ?></p>
                                        </div>
                                        <div class="col-md-6">
                                            <p class="mb-1"><strong>Uploaded:</strong> <?php echo isset($video['created_at']) ? date('F d, Y H:i', strtotime($video['created_at'])) : 'Unknown date'; ?></p>
                                            
                                            <?php if ($hasDurationColumn && isset($video['duration'])): ?>
                                            <p class="mb-1"><strong>Duration:</strong> <?php echo $video['duration']; ?></p>
                                            <?php endif; ?>
                                            

                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-lg-4">
                        <div class="card border-0 rounded-4 shadow-sm mb-4">
                            <div class="card-header bg-white py-3 border-0">
                                <h5 class="mb-0 fw-bold">Quick Actions</h5>
                            </div>
                            <div class="card-body p-4">
                                <div class="d-grid gap-2">
                                    <a href="edit_video.php?id=<?php echo $videoId; ?>" class="btn btn-primary rounded-3">
                                        <i class="bx bx-edit me-2"></i> Edit Video
                                    </a>
                                    
                                    <?php if($hasApprovedColumn): ?>
                                        <?php if(!$video['approved']): ?>
                                        <a href="approve_video.php?id=<?php echo $videoId; ?>" class="btn btn-success rounded-3">
                                            <i class="bx bx-check-circle me-2"></i> Approve Video
                                        </a>
                                        <?php else: ?>
                                        <a href="unapprove_video.php?id=<?php echo $videoId; ?>" class="btn btn-warning rounded-3">
                                            <i class="bx bx-x-circle me-2"></i> Unapprove Video
                                        </a>
                                        <?php endif; ?>
                                    <?php endif; ?>
                                    <button class="btn btn-danger rounded-3" data-bs-toggle="modal" data-bs-target="#deleteVideoModal">
                                        <i class="bx bx-trash me-2"></i> Delete Video
                                    </button>
                                </div>
                            </div>
                        </div>
                        

                    </div>
                </div>
                
                <?php else: ?>
                <!-- Not Found Content -->
                <div class="d-sm-flex justify-content-between align-items-center mb-4">
                    <div class="page-header-wrapper">
                        <h2 class="mb-0 fw-bold">
                            <div class="d-flex align-items-center">
                                <div class="icon-wrapper bg-warning-subtle rounded-circle p-3 me-3" style="width: 48px; height: 48px; display: flex; align-items: center; justify-content: center;">
                                    <i class="bx bx-error-circle fs-3 text-warning"></i>
                                </div>
                                Video Not Found
                            </div>
                        </h2>
                        <p class="text-muted mt-2"><?php echo $errorMessage; ?></p>
                    </div>
                </div>
                
                <div class="not-found-container">
                    <div class="not-found-content">
                        <div class="error-icon">
                            <i class="bx bx-video-off"></i>
                        </div>
                        <h1 class="error-code">Not Found</h1>
                        <div class="error-divider"></div>
                        <h3 class="mb-4">The requested video could not be found</h3>
                        <p class="text-muted mb-1">The video you're trying to view might have been removed, deleted, or never existed.</p>
                        <p class="text-muted mb-4">Please check the video ID and try again.</p>
                        
                        <div class="error-url">
                            <?php echo htmlspecialchars($requestedUrl); ?>
                        </div>
                        
                        <div class="action-buttons">
                            <a href="javascript:history.back()" class="btn btn-outline-secondary">
                                <i class="bx bx-arrow-back me-2"></i>Go Back
                            </a>
                            <a href="videos.php" class="btn btn-primary">
                                <i class="bx bx-video me-2"></i>All Videos
                            </a>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </section>
    
    <!-- Delete Video Modal -->
    <?php if ($videoExists): ?>
    <div class="modal fade" id="deleteVideoModal" tabindex="-1" aria-labelledby="deleteVideoModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-0 rounded-4">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title" id="deleteVideoModalLabel">Delete Video</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-4">
                    <p>Are you sure you want to delete this video?</p>
                    <p class="text-danger">This action cannot be undone. The video will be permanently removed from the system.</p>
                </div>
                <div class="modal-footer border-0">
                    <button type="button" class="btn btn-outline-secondary rounded-pill px-4" data-bs-dismiss="modal">Cancel</button>
                    <a href="delete_video.php?id=<?php echo $videoId; ?>" class="btn btn-danger rounded-pill px-4">Delete Permanently</a>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Sidebar toggle functionality
        const sidebarToggle = document.querySelector('.sidebar-toggle');
        const sidebarToggleMobile = document.querySelector('.sidebar-toggle-mobile');
        const sidebar = document.querySelector('.sidebar');
        
        if (sidebarToggle) {
            sidebarToggle.addEventListener('click', () => {
                sidebar.classList.toggle('close');
            });
        }
        
        if (sidebarToggleMobile) {
            sidebarToggleMobile.addEventListener('click', () => {
                sidebar.classList.toggle('show');
            });
        }
        
        // Responsive sidebar behavior
        function checkScreenSize() {
            if (window.innerWidth < 768) {
                sidebar.classList.remove('close');
                sidebar.classList.remove('show');
            } else if (window.innerWidth < 992) {
                sidebar.classList.add('close');
                sidebar.classList.remove('show');
            }
        }
        
        // Initial check and listen for window resize
        checkScreenSize();
        window.addEventListener('resize', checkScreenSize);
    });
    </script>
</body>
</html>
