<?php
/**
 * Manual Email Queue Processor
 * Process queued emails immediately
 */

require_once '../auth/functions.php';
require_once '../config/fast_email_service.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Bridge database connections for email service
if (!isset($pdo)) {
    try {
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        die("Failed to create PDO connection: " . $e->getMessage());
    }
}

$message = '';
$processed_emails = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['process_queue'])) {
    try {
        // Get pending emails from queue
        $stmt = $pdo->prepare("
            SELECT id, to_email, subject, body, priority, status, attempts, error_message, created_at
            FROM email_queue 
            WHERE status IN ('queued', 'failed') AND attempts < 3
            ORDER BY priority DESC, created_at ASC
            LIMIT 10
        ");
        $stmt->execute();
        $emails = $stmt->fetchAll();
        
        if (empty($emails)) {
            $message = '<div class="alert alert-info">No emails in queue to process.</div>';
        } else {
            foreach ($emails as $email) {
                $processed_emails[] = [
                    'id' => $email['id'],
                    'to' => $email['to_email'],
                    'subject' => $email['subject'],
                    'status' => $email['status'],
                    'attempts' => $email['attempts']
                ];
                
                // Update status to sending
                $update_stmt = $pdo->prepare("UPDATE email_queue SET status = 'sending', attempts = attempts + 1 WHERE id = ?");
                $update_stmt->execute([$email['id']]);
                
                // Try to send the email
                $result = sendEmailNow($email['to_email'], $email['subject'], $email['body']);
                
                if ($result) {
                    // Mark as sent
                    $sent_stmt = $pdo->prepare("UPDATE email_queue SET status = 'sent', sent_at = NOW() WHERE id = ?");
                    $sent_stmt->execute([$email['id']]);
                    $processed_emails[count($processed_emails)-1]['result'] = 'sent';
                } else {
                    // Mark as failed
                    $failed_stmt = $pdo->prepare("UPDATE email_queue SET status = 'failed', error_message = 'Send failed' WHERE id = ?");
                    $failed_stmt->execute([$email['id']]);
                    $processed_emails[count($processed_emails)-1]['result'] = 'failed';
                }
            }
            
            $sent_count = count(array_filter($processed_emails, function($e) { return $e['result'] === 'sent'; }));
            $failed_count = count(array_filter($processed_emails, function($e) { return $e['result'] === 'failed'; }));
            
            $message = '<div class="alert alert-success">
                Email processing complete! 
                📤 Sent: ' . $sent_count . ' | ❌ Failed: ' . $failed_count . '
            </div>';
        }
        
    } catch (Exception $e) {
        $message = '<div class="alert alert-danger">Error processing emails: ' . $e->getMessage() . '</div>';
    }
}

// Get current queue status
$queue_stats = [];
try {
    $stats_stmt = $pdo->query("
        SELECT status, COUNT(*) as count 
        FROM email_queue 
        GROUP BY status
        ORDER BY status
    ");
    $queue_stats = $stats_stmt->fetchAll();
} catch (Exception $e) {
    $queue_stats = [['status' => 'error', 'count' => 'Unable to fetch stats']];
}

// Get recent emails
$recent_emails = [];
try {
    $recent_stmt = $pdo->query("
        SELECT id, to_email, subject, status, attempts, created_at, sent_at, error_message
        FROM email_queue 
        ORDER BY created_at DESC 
        LIMIT 20
    ");
    $recent_emails = $recent_stmt->fetchAll();
} catch (Exception $e) {
    $recent_emails = [];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Email Queue Processor - Admin Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/boxicons@2.0.7/css/boxicons.min.css" rel="stylesheet">
</head>
<body class="bg-light">
    <div class="container mt-4">
        <div class="row">
            <div class="col-12">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2><i class="bx bx-mail-send"></i> Email Queue Processor</h2>
                    <a href="courses.php" class="btn btn-secondary">← Back to Courses</a>
                </div>
                
                <?php echo $message; ?>
                
                <!-- Queue Statistics -->
                <div class="row mb-4">
                    <div class="col-12">
                        <div class="card">
                            <div class="card-header">
                                <h5><i class="bx bx-pie-chart-alt"></i> Queue Statistics</h5>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <?php foreach ($queue_stats as $stat): ?>
                                    <div class="col-md-3 mb-3">
                                        <div class="stat-card p-3 border rounded text-center">
                                            <div class="stat-number h3 mb-1"><?php echo $stat['count']; ?></div>
                                            <div class="stat-label text-muted"><?php echo ucfirst($stat['status']); ?></div>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Process Queue -->
                <div class="row mb-4">
                    <div class="col-12">
                        <div class="card">
                            <div class="card-header">
                                <h5><i class="bx bx-play-circle"></i> Process Email Queue</h5>
                            </div>
                            <div class="card-body">
                                <p>Click the button below to manually process pending emails in the queue.</p>
                                <form method="POST">
                                    <button type="submit" name="process_queue" class="btn btn-primary">
                                        <i class="bx bx-send"></i> Process Queue Now
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
                
                <?php if (!empty($processed_emails)): ?>
                <!-- Processing Results -->
                <div class="row mb-4">
                    <div class="col-12">
                        <div class="card">
                            <div class="card-header">
                                <h5><i class="bx bx-list-check"></i> Processing Results</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-sm">
                                        <thead>
                                            <tr>
                                                <th>ID</th>
                                                <th>To</th>
                                                <th>Subject</th>
                                                <th>Previous Status</th>
                                                <th>Attempts</th>
                                                <th>Result</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($processed_emails as $email): ?>
                                            <tr>
                                                <td><?php echo $email['id']; ?></td>
                                                <td><?php echo htmlspecialchars($email['to']); ?></td>
                                                <td><?php echo htmlspecialchars(substr($email['subject'], 0, 50)) . '...'; ?></td>
                                                <td><?php echo $email['status']; ?></td>
                                                <td><?php echo $email['attempts']; ?></td>
                                                <td>
                                                    <?php if ($email['result'] === 'sent'): ?>
                                                        <span class="badge bg-success">✅ Sent</span>
                                                    <?php else: ?>
                                                        <span class="badge bg-danger">❌ Failed</span>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Recent Emails -->
                <div class="row">
                    <div class="col-12">
                        <div class="card">
                            <div class="card-header">
                                <h5><i class="bx bx-history"></i> Recent Emails</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-sm">
                                        <thead>
                                            <tr>
                                                <th>ID</th>
                                                <th>To</th>
                                                <th>Subject</th>
                                                <th>Status</th>
                                                <th>Attempts</th>
                                                <th>Created</th>
                                                <th>Sent</th>
                                                <th>Error</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($recent_emails as $email): ?>
                                            <tr>
                                                <td><?php echo $email['id']; ?></td>
                                                <td><?php echo htmlspecialchars($email['to_email']); ?></td>
                                                <td><?php echo htmlspecialchars(substr($email['subject'], 0, 40)) . '...'; ?></td>
                                                <td>
                                                    <?php 
                                                    $status_colors = [
                                                        'queued' => 'warning',
                                                        'sending' => 'info', 
                                                        'sent' => 'success',
                                                        'failed' => 'danger'
                                                    ];
                                                    $color = $status_colors[$email['status']] ?? 'secondary';
                                                    ?>
                                                    <span class="badge bg-<?php echo $color; ?>"><?php echo $email['status']; ?></span>
                                                </td>
                                                <td><?php echo $email['attempts']; ?></td>
                                                <td><?php echo date('M j, H:i', strtotime($email['created_at'])); ?></td>
                                                <td><?php echo $email['sent_at'] ? date('M j, H:i', strtotime($email['sent_at'])) : '-'; ?></td>
                                                <td><?php echo $email['error_message'] ? htmlspecialchars(substr($email['error_message'], 0, 30)) . '...' : '-'; ?></td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
