<?php
/**
 * Admin Referral Settings Management
 * Panadite Academy - Customize referral rewards and commissions
 */

require_once '../auth/functions.php';
requireRole('admin');

require_once '../config/db_connect.php';

$message = '';
$messageType = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $updates_made = 0;
    
    foreach ($_POST as $setting_name => $setting_value) {
        if ($setting_name !== 'submit') {
            $update_query = "UPDATE referral_settings SET setting_value = ?, updated_at = CURRENT_TIMESTAMP WHERE setting_name = ?";
            $stmt = $conn->prepare($update_query);
            
            if ($stmt) {
                $stmt->bind_param("ss", $setting_value, $setting_name);
                if ($stmt->execute() && $stmt->affected_rows > 0) {
                    $updates_made++;
                }
                $stmt->close();
            }
        }
    }
    
    if ($updates_made > 0) {
        $message = "✅ Referral settings updated successfully! ($updates_made settings changed)";
        $messageType = 'success';
    } else {
        $message = "ℹ️ No changes were made to the settings.";
        $messageType = 'info';
    }
}

// Get current referral settings
$settings_query = "SELECT * FROM referral_settings ORDER BY setting_type, setting_name";
$settings_result = $conn->query($settings_query);
$settings = [];

if ($settings_result) {
    while ($row = $settings_result->fetch_assoc()) {
        $settings[$row['setting_type']][] = $row;
    }
}

// Get referral statistics
$stats_query = "
    SELECT 
        COUNT(*) as total_referrals,
        COUNT(CASE WHEN status = 'completed' THEN 1 END) as successful_referrals,
        SUM(reward_amount) as total_rewards_granted,
        SUM(commission_earned) as total_commissions_earned
    FROM referrals
";
$stats_result = $conn->query($stats_query);
$stats = $stats_result ? $stats_result->fetch_assoc() : [];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Referral Settings - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>

<style>
.admin-content {
    padding: 2rem;
    max-width: 1200px;
    margin: 0 auto;
}

.settings-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 2rem;
    border-radius: 12px;
    margin-bottom: 2rem;
    text-align: center;
}

.settings-grid {
    display: grid;
    grid-template-columns: 1fr;
    gap: 2rem;
    margin-bottom: 2rem;
}

.settings-section {
    background: white;
    border-radius: 12px;
    padding: 1.5rem;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    border: 1px solid #e5e7eb;
}

.settings-section h3 {
    color: #374151;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.form-group {
    margin-bottom: 1rem;
}

.form-group label {
    display: block;
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.5rem;
}

.form-group input {
    width: 100%;
    padding: 0.75rem;
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    font-size: 0.9rem;
    transition: border-color 0.3s;
}

.form-group input:focus {
    outline: none;
    border-color: #3b82f6;
}

.form-group small {
    color: #6b7280;
    font-size: 0.8rem;
    margin-top: 0.25rem;
    display: block;
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    padding: 1.5rem;
    border-radius: 12px;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    text-align: center;
    border: 1px solid #e5e7eb;
}

.stat-number {
    font-size: 2rem;
    font-weight: bold;
    color: #3b82f6;
    display: block;
}

.stat-label {
    color: #6b7280;
    font-size: 0.9rem;
    margin-top: 0.5rem;
}

.save-button {
    background: linear-gradient(135deg, #10b981, #059669);
    color: white;
    padding: 1rem 2rem;
    border: none;
    border-radius: 8px;
    font-size: 1rem;
    font-weight: 600;
    cursor: pointer;
    transition: transform 0.2s;
    width: 100%;
    margin-top: 1rem;
}

.save-button:hover {
    transform: translateY(-2px);
}

.alert {
    padding: 1rem;
    border-radius: 8px;
    margin-bottom: 1rem;
}

.alert.success {
    background: #d1fae5;
    color: #065f46;
    border: 1px solid #a7f3d0;
}

.alert.info {
    background: #dbeafe;
    color: #1e40af;
    border: 1px solid #93c5fd;
}

@media (max-width: 768px) {
    .settings-grid {
        grid-template-columns: 1fr;
    }
}
</style>

        <!-- Dashboard Content -->
        <div class="dashboard-content">
    <!-- Header -->
    <div class="settings-header">
        <h1>🎯 Referral System Settings</h1>
        <p>Configure student discounts, teacher commissions, and referral rewards</p>
    </div>

    <!-- Message Display -->
    <?php if ($message): ?>
    <div class="alert <?= $messageType ?>">
        <?= htmlspecialchars($message) ?>
    </div>
    <?php endif; ?>

    <!-- Statistics -->
    <div class="stats-grid">
        <div class="stat-card">
            <span class="stat-number"><?= number_format($stats['total_referrals'] ?? 0) ?></span>
            <div class="stat-label">Total Referrals</div>
        </div>
        <div class="stat-card">
            <span class="stat-number"><?= number_format($stats['successful_referrals'] ?? 0) ?></span>
            <div class="stat-label">Successful Referrals</div>
        </div>
        <div class="stat-card">
            <span class="stat-number">R<?= number_format($stats['total_rewards_granted'] ?? 0, 2) ?></span>
            <div class="stat-label">Total Rewards Granted</div>
        </div>
        <div class="stat-card">
            <span class="stat-number">R<?= number_format($stats['total_commissions_earned'] ?? 0, 2) ?></span>
            <div class="stat-label">Total Commissions Earned</div>
        </div>
    </div>

    <!-- Settings Form -->
    <form method="POST">
        <div class="settings-grid">
            <!-- Student Discount Settings -->
            <div class="settings-section">
                <h3>🎓 Student Referral Rewards</h3>
                <?php if (isset($settings['student_discount'])): ?>
                    <?php foreach ($settings['student_discount'] as $setting): ?>
                    <div class="form-group">
                        <label for="<?= $setting['setting_name'] ?>">
                            <?= ucwords(str_replace('_', ' ', str_replace(['student_', '_percentage', '_days'], ['', ' %', ' (Days)'], $setting['setting_name']))) ?>
                        </label>
                        <input 
                            type="number" 
                            id="<?= $setting['setting_name'] ?>" 
                            name="<?= $setting['setting_name'] ?>" 
                            value="<?= htmlspecialchars($setting['setting_value']) ?>"
                            step="<?= strpos($setting['setting_name'], 'percentage') !== false ? '0.01' : '1' ?>"
                            min="0"
                            <?= strpos($setting['setting_name'], 'percentage') !== false ? 'max="100"' : '' ?>
                        >
                        <small><?= htmlspecialchars($setting['description']) ?></small>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>

            <!-- Teacher Commission Settings -->
            <div class="settings-section">
                <h3>👨‍🏫 Teacher Referral Commissions</h3>
                <?php if (isset($settings['teacher_commission'])): ?>
                    <?php foreach ($settings['teacher_commission'] as $setting): ?>
                    <div class="form-group">
                        <label for="<?= $setting['setting_name'] ?>">
                            <?= ucwords(str_replace('_', ' ', str_replace(['teacher_', '_percentage', '_months'], ['', ' %', ' (Months)'], $setting['setting_name']))) ?>
                        </label>
                        <input 
                            type="number" 
                            id="<?= $setting['setting_name'] ?>" 
                            name="<?= $setting['setting_name'] ?>" 
                            value="<?= htmlspecialchars($setting['setting_value']) ?>"
                            step="<?= strpos($setting['setting_name'], 'percentage') !== false ? '0.01' : '1' ?>"
                            min="0"
                            <?= strpos($setting['setting_name'], 'percentage') !== false ? 'max="100"' : '' ?>
                        >
                        <small><?= htmlspecialchars($setting['description']) ?></small>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- General Settings -->
        <div class="settings-section">
            <h3>⚙️ General Settings</h3>
            <div class="settings-grid">
                <?php if (isset($settings['general'])): ?>
                    <?php foreach ($settings['general'] as $setting): ?>
                    <div class="form-group">
                        <label for="<?= $setting['setting_name'] ?>">
                            <?= ucwords(str_replace('_', ' ', $setting['setting_name'])) ?>
                        </label>
                        <?php if ($setting['setting_name'] === 'referral_reward_active'): ?>
                        <select id="<?= $setting['setting_name'] ?>" name="<?= $setting['setting_name'] ?>">
                            <option value="1" <?= $setting['setting_value'] == '1' ? 'selected' : '' ?>>Active</option>
                            <option value="0" <?= $setting['setting_value'] == '0' ? 'selected' : '' ?>>Inactive</option>
                        </select>
                        <?php else: ?>
                        <input 
                            type="number" 
                            id="<?= $setting['setting_name'] ?>" 
                            name="<?= $setting['setting_name'] ?>" 
                            value="<?= htmlspecialchars($setting['setting_value']) ?>"
                            min="0"
                        >
                        <?php endif; ?>
                        <small><?= htmlspecialchars($setting['description']) ?></small>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

        <button type="submit" name="submit" class="save-button">
            💾 Save Referral Settings
        </button>
    </form>
</div>
        </div>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>  
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    // Add some interactivity
    document.addEventListener('DOMContentLoaded', function() {
        const inputs = document.querySelectorAll('input[type="number"]');
        
        inputs.forEach(input => {
            input.addEventListener('input', function() {
                if (this.name.includes('percentage') && parseFloat(this.value) > 100) {
                    this.value = 100;
                }
                if (parseFloat(this.value) < 0) {
                    this.value = 0;
                }
            });
        });
    });
    </script>
</body>
</html>
