<?php
require_once '../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Initialize variables
$message = '';
$messageType = '';

// Define backup directory
$backup_dir = '../backup';
if (!file_exists($backup_dir)) {
    mkdir($backup_dir, 0755, true);
}

// Process backup creation request
if (isset($_POST['create_backup'])) {
    try {
        // Generate backup timestamp
        $timestamp = date('Y-m-d_H-i-s');
        $backup_filename = "panadite_backup_$timestamp.sql";
        $backup_path = "$backup_dir/$backup_filename";
        
        // Get database credentials
        $db_host = $conn->host_info ? $conn->host_info : 'localhost';
        $db_name = 'panadite';
        $db_user = 'root'; // Assuming XAMPP default
        $db_pass = '';     // Assuming XAMPP default
        
        // Create backup command
        // Note: This is a simplified approach. In production, you'd use more secure methods
        // and likely integrate with a backup library or service
        $command = "mysqldump --host=$db_host --user=$db_user ";
        if (!empty($db_pass)) {
            $command .= "--password=$db_pass ";
        }
        $command .= "$db_name > $backup_path";
        
        // Execute backup command
        exec($command, $output, $return_var);
        
        if ($return_var === 0 && file_exists($backup_path)) {
            // Create a zip file of uploads directory
            $uploads_zip_filename = "panadite_uploads_$timestamp.zip";
            $uploads_zip_path = "$backup_dir/$uploads_zip_filename";
            
            $zip = new ZipArchive();
            if ($zip->open($uploads_zip_path, ZipArchive::CREATE) === TRUE) {
                // Add uploads directory to zip
                $uploads_dir = '../uploads';
                $files = new RecursiveIteratorIterator(
                    new RecursiveDirectoryIterator($uploads_dir),
                    RecursiveIteratorIterator::LEAVES_ONLY
                );
                
                foreach ($files as $name => $file) {
                    if (!$file->isDir()) {
                        $filePath = $file->getRealPath();
                        $relativePath = substr($filePath, strlen(dirname($uploads_dir)) + 1);
                        
                        $zip->addFile($filePath, $relativePath);
                    }
                }
                
                $zip->close();
                $message = "Backup created successfully. Database and uploads have been backed up.";
                $messageType = "success";
            } else {
                $message = "Database backup created, but failed to create uploads backup.";
                $messageType = "warning";
            }
        } else {
            $message = "Failed to create database backup. Please check MySQL connection and permissions.";
            $messageType = "danger";
        }
    } catch (Exception $e) {
        $message = "Error creating backup: " . $e->getMessage();
        $messageType = "danger";
    }
}

// Process backup deletion request
if (isset($_POST['delete_backup']) && isset($_POST['backup_file'])) {
    $backup_file = $_POST['backup_file'];
    $full_path = realpath("$backup_dir/$backup_file");
    
    // Security check to ensure we're only deleting files within our backup directory
    $backup_dir_real = realpath($backup_dir);
    if ($backup_dir_real && strpos($full_path, $backup_dir_real) === 0 && file_exists($full_path)) {
        if (unlink($full_path)) {
            $message = "Backup file deleted successfully.";
            $messageType = "success";
        } else {
            $message = "Error deleting backup file. Check file permissions.";
            $messageType = "danger";
        }
    } else {
        $message = "Invalid backup file or file does not exist.";
        $messageType = "danger";
    }
}

// Process backup restoration request
if (isset($_POST['restore_backup']) && isset($_POST['backup_file'])) {
    $backup_file = $_POST['backup_file'];
    $full_path = realpath("$backup_dir/$backup_file");
    
    // Security check
    $backup_dir_real = realpath($backup_dir);
    if ($backup_dir_real && strpos($full_path, $backup_dir_real) === 0 && file_exists($full_path)) {
        try {
            // Extract file extension
            $ext = strtolower(pathinfo($backup_file, PATHINFO_EXTENSION));
            
            if ($ext === 'sql') {
                // Get database credentials
                $db_host = $conn->host_info ? $conn->host_info : 'localhost';
                $db_name = 'panadite';
                $db_user = 'root'; // Assuming XAMPP default
                $db_pass = '';     // Assuming XAMPP default
                
                // Create restore command
                $command = "mysql --host=$db_host --user=$db_user ";
                if (!empty($db_pass)) {
                    $command .= "--password=$db_pass ";
                }
                $command .= "$db_name < $full_path";
                
                // Execute restore command
                exec($command, $output, $return_var);
                
                if ($return_var === 0) {
                    $message = "Database backup restored successfully.";
                    $messageType = "success";
                } else {
                    $message = "Failed to restore database backup.";
                    $messageType = "danger";
                }
            } elseif ($ext === 'zip') {
                // Handle uploads zip restoration
                $zip = new ZipArchive();
                if ($zip->open($full_path) === TRUE) {
                    $target_dir = '../';
                    $zip->extractTo($target_dir);
                    $zip->close();
                    $message = "Uploads backup restored successfully.";
                    $messageType = "success";
                } else {
                    $message = "Failed to restore uploads backup.";
                    $messageType = "danger";
                }
            } else {
                $message = "Unknown backup file type.";
                $messageType = "danger";
            }
        } catch (Exception $e) {
            $message = "Error restoring backup: " . $e->getMessage();
            $messageType = "danger";
        }
    } else {
        $message = "Invalid backup file or file does not exist.";
        $messageType = "danger";
    }
}

// Get list of backup files
$backup_files = [];
if (is_dir($backup_dir)) {
    $files = scandir($backup_dir);
    foreach ($files as $file) {
        if ($file !== '.' && $file !== '..' && !is_dir("$backup_dir/$file")) {
            $backup_files[] = [
                'name' => $file,
                'size' => filesize("$backup_dir/$file"),
                'date' => date("Y-m-d H:i:s", filemtime("$backup_dir/$file")),
                'type' => strtolower(pathinfo($file, PATHINFO_EXTENSION))
            ];
        }
    }
    
    // Sort by date (newest first)
    usort($backup_files, function($a, $b) {
        return strtotime($b['date']) - strtotime($a['date']);
    });
}

// Function to format file size
function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Backup - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.4/css/dataTables.bootstrap5.min.css">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        /* Main layout styles */
        :root {
            --sidebar-width: 260px;
            --sidebar-collapsed-width: 70px;
            --topbar-height: 60px;
            --primary-color: #4e73df;
            --secondary-color: #858796;
            --success-color: #1cc88a;
            --info-color: #36b9cc;
            --warning-color: #f6c23e;
            --danger-color: #e74a3b;
            --light-color: #f8f9fc;
            --dark-color: #5a5c69;
        }
        
        body {
            font-family: 'Nunito', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            background-color: #f8f9fc;
            overflow-x: hidden;
        }
        
        /* Sidebar styles from your existing pages */
        .main-content {
            padding-top: 60px !important; /* Exact navbar height */
            margin-top: -10px !important;
        }
        
        .backup-card {
            border: none;
            border-radius: 0.35rem;
            box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
        }
        
        .backup-icon {
            font-size: 1.5rem;
            margin-right: 0.5rem;
        }
        
        .sql-icon {
            color: var(--info-color);
        }
        
        .zip-icon {
            color: var(--warning-color);
        }
        
        .backup-action {
            width: 220px;
        }
        
        .confirmation-modal .modal-body {
            padding: 2rem;
            text-align: center;
        }
        
        .confirmation-modal .icon {
            font-size: 4rem;
            margin-bottom: 1rem;
        }
        
        .icon-danger {
            color: #e74a3b;
        }
        
        .icon-warning {
            color: #f6c23e;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include('components/sidebar.php'); ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include('components/navbar.php'); ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid">
                <!-- Page Header -->
                <div class="d-sm-flex align-items-center justify-content-between mb-4">
                    <h1 class="h3 mb-0 text-gray-800">System Backup</h1>
                    <a href="system_management.php" class="d-none d-sm-inline-block btn btn-sm btn-secondary shadow-sm">
                        <i class="fas fa-arrow-left fa-sm text-white-50"></i> Back to System Management
                    </a>
                </div>
                
                <?php if (!empty($message)): ?>
                    <div class="alert alert-<?= $messageType ?> alert-dismissible fade show" role="alert">
                        <?= $message ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                    
                    <div class="row">
                        <!-- Backup Options -->
                        <div class="col-lg-4 mb-4">
                            <div class="card backup-card">
                                <div class="card-header">
                                    <h5 class="m-0 font-weight-bold text-primary">Backup Options</h5>
                                </div>
                                <div class="card-body">
                                    <p class="mb-4">Create a full backup of your database and uploaded files. This process may take a few minutes depending on the size of your data.</p>
                                    
                                    <form action="" method="post">
                                        <button type="submit" name="create_backup" class="btn btn-primary btn-block w-100">
                                            <i class="fas fa-database me-2"></i> Create New Backup
                                        </button>
                                    </form>
                                    
                                    <hr class="my-4">
                                    
                                    <div class="alert alert-info">
                                        <i class="fas fa-info-circle me-2"></i>
                                        <strong>Important:</strong> Regular backups are recommended to prevent data loss. Keep multiple backup files for added safety.
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Backup Files -->
                        <div class="col-lg-8 mb-4">
                            <div class="card backup-card">
                                <div class="card-header">
                                    <h5 class="m-0 font-weight-bold text-primary">Backup Files</h5>
                                </div>
                                <div class="card-body">
                                    <?php if (count($backup_files) > 0): ?>
                                        <div class="table-responsive">
                                            <table class="table table-hover" id="backupTable">
                                                <thead>
                                                    <tr>
                                                        <th>Filename</th>
                                                        <th>Type</th>
                                                        <th>Size</th>
                                                        <th>Date</th>
                                                        <th class="backup-action">Actions</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php foreach ($backup_files as $file): ?>
                                                        <tr>
                                                            <td>
                                                                <?php if ($file['type'] === 'sql'): ?>
                                                                    <i class="backup-icon sql-icon bx bx-data"></i>
                                                                <?php elseif ($file['type'] === 'zip'): ?>
                                                                    <i class="backup-icon zip-icon bx bxs-file-archive"></i>
                                                                <?php else: ?>
                                                                    <i class="backup-icon bx bxs-file"></i>
                                                                <?php endif; ?>
                                                                <?= htmlspecialchars($file['name']) ?>
                                                            </td>
                                                            <td>
                                                                <?php if ($file['type'] === 'sql'): ?>
                                                                    <span class="badge bg-info">Database</span>
                                                                <?php elseif ($file['type'] === 'zip'): ?>
                                                                    <span class="badge bg-warning text-dark">Files</span>
                                                                <?php else: ?>
                                                                    <span class="badge bg-secondary">Unknown</span>
                                                                <?php endif; ?>
                                                            </td>
                                                            <td><?= formatFileSize($file['size']) ?></td>
                                                            <td><?= $file['date'] ?></td>
                                                            <td class="backup-action">
                                                                <div class="btn-group" role="group">
                                                                    <button type="button" class="btn btn-sm btn-success"
                                                                            data-bs-toggle="modal" 
                                                                            data-bs-target="#restoreBackupModal"
                                                                            data-backup-file="<?= $file['name'] ?>"
                                                                            data-backup-type="<?= $file['type'] ?>">
                                                                        <i class="fas fa-undo"></i> Restore
                                                                    </button>
                                                                    <a href="<?= '../backup/' . $file['name'] ?>" 
                                                                       class="btn btn-sm btn-primary" 
                                                                       download>
                                                                        <i class="fas fa-download"></i> Download
                                                                    </a>
                                                                    <button type="button" class="btn btn-sm btn-danger"
                                                                            data-bs-toggle="modal" 
                                                                            data-bs-target="#deleteBackupModal"
                                                                            data-backup-file="<?= $file['name'] ?>">
                                                                        <i class="fas fa-trash"></i> Delete
                                                                    </button>
                                                                </div>
                                                            </td>
                                                        </tr>
                                                    <?php endforeach; ?>
                                                </tbody>
                                            </table>
                                        </div>
                                    <?php else: ?>
                                        <div class="alert alert-info">
                                            <i class="fas fa-info-circle me-2"></i> No backup files found. Create a new backup to get started.
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Delete Backup Confirmation Modal -->
    <div class="modal fade confirmation-modal" id="deleteBackupModal" tabindex="-1" aria-labelledby="deleteBackupModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-body">
                    <i class="icon icon-danger bx bxs-trash"></i>
                    <h4 class="mb-4">Delete Backup</h4>
                    <p class="mb-4">Are you sure you want to delete this backup file?</p>
                    <p id="deleteBackupName" class="fw-bold mb-4">filename.ext</p>
                    <p class="text-danger mb-4">This action cannot be undone!</p>
                    <form action="" method="post">
                        <input type="hidden" name="backup_file" id="deleteBackupFile" value="">
                        <div class="d-flex justify-content-center gap-3 mt-4">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="submit" name="delete_backup" class="btn btn-danger">Delete Backup</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Restore Backup Confirmation Modal -->
    <div class="modal fade confirmation-modal" id="restoreBackupModal" tabindex="-1" aria-labelledby="restoreBackupModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-body">
                    <i class="icon icon-warning bx bx-sync"></i>
                    <h4 class="mb-4">Restore Backup</h4>
                    <p class="mb-4">Are you sure you want to restore this backup?</p>
                    <p id="restoreBackupName" class="fw-bold mb-2">filename.ext</p>
                    <p id="restoreBackupType" class="mb-4"></p>
                    <div class="alert alert-warning">
                        <strong>Warning:</strong> Restoring a backup will overwrite current data. This action cannot be undone!
                    </div>
                    <form action="" method="post">
                        <input type="hidden" name="backup_file" id="restoreBackupFile" value="">
                        <div class="mb-3">
                            <label for="confirmRestoreText" class="form-label">Type "RESTORE" to proceed:</label>
                            <input type="text" class="form-control" id="confirmRestoreText" required>
                        </div>
                        <div class="d-flex justify-content-center gap-3 mt-4">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="submit" name="restore_backup" id="restoreBackupBtn" class="btn btn-warning" disabled>Restore Backup</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/dataTables.bootstrap5.min.js"></script>
    <script>
        $(document).ready(function() {
            // Initialize DataTables
            $('#backupTable').DataTable({
                "order": [[3, "desc"]], // Sort by date by default (newest first)
                "lengthMenu": [[10, 25, 50, -1], [10, 25, 50, "All"]],
                "language": {
                    "search": "Search backups:",
                    "lengthMenu": "Show _MENU_ backups per page",
                    "info": "Showing _START_ to _END_ of _TOTAL_ backups",
                    "infoEmpty": "No backups found",
                    "infoFiltered": "(filtered from _MAX_ total backups)",
                    "zeroRecords": "No matching backups found"
                }
            });
            
            // Handle delete backup modal data
            $('#deleteBackupModal').on('show.bs.modal', function (event) {
                const button = $(event.relatedTarget);
                const backupFile = button.data('backup-file');
                
                $('#deleteBackupFile').val(backupFile);
                $('#deleteBackupName').text(backupFile);
            });
            
            // Handle restore backup modal data
            $('#restoreBackupModal').on('show.bs.modal', function (event) {
                const button = $(event.relatedTarget);
                const backupFile = button.data('backup-file');
                const backupType = button.data('backup-type');
                
                $('#restoreBackupFile').val(backupFile);
                $('#restoreBackupName').text(backupFile);
                
                if (backupType === 'sql') {
                    $('#restoreBackupType').html('<span class="badge bg-info">Database Backup</span>');
                } else if (backupType === 'zip') {
                    $('#restoreBackupType').html('<span class="badge bg-warning text-dark">Files Backup</span>');
                } else {
                    $('#restoreBackupType').html('<span class="badge bg-secondary">Unknown Backup Type</span>');
                }
                
                $('#confirmRestoreText').val('');
                $('#restoreBackupBtn').prop('disabled', true);
            });
            
            // Handle confirmation text for restore backup
            $('#confirmRestoreText').on('input', function() {
                $('#restoreBackupBtn').prop('disabled', $(this).val() !== "RESTORE");
            });
        });
    </script>
</body>
</html>
