<?php
require_once '../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Initialize response
$response = [
    'success' => false,
    'message' => '',
    'steps' => []
];

// Check for confirmation
if (isset($_POST['confirm_text']) && $_POST['confirm_text'] === 'CONFIRM DELETE ALL') {
    try {
        // Begin transaction for database operations
        $conn->begin_transaction();
        
        // Log progress
        $response['steps'][] = 'Starting system cleanup process...';
        
        // STEP 1: Clean database tables
        $response['steps'][] = 'Step 1: Cleaning database tables...';
        
        // Disable foreign key checks temporarily
        $conn->query("SET foreign_key_checks = 0");
        $response['steps'][] = '- Foreign key checks disabled';
        
        // Array of tables to truncate (in specific order to handle dependencies)
        $tables_to_clean = [
            // Forum related tables
            'forum_replies',
            'forum_topics',
            'forums',
            
            // Course related tables
            'assignment_submissions',
            'assignments',
            'calendar_events',
            'comments',
            'course_purchases',
            'enrollments',
            'material_views',
            'course_materials',
            'ratings',
            'video_purchases',
            'videos',
            'courses',
            'categories',
            
            // Messages
            'messages'
        ];
        
        // Truncate all tables
        foreach ($tables_to_clean as $table) {
            // Check if table exists
            $result = $conn->query("SHOW TABLES LIKE '$table'");
            if ($result->num_rows > 0) {
                $conn->query("TRUNCATE TABLE $table");
                $response['steps'][] = "- Cleaned table: $table";
            }
        }
        
        // Delete all users except admin (user_id = 1)
        $conn->query("DELETE FROM student_profiles");
        $conn->query("DELETE FROM teacher_profiles");
        $conn->query("DELETE FROM users WHERE user_id != 1");
        $response['steps'][] = "- Deleted all users except admin";
        
        // Reset auto increment values
        $tables_to_reset = [
            'users' => 2,
            'categories' => 1,
            'courses' => 1,
            'videos' => 1,
            'comments' => 1,
            'assignments' => 1,
            'forums' => 1,
            'forum_topics' => 1,
            'forum_replies' => 1
        ];
        
        foreach ($tables_to_reset as $table => $start_value) {
            $result = $conn->query("SHOW TABLES LIKE '$table'");
            if ($result->num_rows > 0) {
                $conn->query("ALTER TABLE $table AUTO_INCREMENT = $start_value");
                $response['steps'][] = "- Reset auto increment for $table";
            }
        }
        
        // Check if course_materials table exists and reset it
        $result = $conn->query("SHOW TABLES LIKE 'course_materials'");
        if ($result->num_rows > 0) {
            $conn->query("ALTER TABLE course_materials AUTO_INCREMENT = 1");
            $response['steps'][] = "- Reset auto increment for course_materials";
        }
        
        // Check if material_views table exists and reset it
        $result = $conn->query("SHOW TABLES LIKE 'material_views'");
        if ($result->num_rows > 0) {
            $conn->query("ALTER TABLE material_views AUTO_INCREMENT = 1");
            $response['steps'][] = "- Reset auto increment for material_views";
        }
        
        // Re-enable foreign key checks
        $conn->query("SET foreign_key_checks = 1");
        $response['steps'][] = '- Foreign key checks re-enabled';
        
        // Commit database changes
        $conn->commit();
        $response['steps'][] = 'Database cleanup completed successfully';
        
        // STEP 2: Clean uploaded files
        $response['steps'][] = 'Step 2: Cleaning uploaded files...';
        
        // Define directories to clean
        $directories = [
            '../uploads/materials',
            '../uploads/assignments',
            '../uploads/test',
            '../uploads/videos',
            '../uploads/thumbnails'
        ];
        
        // Function to delete all files in a directory except .gitkeep
        function cleanDirectory($dir, &$response) {
            if (!file_exists($dir)) {
                mkdir($dir, 0755, true);
                $response['steps'][] = "- Created directory: $dir";
                return;
            }
            
            $files_deleted = 0;
            $files = glob($dir . '/*');
            
            foreach ($files as $file) {
                if (is_dir($file)) {
                    cleanDirectory($file, $response);
                } else {
                    $basename = basename($file);
                    if ($basename !== '.gitkeep') {
                        if (unlink($file)) {
                            $files_deleted++;
                        }
                    }
                }
            }
            
            // Create .gitkeep file if it doesn't exist
            if (!file_exists($dir . '/.gitkeep')) {
                file_put_contents($dir . '/.gitkeep', '');
            }
            
            $response['steps'][] = "- Cleaned directory $dir: $files_deleted files deleted";
        }
        
        // Clean each directory
        foreach ($directories as $dir) {
            cleanDirectory($dir, $response);
        }
        
        $response['success'] = true;
        $response['message'] = 'System cleanup completed successfully. All data has been deleted except the admin account.';
    } catch (Exception $e) {
        // Roll back transaction on error
        $conn->rollback();
        $response['success'] = false;
        $response['message'] = 'Error during cleanup process: ' . $e->getMessage();
        $response['steps'][] = 'ERROR: ' . $e->getMessage();
    }
} else {
    $response['message'] = 'Invalid confirmation. System cleanup aborted.';
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Cleanup - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        .results-card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.05);
            overflow: hidden;
            transition: all 0.3s ease;
        }
        
        .log-container {
            max-height: 400px;
            overflow-y: auto;
            background-color: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            font-family: 'Consolas', monospace;
            font-size: 14px;
            border: 1px solid #ebeef4;
        }
        
        .log-entry {
            padding: 5px 0;
            border-bottom: 1px dotted #e3e6f0;
            line-height: 1.6;
        }
        
        .log-entry:last-child {
            border-bottom: none;
        }
        
        .icon-result {
            font-size: 6rem;
            margin-bottom: 1.5rem;
            filter: drop-shadow(0 5px 10px rgba(0, 0, 0, 0.1));
        }
        
        .icon-success {
            color: #1cc88a;
        }
        
        .icon-error {
            color: #e74a3b;
        }
        
        .page-header {
            background: linear-gradient(135deg, #e74a3b 0%, #be2617 100%);
            color: white;
            border-radius: 15px;
            padding: 25px;
            margin-bottom: 30px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }
        
        .page-header h1 {
            font-weight: 600;
            margin: 0;
            font-size: 1.8rem;
        }
        
        .page-header p {
            margin: 5px 0 0;
            opacity: 0.8;
        }
        
        .btn-return {
            border-radius: 50px;
            padding: 10px 25px;
            font-weight: 500;
            letter-spacing: 0.5px;
            transition: all 0.3s ease;
            text-transform: uppercase;
            font-size: 0.85rem;
            background: #4e73df;
            color: white;
        }
        
        .btn-return:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
            background: #375ac2;
            color: white;
        }
    </style>
</head>
<body>
    <!-- Sidebar Include -->
    <?php include('components/sidebar.php'); ?>
    
    <!-- Main Content Section -->
    <section class="main-content">
        <!-- Navbar Include -->  
        <?php include('components/navbar.php'); ?>
        
        <div class="container-fluid py-4">
            <!-- Page Header -->
            <div class="page-header">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h1>System Cleanup Results</h1>
                        <p>View the results of your system cleanup operation</p>
                    </div>
                    <a href="system_management.php" class="btn btn-light btn-sm rounded-pill shadow-sm">
                        <i class="fas fa-arrow-left me-1"></i> Back to System Management
                    </a>
                </div>
            </div>
            
            <div class="row justify-content-center">
                <div class="col-lg-8">
                    <div class="card results-card">
                        <div class="card-body text-center p-5">
                            <?php if ($response['success']): ?>
                                <i class="icon-result icon-success bx bx-check-circle"></i>
                                <h3 class="mb-3 text-success">System Cleanup Successful</h3>
                            <?php else: ?>
                                <i class="icon-result icon-error bx bx-error-circle"></i>
                                <h3 class="mb-3 text-danger">System Cleanup Failed</h3>
                            <?php endif; ?>
                            
                            <p class="mb-4"><?= $response['message'] ?></p>
                            
                            <div class="log-container mb-4 text-start">
                                <?php foreach ($response['steps'] as $step): ?>
                                    <div class="log-entry"><?= $step ?></div>
                                <?php endforeach; ?>
                            </div>
                            
                            <div class="mt-4">
                                <a href="system_management.php" class="btn btn-primary">Return to System Management</a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
    
    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Initialize tooltips
        var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'))
        var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl)
        })
    </script>
</body>
</html>
