<?php
require_once '../auth/functions.php';
require_once '../includes/config.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Check if database connection is available
if (!$conn) {
    die('Database connection failed. Please check your configuration.');
}

// Initialize variables
$message = '';
$messageType = '';
$content_types = [
    'courses' => [
        'title' => 'Courses',
        'icon' => 'bx bxs-book',
        'items' => [],
        'count' => 0,
        'fields' => ['course_id', 'title', 'description', 'teacher_id', 'category_id', 'price', 'is_published', 'created_at']
    ],
    'videos' => [
        'title' => 'Videos',
        'icon' => 'bx bxs-video',
        'items' => [],
        'count' => 0,
        'fields' => ['video_id', 'course_id', 'title', 'description', 'video_path', 'is_published', 'created_at']
    ],
    'assignments' => [
        'title' => 'Assignments',
        'icon' => 'bx bxs-file-doc',
        'items' => [],
        'count' => 0,
        'fields' => ['assignment_id', 'course_id', 'title', 'description', 'due_date', 'created_at']
    ],
    'forums' => [
        'title' => 'Forums',
        'icon' => 'bx bxs-conversation',
        'items' => [],
        'count' => 0,
        'fields' => ['forum_id', 'course_id', 'title', 'description', 'created_at']
    ]
];

// Process deletion requests
if (isset($_POST['delete_content']) && isset($_POST['content_type']) && isset($_POST['content_id'])) {
    $content_type = $_POST['content_type'];
    $content_id = (int)$_POST['content_id'];
    $id_field = substr($content_type, 0, -1) . '_id'; // e.g., courses -> course_id
    
    try {
        // Begin transaction
        $conn->begin_transaction();
        
        // Get content info for logging and file deletion
        $stmt = $conn->prepare("SELECT * FROM $content_type WHERE $id_field = ?");
        if ($stmt === false) {
            throw new Exception("Failed to prepare select statement: " . $conn->error);
        }
        $stmt->bind_param('i', $content_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $content = $result->fetch_assoc();
        $stmt->close();
        
        if (!$content) {
            throw new Exception("Content not found");
        }
        
        // Handle specific content type deletions
        switch ($content_type) {
            case 'courses':
                // Delete course thumbnail if exists
                if (!empty($content['thumbnail_path']) && file_exists('../' . $content['thumbnail_path'])) {
                    unlink('../' . $content['thumbnail_path']);
                }
                
                // Get and delete course materials
                $stmt = $conn->prepare("SELECT material_path FROM course_materials WHERE course_id = ?");
                if ($stmt === false) {
                    error_log("Prepare failed for course_materials query: " . $conn->error);
                    // Continue without materials deletion if table doesn't exist
                } else {
                    $stmt->bind_param('i', $content_id);
                    $stmt->execute();
                    $materials = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
                    $stmt->close();
                    
                    foreach ($materials as $material) {
                        if (!empty($material['material_path']) && file_exists('../' . $material['material_path'])) {
                            unlink('../' . $material['material_path']);
                        }
                    }
                }
                
                // Delete course videos (recursively handled by DB constraints)
                break;
                
            case 'videos':
                // Delete video file if exists
                if (!empty($content['video_path']) && file_exists('../' . $content['video_path'])) {
                    unlink('../' . $content['video_path']);
                }
                
                // Delete thumbnail if exists
                if (!empty($content['thumbnail_path']) && file_exists('../' . $content['thumbnail_path'])) {
                    unlink('../' . $content['thumbnail_path']);
                }
                break;
                
            case 'assignments':
                // Get and delete assignment submissions
                $stmt = $conn->prepare("SELECT file_path FROM assignment_submissions WHERE assignment_id = ?");
                if ($stmt === false) {
                    error_log("Prepare failed for assignment_submissions query: " . $conn->error);
                    // Continue without submissions deletion if table doesn't exist
                } else {
                    $stmt->bind_param('i', $content_id);
                    $stmt->execute();
                    $submissions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
                    $stmt->close();
                    
                    foreach ($submissions as $submission) {
                        if (!empty($submission['file_path']) && file_exists('../' . $submission['file_path'])) {
                            unlink('../' . $submission['file_path']);
                        }
                    }
                }
                break;
        }
        
        // Delete the content from database
        $stmt = $conn->prepare("DELETE FROM $content_type WHERE $id_field = ?");
        if ($stmt === false) {
            throw new Exception("Failed to prepare delete statement: " . $conn->error);
        }
        $stmt->bind_param('i', $content_id);
        $stmt->execute();
        
        if ($stmt->affected_rows > 0) {
            // Commit transaction
            $conn->commit();
            $message = ucfirst(substr($content_type, 0, -1)) . " deleted successfully.";
            $messageType = "success";
        } else {
            throw new Exception("No content was deleted");
        }
        $stmt->close();
    } catch (Exception $e) {
        // Rollback on error
        $conn->rollback();
        $message = "Error deleting content: " . $e->getMessage();
        $messageType = "danger";
    }
}

// Load content data
foreach ($content_types as $type => &$content_info) {
    // Count total items
    $result = $conn->query("SELECT COUNT(*) as count FROM $type");
    $content_info['count'] = $result->fetch_assoc()['count'];
    
    // Get items (limit to 50)
    $query = "SELECT * FROM $type ORDER BY created_at DESC LIMIT 50";
    $result = $conn->query($query);
    
    if ($result) {
        $content_info['items'] = $result->fetch_all(MYSQLI_ASSOC);
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Content Management - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.4/css/dataTables.bootstrap5.min.css">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        /* Main layout styles */
        :root {
            --sidebar-width: 260px;
            --sidebar-collapsed-width: 70px;
            --topbar-height: 60px;
            --primary-color: #4e73df;
            --secondary-color: #858796;
            --success-color: #1cc88a;
            --info-color: #36b9cc;
            --warning-color: #f6c23e;
            --danger-color: #e74a3b;
            --light-color: #f8f9fc;
            --dark-color: #5a5c69;
        }
        
        body {
            font-family: 'Nunito', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            background-color: #f8f9fc;
            overflow-x: hidden;
        }
        
        /* Sidebar styles from your existing pages */
        .main-content {
            padding-top: 60px !important; /* Exact navbar height */
            margin-top: -10px !important;
        }
        
        .content-card {
            border: none;
            border-radius: 0.35rem;
            box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
            margin-bottom: 1.5rem;
            transition: all 0.3s ease;
        }
        
        .card-header {
            background-color: #f8f9fc;
            border-bottom: 1px solid #e3e6f0;
            padding: 1rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .content-icon {
            font-size: 2.5rem;
            margin-right: 1rem;
            color: var(--primary-color);
        }
        
        .content-info {
            display: flex;
            align-items: center;
        }
        
        .content-row:hover {
            background-color: rgba(78, 115, 223, 0.05);
        }
        
        .confirmation-modal .modal-body {
            padding: 2rem;
            text-align: center;
        }
        
        .confirmation-modal .icon {
            font-size: 4rem;
            color: #e74a3b;
            margin-bottom: 1rem;
        }
        
        /* DataTables custom styling */
        .dataTables_wrapper .dataTables_length,
        .dataTables_wrapper .dataTables_filter,
        .dataTables_wrapper .dataTables_info {
            padding: 1rem;
        }
        
        .dataTables_wrapper .dataTables_paginate {
            padding: 1rem;
        }
        
        .dataTables_wrapper .dataTables_paginate .paginate_button.current {
            background: var(--primary-color) !important;
            color: white !important;
            border: none;
        }
        
        .dataTables_wrapper .dataTables_paginate .paginate_button:hover {
            background: var(--primary-color) !important;
            color: white !important;
        }
        
        .nav-pills .nav-link {
            color: var(--dark-color);
        }
        
        .nav-pills .nav-link.active {
            background-color: var(--primary-color);
            color: white;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include('components/sidebar.php'); ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include('components/navbar.php'); ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid">
                <!-- Page Header -->
                <div class="d-sm-flex align-items-center justify-content-between mb-4">
                    <h1 class="h3 mb-0 text-gray-800">Content Management</h1>
                    <a href="system_management.php" class="d-none d-sm-inline-block btn btn-sm btn-secondary shadow-sm">
                        <i class="fas fa-arrow-left fa-sm text-white-50"></i> Back to System Management
                    </a>
                </div>
                
                <?php if (!empty($message)): ?>
                    <div class="alert alert-<?= $messageType ?> alert-dismissible fade show" role="alert">
                        <?= $message ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                    
                    <!-- Content Type Tabs -->
                    <ul class="nav nav-pills mb-4" id="contentTabs" role="tablist">
                        <?php $active = true; foreach ($content_types as $type => $content): ?>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link <?= $active ? 'active' : '' ?>" 
                                        id="<?= $type ?>-tab" 
                                        data-bs-toggle="pill" 
                                        data-bs-target="#<?= $type ?>" 
                                        type="button" 
                                        role="tab" 
                                        aria-controls="<?= $type ?>" 
                                        aria-selected="<?= $active ? 'true' : 'false' ?>">
                                    <i class="<?= $content['icon'] ?>"></i> <?= $content['title'] ?>
                                    <span class="badge bg-primary rounded-pill ms-1"><?= $content['count'] ?></span>
                                </button>
                            </li>
                            <?php $active = false; ?>
                        <?php endforeach; ?>
                    </ul>
                    
                    <!-- Content Type Panes -->
                    <div class="tab-content" id="contentTabsContent">
                        <?php $active = true; foreach ($content_types as $type => $content): ?>
                            <div class="tab-pane fade <?= $active ? 'show active' : '' ?>" 
                                 id="<?= $type ?>" 
                                 role="tabpanel" 
                                 aria-labelledby="<?= $type ?>-tab">
                                <div class="card content-card mb-4">
                                    <div class="card-header">
                                        <div class="content-info">
                                            <i class="content-icon <?= $content['icon'] ?>"></i>
                                            <div>
                                                <h5 class="mb-0"><?= $content['title'] ?></h5>
                                                <small class="text-muted">
                                                    <?= $content['count'] ?> items
                                                </small>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="card-body">
                                        <?php if (count($content['items']) > 0): ?>
                                            <div class="table-responsive">
                                                <table class="table table-hover contentTable">
                                                    <thead>
                                                        <tr>
                                                            <?php foreach ($content['fields'] as $field): ?>
                                                                <th><?= ucwords(str_replace('_', ' ', $field)) ?></th>
                                                            <?php endforeach; ?>
                                                            <th>Actions</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        <?php foreach ($content['items'] as $item): ?>
                                                            <tr class="content-row">
                                                                <?php foreach ($content['fields'] as $field): ?>
                                                                    <td>
                                                                        <?php 
                                                                        // Special handling for certain fields
                                                                        if ($field === 'description' && strlen($item[$field]) > 50) {
                                                                            echo htmlspecialchars(substr($item[$field], 0, 50)) . '...';
                                                                        } elseif ($field === 'is_published') {
                                                                            echo $item[$field] ? '<span class="badge bg-success">Published</span>' : '<span class="badge bg-secondary">Draft</span>';
                                                                        } elseif ($field === 'created_at' || $field === 'due_date') {
                                                                            echo date('Y-m-d', strtotime($item[$field]));
                                                                        } elseif ($field === 'video_path' || $field === 'thumbnail_path') {
                                                                            echo !empty($item[$field]) ? '<i class="bx bx-check text-success"></i>' : '<i class="bx bx-x text-danger"></i>';
                                                                        } else {
                                                                            echo htmlspecialchars($item[$field] ?? 'N/A');
                                                                        }
                                                                        ?>
                                                                    </td>
                                                                <?php endforeach; ?>
                                                                <td>
                                                                    <button type="button" class="btn btn-sm btn-danger" 
                                                                            data-bs-toggle="modal" 
                                                                            data-bs-target="#deleteContentModal" 
                                                                            data-content-type="<?= $type ?>"
                                                                            data-content-id="<?= $item[substr($type, 0, -1) . '_id'] ?>"
                                                                            data-content-title="<?= htmlspecialchars($item['title']) ?>">
                                                                        <i class="fas fa-trash"></i> Delete
                                                                    </button>
                                                                </td>
                                                            </tr>
                                                        <?php endforeach; ?>
                                                    </tbody>
                                                </table>
                                                
                                                <?php if ($content['count'] > count($content['items'])): ?>
                                                    <div class="alert alert-info">
                                                        <i class="fas fa-info-circle"></i> 
                                                        Showing <?= count($content['items']) ?> of <?= $content['count'] ?> total items.
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        <?php else: ?>
                                            <div class="alert alert-info">
                                                <i class="fas fa-info-circle"></i> No items found in this category.
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <?php $active = false; ?>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Delete Content Confirmation Modal -->
    <div class="modal fade confirmation-modal" id="deleteContentModal" tabindex="-1" aria-labelledby="deleteContentModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-body">
                    <i class="icon bx bxs-trash"></i>
                    <h4 class="mb-4">Delete Content</h4>
                    <p class="mb-4">Are you sure you want to delete this item?</p>
                    <p id="deleteContentTitle" class="fw-bold mb-4">Content Title</p>
                    <p class="text-danger mb-4">This action cannot be undone! All associated data will also be deleted.</p>
                    <form action="" method="post">
                        <input type="hidden" name="content_type" id="deleteContentType" value="">
                        <input type="hidden" name="content_id" id="deleteContentId" value="">
                        <div class="d-flex justify-content-center gap-3 mt-4">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="submit" name="delete_content" class="btn btn-danger">Delete Content</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/dataTables.bootstrap5.min.js"></script>
    <script>
        $(document).ready(function() {
            // Initialize DataTables
            $('.contentTable').DataTable({
                "order": [[0, "desc"]], // Sort by ID by default
                "lengthMenu": [[10, 25, 50, -1], [10, 25, 50, "All"]],
                "language": {
                    "search": "Search content:",
                    "lengthMenu": "Show _MENU_ items per page",
                    "info": "Showing _START_ to _END_ of _TOTAL_ items",
                    "infoEmpty": "No items found",
                    "infoFiltered": "(filtered from _MAX_ total items)",
                    "zeroRecords": "No matching items found"
                }
            });
            
            // Handle delete content modal data
            $('#deleteContentModal').on('show.bs.modal', function (event) {
                const button = $(event.relatedTarget);
                const contentType = button.data('content-type');
                const contentId = button.data('content-id');
                const contentTitle = button.data('content-title');
                
                $('#deleteContentType').val(contentType);
                $('#deleteContentId').val(contentId);
                $('#deleteContentTitle').text(contentTitle);
            });
        });
    </script>
</body>
</html>
