<?php
require_once '../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Process file deletion actions
$message = '';
$messageType = '';

// Define upload directories
$upload_dirs = [
    'materials' => [
        'path' => '../uploads/materials',
        'title' => 'Course Materials',
        'icon' => 'bx bxs-file-pdf'
    ],
    'assignments' => [
        'path' => '../uploads/assignments',
        'title' => 'Assignment Submissions',
        'icon' => 'bx bxs-file-doc'
    ],
    'videos' => [
        'path' => '../uploads/videos',
        'title' => 'Video Files',
        'icon' => 'bx bxs-video'
    ],
    'thumbnails' => [
        'path' => '../uploads/thumbnails',
        'title' => 'Thumbnail Images',
        'icon' => 'bx bxs-image'
    ],
    'test' => [
        'path' => '../uploads/test',
        'title' => 'Test Files',
        'icon' => 'bx bxs-file'
    ]
];

// Handle file deletion
if (isset($_POST['delete_file']) && isset($_POST['file_path']) && !empty($_POST['file_path'])) {
    $file_path = $_POST['file_path'];
    $full_path = realpath($file_path);
    
    // Security check to ensure we're only deleting files within our upload directories
    $is_valid_path = false;
    foreach ($upload_dirs as $dir) {
        $base_path = realpath($dir['path']);
        if ($base_path && strpos($full_path, $base_path) === 0) {
            $is_valid_path = true;
            break;
        }
    }
    
    if ($is_valid_path && file_exists($full_path) && is_file($full_path)) {
        if (unlink($full_path)) {
            $message = "File deleted successfully.";
            $messageType = "success";
        } else {
            $message = "Error deleting file. Check file permissions.";
            $messageType = "danger";
        }
    } else {
        $message = "Invalid file path or file does not exist.";
        $messageType = "danger";
    }
}

// Handle directory cleanup (delete all files in directory)
if (isset($_POST['clean_directory']) && isset($_POST['directory']) && array_key_exists($_POST['directory'], $upload_dirs)) {
    $dir_key = $_POST['directory'];
    $dir_path = $upload_dirs[$dir_key]['path'];
    $files_deleted = 0;
    $errors = 0;
    
    if (is_dir($dir_path)) {
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir_path, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );
        
        foreach ($files as $fileinfo) {
            if ($fileinfo->isFile()) {
                $filename = $fileinfo->getFilename();
                if ($filename !== '.gitkeep') {
                    if (unlink($fileinfo->getRealPath())) {
                        $files_deleted++;
                    } else {
                        $errors++;
                    }
                }
            }
        }
        
        if ($errors == 0) {
            $message = "All files deleted successfully from {$upload_dirs[$dir_key]['title']}. Total files deleted: $files_deleted.";
            $messageType = "success";
        } else {
            $message = "$files_deleted files deleted, but encountered errors with $errors files.";
            $messageType = "warning";
        }
    } else {
        $message = "Directory not found or is not accessible.";
        $messageType = "danger";
    }
}

// Function to get directory size
function getDirSize($dir) {
    $size = 0;
    if (is_dir($dir)) {
        foreach (new RecursiveIteratorIterator(new RecursiveDirectoryIterator($dir)) as $file) {
            if ($file->isFile() && $file->getFilename() !== '.gitkeep') {
                $size += $file->getSize();
            }
        }
    }
    return $size;
}

// Function to format file size
function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}

// Get file information for each directory
$directories = [];
foreach ($upload_dirs as $key => $dir) {
    $file_count = 0;
    $dir_size = 0;
    $files = [];
    
    if (is_dir($dir['path'])) {
        $dir_size = getDirSize($dir['path']);
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir['path'], RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );
        
        foreach ($iterator as $fileinfo) {
            if ($fileinfo->isFile() && $fileinfo->getFilename() !== '.gitkeep') {
                $file_count++;
                
                // Only add up to 100 files to prevent UI overload
                if (count($files) < 100) {
                    $files[] = [
                        'name' => $fileinfo->getFilename(),
                        'path' => $fileinfo->getRealPath(),
                        'size' => formatFileSize($fileinfo->getSize()),
                        'modified' => date("Y-m-d H:i:s", $fileinfo->getMTime())
                    ];
                }
            }
        }
        
        // Sort files by modified date (newest first)
        usort($files, function($a, $b) {
            return strtotime($b['modified']) - strtotime($a['modified']);
        });
    }
    
    $directories[$key] = [
        'title' => $dir['title'],
        'icon' => $dir['icon'],
        'path' => $dir['path'],
        'file_count' => $file_count,
        'size' => formatFileSize($dir_size),
        'files' => $files
    ];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>File Management - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.4/css/dataTables.bootstrap5.min.css">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        .dir-card {
            border: none;
            border-radius: 0.35rem;
            box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
            margin-bottom: 1.5rem;
            transition: all 0.3s ease;
        }
        
        .dir-card:hover {
            transform: translateY(-5px);
        }
        
        .card-header {
            background-color: #f8f9fc;
            border-bottom: 1px solid #e3e6f0;
            padding: 1rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .dir-icon {
            font-size: 2.5rem;
            margin-right: 1rem;
            color: var(--primary-color);
        }
        
        .dir-info {
            display: flex;
            align-items: center;
        }
        
        .file-actions {
            display: flex;
            align-items: center;
        }
        
        .file-row:hover {
            background-color: rgba(78, 115, 223, 0.05);
        }
        
        .file-icon {
            font-size: 1.2rem;
            margin-right: 0.5rem;
        }
        
        .confirmation-modal .modal-body {
            padding: 2rem;
            text-align: center;
        }
        
        .confirmation-modal .icon {
            font-size: 4rem;
            color: #e74a3b;
            margin-bottom: 1rem;
        }
        
        .nav-pills .nav-link {
            color: var(--dark-color);
        }
        
        .nav-pills .nav-link.active {
            background-color: var(--primary-color);
            color: white;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include('components/sidebar.php'); ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include('components/navbar.php'); ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid">
                <!-- Page Header -->
                <div class="d-sm-flex align-items-center justify-content-between mb-4">
                    <h1 class="h3 mb-0 text-gray-800">File Management</h1>
                    <a href="system_management.php" class="d-none d-sm-inline-block btn btn-sm btn-secondary shadow-sm">
                        <i class="fas fa-arrow-left fa-sm text-white-50"></i> Back to System Management
                    </a>
                </div>
                
                <?php if (!empty($message)): ?>
                    <div class="alert alert-<?= $messageType ?> alert-dismissible fade show" role="alert">
                        <?= $message ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <!-- Directory Navigation Tabs -->
                <ul class="nav nav-pills mb-4" id="directoryTabs" role="tablist">
                    <?php $active = true; foreach ($directories as $key => $dir): ?>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link <?= $active ? 'active' : '' ?>" 
                                    id="<?= $key ?>-tab" 
                                    data-bs-toggle="pill" 
                                    data-bs-target="#<?= $key ?>" 
                                    type="button" 
                                    role="tab" 
                                    aria-controls="<?= $key ?>" 
                                    aria-selected="<?= $active ? 'true' : 'false' ?>">
                                <i class="<?= $dir['icon'] ?>"></i> <?= $dir['title'] ?>
                            </button>
                        </li>
                        <?php $active = false; ?>
                    <?php endforeach; ?>
                </ul>
                
                <!-- Directory Content Tabs -->
                <div class="tab-content" id="directoryTabsContent">
                    <?php $active = true; foreach ($directories as $key => $dir): ?>
                        <div class="tab-pane fade <?= $active ? 'show active' : '' ?>" 
                             id="<?= $key ?>" 
                             role="tabpanel" 
                             aria-labelledby="<?= $key ?>-tab">
                            <div class="card dir-card mb-4">
                                <div class="card-header">
                                    <div class="dir-info">
                                        <i class="dir-icon <?= $dir['icon'] ?>"></i>
                                        <div>
                                            <h5 class="mb-0"><?= $dir['title'] ?></h5>
                                            <small class="text-muted">
                                                <?= $dir['file_count'] ?> files (<?= $dir['size'] ?>)
                                            </small>
                                        </div>
                                    </div>
                                    <div>
                                        <button type="button" class="btn btn-danger" 
                                                data-bs-toggle="modal" 
                                                data-bs-target="#cleanDirectoryModal" 
                                                data-directory="<?= $key ?>"
                                                data-dirtitle="<?= $dir['title'] ?>"
                                                data-filecount="<?= $dir['file_count'] ?>">
                                            <i class="fas fa-trash"></i> Clean Directory
                                        </button>
                                    </div>
                                </div>
                                <div class="card-body">
                                    <?php if (count($dir['files']) > 0): ?>
                                        <div class="table-responsive">
                                            <table class="table table-hover fileTable">
                                                <thead>
                                                    <tr>
                                                        <th>File Name</th>
                                                        <th>Size</th>
                                                        <th>Modified</th>
                                                        <th>Actions</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php foreach ($dir['files'] as $file): ?>
                                                        <tr class="file-row">
                                                            <td>
                                                                <i class="file-icon <?= $dir['icon'] ?>"></i>
                                                                <?= htmlspecialchars($file['name']) ?>
                                                            </td>
                                                            <td><?= $file['size'] ?></td>
                                                            <td><?= $file['modified'] ?></td>
                                                            <td>
                                                                <button type="button" class="btn btn-sm btn-danger" 
                                                                        data-bs-toggle="modal" 
                                                                        data-bs-target="#deleteFileModal" 
                                                                        data-filepath="<?= $file['path'] ?>"
                                                                        data-filename="<?= htmlspecialchars($file['name']) ?>">
                                                                    <i class="fas fa-trash"></i> Delete
                                                                </button>
                                                            </td>
                                                        </tr>
                                                    <?php endforeach; ?>
                                                </tbody>
                                            </table>
                                            
                                            <?php if ($dir['file_count'] > count($dir['files'])): ?>
                                                <div class="alert alert-info">
                                                    <i class="fas fa-info-circle"></i> 
                                                    Showing <?= count($dir['files']) ?> of <?= $dir['file_count'] ?> total files.
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    <?php else: ?>
                                        <div class="alert alert-info">
                                            <i class="fas fa-info-circle"></i> No files found in this directory.
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                        <?php $active = false; ?>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </section>
    
    <!-- Delete File Confirmation Modal -->
    <div class="modal fade confirmation-modal" id="deleteFileModal" tabindex="-1" aria-labelledby="deleteFileModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-body">
                    <i class="icon bx bxs-trash"></i>
                    <h4 class="mb-4">Delete File</h4>
                    <p class="mb-4">Are you sure you want to delete this file?</p>
                    <p id="deleteFileName" class="fw-bold mb-4">filename.ext</p>
                    <p class="text-danger mb-4">This action cannot be undone!</p>
                    <form action="" method="post">
                        <input type="hidden" name="file_path" id="deleteFilePath" value="">
                        <div class="d-flex justify-content-center gap-3 mt-4">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="submit" name="delete_file" class="btn btn-danger">Delete File</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Clean Directory Confirmation Modal -->
    <div class="modal fade confirmation-modal" id="cleanDirectoryModal" tabindex="-1" aria-labelledby="cleanDirectoryModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-body">
                    <i class="icon bx bxs-trash"></i>
                    <h4 class="mb-4">Clean Directory</h4>
                    <p class="mb-4">Are you sure you want to delete <span id="cleanDirFileCount">0</span> files from:</p>
                    <p id="cleanDirName" class="fw-bold mb-4">Directory Name</p>
                    <p class="text-danger mb-4">This action cannot be undone!</p>
                    <form action="" method="post">
                        <input type="hidden" name="directory" id="cleanDirectoryKey" value="">
                        <div class="mb-3">
                            <label for="confirmCleanText" class="form-label">Type "CONFIRM" to proceed:</label>
                            <input type="text" class="form-control" id="confirmCleanText" required>
                        </div>
                        <div class="d-flex justify-content-center gap-3 mt-4">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="submit" name="clean_directory" id="cleanDirectoryBtn" class="btn btn-danger" disabled>Clean Directory</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/dataTables.bootstrap5.min.js"></script>
    <script>
        $(document).ready(function() {
            // Initialize DataTables
            $('.fileTable').DataTable({
                "order": [[2, "desc"]], // Sort by modified date by default
                "lengthMenu": [[10, 25, 50, -1], [10, 25, 50, "All"]],
                "language": {
                    "search": "Search files:",
                    "lengthMenu": "Show _MENU_ files per page",
                    "info": "Showing _START_ to _END_ of _TOTAL_ files",
                    "infoEmpty": "No files found",
                    "infoFiltered": "(filtered from _MAX_ total files)",
                    "zeroRecords": "No matching files found"
                }
            });
            
            // Handle delete file modal data
            $('#deleteFileModal').on('show.bs.modal', function (event) {
                const button = $(event.relatedTarget);
                const filePath = button.data('filepath');
                const fileName = button.data('filename');
                
                $('#deleteFilePath').val(filePath);
                $('#deleteFileName').text(fileName);
            });
            
            // Handle clean directory modal data
            $('#cleanDirectoryModal').on('show.bs.modal', function (event) {
                const button = $(event.relatedTarget);
                const directory = button.data('directory');
                const dirTitle = button.data('dirtitle');
                const fileCount = button.data('filecount');
                
                $('#cleanDirectoryKey').val(directory);
                $('#cleanDirName').text(dirTitle);
                $('#cleanDirFileCount').text(fileCount);
                $('#confirmCleanText').val('');
                $('#cleanDirectoryBtn').prop('disabled', true);
            });
            
            // Handle confirmation text for clean directory
            $('#confirmCleanText').on('input', function() {
                $('#cleanDirectoryBtn').prop('disabled', $(this).val() !== "CONFIRM");
            });
        });
    </script>
</body>
</html>
