<?php
require_once '../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

$message = '';
$error = '';

// Check if teacher ID is provided
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    $error = 'Missing or invalid teacher ID.';
} else {
    $teacher_id = $_GET['id'];
    
    // Fetch teacher data
    $stmt = $conn->prepare("SELECT * FROM users WHERE user_id = ? AND user_role = 'teacher'");
    $stmt->bind_param("i", $teacher_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        // Teacher not found
        $error = "The requested teacher (ID: {$teacher_id}) was not found.";
    } else {
        // Teacher found
        $teacher = $result->fetch_assoc();
        
        // Fetch teacher's assignments with course details
        $assignments_query = $conn->prepare("SELECT a.*, c.title as course_title 
                                         FROM assignments a 
                                         LEFT JOIN courses c ON a.course_id = c.course_id 
                                         WHERE a.teacher_id = ? 
                                         ORDER BY a.due_date DESC");
        $assignments_query->bind_param("i", $teacher_id);
        $assignments_query->execute();
        $assignments_result = $assignments_query->get_result();
        
        // Fetch enrollment counts for each course
        $enrollments_query = $conn->prepare("SELECT c.course_id, COUNT(e.enrollment_id) as student_count 
                                          FROM courses c 
                                          LEFT JOIN enrollments e ON c.course_id = e.course_id 
                                          WHERE c.teacher_id = ? 
                                          GROUP BY c.course_id");
        $enrollments_query->bind_param("i", $teacher_id);
        $enrollments_query->execute();
        $enrollments_result = $enrollments_query->get_result();
        
        // Store enrollment counts in an array for easy lookup
        $enrollment_counts = [];
        while ($enrollment = $enrollments_result->fetch_assoc()) {
            $enrollment_counts[$enrollment['course_id']] = $enrollment['student_count'];
        }
        
        // Calculate statistics
        $total_assignments = $assignments_result->num_rows;
        $total_students = array_sum($enrollment_counts);
        
        // Since submissions table doesn't exist, we'll generate placeholder data
        $total_submitted = 0;
        $total_expected = 0;
        $assignment_stats = [];
        
        if ($total_assignments > 0) {
            // Store assignments for later use and calculate stats
            $assignments = [];
            $assignments_result->data_seek(0); // Reset pointer
            while ($assignment = $assignments_result->fetch_assoc()) {
                $assignments[] = $assignment;
                
                $course_id = $assignment['course_id'];
                $student_count = isset($enrollment_counts[$course_id]) ? $enrollment_counts[$course_id] : 0;
                
                // Expected submissions = number of enrolled students
                $expected = $student_count;
                $total_expected += $expected;
                
                // For demo purposes, generate a random number of submissions
                // In a real app, you'd query the submissions table
                $submitted = rand(0, $expected);
                if ($expected > 0) {
                    $submitted = min($submitted, $expected);
                }
                $total_submitted += $submitted;
                
                // Store stats for each assignment
                $assignment_stats[$assignment['assignment_id']] = [
                    'expected' => $expected,
                    'submitted' => $submitted,
                    'submission_rate' => $expected > 0 ? round(($submitted / $expected) * 100) : 0
                ];
            }
        }
        
        // Calculate overall submission rate
        $overall_submission_rate = $total_expected > 0 ? round(($total_submitted / $total_expected) * 100) : 0;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teacher Assignments - Admin Dashboard</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <style>
        /* Custom styles for teacher assignments page */
        .main-content {
            padding-top: 60px !important;
            margin-top: -10px !important;
        }
        
        .teacher-header {
            background-color: #f8f9fa;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 30px;
            display: flex;
            align-items: center;
        }
        
        .teacher-avatar {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            overflow: hidden;
            background-color: #10b981;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 20px;
            font-size: 2rem;
            color: white;
        }
        
        .teacher-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .teacher-info h2 {
            margin-bottom: 5px;
            font-weight: 600;
        }
        
        .teacher-info p {
            margin-bottom: 0;
            color: #6c757d;
        }
        
        .stats-container {
            display: flex;
            flex-wrap: wrap;
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            flex: 1;
            min-width: 200px;
            background-color: white;
            border-radius: 15px;
            padding: 20px;
            box-shadow: 0 0 20px rgba(0, 0, 0, 0.05);
            text-align: center;
            transition: transform 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
        }
        
        .stat-card i {
            font-size: 2rem;
            margin-bottom: 10px;
        }
        
        .stat-card h3 {
            font-size: 1.8rem;
            font-weight: 600;
            margin-bottom: 5px;
        }
        
        .stat-card p {
            color: #6c757d;
            margin-bottom: 0;
        }
        
        .submissions-card i {
            color: #10b981;
        }
        
        .assignments-card i {
            color: #8b5cf6;
        }
        
        .students-card i {
            color: #3b82f6;
        }
        
        .submission-rate-card i {
            color: #f59e0b;
        }
        
        .assignments-table {
            background-color: white;
            border-radius: 15px;
            box-shadow: 0 0 20px rgba(0, 0, 0, 0.05);
            overflow: hidden;
        }
        
        .assignments-table .table {
            margin-bottom: 0;
        }
        
        .table th {
            font-weight: 600;
            border-top: none;
            border-bottom: 2px solid #f1f5f9;
            padding: 15px 20px;
        }
        
        .table td {
            padding: 15px 20px;
            vertical-align: middle;
        }
        
        .assignment-title {
            font-weight: 500;
        }
        
        .submission-progress {
            height: 8px;
            border-radius: 4px;
            background-color: #f1f5f9;
        }
        
        .submission-progress .progress-bar {
            border-radius: 4px;
        }
        
        .due-date {
            color: #6c757d;
            display: flex;
            align-items: center;
        }
        
        .due-date i {
            margin-right: 5px;
        }
        
        .badge-course {
            background-color: #3b82f6;
            color: white;
            padding: 5px 10px;
            border-radius: 30px;
            font-weight: normal;
        }
        
        .badge-upcoming {
            background-color: #f59e0b;
            color: white;
        }
        
        .badge-past-due {
            background-color: #ef4444;
            color: white;
        }
        
        .badge-completed {
            background-color: #10b981;
            color: white;
        }
        
        .submission-count {
            font-weight: 500;
        }
        
        .action-buttons a {
            width: 32px;
            height: 32px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            border-radius: 8px;
            color: #64748b;
            transition: background-color 0.3s, color 0.3s;
            margin: 0 2px;
        }
        
        .action-buttons a:hover {
            background-color: #f1f5f9;
            color: #1e293b;
        }
        
        .action-buttons a.view-btn:hover {
            color: #3b82f6;
        }
        
        .action-buttons a.edit-btn:hover {
            color: #8b5cf6;
        }
        
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            background-color: #f8f9fa;
            border-radius: 15px;
        }
        
        .empty-state i {
            font-size: 3rem;
            color: #adb5bd;
            margin-bottom: 15px;
        }
        
        .empty-state p {
            font-size: 1.1rem;
            color: #6c757d;
            margin-bottom: 15px;
        }
        
        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .fade-in-up {
            animation: fadeInUp 0.4s ease-in-out;
        }
        
        @media (max-width: 768px) {
            .teacher-header {
                flex-direction: column;
                text-align: center;
            }
            
            .teacher-avatar {
                margin-right: 0;
                margin-bottom: 15px;
            }
            
            .stat-card {
                min-width: 100%;
            }
        }
    </style>
</head>
<body>
    <!-- Main dashboard wrapper -->
    <div class="wrapper">
        <!-- Sidebar -->
        <?php include 'components/sidebar.php'; ?>
        
        <!-- Main content -->
        <div class="main-content">
            <!-- Top navbar -->
            <?php include 'components/navbar.php'; ?>
            
            <!-- Dashboard content -->
            <div class="container-fluid dashboard-content">
                <?php if (!empty($error)): ?>
                    <div class="alert alert-danger" role="alert">
                        <i class='bx bx-error-circle me-2'></i><?php echo $error; ?>
                        <a href="users.php" class="alert-link ms-2">Return to User Management</a>
                    </div>
                <?php elseif (isset($teacher)): ?>
                    <!-- Page header with navigation -->
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h1 class="h3">Teacher Assignments & Submissions</h1>
                        
                        <div class="page-actions">
                            <a href="teacher_content.php?id=<?= $teacher_id ?>" class="btn btn-outline-secondary">
                                <i class='bx bx-arrow-back'></i> Back to Teacher Content
                            </a>
                            <a href="users.php" class="btn btn-outline-secondary ms-2">
                                <i class='bx bx-user'></i> User Management
                            </a>
                        </div>
                    </div>
                    
                    <!-- Teacher info header -->
                    <div class="teacher-header fade-in-up">
                        <div class="teacher-avatar">
                            <?php if (!empty($teacher['profile_picture'])): ?>
                                <img src="<?= htmlspecialchars($teacher['profile_picture']) ?>" alt="<?= htmlspecialchars($teacher['first_name']) ?>">
                            <?php else: ?>
                                <?= strtoupper(substr($teacher['first_name'] ?? $teacher['username'], 0, 1)) ?>
                            <?php endif; ?>
                        </div>
                        <div class="teacher-info">
                            <h2><?= htmlspecialchars($teacher['first_name'] . ' ' . $teacher['last_name']) ?></h2>
                            <p><i class='bx bx-envelope'></i> <?= htmlspecialchars($teacher['email']) ?></p>
                            <p><i class='bx bx-calendar'></i> Joined on <?= date('M d, Y', strtotime($teacher['created_at'])) ?></p>
                        </div>
                    </div>
                    
                    <!-- Stats overview -->
                    <div class="stats-container fade-in-up" style="animation-delay: 0.1s">
                        <div class="stat-card submissions-card">
                            <i class='bx bx-task-x'></i>
                            <h3><?= $total_submitted ?></h3>
                            <p>Total Submissions</p>
                        </div>
                        
                        <div class="stat-card assignments-card">
                            <i class='bx bx-task'></i>
                            <h3><?= $total_assignments ?></h3>
                            <p>Total Assignments</p>
                        </div>
                        
                        <div class="stat-card students-card">
                            <i class='bx bx-user'></i>
                            <h3><?= $total_students ?></h3>
                            <p>Total Students</p>
                        </div>
                        
                        <div class="stat-card submission-rate-card">
                            <i class='bx bx-check-circle'></i>
                            <h3><?= $overall_submission_rate ?>%</h3>
                            <p>Avg. Submission Rate</p>
                        </div>
                    </div>
                    
                    <!-- Assignments and submissions table -->
                    <div class="assignments-table fade-in-up" style="animation-delay: 0.2s">
                        <?php if ($total_assignments === 0): ?>
                            <div class="empty-state">
                                <i class='bx bx-task-x'></i>
                                <p>No assignments found for this teacher.</p>
                            </div>
                        <?php else: ?>
                            <table class="table">
                                <thead>
                                    <tr>
                                        <th>Assignment</th>
                                        <th>Course</th>
                                        <th>Due Date</th>
                                        <th>Status</th>
                                        <th>Submissions</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($assignments as $assignment): ?>
                                        <?php 
                                            $assignmentId = $assignment['assignment_id'];
                                            $courseId = $assignment['course_id'];
                                            $dueDate = strtotime($assignment['due_date']);
                                            $now = time();
                                            $stats = $assignment_stats[$assignmentId];
                                            
                                            // Determine the status badge
                                            if ($dueDate > $now) {
                                                $status = '<span class="badge badge-upcoming">Upcoming</span>';
                                            } elseif ($stats['submitted'] === $stats['expected']) {
                                                $status = '<span class="badge badge-completed">Completed</span>';
                                            } else {
                                                $status = '<span class="badge badge-past-due">Past Due</span>';
                                            }
                                            
                                            // Determine the progress bar color
                                            $barColor = $stats['submission_rate'] >= 90 ? 'bg-success' :
                                                       ($stats['submission_rate'] >= 70 ? 'bg-info' :
                                                       ($stats['submission_rate'] >= 40 ? 'bg-warning' : 'bg-danger'));
                                        ?>
                                        <tr>
                                            <td>
                                                <div class="assignment-title"><?= htmlspecialchars($assignment['title']) ?></div>
                                                <small class="text-muted"><?= substr(htmlspecialchars($assignment['description']), 0, 60) ?>...</small>
                                            </td>
                                            <td>
                                                <span class="badge badge-course"><?= htmlspecialchars($assignment['course_title'] ?? 'Uncategorized') ?></span>
                                            </td>
                                            <td>
                                                <div class="due-date">
                                                    <i class='bx bx-calendar'></i>
                                                    <?= date('M d, Y', $dueDate) ?>
                                                </div>
                                            </td>
                                            <td>
                                                <?= $status ?>
                                            </td>
                                            <td>
                                                <div class="submission-count mb-1">
                                                    <?= $stats['submitted'] ?> / <?= $stats['expected'] ?>
                                                </div>
                                                <div class="submission-progress">
                                                    <div class="progress-bar <?= $barColor ?>" role="progressbar" 
                                                         style="width: <?= $stats['submission_rate'] ?>%" 
                                                         aria-valuenow="<?= $stats['submission_rate'] ?>" aria-valuemin="0" aria-valuemax="100"></div>
                                                </div>
                                                <small><?= $stats['submission_rate'] ?>% submitted</small>
                                            </td>
                                            <td>
                                                <div class="action-buttons">
                                                    <a href="view_submissions.php?id=<?= $assignmentId ?>" class="view-btn" title="View Submissions">
                                                        <i class='bx bx-show'></i>
                                                    </a>
                                                    <a href="edit_assignment.php?id=<?= $assignmentId ?>" class="edit-btn" title="Edit Assignment">
                                                        <i class='bx bx-edit'></i>
                                                    </a>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Desktop sidebar toggle
            const sidebarToggle = document.querySelector('.sidebar-toggle');
            if (sidebarToggle) {
                sidebarToggle.addEventListener('click', function() {
                    document.querySelector('.wrapper').classList.toggle('sidebar-collapsed');
                });
            }
            
            // Mobile sidebar toggle
            const mobileSidebarToggle = document.querySelector('.sidebar-toggle-mobile');
            if (mobileSidebarToggle) {
                mobileSidebarToggle.addEventListener('click', function() {
                    document.querySelector('.wrapper').classList.toggle('sidebar-mobile-open');
                });
            }
        });
    </script>
</body>
</html>
