<?php
require_once '../auth/functions.php';

// Function to calculate time ago
function timeAgo($timestamp) {
    $time = strtotime($timestamp);
    $curr_time = time();
    $time_diff = $curr_time - $time;
    
    if ($time_diff < 60) {
        return 'Just now';
    } elseif ($time_diff < 3600) {
        return floor($time_diff / 60) . ' min ago';
    } elseif ($time_diff < 86400) {
        return floor($time_diff / 3600) . ' hr ago';
    } elseif ($time_diff < 604800) {
        return floor($time_diff / 86400) . ' day ago';
    } else {
        return date('M j, Y', $time);
    }
}

// Ensure user is logged in as an admin
requireRole('admin');

$message = '';
$error = '';

// Check if teacher ID is provided
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    $error = 'Missing or invalid teacher ID.';
} else {
    $teacher_id = $_GET['id'];
    
    // Fetch teacher data
    $stmt = $conn->prepare("SELECT * FROM users WHERE user_id = ? AND user_role = 'teacher'");
    $stmt->bind_param("i", $teacher_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        // Teacher not found
        $error = "The requested teacher (ID: {$teacher_id}) was not found.";
    } else {
        // Teacher found
        $teacher = $result->fetch_assoc();
        
        // Fetch teacher's courses with comprehensive data for accordion display
        $courses = [];
        $total_videos = 0;
        $total_assignments = 0;
        
        // Get all courses for this teacher
        $courses_query = $conn->prepare("
            SELECT c.*, 
                   (SELECT COUNT(*) FROM videos v WHERE v.course_id = c.course_id AND v.is_published = 1) as video_count,
                   (SELECT COUNT(*) FROM enrollments e WHERE e.course_id = c.course_id) as student_count,
                   (SELECT COUNT(*) FROM assignments a WHERE a.course_id = c.course_id) as assignment_count,
                   (SELECT COUNT(*) FROM quizzes q WHERE q.course_id = c.course_id AND q.is_published = 1) as quiz_count
            FROM courses c 
            WHERE c.teacher_id = ? 
            ORDER BY c.created_at DESC
        ");
        $courses_query->bind_param("i", $teacher_id);
        $courses_query->execute();
        $courses_result = $courses_query->get_result();
        
        while ($course = $courses_result->fetch_assoc()) {
            // Get videos for this course (similar to student course-detail.php)
            $video_query = $conn->prepare("
                SELECT v.*, 
                       v.video_id as lesson_id,
                       0 as watch_progress,
                       0 as completed,
                       0 as last_position
                FROM videos v
                WHERE v.course_id = ? AND v.is_published = 1
                ORDER BY v.video_order ASC, v.created_at ASC
            ");
            $video_query->bind_param("i", $course['course_id']);
            $video_query->execute();
            $video_result = $video_query->get_result();
            
            $course_videos = [];
            while ($video = $video_result->fetch_assoc()) {
                // Fix thumbnail path for admin access
                if (!empty($video['thumbnail'])) {
                    if (strpos($video['thumbnail'], 'http') === 0) {
                        $video['thumbnail'] = $video['thumbnail'];
                    } else {
                        $video['thumbnail'] = '../' . ltrim($video['thumbnail'], './');
                    }
                } else {
                    $video['thumbnail'] = '../assets/images/default-video-thumbnail.jpg';
                }
                
                // Get quizzes for this video
                $video['quizzes'] = [];
                $quiz_query = $conn->prepare("SELECT quiz_id, title, time_limit, passing_score FROM quizzes WHERE video_id = ? AND is_published = 1");
                if ($quiz_query) {
                    $quiz_query->bind_param("i", $video['video_id']);
                    $quiz_query->execute();
                    $quiz_result = $quiz_query->get_result();
                    while ($quiz = $quiz_result->fetch_assoc()) {
                        $video['quizzes'][] = $quiz;
                    }
                    $quiz_query->close();
                }
                
                // Get assignments for this video
                $video['assignments'] = [];
                $assignment_query = $conn->prepare("SELECT assignment_id, title, due_date FROM assignments WHERE video_id = ?");
                if ($assignment_query) {
                    $assignment_query->bind_param("i", $video['video_id']);
                    $assignment_query->execute();
                    $assignment_result = $assignment_query->get_result();
                    while ($assignment = $assignment_result->fetch_assoc()) {
                        $video['assignments'][] = $assignment;
                    }
                    $assignment_query->close();
                }
                
                // Get materials for this video
                $video['materials'] = [];
                $materials_query = $conn->prepare("SELECT material_id, title, file_path, file_type FROM course_materials WHERE video_id = ?");
                if ($materials_query) {
                    $materials_query->bind_param("i", $video['video_id']);
                    $materials_query->execute();
                    $materials_result = $materials_query->get_result();
                    while ($material = $materials_result->fetch_assoc()) {
                        $video['materials'][] = $material;
                    }
                    $materials_query->close();
                }
                
                $course_videos[] = $video;
            }
            $video_query->close();
            
            $course['videos'] = $course_videos;
            $courses[] = $course;
            
            $total_videos += count($course_videos);
            $total_assignments += $course['assignment_count'];
        }
        $courses_query->close();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teacher Content - Admin Dashboard</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <style>
        /* Custom styles for teacher content page */
        .main-content {
            padding-top: 60px !important;
            margin-top: -10px !important;
        }
        
        .content-section {
            margin-bottom: 2.5rem;
            animation: fadeInUp 0.4s ease-in-out;
        }
        
        .content-card {
            border: none;
            box-shadow: 0 0 20px rgba(0, 0, 0, 0.05);
            border-radius: 15px;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            height: 100%;
            overflow: hidden;
        }
        
        .content-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
        }
        
        .content-card .card-img-top {
            height: 160px;
            object-fit: cover;
        }
        
        .content-section-header {
            display: flex;
            align-items: center;
            margin-bottom: 1.5rem;
        }
        
        .section-icon {
            width: 40px;
            height: 40px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 15px;
            font-size: 1.4rem;
            color: white;
        }
        
        .teacher-header {
            background-color: #f8f9fa;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 30px;
            display: flex;
            align-items: center;
        }
        
        .teacher-avatar {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            overflow: hidden;
            background-color: #10b981;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 20px;
            font-size: 2rem;
            color: white;
        }
        
        .teacher-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .teacher-info h2 {
            margin-bottom: 5px;
            font-weight: 600;
        }
        
        .teacher-info p {
            margin-bottom: 0;
            color: #6c757d;
        }
        
        .teacher-stats {
            display: flex;
            margin-top: 10px;
        }
        
        .stat-item {
            margin-right: 20px;
            text-align: center;
            background-color: #fff;
            border-radius: 10px;
            padding: 10px 15px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }
        
        .stat-item i {
            color: #10b981;
            font-size: 1.5rem;
            margin-bottom: 5px;
        }
        
        .stat-item span {
            display: block;
            font-weight: 500;
        }
        
        .stat-item small {
            color: #6c757d;
        }
        
        .video-card .card-img-overlay {
            background: linear-gradient(to top, rgba(0,0,0,0.8) 0%, rgba(0,0,0,0) 50%);
            display: flex;
            flex-direction: column;
            justify-content: flex-end;
        }
        
        .video-card .play-icon {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            width: 50px;
            height: 50px;
            background-color: rgba(255,255,255,0.2);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 24px;
        }
        
        .video-card .play-icon i {
            margin-left: 3px;
        }
        
        .video-card .card-img-overlay h5 {
            color: white;
            margin-bottom: 0;
        }
        
        .video-card .card-img-overlay p {
            color: rgba(255,255,255,0.8);
            margin-bottom: 0;
            font-size: 0.9rem;
        }
        
        .course-card .card-body, .assignment-card .card-body {
            padding: 20px;
        }
        
        .course-card .course-meta, .assignment-card .assignment-meta {
            margin-top: 15px;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }
        
        .course-card .meta-item, .assignment-card .meta-item {
            display: flex;
            align-items: center;
            color: #6c757d;
            font-size: 0.9rem;
        }
        
        .course-card .meta-item i, .assignment-card .meta-item i {
            margin-right: 5px;
            font-size: 1rem;
        }
        
        .badge-published {
            background-color: #10b981;
            color: white;
        }
        
        .badge-draft {
            background-color: #6c757d;
            color: white;
        }
        
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            background-color: #f8f9fa;
            border-radius: 15px;
        }
        
        .empty-state i {
            font-size: 3rem;
            color: #adb5bd;
            margin-bottom: 15px;
        }
        
        .empty-state p {
            font-size: 1.1rem;
            color: #6c757d;
            margin-bottom: 15px;
            }
            
            .content-card {
                border: none;
                box-shadow: 0 0 20px rgba(0, 0, 0, 0.05);
                border-radius: 15px;
                transition: transform 0.3s ease, box-shadow 0.3s ease;
                height: 100%;
                overflow: hidden;
            }
            
            .content-card:hover {
                transform: translateY(-5px);
                box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            }
            
            .content-card .card-img-top {
                height: 160px;
                object-fit: cover;
            }
            
            .content-section-header {
                display: flex;
                align-items: center;
                margin-bottom: 1.5rem;
            }
            
            .section-icon {
                width: 40px;
                height: 40px;
                border-radius: 10px;
                display: flex;
                align-items: center;
                justify-content: center;
                margin-right: 15px;
                font-size: 1.4rem;
                color: white;
            }
            
            .teacher-header {
                background-color: #f8f9fa;
                border-radius: 15px;
                padding: 20px;
                margin-bottom: 30px;
                display: flex;
                align-items: center;
            }
            
            .teacher-avatar {
                width: 80px;
                height: 80px;
                border-radius: 50%;
                overflow: hidden;
                background-color: #10b981;
                display: flex;
                align-items: center;
                justify-content: center;
                margin-right: 20px;
                font-size: 2rem;
                color: white;
            }
            
            .teacher-avatar img {
                width: 100%;
                height: 100%;
                object-fit: cover;
            }
            
            .teacher-info h2 {
                margin-bottom: 5px;
                font-weight: 600;
            }
            
            .teacher-info p {
                margin-bottom: 0;
                color: #6c757d;
            }
            
            .teacher-stats {
                display: flex;
                gap: 15px;
                margin-top: 15px;
                justify-content: flex-start;
            }
            
            .stat-item {
                width: 140px;
                text-align: center;
                background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
                border-radius: 15px;
                padding: 18px 12px;
                box-shadow: 0 4px 15px rgba(0,0,0,0.08);
                border: 1px solid #f59e0b20;
                transition: all 0.3s ease;
            }
            
            .stat-item:hover {
                transform: translateY(-3px);
                box-shadow: 0 8px 25px rgba(0,0,0,0.12);
            }
            
            .stat-item i {
                color: #d97706;
                font-size: 2rem;
                margin-bottom: 10px;
                display: block;
            }
            
            .stat-item span {
                display: block;
                font-weight: 600;
                font-size: 1.8rem;
                color: #92400e;
                margin-bottom: 5px;
            }
            
            .stat-item small {
                color: #b45309;
                font-weight: 500;
                font-size: 0.9rem;
                text-transform: uppercase;
                letter-spacing: 0.5px;
            }
            
            .video-card .card-img-overlay {
                background: linear-gradient(to top, rgba(0,0,0,0.8) 0%, rgba(0,0,0,0) 50%);
                display: flex;
                flex-direction: column;
                justify-content: flex-end;
            }
            
            .video-card .play-icon {
                position: absolute;
                top: 50%;
                left: 50%;
                transform: translate(-50%, -50%);
                width: 50px;
                height: 50px;
                background-color: rgba(255,255,255,0.2);
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                color: white;
                font-size: 24px;
            }
            
            .video-card .play-icon i {
                margin-left: 3px;
            }
            
            .video-card .card-img-overlay h5 {
                color: white;
                margin-bottom: 0;
            }
            
            .video-card .card-img-overlay p {
                color: rgba(255,255,255,0.8);
                margin-bottom: 0;
                font-size: 0.9rem;
            }
            
            .course-card .card-body, .assignment-card .card-body {
                padding: 20px;
            }
            
            .course-card .course-meta, .assignment-card .assignment-meta {
                margin-top: 15px;
                display: flex;
                align-items: center;
                justify-content: space-between;
            }
            
            .course-card .meta-item, .assignment-card .meta-item {
                display: flex;
                align-items: center;
                color: #6c757d;
                font-size: 0.9rem;
            }
            
            .course-card .meta-item i, .assignment-card .meta-item i {
                margin-right: 5px;
                font-size: 1rem;
            }
            
            .badge-published {
                background-color: #10b981;
                color: white;
            }
            
            .badge-draft {
                background-color: #6c757d;
                color: white;
            }
            
            .empty-state {
                text-align: center;
                padding: 40px 20px;
                background-color: #f8f9fa;
                border-radius: 15px;
            }
            
            .empty-state i {
                font-size: 3rem;
                color: #adb5bd;
                margin-bottom: 15px;
            }
            
            .empty-state p {
                font-size: 1.1rem;
                color: #6c757d;
                margin-bottom: 15px;
            }
            
            @keyframes fadeInUp {
                from {
                    opacity: 0;
                    transform: translateY(30px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }
            
            /* Accordion Styling (Based on Student Course Detail) */
            .course-content-sidebar {
                max-width: 100%;
            }
            
            .accordion-item {
                border: 1px solid #e9ecef;
                border-radius: 12px !important;
                margin-bottom: 15px;
                box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            }
            
            .accordion-button {
                background: white;
                border: none;
                padding: 20px;
                font-weight: 500;
                border-radius: 12px !important;
            }
            
            .accordion-button:not(.collapsed) {
                background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
                color: #92400e;
                box-shadow: none;
            }
            
            .accordion-button:focus {
                border-color: #f59e0b;
                box-shadow: 0 0 0 0.25rem rgba(245, 158, 11, 0.25);
            }
            
            .accordion-body {
                background: #f8f9fa;
                border-radius: 0 0 12px 12px;
            }
            
            .lesson-item {
                transition: all 0.3s ease;
            }
            
            .lesson-item:hover {
                box-shadow: 0 4px 15px rgba(0,0,0,0.1);
                transform: translateY(-2px);
            }
            
            .lesson-header {
                transition: all 0.3s ease;
            }
            
            .lesson-thumbnail-small img {
                transition: all 0.3s ease;
            }
            
            .lesson-item:hover .lesson-thumbnail-small img {
                transform: scale(1.05);
            }
            
            .badge-quiz-accordion {
                background: linear-gradient(135deg, #fbbf24 0%, #f59e0b 100%) !important;
                color: #92400e !important;
            }
            
            .content-item {
                padding: 8px;
                background: white;
                border-radius: 6px;
                border-left: 3px solid;
            }
            
            .content-item:nth-child(odd) {
                background: #f8f9fa;
            }
            
            .course-thumbnail-small img, .lesson-thumbnail-small img {
                border: 2px solid white;
                box-shadow: 0 2px 8px rgba(0,0,0,0.15);
            }
            
            .course-description {
                border-left: 4px solid #f59e0b;
            }
            
            .lesson-actions .btn {
                transition: all 0.3s ease;
            }
            
            .lesson-actions .btn:hover {
                transform: translateY(-2px);
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            }
            
            .accordion-title-text {
                font-size: 1.1rem;
                line-height: 1.3;
            }
            
            /* Responsive adjustments */
            @media (max-width: 768px) {
                .course-thumbnail-small, .lesson-thumbnail-small {
                    margin-bottom: 10px;
                }
                
                .accordion-title-text {
                    font-size: 1rem;
                    white-space: normal;
                    overflow: visible;
                    text-overflow: unset;
                    max-width: 100%;
                }
                
                .lesson-actions {
                    margin-top: 10px;
                }
            }
            
            @media (max-width: 768px) {
                .teacher-header {
                    flex-direction: column;
                    text-align: center;
                }
            
            .teacher-avatar {
                margin-right: 0;
                margin-bottom: 15px;
            }
            
            .teacher-stats {
                flex-wrap: wrap;
                justify-content: center;
            }
            
            .stat-item {
                margin: 5px;
            }
        }
    </style>
</head>
<body>
    <!-- Main dashboard wrapper -->
    <div class="wrapper">
        <!-- Sidebar -->
        <?php include 'components/sidebar.php'; ?>
        
        <!-- Main content -->
        <div class="main-content">
            <!-- Top navbar -->
            <?php include 'components/navbar.php'; ?>
            
            <!-- Dashboard content -->
            <div class="container-fluid dashboard-content">
                <?php if (!empty($error)): ?>
                    <div class="alert alert-danger" role="alert">
                        <i class='bx bx-error-circle me-2'></i><?php echo $error; ?>
                        <a href="users.php" class="alert-link ms-2">Return to User Management</a>
                    </div>
                <?php elseif (isset($teacher)): ?>
                    <!-- Teacher info header -->
                    <div class="teacher-header">
                        <div class="teacher-avatar">
                            <?php if (!empty($teacher['profile_picture'])): ?>
                                <img src="../<?= htmlspecialchars($teacher['profile_picture']) ?>" alt="<?= htmlspecialchars($teacher['first_name']) ?>">
                            <?php else: ?>
                                <?= strtoupper(substr($teacher['first_name'] ?? $teacher['username'], 0, 1)) ?>
                            <?php endif; ?>
                        </div>
                        <div class="teacher-info">
                            <h2><?= htmlspecialchars($teacher['first_name'] . ' ' . $teacher['last_name']) ?></h2>
                            <p><i class='bx bx-envelope'></i> <?= htmlspecialchars($teacher['email']) ?></p>
                            <p><i class='bx bx-calendar'></i> Joined on <?= date('M d, Y', strtotime($teacher['created_at'])) ?></p>
                            
                            <div class="teacher-stats">
                                <div class="stat-item">
                                    <i class='bx bx-video'></i>
                                    <span><?= $total_videos ?></span>
                                    <small>Videos</small>
                                </div>
                                <div class="stat-item">
                                    <i class='bx bx-book-open'></i>
                                    <span><?= count($courses) ?></span>
                                    <small>Courses</small>
                                </div>


                            </div>
                        </div>
                    </div>
                    
                    <!-- Course Content Accordion Section (Based on Student Course Detail) -->
                    <div class="course-wrapper">
                        <div class="container-fluid mt-4">
                            <div class="row">
                                <!-- Course Content Accordion Sidebar -->
                                <div class="col-12">
                                    <div class="content-section-header mb-4">
                                        <div class="section-icon" style="background-color: #3b82f6;">
                                            <i class='bx bx-book-open'></i>
                                        </div>
                                        <h3>Teacher Content Structure</h3>
                                        <p class="text-muted">Browse all courses, lessons, quizzes, and materials created by this teacher</p>
                                    </div>
                                    
                                    <div class="course-content-sidebar">
                                        <div class="accordion" id="teacherContentAccordion">
                                            
                                            <?php if (empty($courses)): ?>
                                                <!-- No Courses Available -->
                                                <div class="accordion-item">
                                                    <div class="empty-state p-4">
                                                        <div class="text-center text-muted">
                                                            <i class="bx bx-book-off fs-1"></i>
                                                            <p class="mt-2 mb-0">No courses available</p>
                                                        </div>
                                                    </div>
                                                </div>
                                            <?php else: ?>
                                                <!-- Individual Course Accordions -->
                                                <?php foreach ($courses as $course_index => $course): ?>
                                                    <div class="accordion-item course-accordion">
                                                        <h2 class="accordion-header" id="headingCourse<?= $course['course_id'] ?>">
                                                            <button class="accordion-button <?= $course_index === 0 ? '' : 'collapsed' ?>" type="button" data-bs-toggle="collapse" data-bs-target="#collapseCourse<?= $course['course_id'] ?>" aria-expanded="<?= $course_index === 0 ? 'true' : 'false' ?>" aria-controls="collapseCourse<?= $course['course_id'] ?>">
                                                                <div class="d-flex align-items-center w-100">
                                                                    <div class="course-thumbnail-small me-3">
                                                                        <?php if (!empty($course['thumbnail'])): ?>
                                                                            <img src="../<?= htmlspecialchars($course['thumbnail']) ?>" alt="<?= htmlspecialchars($course['title']) ?>" class="rounded" style="width: 50px; height: 40px; object-fit: cover;">
                                                                        <?php else: ?>
                                                                            <div class="thumbnail-placeholder-small d-flex align-items-center justify-content-center" style="width: 50px; height: 40px; background: #f8f9fa; border-radius: 4px;">
                                                                                <i class="bx bx-book text-muted"></i>
                                                                            </div>
                                                                        <?php endif; ?>
                                                                    </div>
                                                                    <div class="flex-grow-1" style="min-width: 0;">
                                                                        <div class="fw-semibold accordion-title-text" style="white-space: nowrap; overflow: hidden; text-overflow: ellipsis; max-width: calc(100% - 150px);"><?= htmlspecialchars($course['title']) ?></div>
                                                                        <div class="d-flex align-items-center justify-content-between">
                                                                            <small class="text-muted">
                                                                                <i class="bx bx-video"></i> <?= count($course['videos']) ?> lessons
                                                                                <span class="mx-2">•</span>
                                                                                <i class="bx bx-users"></i> <?= $course['student_count'] ?> students
                                                                            </small>
                                                                        </div>
                                                                        <div class="d-flex align-items-center flex-wrap mt-1">
                                                                            <?php if ($course['quiz_count'] > 0): ?>
                                                                                <span class="badge badge-quiz-accordion me-1" style="font-size: 0.6rem;" title="Quizzes Available">
                                                                                    <i class="bx bx-help-circle"></i> <?= $course['quiz_count'] ?> Quizzes
                                                                                </span>
                                                                            <?php endif; ?>
                                                                            <?php if ($course['assignment_count'] > 0): ?>
                                                                                <span class="badge bg-warning me-1" style="font-size: 0.6rem;" title="Assignments Available">
                                                                                    <i class="bx bx-task"></i> <?= $course['assignment_count'] ?> Tasks
                                                                                </span>
                                                                            <?php endif; ?>
                                                                            <span class="badge <?= $course['is_published'] ? 'bg-success' : 'bg-secondary' ?> me-1" style="font-size: 0.6rem;">
                                                                                <?= $course['is_published'] ? 'Published' : 'Draft' ?>
                                                                            </span>
                                                                        </div>
                                                                    </div>
                                                                </div>
                                                            </button>
                                                        </h2>
                                                        <div id="collapseCourse<?= $course['course_id'] ?>" class="accordion-collapse collapse <?= $course_index === 0 ? 'show' : '' ?>" aria-labelledby="headingCourse<?= $course['course_id'] ?>" data-bs-parent="#teacherContentAccordion">
                                                            <div class="accordion-body">
                                                                <!-- Course Description -->
                                                                <?php if (!empty($course['description'])): ?>
                                                                    <div class="course-description mb-3 p-3" style="background: #f8f9fa; border-radius: 8px;">
                                                                        <small class="text-muted"><?= htmlspecialchars(substr($course['description'], 0, 200)) . (strlen($course['description']) > 200 ? '...' : '') ?></small>
                                                                    </div>
                                                                <?php endif; ?>
                                                                

                                                                
                                                                <!-- Lessons Accordion -->
                                                                <?php if (empty($course['videos'])): ?>
                                                                    <div class="text-center text-muted py-3">
                                                                        <i class="bx bx-video-off fs-4"></i>
                                                                        <p class="mt-2 mb-0">No lessons in this course yet</p>
                                                                    </div>
                                                                <?php else: ?>
                                                                    <div class="lessons-accordion">
                                                                        <h6 class="text-muted mb-3">
                                                                            <i class="bx bx-video me-1"></i> Course Lessons (<?= count($course['videos']) ?>)
                                                                        </h6>
                                                                        
                                                                        <?php foreach ($course['videos'] as $lesson_index => $lesson): ?>
                                                                            <div class="lesson-item mb-3" style="border: 1px solid #e9ecef; border-radius: 8px; overflow: hidden;">
                                                                                <div class="lesson-header p-3" style="background: #f8f9fa;">
                                                                                    <div class="d-flex align-items-center">
                                                                                        <div class="lesson-thumbnail-small me-3">
                                                                                            <img src="<?= htmlspecialchars($lesson['thumbnail']) ?>" alt="Lesson <?= $lesson_index + 1 ?>" class="rounded" style="width: 60px; height: 40px; object-fit: cover;" onerror="this.onerror=null; this.src='../assets/images/default-video-thumbnail.jpg';">
                                                                                        </div>
                                                                                        <div class="flex-grow-1">
                                                                                            <h6 class="mb-1"><?= htmlspecialchars($lesson['title']) ?></h6>
                                                                                            <div class="d-flex align-items-center gap-2">
                                                                                                <small class="text-muted">
                                                                                                    <i class="bx bx-time"></i> <?= !empty($lesson['duration']) ? htmlspecialchars($lesson['duration']) : 'No duration set' ?>
                                                                                                </small>
                                                                                                <span class="badge <?= $lesson['is_published'] ? 'bg-success' : 'bg-secondary' ?>" style="font-size: 0.7rem;">
                                                                                                    <?= $lesson['is_published'] ? 'Published' : 'Draft' ?>
                                                                                                </span>
                                                                                            </div>
                                                                                        </div>
                                                                                        <div class="lesson-actions">
                                                                                            <?php if (!empty($lesson['quizzes'])): ?>
                                                                                                <a href="view_quiz_content.php?lesson_id=<?= $lesson['video_id'] ?>" class="btn btn-sm btn-outline-info">
                                                                                                    <i class="bx bx-help-circle"></i> View Quiz
                                                                                                </a>
                                                                                            <?php else: ?>
                                                                                                <span class="text-muted small">
                                                                                                    <i class="bx bx-info-circle"></i> No quiz available
                                                                                                </span>
                                                                                            <?php endif; ?>
                                                                                        </div>
                                                                                    </div>
                                                                                </div>
                                                                                
                                                                                <!-- Lesson Content (Quizzes, Assignments, Materials) -->
                                                                                <?php if (!empty($lesson['quizzes']) || !empty($lesson['assignments']) || !empty($lesson['materials'])): ?>
                                                                                    <div class="lesson-content p-3">
                                                                                        <div class="row">
                                                                                            <!-- Quizzes -->
                                                                                            <?php if (!empty($lesson['quizzes'])): ?>
                                                                                                <div class="col-md-4 mb-2">
                                                                                                    <div class="content-section">
                                                                                                        <h6 class="text-primary mb-2">
                                                                                                            <i class="bx bx-help-circle me-1"></i> 
                                                                                                            Quizzes (<?= count($lesson['quizzes']) ?>)
                                                                                                        </h6>
                                                                                                        <?php foreach ($lesson['quizzes'] as $quiz): ?>
                                                                                                            <div class="content-item small mb-1">
                                                                                                                <i class="bx bx-help-circle text-primary me-1"></i>
                                                                                                                <?= htmlspecialchars($quiz['title']) ?>
                                                                                                                <small class="text-muted">(<?= $quiz['time_limit'] ?>min)</small>
                                                                                                            </div>
                                                                                                        <?php endforeach; ?>
                                                                                                    </div>
                                                                                                </div>
                                                                                            <?php endif; ?>
                                                                                            
                                                                                            <!-- Assignments -->
                                                                                            <?php if (!empty($lesson['assignments'])): ?>
                                                                                                <div class="col-md-4 mb-2">
                                                                                                    <div class="content-section">
                                                                                                        <h6 class="text-warning mb-2">
                                                                                                            <i class="bx bx-task me-1"></i> 
                                                                                                            Assignments (<?= count($lesson['assignments']) ?>)
                                                                                                        </h6>
                                                                                                        <?php foreach ($lesson['assignments'] as $assignment): ?>
                                                                                                            <div class="content-item small mb-1">
                                                                                                                <i class="bx bx-task text-warning me-1"></i>
                                                                                                                <?= htmlspecialchars($assignment['title']) ?>
                                                                                                                <?php if (!empty($assignment['due_date'])): ?>
                                                                                                                    <small class="text-muted">(Due: <?= date('M d', strtotime($assignment['due_date'])) ?>)</small>
                                                                                                                <?php endif; ?>
                                                                                                            </div>
                                                                                                        <?php endforeach; ?>
                                                                                                    </div>
                                                                                                </div>
                                                                                            <?php endif; ?>
                                                                                            
                                                                                            <!-- Materials -->
                                                                                            <?php if (!empty($lesson['materials'])): ?>
                                                                                                <div class="col-md-4 mb-2">
                                                                                                    <div class="content-section">
                                                                                                        <h6 class="text-success mb-2">
                                                                                                            <i class="bx bx-folder me-1"></i> 
                                                                                                            Materials (<?= count($lesson['materials']) ?>)
                                                                                                        </h6>
                                                                                                        <?php foreach ($lesson['materials'] as $material): ?>
                                                                                                            <div class="content-item small mb-1">
                                                                                                                <i class="bx bx-file text-success me-1"></i>
                                                                                                                <?= htmlspecialchars($material['title']) ?>
                                                                                                                <small class="text-muted">(<?= strtoupper($material['file_type'] ?? 'FILE') ?>)</small>
                                                                                                            </div>
                                                                                                        <?php endforeach; ?>
                                                                                                    </div>
                                                                                                </div>
                                                                                            <?php endif; ?>
                                                                                        </div>
                                                                                    </div>
                                                                                <?php endif; ?>
                                                                            </div>
                                                                        <?php endforeach; ?>
                                                                    </div>
                                                                <?php endif; ?>
                                                            </div>
                                                        </div>
                                                    </div>
                                                <?php endforeach; ?>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    

                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Desktop sidebar toggle
            const sidebarToggle = document.querySelector('.sidebar-toggle');
            if (sidebarToggle) {
                sidebarToggle.addEventListener('click', function() {
                    document.querySelector('.wrapper').classList.toggle('sidebar-collapsed');
                });
            }
            
            // Mobile sidebar toggle
            const mobileSidebarToggle = document.querySelector('.sidebar-toggle-mobile');
            if (mobileSidebarToggle) {
                mobileSidebarToggle.addEventListener('click', function() {
                    document.querySelector('.wrapper').classList.toggle('sidebar-mobile-open');
                });
            }
        });
    </script>
</body>
</html>
