<?php
// Test page for Revenue Sharing System
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/components/admin_layout.php';
require_once __DIR__ . '/../includes/db_connection.php';
require_once __DIR__ . '/../includes/purchase_processor.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_id'])) {
    header("Location: ../auth/admin_login.php");
    exit();
}

$message = '';
$test_results = [];

// Handle test purchase simulation
if (isset($_POST['test_purchase'])) {
    $user_id = (int)$_POST['user_id'];
    $course_id = (int)$_POST['course_id'];
    $amount = (float)$_POST['amount'];
    
    $processor = new PurchaseProcessor($conn);
    $result = $processor->processPurchase(
        $user_id, 
        $course_id, 
        $amount, 
        'test_payment', 
        'TEST_' . time()
    );
    
    if ($result['success']) {
        $message = "✅ Test purchase processed successfully!";
        $test_results = $result;
    } else {
        $message = "❌ Error: " . $result['error'];
    }
}

// Get sample data for testing
$users = [];
$courses = [];

try {
    // Get sample users
    $stmt = $conn->prepare("SELECT user_id, first_name, last_name, email FROM users WHERE role = 'student' LIMIT 10");
    $stmt->execute();
    $users = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    
    // Get sample courses
    $stmt = $conn->prepare("
        SELECT c.course_id, c.title, c.price, 
               CONCAT(u.first_name, ' ', u.last_name) as teacher_name
        FROM courses c
        LEFT JOIN users u ON c.teacher_id = u.user_id 
        WHERE c.is_published = 1
        LIMIT 10
    ");
    $stmt->execute();
    $courses = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    
} catch (Exception $e) {
    $message = "Error loading test data: " . $e->getMessage();
}

// Get current revenue settings
$processor = new PurchaseProcessor($conn);
$revenue_stats = $processor->getRevenueStats();

// Set current page for sidebar
$current_page = 'test-revenue-system.php';
?>

<style>
    /* Test Page Styles */
    * {
        font-family: 'Inter', 'Poppins', sans-serif !important;
    }

    :root {
        --primary-color: #4361ee;
        --success-color: #22c55e;
        --warning-color: #f59e0b;
        --danger-color: #ef4444;
        --light-color: #f8fafc;
        --dark-color: #1f2937;
    }

    body {
        background-color: var(--light-color) !important;
    }

    .test-container {
        padding: 0 30px !important;
    }
    
    .page-header {
        background: white !important;
        border: 1px solid #f1f5f9 !important;
        border-radius: 12px !important;
        padding: 24px !important;
        margin-bottom: 32px !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
    }
    
    .page-title {
        font-size: 24px !important;
        font-weight: 700 !important;
        color: #1f2937 !important;
        margin-bottom: 8px !important;
        display: flex !important;
        align-items: center !important;
    }

    .page-title i {
        color: var(--primary-color) !important;
        font-size: 28px !important;
        margin-right: 12px !important;
    }
    
    .test-form {
        background: white !important;
        border: 1px solid #f1f5f9 !important;
        border-radius: 12px !important;
        padding: 24px !important;
        margin-bottom: 24px !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
    }
    
    .form-group {
        margin-bottom: 20px !important;
    }
    
    .form-label {
        font-size: 14px !important;
        font-weight: 500 !important;
        color: #1f2937 !important;
        margin-bottom: 6px !important;
        display: block !important;
    }
    
    .form-select, .form-input {
        width: 100% !important;
        padding: 10px 12px !important;
        border: 1px solid #d1d5db !important;
        border-radius: 6px !important;
        font-size: 14px !important;
        transition: all 0.3s ease !important;
    }
    
    .form-select:focus, .form-input:focus {
        outline: none !important;
        border-color: var(--primary-color) !important;
        box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1) !important;
    }
    
    .btn-test {
        background: var(--primary-color) !important;
        color: white !important;
        padding: 12px 20px !important;
        border: none !important;
        border-radius: 6px !important;
        font-weight: 500 !important;
        cursor: pointer !important;
        transition: all 0.3s ease !important;
        display: inline-flex !important;
        align-items: center !important;
        gap: 8px !important;
    }
    
    .btn-test:hover {
        background: #3f37c9 !important;
        transform: translateY(-1px) !important;
    }
    
    .results-container {
        background: white !important;
        border: 1px solid #f1f5f9 !important;
        border-radius: 12px !important;
        padding: 24px !important;
        margin-bottom: 24px !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
    }
    
    .alert {
        padding: 12px 16px !important;
        border-radius: 6px !important;
        margin-bottom: 20px !important;
        font-size: 14px !important;
        display: flex !important;
        align-items: center !important;
        gap: 8px !important;
    }
    
    .alert-success {
        background: #f0fdf4 !important;
        color: #166534 !important;
        border: 1px solid #bbf7d0 !important;
    }
    
    .alert-danger {
        background: #fef2f2 !important;
        color: #dc2626 !important;
        border: 1px solid #fecaca !important;
    }
    
    .stats-grid {
        display: grid !important;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)) !important;
        gap: 16px !important;
        margin-bottom: 24px !important;
    }
    
    .stat-card {
        background: white !important;
        border: 1px solid #f1f5f9 !important;
        border-radius: 8px !important;
        padding: 16px !important;
        text-align: center !important;
    }
    
    .stat-value {
        font-size: 24px !important;
        font-weight: 700 !important;
        color: var(--primary-color) !important;
        margin-bottom: 4px !important;
    }
    
    .stat-label {
        font-size: 12px !important;
        color: #6b7280 !important;
        text-transform: uppercase !important;
        font-weight: 500 !important;
    }
    
    .result-item {
        padding: 8px 12px !important;
        background: #f8fafc !important;
        border-radius: 4px !important;
        margin-bottom: 8px !important;
        font-size: 14px !important;
        display: flex !important;
        justify-content: space-between !important;
    }
    
    .result-label {
        font-weight: 500 !important;
        color: #374151 !important;
    }
    
    .result-value {
        font-weight: 600 !important;
        color: var(--primary-color) !important;
    }
</style>

<div class="test-container">
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">
            <i class='bx bx-test-tube'></i>
            Revenue System Testing
        </h1>
        <p style="color: #6b7280; font-size: 14px; margin: 0;">
            Test the revenue sharing system and see how purchases are processed and split between admin and teachers.
        </p>
    </div>

    <?php if ($message): ?>
        <div class="alert <?= strpos($message, '✅') === 0 ? 'alert-success' : 'alert-danger' ?>">
            <?= $message ?>
        </div>
    <?php endif; ?>

    <!-- Current Revenue Stats -->
    <div class="results-container">
        <h3 style="margin-bottom: 16px; color: #1f2937; font-size: 18px; font-weight: 600;">
            <i class='bx bx-chart-line' style="color: var(--primary-color); margin-right: 8px;"></i>
            Current Revenue Statistics
        </h3>
        
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-value">R<?= number_format($revenue_stats['total_revenue'] ?? 0, 2) ?></div>
                <div class="stat-label">Total Revenue</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">R<?= number_format($revenue_stats['admin_earned'] ?? 0, 2) ?></div>
                <div class="stat-label">Admin Earned</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">R<?= number_format($revenue_stats['teacher_earned'] ?? 0, 2) ?></div>
                <div class="stat-label">Teachers Earned</div>
            </div>
            <div class="stat-card">
                <div class="stat-value"><?= number_format($revenue_stats['total_purchases'] ?? 0) ?></div>
                <div class="stat-label">Total Purchases</div>
            </div>
        </div>
    </div>

    <!-- Test Purchase Form -->
    <div class="test-form">
        <h3 style="margin-bottom: 20px; color: #1f2937; font-size: 18px; font-weight: 600;">
            <i class='bx bx-play-circle' style="color: var(--primary-color); margin-right: 8px;"></i>
            Simulate Course Purchase
        </h3>
        
        <form method="POST">
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px;">
                <div class="form-group">
                    <label class="form-label">Select Student</label>
                    <select name="user_id" class="form-select" required>
                        <option value="">Choose a student...</option>
                        <?php foreach ($users as $user): ?>
                            <option value="<?= $user['user_id'] ?>">
                                <?= htmlspecialchars($user['first_name'] . ' ' . $user['last_name']) ?> 
                                (<?= htmlspecialchars($user['email']) ?>)
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label class="form-label">Select Course</label>
                    <select name="course_id" class="form-select" required onchange="updatePrice(this)">
                        <option value="">Choose a course...</option>
                        <?php foreach ($courses as $course): ?>
                            <option value="<?= $course['course_id'] ?>" data-price="<?= $course['price'] ?>">
                                <?= htmlspecialchars($course['title']) ?> - 
                                by <?= htmlspecialchars($course['teacher_name']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label class="form-label">Purchase Amount (R)</label>
                    <input type="number" name="amount" class="form-input" step="0.01" min="0" required id="amount_input">
                </div>

                <div class="form-group" style="display: flex; align-items: end;">
                    <button type="submit" name="test_purchase" class="btn-test">
                        <i class='bx bx-credit-card'></i>
                        Process Test Purchase
                    </button>
                </div>
            </div>
        </form>
    </div>

    <!-- Test Results -->
    <?php if (!empty($test_results)): ?>
        <div class="results-container">
            <h3 style="margin-bottom: 20px; color: #1f2937; font-size: 18px; font-weight: 600;">
                <i class='bx bx-check-circle' style="color: var(--success-color); margin-right: 8px;"></i>
                Purchase Processing Results
            </h3>
            
            <div class="result-item">
                <span class="result-label">Purchase ID:</span>
                <span class="result-value">#<?= $test_results['purchase_id'] ?></span>
            </div>
            
            <div class="result-item">
                <span class="result-label">Course:</span>
                <span class="result-value"><?= htmlspecialchars($test_results['course_title']) ?></span>
            </div>
            
            <div class="result-item">
                <span class="result-label">Teacher:</span>
                <span class="result-value"><?= htmlspecialchars($test_results['teacher_name']) ?></span>
            </div>
            
            <div class="result-item">
                <span class="result-label">Total Amount:</span>
                <span class="result-value">R<?= number_format($test_results['revenue_split']['total_amount'], 2) ?></span>
            </div>
            
            <div class="result-item">
                <span class="result-label">Admin Share (<?= $test_results['revenue_split']['admin_percentage'] ?>%):</span>
                <span class="result-value">R<?= number_format($test_results['revenue_split']['admin_amount'], 2) ?></span>
            </div>
            
            <div class="result-item">
                <span class="result-label">Teacher Share (<?= $test_results['revenue_split']['teacher_percentage'] ?>%):</span>
                <span class="result-value">R<?= number_format($test_results['revenue_split']['teacher_amount'], 2) ?></span>
            </div>
        </div>
    <?php endif; ?>

    <!-- Navigation -->
    <div style="text-align: center; margin-top: 32px;">
        <a href="revenue-settings.php" style="display: inline-flex; align-items: center; gap: 8px; background: var(--primary-color); color: white; padding: 12px 20px; border-radius: 6px; text-decoration: none; font-weight: 500; margin-right: 12px;">
            <i class='bx bx-slider-alt'></i> Configure Revenue Settings
        </a>
        <a href="setup-revenue-system.php" style="display: inline-flex; align-items: center; gap: 8px; background: #6b7280; color: white; padding: 12px 20px; border-radius: 6px; text-decoration: none; font-weight: 500;">
            <i class='bx bx-cog'></i> Setup System
        </a>
    </div>
</div>

<script>
function updatePrice(selectElement) {
    const selectedOption = selectElement.options[selectElement.selectedIndex];
    const price = selectedOption.getAttribute('data-price');
    if (price) {
        document.getElementById('amount_input').value = price;
    }
}
</script>

<?php
require_once __DIR__ . '/components/admin_layout_end.php';
?>
