<?php
session_start();
require_once '../config/database.php';

// Check if user is logged in and is an admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

$lesson_id = filter_input(INPUT_GET, 'lesson_id', FILTER_VALIDATE_INT);

if (!$lesson_id) {
    header('Location: teacher_content.php');
    exit();
}

// Get lesson details
$lesson_query = $conn->prepare("SELECT v.*, c.title as course_title FROM videos v LEFT JOIN courses c ON v.course_id = c.course_id WHERE v.video_id = ?");
$lesson_query->bind_param("i", $lesson_id);
$lesson_query->execute();
$lesson_result = $lesson_query->get_result();

if ($lesson_result->num_rows === 0) {
    header('Location: teacher_content.php');
    exit();
}

$lesson = $lesson_result->fetch_assoc();
$lesson_query->close();

// Get all quizzes for this lesson
$quiz_query = $conn->prepare("SELECT * FROM quizzes WHERE video_id = ? AND is_published = 1 ORDER BY created_at ASC");
$quiz_query->bind_param("i", $lesson_id);
$quiz_query->execute();
$quiz_result = $quiz_query->get_result();

$quizzes = [];
while ($quiz = $quiz_result->fetch_assoc()) {
    // Get quiz questions
    $question_query = $conn->prepare("SELECT * FROM quiz_questions WHERE quiz_id = ? ORDER BY question_order ASC");
    $question_query->bind_param("i", $quiz['quiz_id']);
    $question_query->execute();
    $question_result = $question_query->get_result();
    
    $questions = [];
    while ($question = $question_result->fetch_assoc()) {
        // Get question options
        $option_query = $conn->prepare("SELECT * FROM quiz_options WHERE question_id = ? ORDER BY option_order ASC");
        $option_query->bind_param("i", $question['question_id']);
        $option_query->execute();
        $option_result = $option_query->get_result();
        
        $options = [];
        while ($option = $option_result->fetch_assoc()) {
            $options[] = $option;
        }
        $option_query->close();
        
        $question['options'] = $options;
        $questions[] = $question;
    }
    $question_query->close();
    
    $quiz['questions'] = $questions;
    $quizzes[] = $quiz;
}
$quiz_query->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>View Quiz Content - Admin Dashboard</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <style>
        .main-content {
            padding-top: 60px !important;
            margin-top: -10px !important;
        }
        
        .quiz-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 15px;
            padding: 30px;
            margin-bottom: 30px;
            text-align: center;
        }
        
        .lesson-info {
            background-color: #f8f9fa;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 30px;
            border-left: 5px solid #667eea;
        }
        
        .quiz-card {
            border: none;
            box-shadow: 0 0 20px rgba(0, 0, 0, 0.05);
            border-radius: 15px;
            margin-bottom: 30px;
            overflow: hidden;
        }
        
        .quiz-card-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
        }
        
        .question-card {
            border-left: 4px solid #667eea;
            margin-bottom: 20px;
        }
        
        .option-item {
            padding: 10px 15px;
            margin-bottom: 8px;
            border-radius: 8px;
            border: 1px solid #e9ecef;
            transition: all 0.3s ease;
        }
        
        .option-item.correct {
            background-color: #d4edda;
            border-color: #c3e6cb;
            color: #155724;
        }
        
        .option-item.correct::before {
            content: "✓ ";
            font-weight: bold;
            color: #28a745;
        }
        
        .quiz-info-badges {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
            margin-bottom: 20px;
        }
        
        .info-badge {
            background: rgba(255, 255, 255, 0.2);
            padding: 8px 15px;
            border-radius: 20px;
            font-size: 0.9rem;
        }
        
        .back-button {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            color: white;
            padding: 10px 20px;
            border-radius: 25px;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            transition: all 0.3s ease;
        }
        
        .back-button:hover {
            color: white;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        }
        
        .no-quiz-state {
            text-align: center;
            padding: 60px 20px;
            background-color: #f8f9fa;
            border-radius: 15px;
        }
        
        .no-quiz-state i {
            font-size: 4rem;
            color: #adb5bd;
            margin-bottom: 20px;
        }
        
        @media (max-width: 768px) {
            .quiz-header {
                padding: 20px;
            }
            
            .lesson-info {
                padding: 15px;
            }
            
            .quiz-info-badges {
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="wrapper">
        <?php include 'includes/sidebar.php'; ?>
        
        <div class="main-content">
            <div class="container-fluid">
                <!-- Header -->
                <div class="quiz-header">
                    <h1><i class="bx bx-help-circle me-2"></i>Quiz Content Review</h1>
                    <p class="mb-0">Administrative view of quiz content - Read-only mode</p>
                </div>
                
                <!-- Back Button -->
                <div class="mb-4">
                    <a href="javascript:history.back()" class="back-button">
                        <i class="bx bx-arrow-back me-2"></i>Back to Teacher Content
                    </a>
                </div>
                
                <!-- Lesson Information -->
                <div class="lesson-info">
                    <div class="d-flex align-items-center mb-3">
                        <div class="me-3">
                            <?php if (!empty($lesson['thumbnail'])): ?>
                                <img src="../<?= htmlspecialchars($lesson['thumbnail']) ?>" alt="Lesson Thumbnail" class="rounded" style="width: 60px; height: 45px; object-fit: cover;">
                            <?php else: ?>
                                <div class="d-flex align-items-center justify-content-center bg-secondary text-white rounded" style="width: 60px; height: 45px;">
                                    <i class="bx bx-video"></i>
                                </div>
                            <?php endif; ?>
                        </div>
                        <div>
                            <h4 class="mb-1"><?= htmlspecialchars($lesson['title']) ?></h4>
                            <p class="text-muted mb-0">
                                <i class="bx bx-book me-1"></i>Course: <?= htmlspecialchars($lesson['course_title'] ?? 'Unknown Course') ?>
                                <span class="mx-2">•</span>
                                <i class="bx bx-time me-1"></i>Duration: <?= htmlspecialchars($lesson['duration'] ?? 'Not specified') ?>
                            </p>
                        </div>
                    </div>
                </div>
                
                <!-- Quiz Content -->
                <?php if (empty($quizzes)): ?>
                    <div class="no-quiz-state">
                        <i class="bx bx-help-circle"></i>
                        <h3>No Quizzes Available</h3>
                        <p class="text-muted mb-0">This lesson doesn't have any published quizzes yet.</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($quizzes as $quiz_index => $quiz): ?>
                        <div class="quiz-card">
                            <div class="quiz-card-header">
                                <h3 class="mb-3">
                                    <i class="bx bx-help-circle me-2"></i>
                                    Quiz <?= $quiz_index + 1 ?>: <?= htmlspecialchars($quiz['title']) ?>
                                </h3>
                                
                                <div class="quiz-info-badges">
                                    <span class="info-badge">
                                        <i class="bx bx-time me-1"></i>
                                        Time Limit: <?= $quiz['time_limit'] ?> minutes
                                    </span>
                                    <span class="info-badge">
                                        <i class="bx bx-target-lock me-1"></i>
                                        Passing Score: <?= $quiz['passing_score'] ?>%
                                    </span>
                                    <span class="info-badge">
                                        <i class="bx bx-question-mark me-1"></i>
                                        <?= count($quiz['questions']) ?> Questions
                                    </span>
                                </div>
                                
                                <?php if (!empty($quiz['description'])): ?>
                                    <p class="mb-0 opacity-75">
                                        <i class="bx bx-info-circle me-1"></i>
                                        <?= htmlspecialchars($quiz['description']) ?>
                                    </p>
                                <?php endif; ?>
                            </div>
                            
                            <div class="card-body p-4">
                                <?php if (empty($quiz['questions'])): ?>
                                    <div class="text-center text-muted py-3">
                                        <i class="bx bx-question-mark fs-4"></i>
                                        <p class="mt-2 mb-0">No questions available in this quiz</p>
                                    </div>
                                <?php else: ?>
                                    <?php foreach ($quiz['questions'] as $q_index => $question): ?>
                                        <div class="question-card card">
                                            <div class="card-body">
                                                <div class="d-flex align-items-start mb-3">
                                                    <span class="badge bg-primary me-3 mt-1">Q<?= $q_index + 1 ?></span>
                                                    <div class="flex-grow-1">
                                                        <h6 class="mb-2"><?= htmlspecialchars($question['question_text']) ?></h6>
                                                        <small class="text-muted">
                                                            <i class="bx bx-target-lock me-1"></i>
                                                            Type: <?= ucfirst(htmlspecialchars($question['question_type'])) ?>
                                                            <span class="mx-2">•</span>
                                                            <i class="bx bx-award me-1"></i>
                                                            Points: <?= $question['points'] ?>
                                                        </small>
                                                    </div>
                                                </div>
                                                
                                                <?php if (!empty($question['options'])): ?>
                                                    <div class="mt-3">
                                                        <small class="text-muted mb-2 d-block">
                                                            <i class="bx bx-list-ul me-1"></i>Answer Options:
                                                        </small>
                                                        <?php foreach ($question['options'] as $option): ?>
                                                            <div class="option-item <?= $option['is_correct'] ? 'correct' : '' ?>">
                                                                <?= htmlspecialchars($option['option_text']) ?>
                                                                <?php if ($option['is_correct']): ?>
                                                                    <small class="float-end text-success">
                                                                        <strong>Correct Answer</strong>
                                                                    </small>
                                                                <?php endif; ?>
                                                            </div>
                                                        <?php endforeach; ?>
                                                    </div>
                                                <?php endif; ?>
                                                
                                                <?php if (!empty($question['explanation'])): ?>
                                                    <div class="mt-3 p-3 bg-light rounded">
                                                        <small class="text-muted d-block mb-1">
                                                            <i class="bx bx-lightbulb me-1"></i><strong>Explanation:</strong>
                                                        </small>
                                                        <small><?= htmlspecialchars($question['explanation']) ?></small>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Desktop sidebar toggle
            const sidebarToggle = document.querySelector('.sidebar-toggle');
            if (sidebarToggle) {
                sidebarToggle.addEventListener('click', function() {
                    document.querySelector('.wrapper').classList.toggle('sidebar-collapsed');
                });
            }
            
            // Mobile sidebar toggle
            const mobileSidebarToggle = document.querySelector('.sidebar-toggle-mobile');
            if (mobileSidebarToggle) {
                mobileSidebarToggle.addEventListener('click', function() {
                    document.querySelector('.wrapper').classList.toggle('sidebar-mobile-open');
                });
            }
        });
    </script>
</body>
</html>
