/**
 * Course Video Player and Navigation - Udemy Style
 * Handles the video player functionality and navigation between videos
 */

document.addEventListener('DOMContentLoaded', function() {
    // Elements
    const prevLessonBtn = document.getElementById('prevLessonBtn');
    const nextLessonBtn = document.getElementById('nextLessonBtn');
    const markCompletedBtn = document.getElementById('markCompletedBtn');
    const videoPlayer = document.querySelector('video');
    
    // Get the current course and video IDs from URL parameters
    const urlParams = new URLSearchParams(window.location.search);
    const courseId = urlParams.get('id');
    const currentVideoId = urlParams.get('video_id');
    
    // Event listeners for navigation buttons
    if (prevLessonBtn) {
        prevLessonBtn.addEventListener('click', function() {
            navigateToPreviousVideo();
        });
    }
    
    if (nextLessonBtn) {
        nextLessonBtn.addEventListener('click', function() {
            navigateToNextVideo();
        });
    }
    
    if (markCompletedBtn) {
        markCompletedBtn.addEventListener('click', function() {
            markVideoAsCompleted();
        });
    }
    
    // Video events - detect completion
    if (videoPlayer) {
        videoPlayer.addEventListener('ended', function() {
            // Auto-mark as completed when video ends
            markVideoAsCompleted(true);
        });
    }
    
    /**
     * Navigate to the previous video
     */
    function navigateToPreviousVideo() {
        const prevVideoId = document.querySelector('#prevLessonBtn').getAttribute('data-prev-video');
        if (prevVideoId) {
            window.location.href = `?id=${courseId}&video_id=${prevVideoId}`;
        }
    }
    
    /**
     * Navigate to the next video
     */
    function navigateToNextVideo() {
        const nextVideoId = document.querySelector('#nextLessonBtn').getAttribute('data-next-video');
        if (nextVideoId) {
            window.location.href = `?id=${courseId}&video_id=${nextVideoId}`;
        }
    }
    
    /**
     * Mark the current video as completed via AJAX
     * @param {boolean} autoComplete - Whether this was called automatically
     */
    function markVideoAsCompleted(autoComplete = false) {
        if (!currentVideoId || !courseId) return;
        
        // Show spinner on button
        if (markCompletedBtn) {
            markCompletedBtn.innerHTML = '<i class="bx bx-loader bx-spin me-1"></i> Processing...';
            markCompletedBtn.disabled = true;
        }
        
        // Make AJAX request to mark video as completed
        fetch('mark_lesson_completed.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `video_id=${currentVideoId}&course_id=${courseId}&auto_complete=${autoComplete ? 1 : 0}`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Update the UI to show completion
                if (markCompletedBtn) {
                    markCompletedBtn.innerHTML = '<i class="bx bx-check-double me-1"></i> Completed';
                    markCompletedBtn.classList.remove('btn-primary');
                    markCompletedBtn.classList.add('btn-success');
                }
                
                // Update the video item in the sidebar
                const videoItem = document.querySelector(`.lesson-item[href*="video_id=${currentVideoId}"]`);
                if (videoItem) {
                    const statusIcon = videoItem.querySelector('.lesson-status i');
                    if (statusIcon) {
                        statusIcon.className = 'bx bx-check-circle fs-5 text-success';
                    }
                }
                
                // Update progress percentage
                updateProgressPercentage(data.progress_percent);
                
                // If auto-completed, offer to go to next video
                if (autoComplete && data.has_next_video && data.next_video_id) {
                    showNextVideoPrompt(data.next_video_id);
                }
            } else {
                // Handle error
                if (markCompletedBtn) {
                    markCompletedBtn.innerHTML = '<i class="bx bx-error-circle me-1"></i> Failed';
                    setTimeout(() => {
                        markCompletedBtn.innerHTML = '<i class="bx bx-check me-1"></i> Mark as Completed';
                        markCompletedBtn.disabled = false;
                    }, 2000);
                }
            }
        })
        .catch(error => {
            console.error('Error marking lesson as completed:', error);
            if (markCompletedBtn) {
                markCompletedBtn.innerHTML = '<i class="bx bx-check me-1"></i> Mark as Completed';
                markCompletedBtn.disabled = false;
            }
        });
    }
    
    /**
     * Update the progress percentage in the UI
     */
    function updateProgressPercentage(percent) {
        const progressText = document.querySelector('.progress + small');
        const progressBar = document.querySelector('.progress-bar');
        
        if (progressBar) {
            progressBar.style.width = `${percent}%`;
        }
        
        if (progressText) {
            progressText.textContent = `${percent}% complete`;
        }
    }
    
    /**
     * Show a prompt asking if the user wants to continue to the next lesson
     */
    function showNextLessonPrompt() {
        const nextLessonId = document.querySelector('#nextLessonBtn').getAttribute('data-next-lesson');
        if (!nextLessonId) return;
        
        // Create prompt element
        const promptEl = document.createElement('div');
        promptEl.className = 'next-lesson-prompt';
        promptEl.innerHTML = `
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <strong>Lesson completed!</strong> 
                        Ready to continue to the next lesson?
            
            // Add click event for the next button
            const nextBtn = nextPrompt.querySelector('.btn-next');
            if (nextBtn) {
                nextBtn.addEventListener('click', function() {
                    window.location.href = `?id=${courseId}&video_id=${nextVideoId}`;
                });
            }
        }
    }
});
