/**
 * Enhanced Night Sky Animation with Stars and Shooting Stars
 * For Panadite Academy Teacher Dashboard
 */

document.addEventListener('DOMContentLoaded', function() {
    // Select the container for the stars
    const starsContainer = document.querySelector('.stars');
    
    if (!starsContainer) {
        console.error('Stars container not found');
        return;
    }
    
    // Configuration
    const config = {
        stars: {
            count: 100,
            size: { min: 1, max: 3 },
            color: '#ffffff',
            twinkle: { min: 1, max: 3 }
        },
        shootingStars: {
            count: 5,
            length: { min: 80, max: 150 },
            speed: { min: 6, max: 10 },
            angle: 45,
            color: '#ffffff',
            delay: { min: 3, max: 15 }
        }
    };
    
    // Create regular stars
    for (let i = 0; i < config.stars.count; i++) {
        createStar(starsContainer, config.stars);
    }
    
    // Create shooting stars with randomized delays
    for (let i = 0; i < config.shootingStars.count; i++) {
        setTimeout(() => {
            createShootingStar(starsContainer, config.shootingStars);
        }, getRandomInt(500, 2000));
        
        // Create recurring shooting stars
        setInterval(() => {
            createShootingStar(starsContainer, config.shootingStars);
        }, getRandomInt(config.shootingStars.delay.min * 1000, config.shootingStars.delay.max * 1000));
    }
    
    // Initialize sidebar toggle functionality
    initSidebarToggle();
});

/**
 * Create a star element with random properties
 */
function createStar(container, config) {
    const star = document.createElement('div');
    star.classList.add('star');
    
    // Random positioning
    star.style.left = `${getRandomInt(0, 100)}%`;
    star.style.top = `${getRandomInt(0, 100)}%`;
    
    // Random size
    const size = getRandomInt(config.size.min, config.size.max);
    star.style.width = `${size}px`;
    star.style.height = `${size}px`;
    
    // Base styling
    star.style.backgroundColor = config.color;
    star.style.opacity = getRandomFloat(0.5, 1);
    
    // Add twinkling animation
    const twinkleDuration = getRandomFloat(config.twinkle.min, config.twinkle.max);
    star.style.animation = `twinkle ${twinkleDuration}s ease-in-out infinite alternate`;
    star.style.animationDelay = `${getRandomFloat(0, 10)}s`;
    
    // Append to container
    container.appendChild(star);
}

/**
 * Create a shooting star element
 */
function createShootingStar(container, config) {
    const shootingStar = document.createElement('div');
    shootingStar.classList.add('shooting-star');
    
    // Position at random location on the left side
    shootingStar.style.left = `${getRandomInt(-10, 20)}%`;
    shootingStar.style.top = `${getRandomInt(0, 60)}%`;
    
    // Apply random length and angle
    const length = getRandomInt(config.length.min, config.length.max);
    shootingStar.style.width = `${length}px`;
    
    // Apply random speed
    const speed = getRandomFloat(config.speed.min, config.speed.max);
    shootingStar.style.setProperty('--speed', `${speed}s`);
    
    // Set transform for the angle
    const angle = config.angle + getRandomInt(-15, 15);
    shootingStar.style.transform = `rotate(${angle}deg)`;
    
    // Add to container
    container.appendChild(shootingStar);
    
    // Remove after animation completes
    setTimeout(() => {
        container.removeChild(shootingStar);
    }, speed * 1000 + 100);
}

/**
 * Initialize sidebar toggle functionality
 */
function initSidebarToggle() {
    const sidebarToggle = document.getElementById('sidebarToggle');
    const sidebar = document.querySelector('.sidebar');
    const mainContent = document.querySelector('.main-content');
    
    if (!sidebarToggle || !sidebar || !mainContent) {
        console.error('Sidebar toggle elements not found');
        return;
    }
    
    sidebarToggle.addEventListener('click', function() {
        sidebar.classList.toggle('collapsed');
        mainContent.classList.toggle('expanded');
    });
}

/**
 * Helper function to get random integer between min and max (inclusive)
 */
function getRandomInt(min, max) {
    return Math.floor(Math.random() * (max - min + 1)) + min;
}

/**
 * Helper function to get random float between min and max
 */
function getRandomFloat(min, max) {
    return (Math.random() * (max - min) + min).toFixed(2);
}
