<?php
require_once __DIR__ . '/../auth/functions.php';
require_once __DIR__ . '/../config/fast_email_service.php';

// Ensure user is logged in as an admin
requireRole('admin');

$admin_id = $_SESSION['user_id'];

// Initialize debug info
$debug_info = ['steps' => [], 'errors' => [], 'success' => false, 'email_data' => []];

/**
 * Send library document approval email notification to teacher
 * COPIED EXACT APPROACH FROM WORKING COURSE EMAIL SYSTEM
 */
function sendLibraryApprovalEmail($submission_id) {
    global $conn;
    
    try {
        // Get submission and teacher details
        $stmt = $conn->prepare("
            SELECT ls.title, ls.description, ls.file_name, ls.file_type, ls.file_size, ls.submission_date, ls.teacher_id,
                   u.first_name, u.last_name, u.email, u.user_id,
                   cat.category_name
            FROM library_submissions ls 
            JOIN users u ON ls.teacher_id = u.user_id 
            LEFT JOIN categories cat ON ls.category_id = cat.category_id 
            WHERE ls.submission_id = ?
        ");
        $stmt->bind_param("i", $submission_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $submission = $result->fetch_assoc();
        
        if (!$submission) {
            error_log("Submission not found for library approval email: $submission_id");
            return false;
        }
        
        $teacher_name = $submission['first_name'] . ' ' . $submission['last_name'];
        $teacher_email = $submission['email'];
        $document_title = $submission['title'];
        $category = $submission['category_name'] ?? 'General';
        $file_size = round($submission['file_size'] / 1024, 1); // Convert to KB
        $submission_date = date('M j, Y', strtotime($submission['submission_date']));
        
        // Create email content - SAME STRUCTURE AS WORKING COURSE EMAIL
        $subject = "📚 Document Approved: \"$document_title\" is Now Live!";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#28a745; color:white; padding:20px; text-align:center;'>
                <h1>📚 Document Approved!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Congratulations, " . htmlspecialchars($teacher_name) . "! 🎉</h2>
                <p>Your document <strong>\"" . htmlspecialchars($document_title) . "\"</strong> has been approved and is now live!</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Document Details:</h3>
                    <p><strong>Category:</strong> " . htmlspecialchars($category) . "</p>
                    <p><strong>File Size:</strong> " . $file_size . " KB</p>
                    <p><strong>Submitted:</strong> " . $submission_date . "</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/teacher/library.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        View Your Library
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending - EXACT SAME PATTERN AS WORKING COURSE EMAIL
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($teacher_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail - EXACT SAME AS COURSE EMAIL
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($teacher_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending library approval email: " . $e->getMessage());
        return false;
    }
}

/**
 * Send library document rejection email notification to teacher
 */
function sendLibraryRejectionEmail($submission_id, $admin_notes = '') {
    global $conn, $debug_info;
    
    $debug_info['steps'][] = "🔍 Starting library rejection email for submission ID: $submission_id";
    
    try {
        // Get submission and teacher details
        $stmt = $conn->prepare("
            SELECT ls.title, ls.description, ls.file_name, ls.file_type, ls.submission_date,
                   u.first_name, u.last_name, u.email, u.user_id,
                   cat.category_name
            FROM library_submissions ls 
            JOIN users u ON ls.teacher_id = u.user_id 
            LEFT JOIN categories cat ON ls.category_id = cat.category_id 
            WHERE ls.submission_id = ?
        ");
        $stmt->bind_param("i", $submission_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $submission = $result->fetch_assoc();
        
        if (!$submission) {
            $debug_info['errors'][] = "❌ Submission not found for ID: $submission_id";
            error_log("Submission not found for library rejection email: $submission_id");
            return false;
        }
        
        $debug_info['steps'][] = "✅ Document details retrieved for rejection email";
        
        $teacher_name = $submission['first_name'] . ' ' . $submission['last_name'];
        $teacher_email = $submission['email'];
        $document_title = $submission['title'];
        $category = $submission['category_name'] ?? 'General';
        $submission_date = date('M j, Y', strtotime($submission['submission_date']));
        
        // Store email data for debugging
        $debug_info['email_data'] = [
            'teacher_name' => $teacher_name,
            'teacher_email' => $teacher_email,
            'document_title' => $document_title,
            'category' => $category,
            'submission_date' => $submission_date,
            'admin_notes' => $admin_notes
        ];
        
        $debug_info['steps'][] = "📧 Rejection email recipient: $teacher_name ($teacher_email)";
        $debug_info['steps'][] = "📄 Document: $document_title [$category]";
        $debug_info['steps'][] = "📝 Admin notes provided: " . (!empty($admin_notes) ? "✅ Yes" : "❌ No");
        
        // Prepare admin feedback section
        $admin_feedback = '';
        if (!empty($admin_notes)) {
            $admin_feedback = "
            <div style='background:#fff3cd; border:1px solid #ffeaa7; padding:15px; border-radius:5px; margin:20px 0;'>
                <h4>📝 Admin Feedback:</h4>
                <p>" . nl2br(htmlspecialchars($admin_notes)) . "</p>
            </div>";
        }
        
        // Create email content
        $subject = "📚 Document Review: \"$document_title\" Requires Revision";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#dc3545; color:white; padding:20px; text-align:center;'>
                <h1>📋 Document Review</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Hello, " . htmlspecialchars($teacher_name) . "!</h2>
                <p>Thank you for submitting <strong>\"" . htmlspecialchars($document_title) . "\"</strong> to our library. After careful review, we need you to make some revisions before we can approve it for publication.</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Document Details:</h3>
                    <p><strong>Title:</strong> " . htmlspecialchars($document_title) . "</p>
                    <p><strong>Category:</strong> " . htmlspecialchars($category) . "</p>
                    <p><strong>File:</strong> " . htmlspecialchars($submission['file_name']) . "</p>
                    <p><strong>Submitted:</strong> " . $submission_date . "</p>
                </div>
                " . $admin_feedback . "
                <div style='background:#f8d7da; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h4>🔄 Next Steps:</h4>
                    <ul>
                        <li>Review the feedback above (if provided)</li>
                        <li>Make necessary improvements to your document</li>
                        <li>Resubmit your revised document</li>
                        <li>Contact support if you have questions</li>
                    </ul>
                </div>
                <div style='background:#e2e3e5; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h4>💡 Common Issues to Check:</h4>
                    <ul>
                        <li>✓ Content quality and educational value</li>
                        <li>✓ Proper formatting and readability</li>
                        <li>✓ Copyright compliance</li>
                        <li>✓ Appropriate file size and type</li>
                        <li>✓ Clear title and description</li>
                    </ul>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/teacher/upload_document.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block; margin-right:10px;'>
                        Resubmit Document
                    </a>
                    <a href='https://panaditeacademy.com/teacher/support.php' 
                       style='background:#6c757d; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        Get Help
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($teacher_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($teacher_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        $debug_info['errors'][] = "❌ Exception in rejection email: " . $e->getMessage();
        error_log("Error sending library rejection email: " . $e->getMessage());
        return false;
    }
}

// Check if tables exist first
$tables_exist = true;
$required_tables = ['library_submissions', 'public_library_resources', 'library_categories', 'admin_library_logs'];
foreach ($required_tables as $table) {
    $result = $conn->query("SHOW TABLES LIKE '$table'");
    if (!$result || $result->num_rows == 0) {
        $tables_exist = false;
        break;
    }
}

// Handle approval/rejection actions (only if tables exist)
if ($_POST['action'] ?? false) {
    if (!$tables_exist) {
        $error_message = "Database tables not found. Please run the setup script first.";
    } else {
        $submission_id = (int)$_POST['submission_id'];
        $action = $_POST['action'];
        $notes = trim($_POST['admin_notes'] ?? '');
        
        if (in_array($action, ['approve', 'reject'])) {
            $status = $action === 'approve' ? 'approved' : 'rejected';
            
            // Update submission status - FIXED: Use correct primary key 'submission_id'
            $stmt = $conn->prepare("UPDATE library_submissions SET status = ?, admin_notes = ?, review_date = NOW() WHERE submission_id = ?");
            if ($stmt) {
                $stmt->bind_param("ssi", $status, $notes, $submission_id);
                if ($stmt->execute()) {
                    $stmt->close();
                    
                    // If approved, copy to public library
                    if ($action === 'approve') {
                        $stmt = $conn->prepare("
                            INSERT INTO public_library_resources (teacher_id, title, description, category_id, tags, file_path, file_name, file_type, file_size, thumbnail_path, publication_date)
                            SELECT teacher_id, title, description, category_id, tags, file_path, file_name, file_type, file_size, thumbnail_path, NOW()
                            FROM library_submissions WHERE submission_id = ?
                        ");
                        if ($stmt) {
                            $stmt->bind_param("i", $submission_id);
                            $stmt->execute();
                            $stmt->close();
                        }
                        
                        // Send approval email with debugging
                        $email_result = sendLibraryApprovalEmail($submission_id);
                        $debug_info['email_type'] = 'approval';
                        $debug_info['email_sent'] = $email_result;
                    } else {
                        // Send rejection email with admin notes and debugging
                        $email_result = sendLibraryRejectionEmail($submission_id, $notes);
                        $debug_info['email_type'] = 'rejection';
                        $debug_info['email_sent'] = $email_result;
                    }
                    
                    if ($action === 'approve') {
                        if ($email_result) {
                            $message = "Document approved and published! ✉️ Approval email sent to teacher.";
                        } else {
                            $message = "Document approved and published, but failed to send notification email to teacher.";
                        }
                    } else {
                        $email_status = $email_result ? '✉️ Rejection email sent to teacher.' : 'Email failed to send';
                        $message = "Document rejected. $email_status";
                    }
                } else {
                    $error_message = "Database error: " . $stmt->error;
                }
                }
                
                // Log admin action
                $stmt = $conn->prepare("INSERT INTO admin_library_logs (admin_id, submission_id, action, notes) VALUES (?, ?, ?, ?)");
                if ($stmt) {
                    $log_action = $action === 'approve' ? 'approved' : 'rejected';
                    $stmt->bind_param("iiss", $admin_id, $submission_id, $log_action, $notes);
                    $stmt->execute();
                    $stmt->close();
                }
                
                $message = $action === 'approve' ? 'Document approved and published!' : 'Document rejected.';
            } else {
                $error_message = "Database error: " . $conn->error;
            }
        }
    }


// Get pending submissions (only if tables exist)
$pending_submissions = [];
if ($tables_exist) {
    $stmt = $conn->prepare("
        SELECT ls.*, u.username as teacher_name, c.category_name
        FROM library_submissions ls
        LEFT JOIN users u ON ls.teacher_id = u.user_id
        LEFT JOIN categories c ON ls.category_id = c.category_id
        WHERE ls.status = 'pending'
        ORDER BY ls.priority DESC, ls.submission_date ASC
    ");
    if ($stmt) {
        $stmt->execute();
        $pending_submissions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    }
}

// Get statistics (only if tables exist)
$stats = ['total' => 0, 'pending' => 0, 'approved' => 0, 'rejected' => 0];
if ($tables_exist) {
    $result = $conn->query("SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
        SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
        SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected
        FROM library_submissions
    ");
    if ($result) {
        $stats = $result->fetch_assoc();
    }
}
$page_title = 'Library Document Approvals';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/admin-dashboard.css">
    <link rel="stylesheet" href="../assets/css/user-cards.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        .admin-header { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); 
            color: white; 
            padding: 2rem; 
            border-radius: 16px; 
            margin-bottom: 2rem; 
        }
        .submission-card { 
            background: white; 
            border-radius: 12px; 
            padding: 1.5rem; 
            margin-bottom: 1rem; 
            box-shadow: 0 4px 12px rgba(0,0,0,0.1); 
            border-left: 4px solid #e5e7eb; 
        }
        .submission-card.priority-high { border-left-color: #dc2626; }
        .submission-card.priority-medium { border-left-color: #f59e0b; }
        .submission-card.priority-low { border-left-color: #10b981; }
        .status-badge { 
            padding: 0.25rem 0.75rem; 
            border-radius: 20px; 
            font-size: 0.75rem; 
            font-weight: 600; 
        }
        .status-pending { background: #fef3c7; color: #92400e; }
        .btn-approve { background: #10b981; border: none; color: white; }
        .btn-reject { background: #ef4444; border: none; color: white; }
        .stat-card { 
            background: white; 
            padding: 1.5rem; 
            border-radius: 12px; 
            text-align: center; 
            box-shadow: 0 4px 12px rgba(0,0,0,0.05); 
        }
        .stat-number { font-size: 2rem; font-weight: 700; }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <!-- Page Header -->
            <div class="page-header">
                <h1>Library Document Approvals</h1>
                <div>
                    <span class="page-subtitle">Review and approve teacher-submitted educational resources</span>
                </div>
            </div>

            <!-- Admin Header Section -->
            <div class="admin-header">
                <h1><i class="bx bx-shield-check"></i> Library Document Approvals</h1>
                <p class="mb-0">Review and approve teacher-submitted educational resources</p>
            </div>

    <!-- Error Message -->
    <?php if (isset($error_message)): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="bx bx-error-circle"></i> <?= htmlspecialchars($error_message) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <!-- Success Message -->
    <?php if (isset($message)): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <i class="bx bx-check-circle"></i> <?= htmlspecialchars($message) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <!-- Email Debug Information -->
    <?php if (!empty($debug_info['steps']) || !empty($debug_info['errors'])): ?>
        <div class="alert alert-info alert-dismissible fade show">
            <h5><i class="bx bx-bug"></i> Email Debug Information</h5>
            
            <?php if (!empty($debug_info['email_data'])): ?>
                <div style="background: #f8f9fa; padding: 15px; border-radius: 5px; margin: 10px 0;">
                    <h6>📧 Email Details:</h6>
                    <ul style="margin: 0; padding-left: 20px;">
                        <?php foreach ($debug_info['email_data'] as $key => $value): ?>
                            <li><strong><?= ucfirst(str_replace('_', ' ', $key)) ?>:</strong> <?= htmlspecialchars($value) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($debug_info['steps'])): ?>
                <div style="background: #e8f5e8; padding: 15px; border-radius: 5px; margin: 10px 0;">
                    <h6>✅ Processing Steps:</h6>
                    <ul style="margin: 0; padding-left: 20px;">
                        <?php foreach ($debug_info['steps'] as $step): ?>
                            <li><?= htmlspecialchars($step) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($debug_info['errors'])): ?>
                <div style="background: #f8d7da; padding: 15px; border-radius: 5px; margin: 10px 0;">
                    <h6>❌ Errors & Issues:</h6>
                    <ul style="margin: 0; padding-left: 20px;">
                        <?php foreach ($debug_info['errors'] as $error): ?>
                            <li><?= htmlspecialchars($error) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            
            <div style="background: #fff3cd; padding: 15px; border-radius: 5px; margin: 10px 0;">
                <h6>🔧 System Status:</h6>
                <ul style="margin: 0; padding-left: 20px;">
                    <li><strong>Email Service File:</strong> <?= file_exists(__DIR__ . '/../config/fast_email_service.php') ? '✅ Found' : '❌ Missing' ?></li>
                    <li><strong>sendEmailNow Function:</strong> <?= function_exists('sendEmailNow') ? '✅ Available' : '❌ Not Available' ?></li>
                    <li><strong>Native Mail Function:</strong> <?= function_exists('mail') ? '✅ Available' : '❌ Not Available' ?></li>
                    <li><strong>Overall Result:</strong> <?= isset($debug_info['success']) ? ($debug_info['success'] ? '✅ Success' : '❌ Failed') : '❓ Unknown' ?></li>
                </ul>
            </div>
            
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <!-- Setup Required Message -->
    <?php if (!$tables_exist): ?>
        <div class="alert alert-warning">
            <h5><i class="bx bx-error-circle"></i> Setup Required</h5>
            <p class="mb-2">The library system database tables have not been created yet. Please run the setup script to initialize the system.</p>
            <a href="../setup_library_tables.php" class="btn btn-warning" target="_blank">
                <i class="bx bx-cog"></i> Run Setup Script
            </a>
        </div>
    <?php endif; ?>

    <!-- Statistics -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="stat-card">
                <div class="stat-number text-primary"><?= $stats['total'] ?></div>
                <div class="text-muted">Total Submissions</div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card">
                <div class="stat-number text-warning"><?= $stats['pending'] ?></div>
                <div class="text-muted">Pending Review</div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card">
                <div class="stat-number text-success"><?= $stats['approved'] ?></div>
                <div class="text-muted">Approved</div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card">
                <div class="stat-number text-danger"><?= $stats['rejected'] ?></div>
                <div class="text-muted">Rejected</div>
            </div>
        </div>
    </div>

    <!-- Pending Submissions -->
    <div class="row">
        <div class="col-12">
            <h3 class="mb-3"><i class="bx bx-time-five text-warning"></i> Pending Approvals (<?= count($pending_submissions) ?>)</h3>
            
            <?php if (empty($pending_submissions)): ?>
                <div class="text-center py-5">
                    <i class="bx bx-check-circle text-success" style="font-size: 4rem;"></i>
                    <h5 class="text-muted mt-3">All caught up!</h5>
                    <p class="text-muted">No pending submissions to review.</p>
                </div>
            <?php else: ?>
                <?php foreach ($pending_submissions as $submission): ?>
                    <div class="submission-card priority-<?= $submission['priority'] ?>">
                        <div class="row">
                            <div class="col-md-8">
                                <div class="d-flex align-items-start mb-2">
                                    <!-- Thumbnail Display -->
                                    <div class="me-3">
                                        <?php if (!empty($submission['thumbnail_path'])): ?>
                                            <img src="../<?= $submission['thumbnail_path'] ?>" 
                                                 alt="Document Thumbnail" 
                                                 style="width: 80px; height: 60px; object-fit: cover; border-radius: 8px; border: 2px solid #e5e7eb;"
                                                 onerror="this.parentNode.innerHTML='<div class=&quot;p-2 rounded&quot; style=&quot;background-color: #6b728020;&quot;><i class=&quot;bx bx-file&quot; style=&quot;font-size: 1.5rem; color: #6b7280;&quot;></i></div>'">
                                        <?php else: ?>
                                            <div class="p-2 rounded" style="background-color: #6b728020; width: 80px; height: 60px; display: flex; align-items: center; justify-content: center;">
                                                <i class="bx bx-file" style="font-size: 1.5rem; color: #6b7280;"></i>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                    <div class="flex-grow-1">
                                        <h5 class="mb-1"><?= htmlspecialchars($submission['title']) ?></h5>
                                        <p class="text-muted mb-2"><?= htmlspecialchars($submission['description']) ?></p>
                                        <div class="small text-muted">
                                            <span class="me-3"><i class="bx bx-user"></i> <?= htmlspecialchars($submission['teacher_name']) ?></span>
                                            <span class="me-3"><i class="bx bx-calendar"></i> <?= date('M j, Y', strtotime($submission['submission_date'])) ?></span>
                                            <span class="me-3"><i class="bx bx-category"></i> <?= htmlspecialchars($submission['category_name'] ?? 'Uncategorized') ?></span>
                                            <span class="me-3"><i class="bx bx-file"></i> <?= strtoupper($submission['file_type']) ?></span>
                                            <span><i class="bx bx-hdd"></i> <?= number_format($submission['file_size'] / 1024, 1) ?> KB</span>
                                        </div>
                                        <?php if (!empty($submission['tags'])): ?>
                                            <div class="mt-2">
                                                <?php foreach (explode(',', $submission['tags']) as $tag): ?>
                                                    <span class="badge bg-light text-dark me-1"><?= htmlspecialchars(trim($tag)) ?></span>
                                                <?php endforeach; ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="text-end">
                                    <span class="status-badge status-pending mb-2 d-inline-block">Pending Review</span>
                                    <div class="mb-2">
                                        <span class="badge bg-<?= $submission['priority'] === 'high' ? 'danger' : ($submission['priority'] === 'medium' ? 'warning' : 'info') ?>">
                                            <?= ucfirst($submission['priority']) ?> Priority
                                        </span>
                                    </div>
                                    <div class="btn-group d-block">
                                        <button class="btn btn-approve btn-sm me-2" onclick="showApprovalModal(<?= $submission['submission_id'] ?>, 'approve', '<?= htmlspecialchars($submission['title']) ?>')">
                                            <i class="bx bx-check"></i> Approve
                                        </button>
                                        <button class="btn btn-reject btn-sm" onclick="showApprovalModal(<?= $submission['submission_id'] ?>, 'reject', '<?= htmlspecialchars($submission['title']) ?>')">
                                            <i class="bx bx-x"></i> Reject
                                        </button>
                                    </div>
                                    <div class="mt-2">
                                        <a href="../<?= $submission['file_path'] ?>" class="btn btn-outline-primary btn-sm" target="_blank">
                                            <i class="bx bx-download"></i> Preview
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>

<!-- Approval Modal -->
<div class="modal fade" id="approvalModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST">
                <div class="modal-header">
                    <h5 class="modal-title" id="modalTitle">Confirm Action</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p id="modalMessage"></p>
                    <div class="mb-3">
                        <label class="form-label">Admin Notes (Optional)</label>
                        <textarea name="admin_notes" class="form-control" rows="3" placeholder="Add notes for the teacher..."></textarea>
                    </div>
                    <input type="hidden" name="submission_id" id="submissionId">
                    <input type="hidden" name="action" id="actionType">
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn" id="confirmBtn">Confirm</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
function showApprovalModal(submissionId, action, title) {
    document.getElementById('submissionId').value = submissionId;
    document.getElementById('actionType').value = action;
    
    const modal = document.getElementById('approvalModal');
    const modalTitle = document.getElementById('modalTitle');
    const modalMessage = document.getElementById('modalMessage');
    const confirmBtn = document.getElementById('confirmBtn');
    
    if (action === 'approve') {
        modalTitle.textContent = 'Approve Document';
        modalMessage.textContent = `Are you sure you want to approve "${title}"? This will make it publicly available in the library.`;
        confirmBtn.className = 'btn btn-success';
        confirmBtn.innerHTML = '<i class="bx bx-check"></i> Approve';
    } else {
        modalTitle.textContent = 'Reject Document';
        modalMessage.textContent = `Are you sure you want to reject "${title}"? The teacher will be notified.`;
        confirmBtn.className = 'btn btn-danger';
        confirmBtn.innerHTML = '<i class="bx bx-x"></i> Reject';
    }
    
    new bootstrap.Modal(modal).show();
}
</script>

        </div>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>

</body>
</html>
