<?php
/**
 * Login System Diagnostic Tool
 * Panadite Academy - Debug Login Issues
 */

// Start session for debugging
session_start();

// Turn on error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<!DOCTYPE html>
<html lang='en'>
<head>
    <meta charset='UTF-8'>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <title>Login System Diagnostics - Panadite Academy</title>
    <link href='https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css' rel='stylesheet'>
    <link rel='stylesheet' href='https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css'>
    <style>
        .diagnostic-card { margin-bottom: 20px; }
        .status-ok { color: #28a745; }
        .status-warning { color: #ffc107; }
        .status-error { color: #dc3545; }
        .code-block { background: #f8f9fa; padding: 10px; border-radius: 5px; font-family: monospace; }
    </style>
</head>
<body class='bg-light'>
<div class='container mt-4'>
    <h1><i class='fas fa-bug'></i> Login System Diagnostics</h1>
    <p class='text-muted'>Comprehensive debugging tool for Panadite Academy login issues</p>";

// ===== PHP ENVIRONMENT =====
echo "<div class='card diagnostic-card'>
        <div class='card-header'><h5><i class='fas fa-server'></i> PHP & Server Environment</h5></div>
        <div class='card-body'>";

$php_ok = version_compare(PHP_VERSION, '7.4', '>=');
echo "<p><strong>PHP Version:</strong> <span class='" . ($php_ok ? 'status-ok' : 'status-error') . "'>" . PHP_VERSION . "</span></p>";

$server = $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown';
echo "<p><strong>Server Software:</strong> $server</p>";

echo "<p><strong>Current Time:</strong> " . date('Y-m-d H:i:s') . "</p>";
echo "<p><strong>Server Name:</strong> " . ($_SERVER['SERVER_NAME'] ?? 'Unknown') . "</p>";
echo "<p><strong>Document Root:</strong> " . ($_SERVER['DOCUMENT_ROOT'] ?? 'Unknown') . "</p>";

echo "</div></div>";

// ===== SESSION DIAGNOSTICS =====
echo "<div class='card diagnostic-card'>
        <div class='card-header'><h5><i class='fas fa-cookie-bite'></i> Session Diagnostics</h5></div>
        <div class='card-body'>";

echo "<p><strong>Session Status:</strong> ";
$session_status = session_status();
switch($session_status) {
    case PHP_SESSION_DISABLED:
        echo "<span class='status-error'>DISABLED (PHP_SESSION_DISABLED)</span>";
        break;
    case PHP_SESSION_NONE:
        echo "<span class='status-warning'>NONE (PHP_SESSION_NONE)</span>";
        break;
    case PHP_SESSION_ACTIVE:
        echo "<span class='status-ok'>ACTIVE (PHP_SESSION_ACTIVE)</span>";
        break;
    default:
        echo "<span class='status-error'>Unknown: $session_status</span>";
}
echo "</p>";

echo "<p><strong>Session ID:</strong> " . (session_id() ?: '<span class="status-warning">No session ID</span>') . "</p>";
echo "<p><strong>Session Name:</strong> " . session_name() . "</p>";
echo "<p><strong>Session Save Path:</strong> " . session_save_path() . "</p>";

// Check if session save path is writable
$save_path = session_save_path();
if (empty($save_path)) {
    $save_path = sys_get_temp_dir();
}
$writable = is_writable($save_path);
echo "<p><strong>Save Path Writable:</strong> <span class='" . ($writable ? 'status-ok' : 'status-error') . "'>" . ($writable ? 'YES' : 'NO') . "</span></p>";

echo "<p><strong>Cookie Settings:</strong></p>";
echo "<ul>";
$cookie_params = session_get_cookie_params();
foreach($cookie_params as $key => $value) {
    echo "<li><strong>$key:</strong> " . (is_bool($value) ? ($value ? 'true' : 'false') : $value) . "</li>";
}
echo "</ul>";

echo "<p><strong>Current Session Data:</strong></p>";
echo "<div class='code-block'>" . htmlspecialchars(print_r($_SESSION, true)) . "</div>";

echo "</div></div>";

// ===== DATABASE CONNECTION TEST =====
echo "<div class='card diagnostic-card'>
        <div class='card-header'><h5><i class='fas fa-database'></i> Database Connection Test</h5></div>
        <div class='card-body'>";

try {
    require_once __DIR__ . '/../includes/site_config.php';
    require_once __DIR__ . '/../includes/db_connection.php';
    
    if (isset($conn) && $conn) {
        if ($conn->connect_error) {
            echo "<p><strong>Connection Status:</strong> <span class='status-error'>FAILED - " . $conn->connect_error . "</span></p>";
        } else {
            echo "<p><strong>Connection Status:</strong> <span class='status-ok'>SUCCESS</span></p>";
            
            // Test a simple query
            $result = $conn->query("SELECT 1 as test");
            if ($result) {
                echo "<p><strong>Query Test:</strong> <span class='status-ok'>SUCCESS</span></p>";
                
                // Check if users table exists
                $table_check = $conn->query("SHOW TABLES LIKE 'users'");
                if ($table_check && $table_check->num_rows > 0) {
                    echo "<p><strong>Users Table:</strong> <span class='status-ok'>EXISTS</span></p>";
                    
                    // Count users
                    $user_count = $conn->query("SELECT COUNT(*) as count FROM users");
                    if ($user_count) {
                        $count = $user_count->fetch_assoc()['count'];
                        echo "<p><strong>User Count:</strong> $count users in database</p>";
                    }
                } else {
                    echo "<p><strong>Users Table:</strong> <span class='status-error'>NOT FOUND</span></p>";
                }
            } else {
                echo "<p><strong>Query Test:</strong> <span class='status-error'>FAILED - " . $conn->error . "</span></p>";
            }
        }
    } else {
        echo "<p><strong>Connection Status:</strong> <span class='status-error'>NO CONNECTION OBJECT</span></p>";
    }
} catch (Exception $e) {
    echo "<p><strong>Connection Status:</strong> <span class='status-error'>EXCEPTION - " . $e->getMessage() . "</span></p>";
}

echo "</div></div>";

// ===== FILE PERMISSIONS =====
echo "<div class='card diagnostic-card'>
        <div class='card-header'><h5><i class='fas fa-file-alt'></i> File Permissions & Paths</h5></div>
        <div class='card-body'>";

$files_to_check = [
    'login.php' => __DIR__ . '/login.php',
    'functions.php' => __DIR__ . '/functions.php',
    'db_connection.php' => __DIR__ . '/../includes/db_connection.php',
    'site_config.php' => __DIR__ . '/../includes/site_config.php',
    'session_start.php' => __DIR__ . '/../includes/session_start.php'
];

foreach ($files_to_check as $name => $path) {
    $exists = file_exists($path);
    $readable = $exists ? is_readable($path) : false;
    
    echo "<p><strong>$name:</strong> ";
    if ($exists) {
        echo "<span class='status-ok'>EXISTS</span>";
        echo $readable ? " <span class='status-ok'>READABLE</span>" : " <span class='status-error'>NOT READABLE</span>";
        echo " <small class='text-muted'>($path)</small>";
    } else {
        echo "<span class='status-error'>NOT FOUND</span> <small class='text-muted'>($path)</small>";
    }
    echo "</p>";
}

echo "</div></div>";

// ===== REQUEST INFORMATION =====
echo "<div class='card diagnostic-card'>
        <div class='card-header'><h5><i class='fas fa-globe'></i> Request Information</h5></div>
        <div class='card-body'>";

echo "<p><strong>Request Method:</strong> " . ($_SERVER['REQUEST_METHOD'] ?? 'Unknown') . "</p>";
echo "<p><strong>Request URI:</strong> " . ($_SERVER['REQUEST_URI'] ?? 'Unknown') . "</p>";
echo "<p><strong>HTTP Host:</strong> " . ($_SERVER['HTTP_HOST'] ?? 'Unknown') . "</p>";
echo "<p><strong>HTTPS:</strong> " . (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'YES' : 'NO') . "</p>";

echo "<p><strong>POST Data:</strong></p>";
echo "<div class='code-block'>" . htmlspecialchars(print_r($_POST, true)) . "</div>";

echo "<p><strong>GET Data:</strong></p>";
echo "<div class='code-block'>" . htmlspecialchars(print_r($_GET, true)) . "</div>";

echo "<p><strong>Cookies:</strong></p>";
echo "<div class='code-block'>" . htmlspecialchars(print_r($_COOKIE, true)) . "</div>";

echo "</div></div>";

// ===== EMERGENCY CREDENTIAL TEST =====
echo "<div class='card diagnostic-card'>
        <div class='card-header'><h5><i class='fas fa-key'></i> Emergency Credential Test</h5></div>
        <div class='card-body'>";

echo "<p>Testing emergency credentials logic:</p>";

$test_credentials = [
    ['username' => 'admin', 'password' => 'admin123'],
    ['username' => 'teacher', 'password' => 'teacher123'],
    ['username' => 'student', 'password' => 'student123']
];

foreach ($test_credentials as $cred) {
    $username = $cred['username'];
    $password = $cred['password'];
    
    $is_emergency = (($username === 'admin' && $password === 'admin123') || 
                    ($username === 'teacher' && $password === 'teacher123') || 
                    ($username === 'student' && $password === 'student123'));
    
    echo "<p><strong>$username / $password:</strong> ";
    echo $is_emergency ? "<span class='status-ok'>VALID EMERGENCY CREDENTIAL</span>" : "<span class='status-error'>INVALID</span>";
    echo "</p>";
}

echo "</div></div>";

// ===== FORM TEST =====
echo "<div class='card diagnostic-card'>
        <div class='card-header'><h5><i class='fas fa-vial'></i> Live Form Test</h5></div>
        <div class='card-body'>";

echo "<p>Test the login form directly from this diagnostic page:</p>";

echo "<form method='POST' action='login.php' target='_blank' class='mt-3'>
        <div class='row'>
            <div class='col-md-4'>
                <label>Username:</label>
                <input type='text' name='username' class='form-control' value='admin'>
            </div>
            <div class='col-md-4'>
                <label>Password:</label>
                <input type='text' name='password' class='form-control' value='admin123'>
            </div>
            <div class='col-md-4'>
                <label>&nbsp;</label><br>
                <button type='submit' class='btn btn-primary'>Test Login</button>
            </div>
        </div>
      </form>";

echo "<p class='mt-3'><small class='text-muted'>This will open the login page in a new window with the form data submitted.</small></p>";

echo "</div></div>";

// ===== ERROR LOG TAIL =====
echo "<div class='card diagnostic-card'>
        <div class='card-header'><h5><i class='fas fa-file-medical-alt'></i> Recent Error Logs</h5></div>
        <div class='card-body'>";

$error_log_paths = [
    ini_get('error_log'),
    '/var/log/apache2/error.log',
    '/var/log/nginx/error.log',
    $_SERVER['DOCUMENT_ROOT'] . '/error.log',
    '/tmp/php_errors.log'
];

$found_logs = false;
foreach ($error_log_paths as $log_path) {
    if ($log_path && file_exists($log_path) && is_readable($log_path)) {
        $found_logs = true;
        echo "<h6>$log_path</h6>";
        $lines = file($log_path);
        if ($lines) {
            $recent_lines = array_slice($lines, -20); // Get last 20 lines
            echo "<div class='code-block' style='max-height: 200px; overflow-y: auto;'>";
            foreach ($recent_lines as $line) {
                echo htmlspecialchars($line);
            }
            echo "</div>";
        }
        break; // Only show the first found log
    }
}

if (!$found_logs) {
    echo "<p class='text-warning'>No accessible error logs found. Check server configuration.</p>";
}

echo "</div></div>";

echo "</div>
<script src='https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js'></script>
</body>
</html>";
?>
