<?php
/**
 * Migration script to upgrade users to secure referral codes
 */

header('Content-Type: text/html');
require_once '../includes/secure_referral_generator.php';

// Database configuration  
$is_local = ($_SERVER['HTTP_HOST'] == 'localhost' || 
            strpos($_SERVER['HTTP_HOST'], '127.0.0.1') !== false ||
            strpos($_SERVER['HTTP_HOST'], '192.168.') !== false);

if ($is_local) {
    $db_host = 'localhost'; $db_name = 'panadite_academy'; $db_user = 'root'; $db_pass = '';
} else {
    $db_host = 'localhost'; $db_name = 'twpurnyc_panadite_DB'; $db_user = 'twpurnyc_149586Alain'; $db_pass = '149586Alain#';
}

try {
    $conn = new mysqli($db_host, $db_user, $db_pass, $db_name);
    if ($conn->connect_error) {
        die("Connection failed: " . $conn->connect_error);
    }
} catch (Exception $e) {
    die("Database error: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>🔐 Secure Referral Code Migration</title>
    <style>
        body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; margin: 0; padding: 20px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); min-height: 100vh; }
        .container { max-width: 1200px; margin: 0 auto; background: white; border-radius: 15px; box-shadow: 0 10px 30px rgba(0,0,0,0.3); overflow: hidden; }
        .header { background: linear-gradient(135deg, #ff6b6b 0%, #ee5a24 100%); color: white; padding: 30px; text-align: center; }
        .content { padding: 30px; }
        .section { margin-bottom: 30px; padding: 20px; border-radius: 10px; border-left: 5px solid #667eea; }
        .secure { background: #e8f5e8; border-color: #28a745; }
        .legacy { background: #fff3cd; border-color: #ffc107; }
        .info { background: #d1ecf1; border-color: #17a2b8; }
        .error { background: #f8d7da; border-color: #dc3545; }
        .code-example { font-family: 'Courier New', monospace; background: #f8f9fa; padding: 15px; border-radius: 5px; margin: 10px 0; border: 1px solid #dee2e6; }
        .secure-code { color: #28a745; font-weight: bold; }
        .legacy-code { color: #fd7e14; font-weight: bold; }
        .btn { padding: 12px 24px; border: none; border-radius: 5px; cursor: pointer; font-size: 16px; margin: 5px; text-decoration: none; display: inline-block; }
        .btn-success { background: #28a745; color: white; }
        .btn-info { background: #17a2b8; color: white; }
        .btn-warning { background: #ffc107; color: #212529; }
        .stats { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin: 20px 0; }
        .stat-card { background: #f8f9fa; padding: 20px; border-radius: 10px; text-align: center; border: 1px solid #dee2e6; }
        .stat-number { font-size: 2em; font-weight: bold; color: #667eea; }
        .comparison { display: grid; grid-template-columns: 1fr 1fr; gap: 30px; margin: 30px 0; }
        .vs { text-align: center; align-self: center; font-size: 2em; color: #667eea; font-weight: bold; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🔐 Secure Referral Code Migration</h1>
            <p>Upgrading from predictable codes to cryptographically secure referral system</p>
        </div>
        
        <div class="content">
            <div class="comparison">
                <div class="section legacy">
                    <h3>❌ OLD SYSTEM (Insecure)</h3>
                    <div class="code-example">
                        <div class="legacy-code">ALKI2315</div>
                        <small>AL(ain) + KI(tumaini) + 23(user_id) + 15(random)</small>
                    </div>
                    <ul>
                        <li>Names are often public information</li>
                        <li>User IDs are sequential and guessable</li>
                        <li>Only 90 possible random combinations (10-99)</li>
                        <li>Hackers can generate codes for any user</li>
                        <li>Pattern is easily reverse-engineered</li>
                    </ul>
                </div>
                
                <div class="section secure">
                    <h3>✅ NEW SYSTEM (Secure)</h3>
                    <div class="code-example">
                        <div class="secure-code">STU4A7B9E2F1C8D</div>
                        <small>STU + 12 cryptographic hex digits</small>
                    </div>
                    <ul>
                        <li>Cryptographically secure using HMAC-SHA256</li>
                        <li>No user information revealed in code</li>
                        <li>16^12 possible combinations (281 trillion)</li>
                        <li>Impossible to generate without secret key</li>
                        <li>Stored securely in dedicated table</li>
                    </ul>
                </div>
            </div>

            <?php
            // Get current statistics
            $totalUsers = 0; $secureUsers = 0; $legacyUsers = 0;
            
            $userCountQuery = "SELECT COUNT(*) as total FROM users WHERE user_role IN ('student', 'teacher')";
            $result = $conn->query($userCountQuery);
            if ($result) {
                $totalUsers = $result->fetch_assoc()['total'];
            }
            
            $secureCountQuery = "SELECT COUNT(DISTINCT user_id) as secure FROM secure_referral_codes WHERE is_active = 1";
            $result = $conn->query($secureCountQuery);
            if ($result) {
                $secureUsers = $result->fetch_assoc()['secure'];
            }
            
            $legacyUsers = $totalUsers - $secureUsers;
            ?>

            <div class="section info">
                <h3>📊 Current System Status</h3>
                <div class="stats">
                    <div class="stat-card">
                        <div class="stat-number"><?php echo $totalUsers; ?></div>
                        <div>Total Users</div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-number" style="color: #28a745;"><?php echo $secureUsers; ?></div>
                        <div>Secure Codes</div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-number" style="color: #ffc107;"><?php echo $legacyUsers; ?></div>
                        <div>Legacy Users</div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-number" style="color: #17a2b8;"><?php echo $totalUsers > 0 ? round(($secureUsers / $totalUsers) * 100) : 0; ?>%</div>
                        <div>Migration Progress</div>
                    </div>
                </div>
            </div>

            <div class="section info">
                <h3>🎯 Test Secure Validation</h3>
                <p>Test both secure and legacy referral codes to see the new system in action:</p>
                
                <div style="margin: 20px 0;">
                    <input type="text" id="testCode" placeholder="Enter referral code" 
                           style="padding: 10px; border: 1px solid #ccc; border-radius: 5px; width: 250px; font-size: 16px;">
                    <button class="btn btn-info" onclick="testCode()">🧪 Test Code</button>
                </div>
                
                <div id="testResult" style="margin-top: 20px;"></div>
                
                <h4>🔍 Sample Codes to Test:</h4>
                <?php
                // Show some sample codes
                echo '<div class="code-example">';
                
                // Legacy code
                echo '<strong>Legacy Code:</strong> <span class="legacy-code">ALKI2315</span> 
                      <button class="btn btn-warning" onclick="document.getElementById(\'testCode\').value=\'ALKI2315\'; testCode();">Test</button><br><br>';
                
                // Generate a sample secure code
                $sampleUsers = $conn->query("SELECT user_id, first_name, last_name, email, user_role FROM users WHERE user_role IN ('student', 'teacher') LIMIT 1");
                if ($sampleUsers && $sampleUsers->num_rows > 0) {
                    $user = $sampleUsers->fetch_assoc();
                    $secureCode = SecureReferralGenerator::generateSecureCode($user['user_id'], $user['user_role'], $user['email']);
                    echo '<strong>Secure Code:</strong> <span class="secure-code">' . $secureCode . '</span> 
                          <button class="btn btn-success" onclick="document.getElementById(\'testCode\').value=\'' . $secureCode . '\'; testCode();">Test</button>';
                }
                echo '</div>';
                ?>
            </div>

            <div class="section">
                <h3>🚀 Migration Actions</h3>
                <p>Choose how you want to migrate users to the secure system:</p>
                
                <a href="?action=migrate_all" class="btn btn-success">
                    🔄 Migrate All Users to Secure Codes
                </a>
                
                <a href="?action=show_samples" class="btn btn-info">
                    👁️ Show Sample Secure Codes
                </a>
                
                <a href="../student/referrals.php" class="btn btn-warning">
                    📊 View Student Dashboard
                </a>
            </div>

            <?php
            // Handle migration actions
            if (isset($_GET['action'])) {
                echo '<div class="section">';
                
                if ($_GET['action'] === 'migrate_all') {
                    echo '<h3>🔄 Migrating Users to Secure Codes</h3>';
                    
                    $users = $conn->query("SELECT user_id, first_name, last_name, email, user_role FROM users WHERE user_role IN ('student', 'teacher')");
                    $migrated = 0;
                    $errors = 0;
                    
                    if ($users) {
                        while ($user = $users->fetch_assoc()) {
                            $secureCode = SecureReferralGenerator::generateAndStore(
                                $user['user_id'], 
                                $user['user_role'], 
                                $user['email'], 
                                $conn
                            );
                            
                            if ($secureCode) {
                                echo "<p>✅ {$user['first_name']} {$user['last_name']} → <span class='secure-code'>$secureCode</span></p>";
                                $migrated++;
                            } else {
                                echo "<p>❌ Failed to migrate {$user['first_name']} {$user['last_name']}</p>";
                                $errors++;
                            }
                        }
                    }
                    
                    echo "<h4>Migration Complete: $migrated users migrated, $errors errors</h4>";
                    
                } elseif ($_GET['action'] === 'show_samples') {
                    echo '<h3>👁️ Sample Secure Codes</h3>';
                    
                    $users = $conn->query("SELECT user_id, first_name, last_name, user_role FROM users WHERE user_role IN ('student', 'teacher') LIMIT 5");
                    if ($users) {
                        while ($user = $users->fetch_assoc()) {
                            $existing = SecureReferralGenerator::getOrCreateSecureCode($user['user_id'], $conn);
                            if ($existing) {
                                echo "<p>{$user['first_name']} {$user['last_name']} ({$user['user_role']}) → <span class='secure-code'>$existing</span></p>";
                            }
                        }
                    }
                }
                
                echo '</div>';
            }
            ?>
        </div>
    </div>

    <script>
    function testCode() {
        const code = document.getElementById('testCode').value.trim();
        const resultDiv = document.getElementById('testResult');
        
        if (!code) {
            resultDiv.innerHTML = '<div class="section error">Please enter a referral code</div>';
            return;
        }
        
        resultDiv.innerHTML = '<div class="section info">🔄 Testing code: <strong>' + code + '</strong>...</div>';
        
        fetch('../auth/validate_referral.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ code: code })
        })
        .then(response => response.text())
        .then(text => {
            try {
                const data = JSON.parse(text);
                
                if (data.success) {
                    const codeType = code.match(/^(TCH|STU)[A-F0-9]{12}$/) ? 'Secure' : 'Legacy';
                    const typeClass = codeType === 'Secure' ? 'secure' : 'legacy';
                    
                    resultDiv.innerHTML = `
                        <div class="section ${typeClass}">
                            <h4>✅ Valid ${codeType} Referral Code</h4>
                            <p><strong>Referrer:</strong> ${data.referrer.name}</p>
                            <p><strong>Role:</strong> ${data.referrer.role}</p>
                            <p><strong>Reward:</strong> ${data.reward.discount} discount</p>
                        </div>`;
                } else {
                    resultDiv.innerHTML = `
                        <div class="section error">
                            <h4>❌ Invalid Code</h4>
                            <p>${data.message}</p>
                        </div>`;
                }
            } catch (e) {
                resultDiv.innerHTML = `
                    <div class="section error">
                        <h4>⚠️ Server Error</h4>
                        <pre>${text}</pre>
                    </div>`;
            }
        })
        .catch(error => {
            resultDiv.innerHTML = `
                <div class="section error">
                    <h4>🔥 Network Error</h4>
                    <p>${error.message}</p>
                </div>`;
        });
    }
    </script>
</body>
</html>
