<?php
/**
 * Panadite Academy - Complete Registration System
 * With Email Verification, Referral Processing, and Role-Based Registration
 */

// Include SEO headers with favicon for registration page
$seo_title = 'Register - Panadite Academy | Create Your Learning Account';
$seo_description = 'Join Panadite Academy today! Create your account to access professional courses, expert instructors, and advance your career with our online learning platform.';
require_once __DIR__ . '/../includes/seo_header.php';

require_once __DIR__ . '/functions.php';

// Include referral processor if it exists
if (file_exists(__DIR__ . '/../includes/referral_processor.php')) {
    require_once __DIR__ . '/../includes/referral_processor.php';
}

// Check if already logged in
if (isLoggedIn()) {
    // Redirect based on user role
    $role = $_SESSION['user_role'];
    if ($role === 'admin') {
        header('Location: /admin/dashboard.php');
    } else if ($role === 'teacher') {
        header('Location: /teacher/dashboard.php');
    } else {
        header('Location: /student/dashboard.php');
    }
    exit();
}

$error = null;
$success = null;
$referral_code = $_GET['ref'] ?? '';
$selected_role = $_GET['role'] ?? 'student';

// Handle registration form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username']);
    $email = trim($_POST['email']);
    $password = $_POST['password'];
    $confirmPassword = $_POST['confirm_password'];
    $firstName = trim($_POST['first_name']);
    $lastName = trim($_POST['last_name']);
    $role = $_POST['role']; // student or teacher
    $referralCode = trim($_POST['referral_code'] ?? '');
    
    // Student personal details
    $title = trim($_POST['title'] ?? '');
    $cellNumber = trim($_POST['cell_number'] ?? '');
    $idNumber = trim($_POST['id_number'] ?? '');
    $suburb = trim($_POST['suburb'] ?? '');
    $province = trim($_POST['province'] ?? '');
    $communicationMethod = trim($_POST['communication_method'] ?? '');
    
    // Learning preferences
    $ownsLaptop = trim($_POST['owns_laptop'] ?? '');
    $trainingPreference = trim($_POST['training_preference'] ?? '');
    $jobFieldInterest = trim($_POST['job_field_interest'] ?? '');
    $jobFieldOther = trim($_POST['job_field_other'] ?? '');
    
    // Background information
    $educationStatus = trim($_POST['education_status'] ?? '');
    $employmentStatus = trim($_POST['employment_status'] ?? '');
    $heardAboutUs = trim($_POST['heard_about_us'] ?? '');
    $referralPersonName = trim($_POST['referral_person_name'] ?? '');
    $bio = trim($_POST['bio'] ?? '');
    
    // Teacher-specific fields
    $courseTitle = trim($_POST['course_title'] ?? '');
    $courseDescription = trim($_POST['course_description'] ?? '');
    $qualifications = trim($_POST['qualifications'] ?? '');
    $experience = trim($_POST['experience'] ?? '');
    $linkedinProfile = trim($_POST['linkedin_profile'] ?? '');
    $portfolioUrl = trim($_POST['portfolio_url'] ?? '');
    
    // Validate passwords match
    if ($password !== $confirmPassword) {
        $error = 'Passwords do not match';
    } elseif (strlen($password) < 8) {
        $error = 'Password must be at least 8 characters long';
    } elseif (strlen($username) < 5) {
        $error = 'Username must be at least 5 characters long';
    } else {
        // Use email verification service instead of direct registration
        {
            // Include email verification service
            require_once '../config/email_verification_service.php';
            
            // Check if username or email already exists
            $existingCheck = checkExistingUser($username, $email);
            if (!$existingCheck['success']) {
                $error = $existingCheck['message'];
            } else {
                // Prepare all additional data for verification storage
                $additionalData = [
                    'title' => $title,
                    'cell_number' => $cellNumber,
                    'id_number' => $idNumber,
                    'suburb' => $suburb,
                    'province' => $province,
                    'communication_method' => $communicationMethod,
                    'owns_laptop' => $ownsLaptop,
                    'training_preference' => $trainingPreference,
                    'job_field_interest' => $jobFieldInterest,
                    'job_field_other' => $jobFieldOther,
                    'education_status' => $educationStatus,
                    'employment_status' => $employmentStatus,
                    'heard_about_us' => $heardAboutUs,
                    'referral_person_name' => $referralPersonName,
                    'bio' => $bio
                ];
                
                // Add teacher-specific data if role is teacher
                if ($role === 'teacher') {
                    $additionalData['course_title'] = $courseTitle;
                    $additionalData['course_description'] = $courseDescription;
                    $additionalData['qualifications'] = $qualifications;
                    $additionalData['experience'] = $experience;
                    $additionalData['linkedin_profile'] = $linkedinProfile;
                    $additionalData['portfolio_url'] = $portfolioUrl;
                }
                
                // Store pending registration for email verification
                $result = storePendingRegistration(
                    $username, 
                    $email, 
                    $password, 
                    $firstName, 
                    $lastName, 
                    $role, 
                    $additionalData,
                    $referralCode
                );
                
                if ($result['success']) {
                    // Send verification email
                    $emailResult = sendVerificationEmail($email, $firstName, $lastName, $result['token']);
                    
                    if ($emailResult['success']) {
                        $success = 'Registration initiated successfully! Please check your email to verify your account. The verification link expires in 10 minutes.';
                        $showVerificationMessage = true;
                    } else {
                        $error = 'Registration successful but failed to send verification email. Please contact support.';
                    }
                } else {
                    $error = $result['message'];
                }
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        body {
            background-color: #f8f9fa;
            position: relative;
            overflow-x: hidden;
        }
        
        /* Floating Education Icons */
        .floating-icon {
            position: absolute;
            opacity: 0.05; /* Reduced opacity to make them much lighter */
            z-index: 0;
            pointer-events: none;
            color: #276BE4;
            filter: drop-shadow(0 3px 5px rgba(39, 107, 228, 0.1)); /* Light shadow */
        }
        
        .register-container {
            max-width: 600px;
            margin: 40px auto;
            padding: 25px;
            background-color: #fff;
            border-radius: 10px;
            box-shadow: 0 0 20px rgba(0,0,0,0.1);
        }
        .logo-container {
            text-align: center;
            margin-bottom: 20px;
        }
        .logo-container img {
            max-width: 180px;
        }
        .register-form .form-control {
            height: 45px;
            border-radius: 5px;
            box-shadow: none;
            margin-bottom: 15px;
        }
        .btn-register {
            background-color: #276BE4;
            border-color: #276BE4;
            color: #fff;
            height: 45px;
            border-radius: 5px;
            font-weight: 600;
        }
        .btn-register:hover {
            background-color: #1c54b8;
        }
        .register-links {
            text-align: center;
            margin-top: 20px;
        }
        .register-links a {
            color: #276BE4;
            text-decoration: none;
        }
        .register-links a:hover {
            text-decoration: underline;
        }
        .tagline {
            text-align: center;
            color: #FFA915;
            font-style: italic;
            margin-bottom: 20px;
        }
        .role-selector {
            display: flex;
            justify-content: center;
            margin-bottom: 20px;
        }
        .role-option {
            flex: 1;
            text-align: center;
            padding: 15px;
            border: 2px solid #e9ecef;
            border-radius: 5px;
            margin: 0 10px;
            cursor: pointer;
            transition: all 0.3s;
        }
        .role-option:hover {
            border-color: #FFA915;
        }
        .role-option.active {
            border-color: #276BE4;
            background-color: rgba(39, 107, 228, 0.1);
        }
        .role-option i {
            font-size: 24px;
            margin-bottom: 10px;
            color: #276BE4;
        }
        
        /* Password strength styles */
        .password-strength-meter {
            height: 5px;
            background-color: #e9ecef;
            margin-top: -10px;
            margin-bottom: 15px;
            border-radius: 3px;
            overflow: hidden;
        }
        
        .password-strength-meter-fill {
            height: 100%;
            border-radius: 3px;
            transition: width 0.3s ease, background-color 0.3s ease;
            width: 0%;
        }
        
        .strength-weak { background-color: #ff4d4d; }
        .strength-medium { background-color: #ffa500; }
        .strength-good { background-color: #2ecc71; }
        
        /* Password toggles */
        .password-container {
            position: relative;
        }
        
        .toggle-password {
            position: absolute;
            right: 10px;
            top: 13px;
            cursor: pointer;
            color: #6c757d;
        }
        
        /* Input field hover effect */
        .register-form .form-control:focus {
            border-color: #276BE4;
            box-shadow: 0 0 5px rgba(39, 107, 228, 0.2);
        }
        
        /* Animation styles */
        .fade-in {
            animation: fadeIn 0.8s;
        }
        
        .slide-up {
            animation: slideInUp 0.5s;
        }
        
        .logo-wrapper {
            position: relative;
            height: 120px;
            display: flex;
            justify-content: center;
            align-items: center;
            overflow: hidden;
        }
        
        .logo-wrapper img {
            max-height: 100%;
            max-width: 100%;
            object-fit: contain;
        }
    </style>
</head>
<body>
    <div class="floating-icon" style="top: 10%; left: 10%; font-size: 40px;"><i class="fas fa-book"></i></div>
    <div class="floating-icon" style="top: 20%; right: 15%; font-size: 35px;"><i class="fas fa-graduation-cap"></i></div>
    <div class="floating-icon" style="top: 60%; left: 5%; font-size: 45px;"><i class="fas fa-pencil-alt"></i></div>
    <div class="floating-icon" style="bottom: 20%; right: 10%; font-size: 50px;"><i class="fas fa-lightbulb"></i></div>
    <div class="floating-icon" style="bottom: 35%; left: 40%; font-size: 30px;"><i class="fas fa-brain"></i></div>
    
    <!-- Include the modern navbar -->
    <?php include '../includes/navbar.php'; ?>
    
    <div class="container">
        <div class="register-container animate__animated animate__fadeIn">
            <div class="logo-container">
                <div class="logo-wrapper">
                    <img src="/assets/images/logo.jpeg" alt="Panadite Academy Logo" class="animate__animated animate__zoomIn">
                </div>
                <p class="tagline animate__animated animate__fadeIn animate__delay-1s">Get ahead of the curve</p>
            </div>
            
            <?php if (!empty($error)): ?>
                <div class="alert alert-danger animate__animated animate__fadeIn" role="alert">
                    <i class="fas fa-exclamation-triangle"></i> <?php echo htmlspecialchars($error); ?>
                </div>
            <?php elseif (!empty($success)): ?>
                <div class="alert alert-success animate__animated animate__fadeIn" role="alert">
                    <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($success); ?>
                    
                    <?php if (isset($showVerificationMessage)): ?>
                        <hr class="my-3">
                        <div class="d-flex align-items-center mb-2">
                            <i class="fas fa-envelope text-primary me-2"></i>
                            <strong>Next Steps:</strong>
                        </div>
                        <ol class="mb-2">
                            <li>Check your email inbox (and spam/junk folder)</li>
                            <li>Click the verification link</li>
                            <li>Your account will be created after verification</li>
                        </ol>
                        <div class="text-muted small">
                            <i class="fas fa-clock me-1"></i>
                            Verification link expires in 10 minutes. Didn't receive it? 
                            <a href="#" onclick="showResendOptions()" class="text-decoration-none">Resend verification email</a>
                        </div>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
            
            <form class="register-form" method="POST" action="">
                <div class="role-selector">
                    <div class="role-option <?php echo $selected_role === 'student' ? 'active' : ''; ?>" data-role="student">
                        <i class="fas fa-user-graduate"></i>
                        <h5>Student</h5>
                        <p class="small">Learn from our courses</p>
                    </div>
                    <div class="role-option <?php echo $selected_role === 'teacher' ? 'active' : ''; ?>" data-role="teacher">
                        <i class="fas fa-chalkboard-teacher"></i>
                        <h5>Facilitator</h5>
                        <p class="small">Create and teach courses</p>
                    </div>
                </div>
                <input type="hidden" name="role" id="role" value="<?php echo htmlspecialchars($selected_role); ?>">
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="first_name" class="form-label">First Name</label>
                            <input type="text" class="form-control" id="first_name" name="first_name" required>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="last_name" class="form-label">Last Name</label>
                            <input type="text" class="form-control" id="last_name" name="last_name" required>
                        </div>
                    </div>
                </div>
                
                <div class="mb-3">
                    <label for="username" class="form-label">Username</label>
                    <input type="text" class="form-control" id="username" name="username" required>
                </div>
                
                <div class="mb-3">
                    <label for="email" class="form-label">Email</label>
                    <input type="email" class="form-control" id="email" name="email" value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>" required>
                </div>
                
                <!-- Referral Code Input -->
                <div class="mb-3">
                    <label for="referral_code" class="form-label">
                        <i class="fas fa-gift me-2"></i>Referral Code <span class="text-muted">(Optional)</span>
                    </label>
                    <input type="text" class="form-control" id="referral_code" name="referral_code" 
                           value="<?php echo htmlspecialchars($referral_code ?: ($_POST['referral_code'] ?? '')); ?>" 
                           placeholder="Enter referral code to get discounts">
                    <div id="referral_loading" style="display: none;" class="mt-2">
                        <i class="fas fa-spinner fa-spin me-2"></i><small class="text-muted">Validating referral code...</small>
                    </div>
                    <div id="referral_info" style="display: none;" class="mt-2 p-3 bg-success bg-opacity-10 border border-success rounded">
                        <div class="d-flex align-items-center">
                            <i class="fas fa-check-circle text-success me-2"></i>
                            <div>
                                <strong id="referrer_name"></strong> <span id="referrer_role" class="badge bg-primary ms-1"></span>
                                <div class="discount-message"><small class="text-success">✅ Get <span id="discount_amount"></span> discount on your first course!</small></div>
                                <div><small class="text-success">✅ Valid referral code from this user!</small></div>
                            </div>
                        </div>
                    </div>
                    <div id="referral_error" style="display: none;" class="mt-2">
                        <small class="text-danger"><i class="fas fa-exclamation-triangle me-1"></i><span id="referral_error_message"></span></small>
                    </div>
                    <small id="referral_help" class="text-muted">Get 5% discount on your first course with a valid referral code!</small>
                </div>
                
                <div class="mb-3">
                    <label for="password" class="form-label">Password</label>
                    <div class="password-container">
                        <input type="password" class="form-control" id="password" name="password" required>
                        <span class="toggle-password" data-target="password">
                            <i class="far fa-eye"></i>
                        </span>
                    </div>
                    <div class="password-strength-meter mt-2">
                        <div class="password-strength-meter-fill"></div>
                    </div>
                    <small id="passwordHelp" class="form-text text-muted">Use at least 8 characters with letters, numbers and symbols</small>
                </div>
                
                <div class="mb-3">
                    <label for="confirm_password" class="form-label">Confirm Password</label>
                    <div class="password-container">
                        <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                        <span class="toggle-password" data-target="confirm_password">
                            <i class="far fa-eye"></i>
                        </span>
                    </div>
                    <small id="passwordMatch" class="form-text"></small>
                </div>
                
                <!-- Student Information Section - shown for all users -->
                <div class="border rounded p-3 mb-3 bg-light">
                    <h5 class="mb-3"><i class="fas fa-user-detail me-2"></i>Personal Details</h5>
                    
                    <div class="row">
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label for="title" class="form-label">Title</label>
                                <select class="form-control" id="title" name="title">
                                    <option value="">Select Title</option>
                                    <option value="Mr" <?php echo ($_POST['title'] ?? '') === 'Mr' ? 'selected' : ''; ?>>Mr</option>
                                    <option value="Ms" <?php echo ($_POST['title'] ?? '') === 'Ms' ? 'selected' : ''; ?>>Ms</option>
                                    <option value="Mrs" <?php echo ($_POST['title'] ?? '') === 'Mrs' ? 'selected' : ''; ?>>Mrs</option>
                                    <option value="Dr" <?php echo ($_POST['title'] ?? '') === 'Dr' ? 'selected' : ''; ?>>Dr</option>
                                    <option value="Prof" <?php echo ($_POST['title'] ?? '') === 'Prof' ? 'selected' : ''; ?>>Prof</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label for="cell_number" class="form-label">Cell Number *</label>
                                <input type="tel" class="form-control" id="cell_number" name="cell_number" 
                                       value="<?php echo htmlspecialchars($_POST['cell_number'] ?? ''); ?>"
                                       placeholder="e.g., 0823456789" required>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label for="id_number" class="form-label">ID Number *</label>
                                <input type="text" class="form-control" id="id_number" name="id_number" 
                                       value="<?php echo htmlspecialchars($_POST['id_number'] ?? ''); ?>"
                                       placeholder="e.g., 9001015800001" required>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="suburb" class="form-label">Suburb</label>
                                <input type="text" class="form-control" id="suburb" name="suburb" 
                                       value="<?php echo htmlspecialchars($_POST['suburb'] ?? ''); ?>"
                                       placeholder="e.g., Pretoria Central">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="province" class="form-label">Province</label>
                                <select class="form-control" id="province" name="province">
                                    <option value="">Select Province</option>
                                    <option value="Eastern Cape" <?php echo ($_POST['province'] ?? '') === 'Eastern Cape' ? 'selected' : ''; ?>>Eastern Cape</option>
                                    <option value="Free State" <?php echo ($_POST['province'] ?? '') === 'Free State' ? 'selected' : ''; ?>>Free State</option>
                                    <option value="Gauteng" <?php echo ($_POST['province'] ?? '') === 'Gauteng' ? 'selected' : ''; ?>>Gauteng</option>
                                    <option value="KwaZulu-Natal" <?php echo ($_POST['province'] ?? '') === 'KwaZulu-Natal' ? 'selected' : ''; ?>>KwaZulu-Natal</option>
                                    <option value="Limpopo" <?php echo ($_POST['province'] ?? '') === 'Limpopo' ? 'selected' : ''; ?>>Limpopo</option>
                                    <option value="Mpumalanga" <?php echo ($_POST['province'] ?? '') === 'Mpumalanga' ? 'selected' : ''; ?>>Mpumalanga</option>
                                    <option value="Northern Cape" <?php echo ($_POST['province'] ?? '') === 'Northern Cape' ? 'selected' : ''; ?>>Northern Cape</option>
                                    <option value="North West" <?php echo ($_POST['province'] ?? '') === 'North West' ? 'selected' : ''; ?>>North West</option>
                                    <option value="Western Cape" <?php echo ($_POST['province'] ?? '') === 'Western Cape' ? 'selected' : ''; ?>>Western Cape</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="communication_method" class="form-label">Preferred Communication Method</label>
                        <select class="form-control" id="communication_method" name="communication_method">
                            <option value="">Select Communication Method</option>
                            <option value="Email" <?php echo ($_POST['communication_method'] ?? '') === 'Email' ? 'selected' : ''; ?>>Email</option>
                            <option value="WhatsApp" <?php echo ($_POST['communication_method'] ?? '') === 'WhatsApp' ? 'selected' : ''; ?>>WhatsApp</option>
                        </select>
                    </div>
                </div>
                
                <!-- Learning Preferences Section -->
                <div class="border rounded p-3 mb-3 bg-light">
                    <h5 class="mb-3"><i class="fas fa-laptop me-2"></i>Learning Preferences</h5>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="owns_laptop" class="form-label">Do you own a Laptop/PC? *</label>
                                <select class="form-control" id="owns_laptop" name="owns_laptop" required>
                                    <option value="">Select Option</option>
                                    <option value="Yes" <?php echo ($_POST['owns_laptop'] ?? '') === 'Yes' ? 'selected' : ''; ?>>Yes</option>
                                    <option value="No" <?php echo ($_POST['owns_laptop'] ?? '') === 'No' ? 'selected' : ''; ?>>No</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="training_preference" class="form-label">Training Preference</label>
                                <select class="form-control" id="training_preference" name="training_preference">
                                    <option value="">Select Preference</option>
                                    <option value="Online" <?php echo ($_POST['training_preference'] ?? '') === 'Online' ? 'selected' : ''; ?>>Online</option>
                                    <option value="Onsite" <?php echo ($_POST['training_preference'] ?? '') === 'Onsite' ? 'selected' : ''; ?>>Onsite</option>
                                    <option value="Hybrid" <?php echo ($_POST['training_preference'] ?? '') === 'Hybrid' ? 'selected' : ''; ?>>Hybrid</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="job_field_interest" class="form-label">Job Field Interest</label>
                        <select class="form-control" id="job_field_interest" name="job_field_interest">
                            <option value="">Select Job Field</option>
                            <option value="Data Capturing" <?php echo ($_POST['job_field_interest'] ?? '') === 'Data Capturing' ? 'selected' : ''; ?>>Data Capturing</option>
                            <option value="Administrative Assistant" <?php echo ($_POST['job_field_interest'] ?? '') === 'Administrative Assistant' ? 'selected' : ''; ?>>Administrative Assistant</option>
                            <option value="Customer Service" <?php echo ($_POST['job_field_interest'] ?? '') === 'Customer Service' ? 'selected' : ''; ?>>Customer Service</option>
                            <option value="IT Support" <?php echo ($_POST['job_field_interest'] ?? '') === 'IT Support' ? 'selected' : ''; ?>>IT Support</option>
                            <option value="Digital Marketing" <?php echo ($_POST['job_field_interest'] ?? '') === 'Digital Marketing' ? 'selected' : ''; ?>>Digital Marketing</option>
                            <option value="Web Development" <?php echo ($_POST['job_field_interest'] ?? '') === 'Web Development' ? 'selected' : ''; ?>>Web Development</option>
                            <option value="Graphic Design" <?php echo ($_POST['job_field_interest'] ?? '') === 'Graphic Design' ? 'selected' : ''; ?>>Graphic Design</option>
                            <option value="Project Management" <?php echo ($_POST['job_field_interest'] ?? '') === 'Project Management' ? 'selected' : ''; ?>>Project Management</option>
                            <option value="Other" <?php echo ($_POST['job_field_interest'] ?? '') === 'Other' ? 'selected' : ''; ?>>Other</option>
                        </select>
                    </div>
                    
                    <div class="mb-3" id="job_field_other_group" style="display: none;">
                        <label for="job_field_other" class="form-label">Please specify other job field</label>
                        <input type="text" class="form-control" id="job_field_other" name="job_field_other" 
                               value="<?php echo htmlspecialchars($_POST['job_field_other'] ?? ''); ?>"
                               placeholder="Specify your job field interest">
                    </div>
                </div>
                
                <!-- Background Information Section -->
                <div class="border rounded p-3 mb-3 bg-light">
                    <h5 class="mb-3"><i class="fas fa-graduation-cap me-2"></i>Background Information</h5>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="education_status" class="form-label">Education Status</label>
                                <select class="form-control" id="education_status" name="education_status">
                                    <option value="">Select Education Level</option>
                                    <option value="High School" <?php echo ($_POST['education_status'] ?? '') === 'High School' ? 'selected' : ''; ?>>High School</option>
                                    <option value="University Degree/Diploma" <?php echo ($_POST['education_status'] ?? '') === 'University Degree/Diploma' ? 'selected' : ''; ?>>University Degree/Diploma</option>
                                    <option value="Postgraduate" <?php echo ($_POST['education_status'] ?? '') === 'Postgraduate' ? 'selected' : ''; ?>>Postgraduate</option>
                                    <option value="Professional Certification" <?php echo ($_POST['education_status'] ?? '') === 'Professional Certification' ? 'selected' : ''; ?>>Professional Certification</option>
                                    <option value="Other" <?php echo ($_POST['education_status'] ?? '') === 'Other' ? 'selected' : ''; ?>>Other</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="employment_status" class="form-label">Employment Status</label>
                                <select class="form-control" id="employment_status" name="employment_status">
                                    <option value="">Select Employment Status</option>
                                    <option value="Employed" <?php echo ($_POST['employment_status'] ?? '') === 'Employed' ? 'selected' : ''; ?>>Employed</option>
                                    <option value="Unemployed" <?php echo ($_POST['employment_status'] ?? '') === 'Unemployed' ? 'selected' : ''; ?>>Unemployed</option>
                                    <option value="Self-employed" <?php echo ($_POST['employment_status'] ?? '') === 'Self-employed' ? 'selected' : ''; ?>>Self-employed</option>
                                    <option value="Student" <?php echo ($_POST['employment_status'] ?? '') === 'Student' ? 'selected' : ''; ?>>Student</option>
                                    <option value="Retired" <?php echo ($_POST['employment_status'] ?? '') === 'Retired' ? 'selected' : ''; ?>>Retired</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="heard_about_us" class="form-label">How did you hear about us?</label>
                                <select class="form-control" id="heard_about_us" name="heard_about_us">
                                    <option value="">Select Option</option>
                                    <option value="Social Media" <?php echo ($_POST['heard_about_us'] ?? '') === 'Social Media' ? 'selected' : ''; ?>>Social Media</option>
                                    <option value="Google Search" <?php echo ($_POST['heard_about_us'] ?? '') === 'Google Search' ? 'selected' : ''; ?>>Google Search</option>
                                    <option value="Friend/Family" <?php echo ($_POST['heard_about_us'] ?? '') === 'Friend/Family' ? 'selected' : ''; ?>>Friend/Family</option>
                                    <option value="Advertisement" <?php echo ($_POST['heard_about_us'] ?? '') === 'Advertisement' ? 'selected' : ''; ?>>Advertisement</option>
                                    <option value="Referral" <?php echo ($_POST['heard_about_us'] ?? '') === 'Referral' ? 'selected' : ''; ?>>Referral</option>
                                    <option value="Other" <?php echo ($_POST['heard_about_us'] ?? '') === 'Other' ? 'selected' : ''; ?>>Other</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="referral_person_name" class="form-label">Referred by (Name) <span class="text-muted">(Optional)</span></label>
                                <input type="text" class="form-control" id="referral_person_name" name="referral_person_name" 
                                       value="<?php echo htmlspecialchars($_POST['referral_person_name'] ?? ''); ?>"
                                       placeholder="Name of person who referred you">
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="bio" class="form-label">Bio <span class="text-muted">(Optional)</span></label>
                        <textarea class="form-control" id="bio" name="bio" rows="3"
                                  placeholder="Tell us about yourself, your interests, and your learning goals..."><?php echo htmlspecialchars($_POST['bio'] ?? ''); ?></textarea>
                        <small class="text-muted">This helps us provide better course recommendations.</small>
                    </div>
                </div>
                
                <!-- Facilitator specific fields - hidden by default -->
                <div id="facilitator-fields" style="display: <?php echo $selected_role === 'teacher' ? 'block' : 'none'; ?>;" class="border rounded p-3 mb-3 bg-light">
                    <h5 class="mb-3"><i class="fas fa-chalkboard-teacher me-2"></i>Facilitator Information</h5>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="course_title" class="form-label">Proposed Course Title *</label>
                                <input type="text" class="form-control" id="course_title" name="course_title" 
                                       value="<?php echo htmlspecialchars($_POST['course_title'] ?? ''); ?>"
                                       placeholder="What course do you want to teach?">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="experience" class="form-label">Years of Experience *</label>
                                <select class="form-control" id="experience" name="experience">
                                    <option value="">Select experience level</option>
                                    <option value="1-2 years" <?php echo ($_POST['experience'] ?? '') === '1-2 years' ? 'selected' : ''; ?>>1-2 years</option>
                                    <option value="3-5 years" <?php echo ($_POST['experience'] ?? '') === '3-5 years' ? 'selected' : ''; ?>>3-5 years</option>
                                    <option value="5-10 years" <?php echo ($_POST['experience'] ?? '') === '5-10 years' ? 'selected' : ''; ?>>5-10 years</option>
                                    <option value="10+ years" <?php echo ($_POST['experience'] ?? '') === '10+ years' ? 'selected' : ''; ?>>10+ years</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="course_description" class="form-label">Course Description *</label>
                        <textarea class="form-control" id="course_description" name="course_description" rows="3" 
                                  placeholder="Brief description of your proposed course (min 50 characters)"><?php echo htmlspecialchars($_POST['course_description'] ?? ''); ?></textarea>
                        <small class="text-muted">Describe what students will learn and the course objectives.</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="qualifications" class="form-label">Your Qualifications *</label>
                        <textarea class="form-control" id="qualifications" name="qualifications" rows="3" 
                                  placeholder="List your relevant qualifications, certifications, and expertise"><?php echo htmlspecialchars($_POST['qualifications'] ?? ''); ?></textarea>
                        <small class="text-muted">Include degrees, certifications, professional experience, etc.</small>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="linkedin_profile" class="form-label">
                                    <i class="fab fa-linkedin me-2"></i>LinkedIn Profile
                                </label>
                                <input type="url" class="form-control" id="linkedin_profile" name="linkedin_profile" 
                                       value="<?php echo htmlspecialchars($_POST['linkedin_profile'] ?? ''); ?>"
                                       placeholder="https://linkedin.com/in/yourprofile">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="portfolio_url" class="form-label">
                                    <i class="fas fa-globe me-2"></i>Portfolio/Website
                                </label>
                                <input type="url" class="form-control" id="portfolio_url" name="portfolio_url" 
                                       value="<?php echo htmlspecialchars($_POST['portfolio_url'] ?? ''); ?>"
                                       placeholder="https://yourwebsite.com">
                            </div>
                        </div>
                    </div>
                    
                    <div class="alert alert-info" role="alert">
                        <i class="fas fa-info-circle me-2"></i> 
                        <strong>Review Process:</strong> All facilitator applications require approval from our admin team. 
                        You'll receive an email notification once your application is reviewed (usually within 24-48 hours).
                    </div>
                </div>
                
                <!-- Terms and Conditions -->
                <div class="mb-3">
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" id="terms_accepted" name="terms_accepted" required>
                        <label class="form-check-label" for="terms_accepted">
                            I agree to the <a href="https://panaditeacademy.com/terms-of-use.php" target="_blank">Terms and Conditions</a> 
                            and <a href="https://panaditeacademy.com/privacy-policy.php" target="_blank">Privacy Policy</a> *
                        </label>
                    </div>
                </div>
                
                <!-- Newsletter Subscription -->
                <div class="mb-3">
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" id="newsletter_subscription" name="newsletter_subscription" checked>
                        <label class="form-check-label" for="newsletter_subscription">
                            <i class="fas fa-envelope me-2"></i>Subscribe to our newsletter for course updates and special offers
                        </label>
                    </div>
                </div>
                
                <div class="d-grid gap-2">
                    <button type="submit" class="btn btn-register">
                        <i class="fas fa-user-plus me-2"></i>
                        <span id="register-btn-text">Create Account & Send Verification Email</span>
                    </button>
                </div>
                
                <div class="mt-3 text-center">
                    <small class="text-muted">
                        <i class="fas fa-shield-alt me-2"></i>
                        We'll send you a verification email to confirm your account. Please check your spam folder if you don't see it.
                    </small>
                </div>
            </form>
            
            <div class="register-links">
                <a href="login.php">Already have an account? Login</a><br>
                <a href="/index.php">Back to Home</a>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/zxcvbn/4.4.2/zxcvbn.js"></script>
    <script>
        $(document).ready(function() {
            // Animate floating icons
            animateFloatingIcons();
            
            // Handle role selection with animation
            $('.role-option').click(function() {
                $('.role-option').removeClass('active');
                $(this).addClass('active animate__animated animate__pulse');
                const role = $(this).data('role');
                $('#role').val(role);
                
                // Update register button text based on role
                if (role === 'teacher') {
                    $('#register-btn-text').text('Apply as Facilitator & Send Verification Email');
                    $('#facilitator-fields').slideDown(300);
                    // Make facilitator fields required
                    $('#course_title, #course_description, #qualifications, #experience').attr('required', true);
                } else {
                    $('#register-btn-text').text('Create Account & Send Verification Email');
                    $('#facilitator-fields').slideUp(300);
                    // Remove required attribute from facilitator fields
                    $('#course_title, #course_description, #qualifications, #experience').removeAttr('required');
                }
            });
            
            // Dynamic referral code validation with AJAX
            var referralTimeout;
            $('#referral_code').on('input', function() {
                var code = $(this).val().trim();
                
                // Clear previous timeout
                clearTimeout(referralTimeout);
                
                // Hide all feedback
                $('#referral_loading, #referral_info, #referral_error').hide();
                $(this).removeClass('is-valid is-invalid');
                
                if (code.length >= 3) {
                    // Show loading state
                    $('#referral_loading').show();
                    
                    // Debounce AJAX call
                    referralTimeout = setTimeout(function() {
                        validateReferralCode(code);
                    }, 500);
                } else if (code.length > 0) {
                    $('#referral_error_message').text('Referral code must be at least 3 characters');
                    $('#referral_error').show();
                    $(this).addClass('is-invalid');
                } else {
                    // Empty code - show default help text
                    $('#referral_help').show();
                }
            });
            
            // Function to validate referral code via AJAX
            function validateReferralCode(code) {
                $.ajax({
                    url: '/auth/validate_referral.php',
                    method: 'POST',
                    data: { code: code },
                    dataType: 'json',
                    timeout: 5000,
                    success: function(response) {
                        $('#referral_loading').hide();
                        
                        if (response.success) {
                            // Show referrer information
                            $('#referrer_name').text(response.referrer.name);
                            $('#referrer_role').text(response.referrer.role).removeClass().addClass('badge bg-' + (response.referrer.role.toLowerCase() === 'teacher' ? 'warning' : 'primary'));
                            
                            // Only show discount if there's an actual discount amount
                            var discountText = response.reward.discount || '0%';
                            var showDiscount = discountText !== '0%' && discountText !== '$0.00' && discountText !== '0';
                            
                            if (showDiscount) {
                                $('#discount_amount').text(discountText);
                                $('.discount-message').show();
                            } else {
                                $('.discount-message').hide();
                            }
                            
                            $('#referral_info').show();
                            $('#referral_code').removeClass('is-invalid').addClass('is-valid');
                            
                            // Update help text
                            $('#referral_help').hide();
                        } else {
                            // Show error
                            $('#referral_error_message').text(response.message || 'Invalid referral code');
                            $('#referral_error').show();
                            $('#referral_code').removeClass('is-valid').addClass('is-invalid');
                        }
                    },
                    error: function(xhr, status, error) {
                        $('#referral_loading').hide();
                        $('#referral_error_message').text('Unable to validate referral code. Please try again.');
                        $('#referral_error').show();
                        $('#referral_code').removeClass('is-valid').addClass('is-invalid');
                        
                        console.error('Referral validation error:', error);
                    }
                });
            }
            
            // Password strength meter
            $('#password').on('keyup', function() {
                var password = $(this).val();
                var result = zxcvbn(password);
                var $meter = $('.password-strength-meter-fill');
                
                // Update the strength meter
                $meter.removeClass('strength-weak strength-medium strength-good');
                
                if (password.length === 0) {
                    $meter.css('width', '0%');
                } else if (result.score < 2) {
                    $meter.addClass('strength-weak').css('width', '33%');
                } else if (result.score < 4) {
                    $meter.addClass('strength-medium').css('width', '66%');
                } else {
                    $meter.addClass('strength-good').css('width', '100%');
                }
            });
            
            // Password matching validation
            $('#confirm_password').on('keyup', function() {
                var password = $('#password').val();
                var confirmPassword = $(this).val();
                var $feedback = $('#passwordMatch');
                
                if (confirmPassword.length > 0) {
                    if (password === confirmPassword) {
                        $feedback.text('Passwords match').removeClass('text-danger').addClass('text-success');
                    } else {
                        $feedback.text('Passwords do not match').removeClass('text-success').addClass('text-danger');
                    }
                } else {
                    $feedback.text('');
                }
            });
            
            // Toggle password visibility
            $('.toggle-password').click(function() {
                var targetId = $(this).data('target');
                var input = $('#' + targetId);
                var icon = $(this).find('i');
                
                if (input.attr('type') === 'password') {
                    input.attr('type', 'text');
                    icon.removeClass('fa-eye').addClass('fa-eye-slash');
                } else {
                    input.attr('type', 'password');
                    icon.removeClass('fa-eye-slash').addClass('fa-eye');
                }
            });
            
            // Teacher field validation
            $('#course_description').on('input', function() {
                var text = $(this).val();
                var minLength = 50;
                var remaining = minLength - text.length;
                
                if (text.length > 0 && text.length < minLength) {
                    $(this).removeClass('is-valid').addClass('is-invalid');
                    $(this).next('.form-text').text('Need ' + remaining + ' more characters for a good description.');
                } else if (text.length >= minLength) {
                    $(this).removeClass('is-invalid').addClass('is-valid');
                    $(this).next('.form-text').text('Describe what students will learn and the course objectives.');
                } else {
                    $(this).removeClass('is-valid is-invalid');
                    $(this).next('.form-text').text('Describe what students will learn and the course objectives.');
                }
            });
            
            // LinkedIn profile validation
            $('#linkedin_profile').on('blur', function() {
                var url = $(this).val();
                if (url && !url.includes('linkedin.com')) {
                    $(this).addClass('is-invalid');
                    $(this).after('<div class="invalid-feedback">Please enter a valid LinkedIn profile URL</div>');
                } else if (url) {
                    $(this).removeClass('is-invalid').addClass('is-valid');
                    $(this).next('.invalid-feedback').remove();
                } else {
                    $(this).removeClass('is-valid is-invalid');
                    $(this).next('.invalid-feedback').remove();
                }
            });
            
            // Form validation before submit
            $('.register-form').on('submit', function(e) {
                var password = $('#password').val();
                var confirmPassword = $('#confirm_password').val();
                var role = $('#role').val();
                var isValid = true;
                
                // Password validation
                if (password !== confirmPassword) {
                    e.preventDefault();
                    $('#passwordMatch').text('Passwords do not match').removeClass('text-success').addClass('text-danger');
                    $('#confirm_password').focus();
                    return false;
                }
                
                if (password.length < 8) {
                    e.preventDefault();
                    $('#passwordHelp').text('Password must be at least 8 characters long').addClass('text-danger');
                    $('#password').focus();
                    return false;
                }
                
                // Teacher-specific validation
                if (role === 'teacher') {
                    var courseTitle = $('#course_title').val().trim();
                    var courseDescription = $('#course_description').val().trim();
                    var qualifications = $('#qualifications').val().trim();
                    var experience = $('#experience').val();
                    
                    if (!courseTitle || courseTitle.length < 5) {
                        e.preventDefault();
                        $('#course_title').focus();
                        alert('Please enter a course title with at least 5 characters.');
                        return false;
                    }
                    
                    if (!courseDescription || courseDescription.length < 50) {
                        e.preventDefault();
                        $('#course_description').focus();
                        alert('Please provide a detailed course description with at least 50 characters.');
                        return false;
                    }
                    
                    if (!qualifications || qualifications.length < 20) {
                        e.preventDefault();
                        $('#qualifications').focus();
                        alert('Please provide detailed information about your qualifications (at least 20 characters).');
                        return false;
                    }
                    
                    if (!experience) {
                        e.preventDefault();
                        $('#experience').focus();
                        alert('Please select your experience level.');
                        return false;
                    }
                }
                
                // Terms acceptance validation
                if (!$('#terms_accepted').is(':checked')) {
                    e.preventDefault();
                    alert('Please accept the Terms and Conditions to continue.');
                    $('#terms_accepted').focus();
                    return false;
                }
                
                // Add loading state to submit button
                $('.btn-register').html('<i class="fas fa-spinner fa-spin me-2"></i>Processing Registration...').prop('disabled', true);
                
                return true;
            });
            
            // Enhanced username validation
            $('#username').on('blur', function() {
                var username = $(this).val().trim();
                var $feedback = $(this).next('.invalid-feedback, .valid-feedback');
                
                $feedback.remove();
                
                if (username && username.length < 5) {
                    $(this).removeClass('is-valid').addClass('is-invalid');
                    $(this).after('<div class="invalid-feedback">Username must be at least 5 characters long</div>');
                } else if (username && !/^[a-zA-Z0-9._-]+$/.test(username)) {
                    $(this).removeClass('is-valid').addClass('is-invalid');
                    $(this).after('<div class="invalid-feedback">Username can only contain letters, numbers, dots, underscores, and hyphens</div>');
                } else if (username) {
                    $(this).removeClass('is-invalid').addClass('is-valid');
                    $(this).after('<div class="valid-feedback">Username looks good!</div>');
                } else {
                    $(this).removeClass('is-valid is-invalid');
                }
            });
            
            // Email validation enhancement
            $('#email').on('blur', function() {
                var email = $(this).val().trim();
                var $feedback = $(this).next('.invalid-feedback, .valid-feedback');
                
                $feedback.remove();
                
                if (email) {
                    var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                    if (emailRegex.test(email)) {
                        $(this).removeClass('is-invalid').addClass('is-valid');
                        $(this).after('<div class="valid-feedback">Email format is valid!</div>');
                    } else {
                        $(this).removeClass('is-valid').addClass('is-invalid');
                        $(this).after('<div class="invalid-feedback">Please enter a valid email address</div>');
                    }
                }
            });
            
            // Auto-fill referral code from URL and validate immediately
            var urlParams = new URLSearchParams(window.location.search);
            var refCode = urlParams.get('ref');
            if (refCode && !$('#referral_code').val()) {
                $('#referral_code').val(refCode).trigger('input');
            }
            
            // Also check on page load if there's already a value
            if ($('#referral_code').val().trim()) {
                $('#referral_code').trigger('input');
            }
            
            // Initialize form based on URL parameters
            if (window.location.search.includes('role=')) {
                var urlParams = new URLSearchParams(window.location.search);
                var roleParam = urlParams.get('role');
                if (roleParam === 'teacher' || roleParam === 'student') {
                    $('.role-option[data-role="' + roleParam + '"]').click();
                }
            }
            
            // Function to show resend options
            window.showResendOptions = function() {
                alert('To resend verification email, please try registering again or contact support if you continue to have issues.');
            };
        });
        
        // Initialize role-based fields on page load
        $(window).on('load', function() {
            var currentRole = $('#role').val();
            if (currentRole === 'teacher') {
                $('#facilitator-fields').show();
                $('#course_title, #course_description, #qualifications, #experience').attr('required', true);
                $('#register-btn-text').text('Apply as Facilitator & Send Verification Email');
            }
        });
        
        // Prevent form resubmission on page refresh
        if (window.history.replaceState) {
            window.history.replaceState(null, null, window.location.href);
        }
        
        // Function to animate the floating icons
        function animateFloatingIcons() {
            $('.floating-icon').each(function(index) {
                var icon = $(this);
                // Set random initial positions
                var startX = Math.random() * 10 - 5; // -5 to +5
                var startY = Math.random() * 10 - 5; // -5 to +5
                
                // Animate with random duration between 5-10 seconds
                var duration = 5000 + Math.random() * 5000;
                
                // Function to create floating animation
                function floatIcon() {
                    // Generate new random positions
                    var moveX = Math.random() * 20 - 10; // -10 to +10
                    var moveY = Math.random() * 20 - 10; // -10 to +10
                    
                    icon.animate({
                        transform: 'translate(' + moveX + 'px, ' + moveY + 'px)'
                    }, duration, 'linear', function() {
                        floatIcon(); // Infinite loop
                    });
                }
                
                // Start with a delay based on index
                setTimeout(function() {
                    icon.css('transform', 'translate(' + startX + 'px, ' + startY + 'px)');
                    floatIcon();
                }, index * 500);
            });
        }
    </script>
</body>
</html>
