<?php
/**
 * Panadite Academy - Password Reset Table Setup
 * Creates the password_reset_tokens table automatically
 */

// Start session and include necessary files
session_start();
require_once __DIR__ . '/../includes/site_config.php';
require_once __DIR__ . '/../includes/db_connection.php';

// Initialize variables
$success = '';
$error = '';
$table_exists = false;

// Check if user is admin (basic security check)
if (!isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'admin') {
    $error = 'Admin access required to run database setup.';
} else {
    // Check if table already exists
    $check_table = "SHOW TABLES LIKE 'password_reset_tokens'";
    $result = $conn->query($check_table);
    
    if ($result && $result->num_rows > 0) {
        $table_exists = true;
        $success = 'Password reset tokens table already exists!';
    } else {
        // Create the password_reset_tokens table
        $create_table_sql = "
        CREATE TABLE `password_reset_tokens` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `user_id` int(11) NOT NULL,
            `token` varchar(255) NOT NULL,
            `expires_at` datetime NOT NULL,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            `used` tinyint(1) DEFAULT 0,
            PRIMARY KEY (`id`),
            UNIQUE KEY `user_id` (`user_id`),
            UNIQUE KEY `token` (`token`),
            KEY `expires_at` (`expires_at`),
            FOREIGN KEY (`user_id`) REFERENCES `users` (`user_id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ";
        
        if ($conn->query($create_table_sql) === TRUE) {
            $success = 'Password reset tokens table created successfully!';
            
            // Also clean up any existing expired tokens (if any)
            $cleanup_sql = "DELETE FROM password_reset_tokens WHERE expires_at < NOW() OR used = 1";
            $conn->query($cleanup_sql);
            
        } else {
            $error = 'Error creating table: ' . $conn->error;
        }
    }
}

// Get table information if it exists
$table_info = [];
if ($table_exists || !empty($success)) {
    $info_sql = "DESCRIBE password_reset_tokens";
    $info_result = $conn->query($info_sql);
    if ($info_result) {
        while ($row = $info_result->fetch_assoc()) {
            $table_info[] = $row;
        }
    }
    
    // Get table count
    $count_result = $conn->query("SELECT COUNT(*) as total FROM password_reset_tokens");
    $total_tokens = $count_result ? $count_result->fetch_assoc()['total'] : 0;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Database Setup - Password Reset Table</title>
    
    <!-- Bootstrap 5.2.3 -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        :root {
            --primary-color: #276BE4;
            --secondary-color: #FFA915;
            --success-color: #28a745;
            --danger-color: #dc3545;
        }

        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        .setup-container {
            max-width: 800px;
            margin: 50px auto;
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.15);
            overflow: hidden;
            position: relative;
        }

        .setup-container::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--primary-color), var(--secondary-color));
        }

        .header {
            background: linear-gradient(135deg, var(--primary-color), #1e5bcc);
            color: white;
            padding: 30px;
            text-align: center;
            position: relative;
        }

        .header h1 {
            margin: 0;
            font-size: 28px;
            font-weight: 700;
        }

        .header p {
            margin: 10px 0 0 0;
            opacity: 0.9;
            font-size: 16px;
        }

        .content {
            padding: 40px;
        }

        .alert {
            border-radius: 12px;
            border: none;
            padding: 20px;
            margin-bottom: 30px;
            font-weight: 500;
        }

        .alert-success {
            background: linear-gradient(135deg, #d4edda, #c3e6cb);
            color: #155724;
            border-left: 4px solid var(--success-color);
        }

        .alert-danger {
            background: linear-gradient(135deg, #f8d7da, #f5c6cb);
            color: #721c24;
            border-left: 4px solid var(--danger-color);
        }

        .table-info {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 12px;
            overflow: hidden;
            margin-top: 30px;
        }

        .table-info-header {
            background: linear-gradient(135deg, #e9ecef, #dee2e6);
            padding: 15px 20px;
            border-bottom: 1px solid #dee2e6;
        }

        .table-info-header h5 {
            margin: 0;
            color: var(--primary-color);
            font-weight: 600;
        }

        .table {
            margin-bottom: 0;
        }

        .table th {
            background-color: #f1f3f4;
            color: #333;
            font-weight: 600;
            border-bottom: 2px solid #dee2e6;
        }

        .stats-card {
            background: linear-gradient(135deg, #e3f2fd, #bbdefb);
            border: 1px solid #2196f3;
            border-radius: 12px;
            padding: 20px;
            text-align: center;
            margin-top: 20px;
        }

        .stats-card h3 {
            color: #1565c0;
            font-size: 36px;
            font-weight: 700;
            margin-bottom: 5px;
        }

        .stats-card p {
            color: #1976d2;
            font-size: 16px;
            margin: 0;
        }

        .btn-custom {
            background: linear-gradient(135deg, var(--primary-color), #1e5bcc);
            border: none;
            border-radius: 12px;
            color: white;
            font-weight: 600;
            padding: 12px 24px;
            transition: all 0.3s ease;
            text-decoration: none;
        }

        .btn-custom:hover {
            background: linear-gradient(135deg, #1e5bcc, #1752b8);
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(39, 107, 228, 0.3);
            color: white;
            text-decoration: none;
        }

        .icon-status {
            font-size: 48px;
            margin-bottom: 20px;
        }

        .status-success {
            color: var(--success-color);
        }

        .status-error {
            color: var(--danger-color);
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="setup-container position-relative">
            <div class="header">
                <h1><i class="fas fa-database me-3"></i>Database Setup</h1>
                <p>Password Reset Table Creation</p>
            </div>
            
            <div class="content">
                <?php if (!isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'admin'): ?>
                    <!-- Access Denied -->
                    <div class="text-center">
                        <div class="icon-status status-error">
                            <i class="fas fa-lock"></i>
                        </div>
                        <h3>Access Denied</h3>
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <?php echo $error; ?>
                        </div>
                        <p class="text-muted">Please login as an administrator to run the database setup.</p>
                        <a href="login.php" class="btn btn-custom mt-3">
                            <i class="fas fa-sign-in-alt me-2"></i>Login
                        </a>
                    </div>
                    
                <?php elseif ($error): ?>
                    <!-- Error State -->
                    <div class="text-center">
                        <div class="icon-status status-error">
                            <i class="fas fa-times-circle"></i>
                        </div>
                        <h3>Setup Failed</h3>
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <?php echo $error; ?>
                        </div>
                    </div>
                    
                <?php else: ?>
                    <!-- Success State -->
                    <div class="text-center">
                        <div class="icon-status status-success">
                            <i class="fas fa-check-circle"></i>
                        </div>
                        <h3>Setup Complete!</h3>
                        <div class="alert alert-success">
                            <i class="fas fa-check-circle me-2"></i>
                            <?php echo $success; ?>
                        </div>
                        
                        <?php if (!empty($table_info)): ?>
                        <div class="table-info">
                            <div class="table-info-header">
                                <h5><i class="fas fa-table me-2"></i>Table Structure: password_reset_tokens</h5>
                            </div>
                            <div class="table-responsive">
                                <table class="table table-striped">
                                    <thead>
                                        <tr>
                                            <th>Field</th>
                                            <th>Type</th>
                                            <th>Null</th>
                                            <th>Key</th>
                                            <th>Default</th>
                                            <th>Extra</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($table_info as $column): ?>
                                        <tr>
                                            <td><code><?php echo htmlspecialchars($column['Field']); ?></code></td>
                                            <td><span class="badge bg-secondary"><?php echo htmlspecialchars($column['Type']); ?></span></td>
                                            <td><?php echo $column['Null'] === 'YES' ? '<span class="text-success">YES</span>' : '<span class="text-danger">NO</span>'; ?></td>
                                            <td><?php echo !empty($column['Key']) ? '<span class="badge bg-primary">' . htmlspecialchars($column['Key']) . '</span>' : ''; ?></td>
                                            <td><?php echo !empty($column['Default']) ? htmlspecialchars($column['Default']) : '<span class="text-muted">NULL</span>'; ?></td>
                                            <td><?php echo !empty($column['Extra']) ? '<span class="badge bg-info">' . htmlspecialchars($column['Extra']) . '</span>' : ''; ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                        
                        <div class="stats-card">
                            <h3><?php echo isset($total_tokens) ? $total_tokens : '0'; ?></h3>
                            <p>Active Reset Tokens</p>
                        </div>
                        <?php endif; ?>
                        
                        <div class="mt-4">
                            <a href="forgot-password.php" class="btn btn-custom me-2">
                                <i class="fas fa-key me-2"></i>Test Forgot Password
                            </a>
                            <a href="../admin/dashboard.php" class="btn btn-outline-secondary">
                                <i class="fas fa-arrow-left me-2"></i>Back to Dashboard
                            </a>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
