<?php
session_start();
require_once 'includes/db_connection.php';
require_once 'includes/id_encryption.php';

// Include SEO headers with favicon for category page
$seo_title = 'Course Categories - Panadite Academy | Browse by Topic';
$seo_description = 'Browse courses by category at Panadite Academy. Find professional training in technology, business, creative arts, and more from expert instructors.';
require_once __DIR__ . '/includes/seo_header.php';

// Get category_id from URL parameter - handle both encrypted and numeric IDs
$category_id = 0;
if (isset($_GET['category_id'])) {
    $id_param = urldecode($_GET['category_id']);
    
    // Try to decrypt the ID first
    $decrypted_id = IdEncryption::decrypt($id_param);
    if ($decrypted_id !== false && is_numeric($decrypted_id)) {
        $category_id = (int)$decrypted_id;
    } else {
        // Fallback to numeric ID for backward compatibility
        $category_id = (int)$_GET['category_id'];
    }
}

// Get current user if logged in
$current_user = null;
if (isset($_SESSION['user_id'])) {
    $user_id = $_SESSION['user_id'];
    $user_query = "SELECT * FROM users WHERE user_id = ?";
    $stmt = $conn->prepare($user_query);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $current_user = $stmt->get_result()->fetch_assoc();
    $stmt->close();
}

// Initialize variables
$category_info = null;
$courses = [];
$total_courses = 0;

// Fetch category information and courses
if ($category_id > 0 && $conn && !$conn->connect_error) {
    // Get category information
    $category_query = "SELECT category_id, category_name FROM categories WHERE category_id = ?";
    $stmt = $conn->prepare($category_query);
    $stmt->bind_param("i", $category_id);
    $stmt->execute();
    $category_result = $stmt->get_result();
    
    if ($category_result->num_rows > 0) {
        $category_info = $category_result->fetch_assoc();
        
        // Get courses in this category
        $courses_query = "SELECT c.*, 
                                 u.first_name, u.last_name,
                                 COUNT(e.enrollment_id) as enrollment_count
                          FROM courses c 
                          LEFT JOIN users u ON c.teacher_id = u.user_id
                          LEFT JOIN enrollments e ON c.course_id = e.course_id
                          WHERE c.category_id = ? AND c.is_published = 1
                          GROUP BY c.course_id
                          ORDER BY c.created_at DESC";
        
        $stmt = $conn->prepare($courses_query);
        $stmt->bind_param("i", $category_id);
        $stmt->execute();
        $courses_result = $stmt->get_result();
        
        while ($row = $courses_result->fetch_assoc()) {
            $courses[] = $row;
        }
        $total_courses = count($courses);
        $stmt->close();
    }
    $stmt->close();
}

// Set page title and meta description
$page_title = $category_info ? $category_info['category_name'] . ' Courses' : 'Category Not Found';
$meta_description = $category_info ? 'Explore ' . $category_info['category_name'] . ' courses at Panadite Academy. Learn from expert instructors with hands-on projects.' : 'Category not found - Browse our course catalog at Panadite Academy.';

// Helper function to format course price
function formatPrice($price) {
    return $price > 0 ? 'R' . number_format($price, 2) : 'Free';
}

// Helper function to truncate description
function truncateText($text, $length = 120) {
    return strlen($text) > $length ? substr($text, 0, $length) . '...' : $text;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($page_title); ?> | Panadite Academy</title>
    <meta name="description" content="<?php echo htmlspecialchars($meta_description); ?>">
    
    <!-- Favicon -->
    <link rel="shortcut icon" href="/assets/images/favicon.ico" type="image/x-icon">
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- Animate.css -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    
    <style>
        /* Modern Panadite Academy Styles */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary-blue: #276BE4;
            --primary-amber: #FFA915;
            --dark-text: #2c3e50;
            --light-text: #6c757d;
            --bg-light: #f8f9fa;
            --white: #ffffff;
            --border-color: #e9ecef;
            --shadow-sm: 0 2px 8px rgba(39, 107, 228, 0.06);
            --shadow-md: 0 4px 16px rgba(39, 107, 228, 0.08);
            --shadow-lg: 0 8px 24px rgba(39, 107, 228, 0.12);
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
            background: var(--bg-light);
            color: var(--dark-text);
            line-height: 1.6;
        }
        
        /* Modern Whitesmoke Hero Section */
        .category-header {
            background: whitesmoke !important;
            color: var(--dark-text) !important;
            padding: 4rem 0 3rem !important;
            position: relative;
            border-bottom: 1px solid var(--border-color);
            overflow: hidden;
        }
        
        .hero-container {
            display: grid;
            grid-template-columns: 1fr 400px;
            gap: 3rem;
            align-items: center;
            position: relative;
            z-index: 2;
        }
        
        .hero-content {
            max-width: 600px;
        }
        
        .hero-image {
            position: relative;
            height: 300px;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.08);
            background: linear-gradient(135deg, #f8f9fa, #e9ecef);
            border: 2px solid var(--border-color);
        }
        
        .hero-image::before {
            content: '';
            position: absolute;
            inset: 0;
            background: linear-gradient(135deg, 
                rgba(39, 107, 228, 0.1) 0%, 
                rgba(30, 91, 204, 0.05) 100%);
            z-index: 1;
        }
        
        .hero-image-content {
            position: absolute;
            inset: 0;
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 2;
            color: var(--dark-text);
        }
        
        .floating-elements {
            position: absolute;
            inset: 0;
            pointer-events: none;
        }
        
        .floating-icon {
            position: absolute;
            color: rgba(255, 169, 21, 0.6);
            animation: float 6s ease-in-out infinite;
        }
        
        .floating-icon:nth-child(1) {
            top: 20%;
            left: 15%;
            font-size: 1.5rem;
            animation-delay: 0s;
        }
        
        .floating-icon:nth-child(2) {
            top: 60%;
            left: 25%;
            font-size: 1.2rem;
            animation-delay: 2s;
        }
        
        .floating-icon:nth-child(3) {
            top: 35%;
            right: 20%;
            font-size: 1.8rem;
            animation-delay: 4s;
        }
        
        .floating-icon:nth-child(4) {
            bottom: 25%;
            right: 15%;
            font-size: 1.3rem;
            animation-delay: 1s;
        }
        
        @keyframes float {
            0%, 100% { transform: translateY(0px) rotate(0deg); opacity: 0.6; }
            50% { transform: translateY(-20px) rotate(5deg); opacity: 1; }
        }
        
        .category-title {
            font-size: 2.8rem !important;
            font-weight: 800 !important;
            margin-bottom: 1rem !important;
            color: var(--dark-text) !important;
            line-height: 1.2;
        }
        
        .category-title .highlight {
            color: var(--primary-amber) !important;
        }
        
        .category-subtitle {
            font-size: 1.1rem !important;
            color: var(--light-text) !important;
            margin-bottom: 2rem !important;
            line-height: 1.6;
            max-width: 500px;
        }
        
        .category-stats {
            font-size: 1rem !important;
            color: var(--light-text) !important;
            margin-bottom: 2rem !important;
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .stat-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            color: var(--primary-blue);
            font-weight: 600;
        }
        
        .breadcrumb-nav {
            background: white !important;
            border-radius: 25px !important;
            padding: 8px 20px !important;
            margin-bottom: 2rem !important;
            display: inline-flex !important;
            align-items: center;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05) !important;
            border: 1px solid var(--border-color) !important;
            font-size: 0.875rem !important;
        }
        
        .breadcrumb-nav a {
            color: var(--primary-blue) !important;
            text-decoration: none;
            margin-right: 4px;
            font-weight: 500;
            transition: color 0.2s ease;
        }
        
        .breadcrumb-nav a:hover {
            color: var(--dark-text) !important;
        }
        
        .breadcrumb-nav span {
            color: var(--light-text) !important;
        }
        
        .breadcrumb-nav i.fa-chevron-right {
            margin: 0 8px;
            font-size: 0.7rem;
            opacity: 0.7;
        }
        
        /* Compact Course Grid - 5 per row */
        .courses-section {
            padding: 3rem 0;
            background: var(--bg-light);
        }
        
        .courses-grid {
            display: grid !important;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)) !important;
            gap: 1.5rem !important;
            margin-top: 0 !important;
            width: 100% !important;
        }
        
        @media (min-width: 1200px) {
            .courses-grid {
                grid-template-columns: repeat(5, 1fr) !important;
            }
        }
        
        @media (min-width: 992px) and (max-width: 1199px) {
            .courses-grid {
                grid-template-columns: repeat(4, 1fr) !important;
            }
        }
        
        @media (min-width: 768px) and (max-width: 991px) {
            .courses-grid {
                grid-template-columns: repeat(3, 1fr) !important;
            }
        }
        
        @media (max-width: 767px) {
            .courses-grid {
                grid-template-columns: repeat(2, 1fr) !important;
            }
        }
        
        @media (max-width: 480px) {
            .courses-grid {
                grid-template-columns: 1fr !important;
            }
        }
        
        /* Compact Course Cards */
        .course-card {
            background: white !important;
            border-radius: 12px !important;
            overflow: hidden !important;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.06) !important;
            transition: all 0.3s ease !important;
            border: 1px solid #e9ecef !important;
            height: 100% !important;
            display: flex !important;
            flex-direction: column !important;
            max-width: 100% !important;
            margin: 0 !important;
        }
        
        .course-card:hover {
            transform: translateY(-3px) !important;
            box-shadow: 0 8px 30px rgba(39, 107, 228, 0.12) !important;
            border-color: rgba(39, 107, 228, 0.2) !important;
        }
        
        .course-thumbnail {
            width: 100%;
            height: 140px;
            background: linear-gradient(135deg, #f8f9fa, #e9ecef);
            position: relative;
            overflow: hidden;
            border-bottom: 1px solid var(--border-color);
        }
        
        .course-thumbnail img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.3s ease;
        }
        
        .course-card:hover .course-thumbnail img {
            transform: scale(1.03);
        }
        
        
        .course-content {
            padding: 1rem;
            flex: 1;
            display: flex;
            flex-direction: column;
        }
        
        .course-title {
            font-size: 1rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
            color: var(--dark-text);
            line-height: 1.3;
            display: -webkit-box;
            -webkit-line-clamp: 1;
            -webkit-box-orient: vertical;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
            min-height: 1.3rem;
        }
        
        .course-title a {
            text-decoration: none;
            color: inherit;
            transition: color 0.2s ease;
        }
        
        .course-title a:hover {
            color: var(--primary-blue);
        }
        
        .course-description {
            color: var(--light-text);
            font-size: 0.8rem;
            line-height: 1.3;
            margin-bottom: 0.75rem;
            display: -webkit-box;
            -webkit-line-clamp: 1;
            -webkit-box-orient: vertical;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        
        /* Add level badge */
        .course-level {
            display: inline-block;
            background: rgba(39, 107, 228, 0.1);
            color: var(--primary-blue);
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 0.7rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
        }
        
        .course-meta {
            display: flex;
            justify-content: flex-start;
            align-items: center;
            margin-bottom: 0.75rem;
            font-size: 0.75rem;
        }
        
        .course-instructor {
            color: var(--light-text);
            font-weight: 500;
            display: flex;
            align-items: center;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        
        .course-instructor i {
            margin-right: 4px;
            font-size: 0.7rem;
            flex-shrink: 0;
            color: var(--primary-blue);
        }
        
        .course-footer {
            margin-top: auto;
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }
        
        .course-price-section {
            display: flex;
            justify-content: center;
            align-items: center;
        }
        
        .course-price {
            font-size: 1.2rem;
            font-weight: 800;
            color: var(--primary-amber);
            text-align: center;
            margin-bottom: 0.5rem;
        }
        
        .course-price.free {
            color: #10b981;
        }
        
        .course-actions {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 6px;
        }
        
        .btn-action {
            padding: 6px 8px;
            border-radius: 4px;
            font-weight: 600;
            text-decoration: none;
            font-size: 0.7rem;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            white-space: nowrap;
            text-align: center;
        }
        
        .btn-view {
            background: linear-gradient(135deg, var(--primary-blue), #1e5bcc);
            color: white;
            border: none;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
        }
        
        .btn-view:hover {
            background: linear-gradient(135deg, #1e5bcc, var(--primary-blue));
            color: white;
            transform: translateY(-1px);
            box-shadow: 0 2px 6px rgba(0, 0, 0, 0.15);
        }
        
        .btn-cart {
            background: white;
            color: var(--primary-blue);
            border: 1px solid var(--border-color);
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
        }
        
        .btn-cart:hover {
            background: var(--primary-blue);
            color: white;
            border-color: var(--primary-blue);
            transform: translateY(-1px);
            box-shadow: 0 2px 6px rgba(0, 0, 0, 0.1);
        }
        
        .btn-action i {
            margin-right: 3px;
            font-size: 0.65rem;
        }
        
        /* Modern Empty State */
        .empty-state {
            text-align: center;
            padding: 5rem 2rem;
            color: var(--light-text);
        }
        
        .empty-state i {
            font-size: 4rem;
            color: #dee2e6;
            margin-bottom: 1.5rem;
        }
        
        .empty-state h3 {
            font-size: 1.5rem;
            margin-bottom: 0.75rem;
            color: var(--dark-text);
            font-weight: 600;
        }
        
        .empty-state p {
            font-size: 1rem;
            margin-bottom: 2rem;
            max-width: 400px;
            margin-left: auto;
            margin-right: auto;
        }
        
        /* Modern Back Button */
        .btn-back {
            background: white !important;
            color: var(--primary-blue) !important;
            border: 1px solid var(--border-color) !important;
            padding: 12px 20px !important;
            border-radius: 8px !important;
            font-weight: 600 !important;
            text-decoration: none !important;
            transition: all 0.3s ease !important;
            display: inline-flex !important;
            align-items: center;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05) !important;
        }
        
        .btn-back:hover {
            background: var(--primary-blue) !important;
            color: white !important;
            transform: translateY(-1px) !important;
            box-shadow: 0 4px 12px rgba(39, 107, 228, 0.15) !important;
        }
        
        .btn-back i {
            margin-right: 8px;
        }
        
        /* Responsive Design */
        @media (max-width: 1024px) {
            .hero-container {
                grid-template-columns: 1fr;
                gap: 2rem;
                text-align: center;
            }
            
            .hero-image {
                height: 250px;
                max-width: 400px;
                margin: 0 auto;
            }
        }
        
        @media (max-width: 768px) {
            .category-header {
                padding: 3rem 0 2rem !important;
            }
            
            .category-title {
                font-size: 2.2rem !important;
            }
            
            .category-subtitle {
                font-size: 1rem !important;
            }
            
            .hero-image {
                height: 200px;
            }
            
            .course-thumbnail {
                height: 200px;
            }
            
            .breadcrumb-nav {
                font-size: 0.8rem;
                padding: 6px 16px;
            }
            
            .floating-icon {
                display: none;
            }
        }
        
        @media (max-width: 500px) {
            .courses-grid {
                grid-template-columns: repeat(2, 1fr) !important;
                gap: 1rem !important;
                display: grid !important;
                width: 100% !important;
            }
            
            .course-thumbnail {
                height: 160px !important;
            }
            
            .course-content {
                padding: 0.75rem !important;
            }
            
            .course-title {
                font-size: 0.9rem !important;
                line-height: 1.2 !important;
            }
            
            .course-description {
                font-size: 0.75rem !important;
            }
        }
        
        @media (max-width: 400px) {
            .courses-grid {
                grid-template-columns: 1fr !important;
                gap: 1rem !important;
                display: grid !important;
                width: 100% !important;
            }
            
            .category-header {
                padding: 2rem 0 1.5rem !important;
            }
            
            .category-title {
                font-size: 1.8rem !important;
            }
            
            .category-subtitle {
                font-size: 0.9rem !important;
            }
            
            .hero-image {
                height: 180px;
            }
            
            .course-content {
                padding: 1rem !important;
            }
            
            .course-thumbnail {
                height: 180px !important;
            }
            
            .course-title {
                font-size: 1rem !important;
                white-space: normal !important;
                -webkit-line-clamp: 2 !important;
            }
            
            .course-description {
                font-size: 0.8rem !important;
                white-space: normal !important;
                -webkit-line-clamp: 2 !important;
            }
            
            .btn-action {
                padding: 8px 12px !important;
                font-size: 0.75rem !important;
            }
            
            .btn-enroll {
                padding: 10px 16px;
                font-size: 0.8rem;
            }
            
            .category-stats {
                flex-direction: column;
                gap: 0.5rem;
            }
            
            .container {
                padding-left: 1rem !important;
                padding-right: 1rem !important;
            }
        }
    </style>
</head>
<body>
    <!-- Include the navbar -->
    <?php include 'includes/navbar.php'; ?>

    <?php if ($category_info): ?>
    <!-- Modern Whitesmoke Hero Section -->
    <header class="category-header">
        <div class="container">
            <div class="breadcrumb-nav">
                <a href="/"><i class="fas fa-home me-1"></i>Home</a>
                <i class="fas fa-chevron-right mx-2" style="font-size: 0.7rem;"></i>
                <a href="/courses.php">Courses</a>
                <i class="fas fa-chevron-right mx-2" style="font-size: 0.7rem;"></i>
                <span><?php echo htmlspecialchars($category_info['category_name']); ?></span>
            </div>
            
            <div class="hero-container">
                <div class="hero-content">
                    <h1 class="category-title">
                        Master <span class="highlight"><?php echo htmlspecialchars(strtolower($category_info['category_name'])); ?></span>
                    </h1>
                    <p class="category-subtitle">
                        Advance your career with comprehensive training in <?php echo htmlspecialchars(strtolower($category_info['category_name'])); ?>. 
                        Join thousands of learners building in-demand skills.
                    </p>
                    
                    <div class="category-stats">
                        <div class="stat-item">
                            <i class="fas fa-user-tie"></i>
                            <span>Expert Instructors</span>
                        </div>
                        <div class="stat-item">
                            <i class="fas fa-video"></i>
                            <span><?php echo $total_courses; ?> course<?php echo $total_courses !== 1 ? 's' : ''; ?></span>
                        </div>
                        <div class="stat-item">
                            <i class="fas fa-users"></i>
                            <span>Active Community</span>
                        </div>
                    </div>
                    
                    <div class="hero-actions">
                        <a href="/courses.php" class="btn-back">
                            <i class="fas fa-arrow-left"></i>Back to All Courses
                        </a>
                    </div>
                </div>
                
                <div class="hero-image">
                    <div class="floating-elements">
                        <i class="fas fa-code floating-icon"></i>
                        <i class="fas fa-shield-alt floating-icon"></i>
                        <i class="fas fa-laptop-code floating-icon"></i>
                        <i class="fas fa-database floating-icon"></i>
                    </div>
                    <div class="hero-image-content">
                        <div style="text-align: center;">
                            <h3 style="font-size: 1.5rem; font-weight: 700; margin: 0;">
                                <?php echo htmlspecialchars($category_info['category_name']); ?>
                            </h3>
                            <p style="margin: 0.5rem 0 0 0; opacity: 0.8; font-size: 0.9rem;">
                                Interactive Learning Experience
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <!-- Courses Section -->
    <section class="courses-section">
        <div class="container">
            <?php if (!empty($courses)): ?>
                <div class="courses-grid">
                    <?php foreach ($courses as $course): 
                        // Encrypt course ID
                        $course_id = $course['course_id'] ?? 0;
                        $encrypted_course_id = class_exists('IdEncryption') ? IdEncryption::encrypt($course_id) : $course_id;
                    ?>
                        <div class="course-card animate__animated animate__fadeInUp" data-course-id="<?php echo htmlspecialchars($encrypted_course_id); ?>">
                            <!-- Course Thumbnail -->
                            <div class="course-thumbnail">
                                <?php if (!empty($course['thumbnail'])): ?>
                                    <img src="<?php echo htmlspecialchars($course['thumbnail']); ?>" 
                                         alt="<?php echo htmlspecialchars($course['title']); ?>">
                                <?php endif; ?>
                            </div>
                            
                            <!-- Course Content -->
                            <div class="course-content">
                            <?php 
                                // Determine course level dynamically
                                $level = 'Beginner'; // default
                                if (isset($course['level'])) {
                                    $level = ucfirst($course['level']);
                                } elseif (isset($course['price'])) {
                                    // Simple logic based on price
                                    if ($course['price'] == 0) {
                                        $level = 'Beginner';
                                    } elseif ($course['price'] <= 1000) {
                                        $level = 'Beginner';
                                    } elseif ($course['price'] <= 3000) {
                                        $level = 'Intermediate';
                                    } else {
                                        $level = 'Advanced';
                                    }
                                }
                            ?>
                            <div class="course-level course-level" data-level="<?php echo $level; ?>"><?php echo htmlspecialchars($level); ?></div>
                            
                            <h3 class="course-title">
                                <a href="course.php?id=<?php echo urlencode($encrypted_course_id); ?>">
                                    <?php echo htmlspecialchars($course['title']); ?>
                                </a>
                            </h3>
                            
                            <p class="course-description">
                                <?php echo htmlspecialchars($course['description'] ?? 'Learn essential skills with this comprehensive course designed by expert instructors.'); ?>
                            </p>
                            
                            <div class="course-meta">
                                <div class="course-instructor">
                                    <i class="fas fa-user-circle"></i>
                                    <span><?php echo htmlspecialchars(($course['first_name'] ?? 'Expert') . ' ' . ($course['last_name'] ?? 'Instructor')); ?></span>
                                </div>
                            </div>
                            
                            <div class="course-footer">
                                <div class="course-price-section">
                                    <div class="course-price<?php echo (($course['price'] ?? 0) <= 0) ? ' free' : ''; ?>" data-price="<?php echo $course['price'] ?? 0; ?>">
                                        <?php echo formatPrice($course['price'] ?? 0); ?>
                                    </div>
                                </div>
                                
                                <div class="course-actions">
                                    <a href="course.php?id=<?php echo urlencode($encrypted_course_id); ?>" class="btn-action btn-view">
                                        <i class="fas fa-eye"></i>
                                        View
                                    </a>
                                    <button class="btn-action btn-cart pa-course-cart-btn" data-course-id="<?php echo htmlspecialchars($encrypted_course_id); ?>">
                                        <i class="fas fa-shopping-cart"></i>
                                        Add to Cart
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <div class="empty-state">
                    <i class="fas fa-book-open"></i>
                    <h3>No courses available</h3>
                    <p>There are currently no published courses in this category.</p>
                    <a href="/courses.php" class="btn btn-primary mt-3">
                        <i class="fas fa-arrow-left me-2"></i>Browse All Courses
                    </a>
                </div>
            <?php endif; ?>
        </div>
    </section>

    <?php else: ?>
    <!-- Category Not Found -->
    <header class="category-header">
        <div class="container text-center">
            <h1 class="category-title">Category Not Found</h1>
            <p class="category-stats">The requested category could not be found.</p>
            <a href="/courses.php" class="btn-back mt-3">
                <i class="fas fa-arrow-left me-2"></i>Browse All Courses
            </a>
        </div>
    </header>
    
    <section class="py-5">
        <div class="container">
            <div class="empty-state">
                <i class="fas fa-exclamation-triangle"></i>
                <h3>Category Not Found</h3>
                <p>The category you're looking for doesn't exist or may have been removed.</p>
                <a href="/courses.php" class="btn btn-primary mt-3">
                    <i class="fas fa-th-large me-2"></i>View All Categories
                </a>
            </div>
        </div>
    </section>
    <?php endif; ?>

    <!-- Include Footer -->
    <?php include 'includes/footer.php'; ?>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Panadite Academy Guest Cart System -->
    <script src="/js/guest-cart-system.js"></script>
    
    <script>
        // Smooth animations on scroll
        document.addEventListener('DOMContentLoaded', function() {
            // Initialize Panadite Academy Cart System
            const cartSystem = new PanaditeAcademyCartSystem();
            
            // Make cart system available globally
            window.paCartSystem = cartSystem;
            
            // Update cart counter on page load
            cartSystem.updateCartCounter();
            
            const observerOptions = {
                threshold: 0.1,
                rootMargin: '0px 0px -50px 0px'
            };
            
            const observer = new IntersectionObserver(function(entries) {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        entry.target.style.animationDelay = `${Math.random() * 0.3}s`;
                        entry.target.classList.add('animate__fadeInUp');
                    }
                });
            }, observerOptions);
            
            // Observe all course cards
            document.querySelectorAll('.course-card').forEach(card => {
                observer.observe(card);
            });
        });
        
        // Add loading states for course links
        document.querySelectorAll('a[href*="course.php"]').forEach(link => {
            link.addEventListener('click', function() {
                this.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i>Loading...';
            });
        });
    </script>
</body>
</html>
