<?php
// CMS Content Editor
// Fix session path issues on production server
if (!is_dir(session_save_path()) || !is_writable(session_save_path())) {
    // Set a proper session save path
    $session_path = sys_get_temp_dir();
    if (is_dir($session_path) && is_writable($session_path)) {
        session_save_path($session_path);
    }
}

// Start session with error handling
try {
    session_start();
} catch (Exception $e) {
    // If sessions still fail, we'll use alternative authentication
    error_log("Session error in CMS content: " . $e->getMessage());
}

require_once '../../includes/db.php';
require_once '../functions/cms_functions.php';

// Check if user is logged in
if (!isset($_SESSION['cms_admin_id'])) {
    header('Location: login.php');
    exit();
}

$admin_user = getCMSAdmin($_SESSION['cms_admin_id']);
if (!$admin_user) {
    session_destroy();
    header('Location: login.php');
    exit();
}

// Get page parameter
$current_page = $_GET['page'] ?? 'homepage';
$allowed_pages = ['homepage', 'courses', 'about', 'teaching-requirements', 'category'];

if (!in_array($current_page, $allowed_pages)) {
    $current_page = 'homepage';
}

// Handle content updates
if ($_POST && isset($_POST['action']) && $_POST['action'] === 'update_content') {
    $section_key = $_POST['section_key'];
    $content_value = $_POST['content_value'];
    $content_type = $_POST['content_type'] ?? 'text';
    
    $success = updateCMSContent($current_page, $section_key, $content_value, $content_type, $_SESSION['cms_admin_id']);
    
    if ($success) {
        $success_message = "Content updated successfully!";
    } else {
        $error_message = "Failed to update content.";
    }
}

// Get current page content
$page_content = getCMSContent($current_page);

// Page configurations
$page_configs = [
    'homepage' => [
        'title' => 'Homepage Content',
        'icon' => 'fas fa-home',
        'sections' => [
            'hero_title' => ['label' => 'Hero Title', 'type' => 'text', 'help' => 'Main heading on homepage'],
            'hero_subtitle' => ['label' => 'Hero Subtitle', 'type' => 'text', 'help' => 'Subtitle below main heading'],
            'hero_description' => ['label' => 'Hero Description', 'type' => 'textarea', 'help' => 'Longer description text'],
            'stats_students' => ['label' => 'Students Count', 'type' => 'text', 'help' => 'Number of students (e.g., 10,000+)'],
            'stats_courses' => ['label' => 'Courses Count', 'type' => 'text', 'help' => 'Number of courses available'],
            'stats_instructors' => ['label' => 'Instructors Count', 'type' => 'text', 'help' => 'Number of instructors'],
            'stats_completion' => ['label' => 'Completion Rate', 'type' => 'text', 'help' => 'Course completion percentage'],
            'about_title' => ['label' => 'About Section Title', 'type' => 'text', 'help' => 'Title for about section'],
            'about_description' => ['label' => 'About Description', 'type' => 'textarea', 'help' => 'About section content']
        ]
    ],
    'courses' => [
        'title' => 'Courses Page Content',
        'icon' => 'fas fa-graduation-cap',
        'sections' => [
            'hero_title' => ['label' => 'Page Title', 'type' => 'html', 'help' => 'Main title (can include HTML)'],
            'hero_subtitle' => ['label' => 'Page Subtitle', 'type' => 'text', 'help' => 'Description below title'],
            'filter_all_text' => ['label' => 'All Courses Filter', 'type' => 'text', 'help' => 'Text for "All" filter tab'],
            'filter_popular_text' => ['label' => 'Popular Filter', 'type' => 'text', 'help' => 'Text for "Popular" filter tab'],
            'filter_new_text' => ['label' => 'New Releases Filter', 'type' => 'text', 'help' => 'Text for "New" filter tab'],
            'filter_featured_text' => ['label' => 'Featured Filter', 'type' => 'text', 'help' => 'Text for "Featured" filter tab'],
            'search_placeholder' => ['label' => 'Search Placeholder', 'type' => 'text', 'help' => 'Placeholder text for search box']
        ]
    ],
    'about' => [
        'title' => 'About Page Content',
        'icon' => 'fas fa-users',
        'sections' => [
            'hero_title' => ['label' => 'Page Title', 'type' => 'text', 'help' => 'Main page title'],
            'hero_subtitle' => ['label' => 'Page Subtitle', 'type' => 'text', 'help' => 'Subtitle description'],
            'mission_title' => ['label' => 'Mission Title', 'type' => 'text', 'help' => 'Mission section heading'],
            'mission_description' => ['label' => 'Mission Description', 'type' => 'textarea', 'help' => 'Mission statement content'],
            'vision_title' => ['label' => 'Vision Title', 'type' => 'text', 'help' => 'Vision section heading'],
            'vision_description' => ['label' => 'Vision Description', 'type' => 'textarea', 'help' => 'Vision statement content']
        ]
    ],
    'teaching-requirements' => [
        'title' => 'Teaching Requirements Content',
        'icon' => 'fas fa-chalkboard-teacher',
        'sections' => [
            'hero_title' => ['label' => 'Page Title', 'type' => 'text', 'help' => 'Main page title'],
            'hero_subtitle' => ['label' => 'Page Subtitle', 'type' => 'text', 'help' => 'Subtitle description'],
            'requirements_title' => ['label' => 'Requirements Section Title', 'type' => 'text', 'help' => 'Title for requirements list'],
            'req_expertise' => ['label' => 'Expertise Requirement', 'type' => 'text', 'help' => 'Expertise requirement text'],
            'req_experience' => ['label' => 'Experience Requirement', 'type' => 'text', 'help' => 'Experience requirement text'],
            'req_communication' => ['label' => 'Communication Requirement', 'type' => 'text', 'help' => 'Communication skills requirement'],
            'req_equipment' => ['label' => 'Equipment Requirement', 'type' => 'text', 'help' => 'Equipment requirement text']
        ]
    ]
];

$current_config = $page_configs[$current_page];

// Create content array for easy access
$content_by_key = [];
foreach ($page_content as $content) {
    $content_by_key[$content['section_key']] = $content['content_value'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $current_config['title']; ?> - Panadite Academy CMS</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../assets/css/admin.css" rel="stylesheet">
</head>
<body>
    <?php include 'components/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'components/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="<?php echo $current_config['icon']; ?> me-2"></i>
                        <?php echo $current_config['title']; ?>
                    </h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <div class="btn-group me-2">
                            <a href="../../<?php echo $current_page === 'homepage' ? 'index.php' : $current_page . '.php'; ?>" 
                               target="_blank" class="btn btn-sm btn-outline-primary">
                                <i class="fas fa-external-link-alt me-1"></i>Preview Page
                            </a>
                        </div>
                    </div>
                </div>

                <!-- Page Selection Tabs -->
                <ul class="nav nav-tabs mb-4">
                    <?php foreach ($page_configs as $page_key => $config): ?>
                        <li class="nav-item">
                            <a class="nav-link <?php echo $page_key === $current_page ? 'active' : ''; ?>" 
                               href="?page=<?php echo $page_key; ?>">
                                <i class="<?php echo $config['icon']; ?> me-1"></i>
                                <?php echo explode(' Content', $config['title'])[0]; ?>
                            </a>
                        </li>
                    <?php endforeach; ?>
                </ul>

                <!-- Success/Error Messages -->
                <?php if (isset($success_message)): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="fas fa-check-circle me-2"></i>
                        <?php echo $success_message; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if (isset($error_message)): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <?php echo $error_message; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Content Editing Form -->
                <div class="row">
                    <div class="col-lg-8">
                        <div class="card shadow">
                            <div class="card-header">
                                <h5 class="mb-0">
                                    <i class="fas fa-edit me-2"></i>
                                    Edit Content Sections
                                </h5>
                            </div>
                            <div class="card-body">
                                <?php foreach ($current_config['sections'] as $section_key => $section_config): ?>
                                    <form method="POST" class="content-form mb-4" data-section="<?php echo $section_key; ?>">
                                        <input type="hidden" name="action" value="update_content">
                                        <input type="hidden" name="section_key" value="<?php echo $section_key; ?>">
                                        <input type="hidden" name="content_type" value="<?php echo $section_config['type']; ?>">
                                        
                                        <div class="content-section border rounded p-3 mb-3">
                                            <div class="d-flex justify-content-between align-items-center mb-2">
                                                <label for="content_<?php echo $section_key; ?>" class="form-label fw-semibold mb-0">
                                                    <?php echo $section_config['label']; ?>
                                                </label>
                                                <button type="submit" class="btn btn-sm btn-success">
                                                    <i class="fas fa-save me-1"></i>Save
                                                </button>
                                            </div>
                                            
                                            <?php if (isset($section_config['help'])): ?>
                                                <small class="text-muted d-block mb-2">
                                                    <i class="fas fa-info-circle me-1"></i>
                                                    <?php echo $section_config['help']; ?>
                                                </small>
                                            <?php endif; ?>
                                            
                                            <?php 
                                            $current_value = $content_by_key[$section_key] ?? '';
                                            $input_type = $section_config['type'];
                                            ?>
                                            
                                            <?php if ($input_type === 'textarea' || $input_type === 'html'): ?>
                                                <textarea name="content_value" id="content_<?php echo $section_key; ?>" 
                                                          class="form-control <?php echo $input_type === 'html' ? 'html-editor' : ''; ?>" 
                                                          rows="<?php echo $input_type === 'html' ? 6 : 3; ?>"
                                                          placeholder="Enter <?php echo strtolower($section_config['label']); ?>..."><?php echo htmlspecialchars($current_value); ?></textarea>
                                            <?php else: ?>
                                                <input type="text" name="content_value" id="content_<?php echo $section_key; ?>" 
                                                       class="form-control" 
                                                       value="<?php echo htmlspecialchars($current_value); ?>"
                                                       placeholder="Enter <?php echo strtolower($section_config['label']); ?>...">
                                            <?php endif; ?>
                                        </div>
                                    </form>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-lg-4">
                        <!-- Preview Panel -->
                        <div class="card shadow mb-4">
                            <div class="card-header">
                                <h6 class="mb-0">
                                    <i class="fas fa-eye me-2"></i>
                                    Quick Actions
                                </h6>
                            </div>
                            <div class="card-body">
                                <div class="d-grid gap-2">
                                    <a href="../../<?php echo $current_page === 'homepage' ? 'index.php' : $current_page . '.php'; ?>" 
                                       target="_blank" class="btn btn-primary btn-sm">
                                        <i class="fas fa-external-link-alt me-1"></i>
                                        Preview Live Page
                                    </a>
                                    <a href="images.php" class="btn btn-success btn-sm">
                                        <i class="fas fa-images me-1"></i>
                                        Manage Images
                                    </a>
                                    <a href="index.php" class="btn btn-secondary btn-sm">
                                        <i class="fas fa-dashboard me-1"></i>
                                        Back to Dashboard
                                    </a>
                                </div>
                            </div>
                        </div>

                        <!-- Page Info -->
                        <div class="card shadow">
                            <div class="card-header">
                                <h6 class="mb-0">
                                    <i class="fas fa-info-circle me-2"></i>
                                    Page Information
                                </h6>
                            </div>
                            <div class="card-body">
                                <p class="text-muted small mb-2">
                                    <strong>Content Sections:</strong> <?php echo count($current_config['sections']); ?>
                                </p>
                                <p class="text-muted small mb-2">
                                    <strong>Last Updated:</strong> 
                                    <?php 
                                    $latest_update = '';
                                    foreach ($page_content as $content) {
                                        if ($content['updated_at'] > $latest_update) {
                                            $latest_update = $content['updated_at'];
                                        }
                                    }
                                    echo $latest_update ? date('M j, Y g:i A', strtotime($latest_update)) : 'Never';
                                    ?>
                                </p>
                                <p class="text-muted small mb-0">
                                    <strong>Editor:</strong> <?php echo htmlspecialchars($admin_user['full_name']); ?>
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="../assets/js/admin.js"></script>
    
    <script>
        // Auto-save functionality (optional)
        document.querySelectorAll('.content-form').forEach(form => {
            const textarea = form.querySelector('textarea, input[type="text"]');
            if (textarea) {
                let saveTimeout;
                textarea.addEventListener('input', function() {
                    clearTimeout(saveTimeout);
                    // Auto-save after 2 seconds of no typing
                    saveTimeout = setTimeout(() => {
                        // Could implement auto-save here
                    }, 2000);
                });
            }
        });

        // Show success message after form submission
        <?php if (isset($success_message)): ?>
            setTimeout(() => {
                const alert = document.querySelector('.alert-success');
                if (alert) {
                    alert.classList.add('fade');
                    setTimeout(() => alert.remove(), 300);
                }
            }, 3000);
        <?php endif; ?>
    </script>
</body>
</html>
