/**
 * Panadite Academy CMS - Inline Content Editor
 * Allows admins to edit content directly on the website
 */

class PanaditeInlineEditor {
    constructor() {
        this.isAdmin = false;
        this.editingElement = null;
        this.originalContent = '';
        this.init();
    }

    async init() {
        // Check if user is logged in as CMS admin
        try {
            const response = await fetch('/cms/functions/check_admin_status.php');
            const data = await response.json();
            this.isAdmin = data.is_admin || false;
            
            if (this.isAdmin) {
                this.enableInlineEditing();
                this.showAdminIndicator();
            }
        } catch (error) {
            console.log('CMS admin check failed:', error);
        }
    }

    enableInlineEditing() {
        // Add CSS for inline editing
        this.addInlineEditingCSS();
        
        // Find all CMS content elements
        const editableElements = document.querySelectorAll('[data-cms-content]');
        
        editableElements.forEach(element => {
            this.makeElementEditable(element);
        });

        console.log(`✅ Inline editing enabled for ${editableElements.length} elements`);
    }

    makeElementEditable(element) {
        // Skip if already made editable
        if (element.classList.contains('cms-editable')) {
            return;
        }

        // Add visual indicators
        element.classList.add('cms-editable');
        element.setAttribute('title', 'Click to edit this content');
        
        // Add edit icon only if it doesn't exist
        let editIcon = element.querySelector('.cms-edit-icon');
        if (!editIcon) {
            editIcon = document.createElement('span');
            editIcon.className = 'cms-edit-icon';
            editIcon.innerHTML = '<i class="fas fa-edit"></i>';
            editIcon.style.display = 'none';
            element.appendChild(editIcon);
        }

        // Store the click handler so we can remove it later
        this.clickHandler = (e) => {
            e.preventDefault();
            e.stopPropagation();
            this.startEditing(element);
        };

        // Show edit icon on hover
        element.addEventListener('mouseenter', () => {
            if (this.editingElement !== element) {
                editIcon.style.display = 'inline-block';
            }
        });

        element.addEventListener('mouseleave', () => {
            if (this.editingElement !== element) {
                editIcon.style.display = 'none';
            }
        });

        // Make clickable for editing
        element.addEventListener('click', this.clickHandler);
    }

    startEditing(element) {
        if (this.editingElement) {
            this.cancelEditing();
        }

        this.editingElement = element;
        this.originalContent = element.innerHTML;
        
        // Remove edit icon temporarily
        const editIcon = element.querySelector('.cms-edit-icon');
        if (editIcon) editIcon.remove();

        // Store original styles
        this.originalStyle = {
            userSelect: element.style.userSelect,
            pointerEvents: element.style.pointerEvents,
            outline: element.style.outline
        };

        // Make element editable with proper styling
        element.contentEditable = true;
        element.classList.add('cms-editing');
        element.style.userSelect = 'text';
        element.style.pointerEvents = 'auto';
        element.style.outline = 'none';
        element.style.cursor = 'text';

        // Clear any existing event listeners that might interfere
        element.removeEventListener('click', this.clickHandler);
        
        // Focus and select all text for easy editing
        element.focus();
        
        // Select all content for easy replacement
        setTimeout(() => {
            if (document.selection) {
                // IE
                const range = document.body.createTextRange();
                range.moveToElementText(element);
                range.select();
            } else if (window.getSelection) {
                // Other browsers
                const range = document.createRange();
                range.selectNodeContents(element);
                const selection = window.getSelection();
                selection.removeAllRanges();
                selection.addRange(range);
            }
        }, 10);

        // Add save/cancel buttons
        this.addEditingControls(element);

        // Handle keyboard shortcuts
        this.keydownHandler = (e) => {
            if (e.key === 'Enter' && e.ctrlKey) {
                e.preventDefault();
                this.saveChanges();
            } else if (e.key === 'Escape') {
                e.preventDefault();
                this.cancelEditing();
            }
        };
        
        element.addEventListener('keydown', this.keydownHandler);
    }

    addEditingControls(element) {
        const controls = document.createElement('div');
        controls.className = 'cms-editing-controls';
        controls.innerHTML = `
            <button type="button" class="cms-save-btn" title="Save (Ctrl+Enter)">
                <i class="fas fa-check"></i> Save
            </button>
            <button type="button" class="cms-cancel-btn" title="Cancel (Escape)">
                <i class="fas fa-times"></i> Cancel
            </button>
        `;

        // Position controls as floating overlay outside containers
        document.body.appendChild(controls);
        this.positionEditingControls(controls, element);

        // Add event listeners
        controls.querySelector('.cms-save-btn').addEventListener('click', () => this.saveChanges());
        controls.querySelector('.cms-cancel-btn').addEventListener('click', () => this.cancelEditing());
        
        // Update position on scroll/resize
        this.positionUpdateHandler = () => this.positionEditingControls(controls, element);
        window.addEventListener('scroll', this.positionUpdateHandler);
        window.addEventListener('resize', this.positionUpdateHandler);
    }

    positionEditingControls(controls, element) {
        const rect = element.getBoundingClientRect();
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        const scrollLeft = window.pageXOffset || document.documentElement.scrollLeft;
        
        // Position controls below and to the right of the element, outside any containers
        const top = rect.bottom + scrollTop + 10;
        const left = Math.min(rect.right + scrollLeft, window.innerWidth - 200); // Ensure it fits on screen
        
        controls.style.position = 'absolute';
        controls.style.top = top + 'px';
        controls.style.left = left + 'px';
        controls.style.zIndex = '10000';
    }

    async saveChanges() {
        if (!this.editingElement) return;

        const element = this.editingElement;
        const newContent = element.innerHTML.replace(/<span class="cms-edit-icon"[^>]*>.*?<\/span>/g, '');
        
        // Get CMS data
        const page = element.getAttribute('data-cms-page') || 'homepage';
        const section = element.getAttribute('data-cms-section') || 'content';

        // Show saving indicator
        this.showSavingIndicator(true);

        try {
            // Use URLSearchParams instead of FormData for better compatibility
            const formData = new URLSearchParams();
            formData.append('page_name', page);
            formData.append('section_key', section);
            formData.append('content', newContent);

            console.log('📤 Sending POST data:', {
                page_name: page,
                section_key: section,
                content: newContent.substring(0, 100) + '...'
            });

            const response = await fetch('/cms/functions/save_content.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded'
                },
                body: formData
            });

            const result = await response.json();

            if (result.success) {
                this.showNotification('✅ Content saved successfully!', 'success');
                this.finishEditing();
            } else {
                throw new Error(result.message || 'Failed to save content');
            }
        } catch (error) {
            console.error('Save error:', error);
            this.showNotification('❌ Failed to save content', 'error');
            this.cancelEditing();
        } finally {
            this.showSavingIndicator(false);
        }
    }

    cancelEditing() {
        if (!this.editingElement) return;

        // Restore original content
        this.editingElement.innerHTML = this.originalContent;
        this.finishEditing();
    }

    finishEditing() {
        if (!this.editingElement) return;

        const element = this.editingElement;

        // Remove editing state
        element.contentEditable = false;
        element.classList.remove('cms-editing');
        
        // Restore original styles
        if (this.originalStyle) {
            element.style.userSelect = this.originalStyle.userSelect;
            element.style.pointerEvents = this.originalStyle.pointerEvents;
            element.style.outline = this.originalStyle.outline;
            element.style.cursor = '';
        }

        // Remove event listeners
        if (this.keydownHandler) {
            element.removeEventListener('keydown', this.keydownHandler);
        }
        
        // Remove scroll/resize listeners
        if (this.positionUpdateHandler) {
            window.removeEventListener('scroll', this.positionUpdateHandler);
            window.removeEventListener('resize', this.positionUpdateHandler);
        }

        // Remove controls
        const controls = document.querySelector('.cms-editing-controls');
        if (controls) controls.remove();

        // Restore edit functionality
        this.makeElementEditable(element);

        // Clear editing element
        this.editingElement = null;
        this.originalContent = '';
        this.originalStyle = null;
        this.keydownHandler = null;
    }

    showAdminIndicator() {
        // Add admin indicator to the page
        const indicator = document.createElement('div');
        indicator.className = 'cms-admin-indicator';
        indicator.innerHTML = `
            <div class="cms-admin-badge">
                <i class="fas fa-edit"></i>
                <span>Admin Mode - Click to Edit Content</span>
                <button class="cms-admin-toggle" onclick="window.cmsEditor.toggleEditMode()">
                    <i class="fas fa-eye"></i>
                </button>
            </div>
        `;
        document.body.appendChild(indicator);
    }

    toggleEditMode() {
        const editableElements = document.querySelectorAll('.cms-editable');
        const isVisible = editableElements.length > 0 && editableElements[0].classList.contains('cms-edit-mode-visible');

        editableElements.forEach(element => {
            if (isVisible) {
                element.classList.remove('cms-edit-mode-visible');
            } else {
                element.classList.add('cms-edit-mode-visible');
            }
        });
    }

    showSavingIndicator(show) {
        let indicator = document.querySelector('.cms-saving-indicator');
        
        if (show && !indicator) {
            indicator = document.createElement('div');
            indicator.className = 'cms-saving-indicator';
            indicator.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Saving...';
            document.body.appendChild(indicator);
        } else if (!show && indicator) {
            indicator.remove();
        }
    }

    showNotification(message, type = 'success') {
        const notification = document.createElement('div');
        notification.className = `cms-notification cms-notification-${type}`;
        notification.innerHTML = message;
        document.body.appendChild(notification);

        // Auto-remove after 3 seconds
        setTimeout(() => {
            if (notification.parentNode) {
                notification.remove();
            }
        }, 3000);
    }

    addInlineEditingCSS() {
        const css = `
            /* CMS Inline Editing Styles */
            .cms-editable {
                position: relative;
                transition: all 0.2s ease;
                border-radius: 4px;
            }

            .cms-editable:hover {
                background-color: rgba(245, 158, 11, 0.1) !important;
                box-shadow: 0 0 0 2px rgba(245, 158, 11, 0.3) !important;
                cursor: pointer;
            }

            .cms-editable.cms-editing {
                background-color: rgba(37, 99, 235, 0.1) !important;
                box-shadow: 0 0 0 2px rgba(37, 99, 235, 0.5) !important;
                outline: none;
                cursor: text;
            }

            .cms-edit-icon {
                position: absolute;
                top: -10px;
                right: -10px;
                background: #f59e0b;
                color: white;
                border-radius: 50%;
                width: 24px;
                height: 24px;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 10px;
                z-index: 1000;
                box-shadow: 0 2px 4px rgba(0,0,0,0.2);
            }

            .cms-editing-controls {
                position: absolute;
                background: white;
                border-radius: 8px;
                padding: 10px;
                box-shadow: 0 8px 25px rgba(0,0,0,0.15);
                border: 1px solid #e5e7eb;
                z-index: 10000;
                display: flex;
                gap: 10px;
                min-width: 150px;
            }

            .cms-save-btn, .cms-cancel-btn {
                padding: 6px 12px;
                border: none;
                border-radius: 4px;
                font-size: 12px;
                font-weight: 500;
                cursor: pointer;
                transition: all 0.2s ease;
            }

            .cms-save-btn {
                background: #22c55e;
                color: white;
            }

            .cms-save-btn:hover {
                background: #16a34a;
            }

            .cms-cancel-btn {
                background: #ef4444;
                color: white;
            }

            .cms-cancel-btn:hover {
                background: #dc2626;
            }

            .cms-admin-indicator {
                position: fixed !important;
                top: 120px !important;
                right: 10px !important;
                z-index: 9999 !important;
            }

            .cms-admin-badge {
                background: linear-gradient(135deg, #2563eb, #f59e0b);
                color: white;
                padding: 8px 16px;
                border-radius: 25px;
                font-size: 12px;
                font-weight: 600;
                display: flex;
                align-items: center;
                gap: 8px;
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            }

            .cms-admin-toggle {
                background: rgba(255,255,255,0.2);
                border: none;
                color: white;
                border-radius: 50%;
                width: 24px;
                height: 24px;
                display: flex;
                align-items: center;
                justify-content: center;
                cursor: pointer;
            }

            .cms-saving-indicator {
                position: fixed;
                bottom: 20px;
                right: 20px;
                background: #2563eb;
                color: white;
                padding: 12px 20px;
                border-radius: 8px;
                font-weight: 500;
                z-index: 9999;
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            }

            .cms-notification {
                position: fixed;
                top: 60px;
                right: 20px;
                padding: 12px 20px;
                border-radius: 8px;
                font-weight: 500;
                z-index: 9999;
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
                animation: slideIn 0.3s ease;
            }

            .cms-notification-success {
                background: #22c55e;
                color: white;
            }

            .cms-notification-error {
                background: #ef4444;
                color: white;
            }

            @keyframes slideIn {
                from { transform: translateX(100%); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
        `;

        const style = document.createElement('style');
        style.textContent = css;
        document.head.appendChild(style);
    }
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.cmsEditor = new PanaditeInlineEditor();
});
