/**
 * Enhanced Inline Image Editor for Panadite Academy CMS
 * Supports both text and image editing with drag & drop upload
 */

// Prevent duplicate class declarations
if (typeof window.InlineImageEditor !== 'undefined') {
    console.log('⚠️ InlineImageEditor already exists, skipping redeclaration');
} else {

class InlineImageEditor {
    constructor() {
        this.isAdminMode = false;
        this.currentImageEditor = null;
        this.init();
    }

    init() {
        this.checkAdminStatus();
        this.setupImageEditing();
        this.addAdminModeIndicator();
    }

    checkAdminStatus() {
        // Check if admin mode is active (look for admin mode indicator)
        const adminMode = document.querySelector('[data-admin-mode]') || 
                         document.body.classList.contains('cms-admin') ||
                         document.body.classList.contains('admin-mode') ||
                         document.querySelector('.cms-admin-badge') ||
                         localStorage.getItem('cms_admin_logged_in') === 'true';
        
        this.isAdminMode = !!adminMode;
        
        if (this.isAdminMode) {
            console.log('🔧 CMS Admin Mode Active - Image editing enabled');
        }
    }

    addAdminModeIndicator() {
        if (!this.isAdminMode) return;

        // Add admin mode badge if not exists
        if (!document.querySelector('.cms-admin-badge')) {
            const badge = document.createElement('div');
            badge.className = 'cms-admin-badge';
            badge.innerHTML = `
                <i class="fas fa-edit"></i>
                <span>CMS Admin Mode</span>
                <small>Click images to edit</small>
            `;
            document.body.appendChild(badge);

            // Add CSS for admin badge
            this.addAdminBadgeCSS();
        }
    }

    addAdminBadgeCSS() {
        if (document.querySelector('#cms-admin-badge-css')) return;

        const css = `
            .cms-admin-badge {
                position: fixed;
                top: 20px;
                right: 20px;
                background: linear-gradient(135deg, #f59e0b, #d97706);
                color: white;
                padding: 12px 20px;
                border-radius: 25px;
                z-index: 9999;
                font-family: 'Inter', sans-serif;
                font-size: 14px;
                font-weight: 500;
                box-shadow: 0 4px 12px rgba(245, 158, 11, 0.3);
                display: flex;
                align-items: center;
                gap: 8px;
                animation: slideInDown 0.5s ease;
            }
            
            .cms-admin-badge i {
                font-size: 16px;
            }
            
            .cms-admin-badge small {
                font-size: 11px;
                opacity: 0.9;
                margin-left: 8px;
            }

            .cms-editable-image {
                cursor: pointer !important;
                position: relative !important;
                transition: all 0.3s ease !important;
                border: 2px dashed transparent !important;
            }

            .cms-editable-image:hover {
                border-color: #f59e0b !important;
                box-shadow: 0 0 0 4px rgba(245, 158, 11, 0.1) !important;
            }

            .cms-editable-image::after {
                content: "📸 Click to edit";
                position: absolute;
                top: 50%;
                left: 50%;
                transform: translate(-50%, -50%);
                background: rgba(0, 0, 0, 0.8);
                color: white;
                padding: 6px 12px;
                border-radius: 20px;
                font-size: 12px;
                font-weight: 500;
                opacity: 0;
                transition: opacity 0.3s ease;
                pointer-events: none;
                z-index: 10;
            }

            .cms-editable-image:hover::after {
                opacity: 1;
            }

            .cms-image-editor-modal {
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0, 0, 0, 0.8);
                z-index: 10000;
                display: flex;
                align-items: center;
                justify-content: center;
                animation: fadeIn 0.3s ease;
            }

            .cms-image-editor-content {
                background: white;
                border-radius: 16px;
                padding: 30px;
                max-width: 500px;
                width: 90%;
                max-height: 90vh;
                overflow-y: auto;
                box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
                animation: slideInUp 0.3s ease;
            }

            .cms-upload-area {
                border: 2px dashed #d1d5db;
                border-radius: 12px;
                padding: 40px 20px;
                text-align: center;
                background: #f9fafb;
                transition: all 0.3s ease;
                cursor: pointer;
                margin: 20px 0;
            }

            .cms-upload-area:hover {
                border-color: #f59e0b;
                background: #fffbeb;
            }

            .cms-upload-area.dragover {
                border-color: #f59e0b;
                background: #fffbeb;
                transform: scale(1.02);
            }

            .cms-current-image {
                max-width: 100%;
                max-height: 200px;
                border-radius: 8px;
                margin-bottom: 20px;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            }

            .cms-editor-buttons {
                display: flex;
                gap: 12px;
                justify-content: flex-end;
                margin-top: 20px;
            }

            .cms-btn {
                padding: 10px 20px;
                border: none;
                border-radius: 8px;
                font-weight: 500;
                cursor: pointer;
                transition: all 0.2s ease;
                font-size: 14px;
            }

            .cms-btn-primary {
                background: linear-gradient(135deg, #2563eb, #1d4ed8);
                color: white;
            }

            .cms-btn-primary:hover {
                transform: translateY(-1px);
                box-shadow: 0 4px 12px rgba(37, 99, 235, 0.3);
            }

            .cms-btn-danger {
                background: linear-gradient(135deg, #dc2626, #b91c1c);
                color: white;
            }

            .cms-btn-danger:hover {
                transform: translateY(-1px);
                box-shadow: 0 4px 12px rgba(220, 38, 38, 0.3);
            }

            .cms-btn-secondary {
                background: #f3f4f6;
                color: #374151;
                border: 1px solid #d1d5db;
            }

            .cms-btn-secondary:hover {
                background: #e5e7eb;
            }

            @keyframes fadeIn {
                from { opacity: 0; }
                to { opacity: 1; }
            }

            @keyframes slideInUp {
                from { 
                    opacity: 0;
                    transform: translateY(30px); 
                }
                to { 
                    opacity: 1;
                    transform: translateY(0); 
                }
            }

            @keyframes slideInDown {
                from { 
                    opacity: 0;
                    transform: translateY(-30px); 
                }
                to { 
                    opacity: 1;
                    transform: translateY(0); 
                }
            }
        `;

        const style = document.createElement('style');
        style.id = 'cms-admin-badge-css';
        style.textContent = css;
        document.head.appendChild(style);
    }

    setupImageEditing() {
        if (this.isAdminMode) {
            this.enableImageEditing();
        }
    }

    enableImageEditing() {
        console.log('🖼️ Image editing enabled');
        this.makeImagesEditable();
        this.setupImageClickHandlers();
    }

    makeImagesEditable() {
        // Target specific image sections like team icons, feature images, etc.
        const imageSelectors = [
            'img[data-cms-image]',
            '.team-icon img',
            '.feature-icon img', 
            '.service-icon img',
            '.about-team img',
            '[data-image-key] img'
        ];

        imageSelectors.forEach(selector => {
            const images = document.querySelectorAll(selector);
            images.forEach(img => {
                this.setupImageForEditing(img);
            });
        });
    }

    setupImageForEditing(img) {
        if (!img.classList.contains('cms-editable-image')) {
            img.classList.add('cms-editable-image');
            
            // Get image key from data attributes or generate one
            const imageKey = img.dataset.imageKey || 
                           img.dataset.cmsImage || 
                           this.generateImageKey(img);
            
            img.dataset.imageKey = imageKey;
        }
    }

    generateImageKey(img) {
        // Generate a unique key based on image context
        const container = img.closest('[class*="team"], [class*="icon"], [class*="feature"]');
        const containerClass = container ? container.className.split(' ')[0] : 'image';
        const index = Array.from(document.querySelectorAll(`${containerClass} img`)).indexOf(img);
        return `${containerClass}_${index}_${Date.now()}`;
    }

    setupImageClickHandlers() {
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('cms-editable-image')) {
                e.preventDefault();
                e.stopPropagation();
                this.openImageEditor(e.target);
            }
        });
    }

    openImageEditor(img) {
        const imageKey = img.dataset.cmsImageKey || img.dataset.imageKey;
        const currentSrc = img.src;
        const currentAlt = img.alt || '';

        // Create Bootstrap Modal
        const modalId = 'cmsImageEditorModal_' + Date.now();
        const modal = document.createElement('div');
        modal.className = 'modal fade';
        modal.id = modalId;
        modal.setAttribute('tabindex', '-1');
        modal.setAttribute('aria-labelledby', modalId + 'Label');
        modal.setAttribute('aria-hidden', 'true');
        
        modal.innerHTML = `
            <div class="modal-dialog modal-lg modal-dialog-centered">
                <div class="modal-content" style="border-radius: 15px; border: none; box-shadow: 0 15px 35px rgba(0,0,0,0.1);">
                    <div class="modal-header" style="background: linear-gradient(135deg, #276BE4, #4F46E5); color: white; border-radius: 15px 15px 0 0;">
                        <h5 class="modal-title fw-bold" id="${modalId}Label">
                            <i class="fas fa-camera me-2"></i>
                            Edit Team Image
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    
                    <div class="modal-body" style="padding: 30px;">
                        <!-- Current Image Preview -->
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <h6 class="fw-bold text-muted mb-3">
                                    <i class="fas fa-eye me-2"></i>Current Image
                                </h6>
                                <div class="current-image-preview text-center p-3" style="background: #f8f9fa; border-radius: 10px; border: 2px dashed #e9ecef;">
                                    <img src="${currentSrc}" class="img-fluid rounded-circle" alt="Current image" 
                                         style="width: 100px; height: 100px; object-fit: cover; border: 3px solid #276BE4;">
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <h6 class="fw-bold text-muted mb-3">
                                    <i class="fas fa-upload me-2"></i>New Image Preview
                                </h6>
                                <div id="imagePreviewContainer" class="text-center p-3" style="background: #f8f9fa; border-radius: 10px; border: 2px dashed #e9ecef; min-height: 120px; display: flex; align-items: center; justify-content: center;">
                                    <div class="text-muted">
                                        <i class="fas fa-image mb-2" style="font-size: 24px; opacity: 0.5;"></i>
                                        <div style="font-size: 14px;">No image selected</div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Upload Area -->
                        <div class="cms-upload-area mb-4" id="cmsUploadArea" 
                             style="border: 3px dashed #276BE4; border-radius: 15px; padding: 40px 20px; text-align: center; background: linear-gradient(135deg, rgba(39, 107, 228, 0.05), rgba(79, 70, 229, 0.05)); cursor: pointer; transition: all 0.3s ease;"
                             onmouseover="this.style.borderColor='#1d4ed8'; this.style.background='linear-gradient(135deg, rgba(39, 107, 228, 0.1), rgba(79, 70, 229, 0.1))';"
                             onmouseout="this.style.borderColor='#276BE4'; this.style.background='linear-gradient(135deg, rgba(39, 107, 228, 0.05), rgba(79, 70, 229, 0.05))';">
                            <div class="upload-icon mb-3" style="font-size: 48px; color: #276BE4;">
                                <i class="fas fa-cloud-upload-alt"></i>
                            </div>
                            <h5 class="fw-bold text-primary mb-2">Upload New Team Image</h5>
                            <p class="mb-3 text-muted">
                                Drag & drop your image here or <strong>click to browse</strong>
                            </p>
                            <p class="small text-muted mb-0">
                                <i class="fas fa-info-circle me-1"></i>
                                Supports: JPEG, PNG, GIF, WebP (Max: 5MB)
                            </p>
                            <input type="file" id="cmsFileInput" accept="image/jpeg,image/png,image/gif,image/webp" style="display: none;">
                        </div>

                        <!-- Alt Text Input -->
                        <div class="form-group">
                            <label class="form-label fw-bold text-muted">
                                <i class="fas fa-universal-access me-2"></i>
                                Alt Text (for accessibility)
                            </label>
                            <input type="text" id="cmsAltText" class="form-control form-control-lg" 
                                   value="${currentAlt}"
                                   placeholder="Describe this image for screen readers..." 
                                   style="border-radius: 10px; border: 2px solid #e9ecef; padding: 12px 16px;">
                            <div class="form-text">
                                <i class="fas fa-lightbulb me-1"></i>
                                Good alt text describes the image content and purpose (e.g., "Professional headshot of marketing team lead")
                            </div>
                        </div>
                    </div>
                    
                    <div class="modal-footer" style="background: #f8f9fa; border-radius: 0 0 15px 15px; padding: 20px 30px;">
                        <button type="button" class="btn btn-secondary btn-lg me-2" data-bs-dismiss="modal" style="border-radius: 10px; padding: 12px 24px;">
                            <i class="fas fa-times me-2"></i>Cancel
                        </button>
                        <button type="button" class="btn btn-outline-danger btn-lg me-2" id="cmsRemoveBtn" style="border-radius: 10px; padding: 12px 24px;">
                            <i class="fas fa-trash me-2"></i>Remove Image
                        </button>
                        <button type="button" class="btn btn-primary btn-lg" id="cmsSaveBtn" disabled style="border-radius: 10px; padding: 12px 24px; background: #276BE4; border: none;">
                            <i class="fas fa-upload me-2"></i>Upload & Save
                        </button>
                    </div>
                </div>
            </div>
        `;

        document.body.appendChild(modal);
        
        // Initialize Bootstrap Modal
        const bootstrapModal = new bootstrap.Modal(modal, {
            backdrop: 'static',
            keyboard: true
        });
        
        // Store references
        this.currentImageEditor = { modal, img, imageKey, bootstrapModal };

        // Show modal
        bootstrapModal.show();

        // Setup event listeners
        this.setupImageEditorEvents();

        // Cleanup when modal is hidden
        modal.addEventListener('hidden.bs.modal', () => {
            modal.remove();
            this.currentImageEditor = null;
        });
    }

    setupImageEditorEvents() {
        const { modal, bootstrapModal } = this.currentImageEditor;
        const uploadArea = modal.querySelector('#cmsUploadArea');
        const fileInput = modal.querySelector('#cmsFileInput');
        const saveBtn = modal.querySelector('#cmsSaveBtn');
        const removeBtn = modal.querySelector('#cmsRemoveBtn');
        const previewContainer = modal.querySelector('#imagePreviewContainer');

        // File input handling
        uploadArea.addEventListener('click', () => fileInput.click());
        fileInput.addEventListener('change', (e) => {
            if (e.target.files[0]) {
                this.handleFileSelect(e.target.files[0]);
            }
        });

        // Drag & drop handling
        uploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadArea.style.borderColor = '#1d4ed8';
            uploadArea.style.background = 'linear-gradient(135deg, rgba(29, 78, 216, 0.15), rgba(79, 70, 229, 0.15))';
        });

        uploadArea.addEventListener('dragleave', (e) => {
            e.preventDefault();
            uploadArea.style.borderColor = '#276BE4';
            uploadArea.style.background = 'linear-gradient(135deg, rgba(39, 107, 228, 0.05), rgba(79, 70, 229, 0.05))';
        });

        uploadArea.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadArea.style.borderColor = '#276BE4';
            uploadArea.style.background = 'linear-gradient(135deg, rgba(39, 107, 228, 0.05), rgba(79, 70, 229, 0.05))';
            
            if (e.dataTransfer.files[0]) {
                this.handleFileSelect(e.dataTransfer.files[0]);
            }
        });

        // Button handlers
        removeBtn.addEventListener('click', () => this.removeImage());
        saveBtn.addEventListener('click', () => this.saveImage());

        // Bootstrap modal will handle the close buttons automatically
        // No need for custom backdrop click handling
    }

    handleFileSelect(file) {
        if (!file) return;

        // Validate file type
        const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
        if (!allowedTypes.includes(file.type)) {
            this.showNotification('Please select a valid image file (JPEG, PNG, GIF, WebP)', 'error');
            return;
        }

        // Validate file size (5MB)
        if (file.size > 5 * 1024 * 1024) {
            this.showNotification('File too large. Please select an image under 5MB', 'error');
            return;
        }

        // Preview the image
        const reader = new FileReader();
        reader.onload = (e) => {
            // Update the preview container instead of replacing upload area
            const previewContainer = this.currentImageEditor.modal.querySelector('#imagePreviewContainer');
            previewContainer.innerHTML = `
                <img src="${e.target.result}" class="img-fluid rounded-circle" 
                     style="width: 100px; height: 100px; object-fit: cover; border: 3px solid #28a745;"
                     alt="New image preview">
            `;
            
            // Update upload area to show success state but keep it functional
            const uploadArea = this.currentImageEditor.modal.querySelector('#cmsUploadArea');
            const uploadIcon = uploadArea.querySelector('.upload-icon');
            const uploadText = uploadArea.querySelector('h5');
            const uploadDesc = uploadArea.querySelector('p');
            
            // Change to success state
            uploadArea.style.borderColor = '#28a745';
            uploadArea.style.background = 'linear-gradient(135deg, rgba(40, 167, 69, 0.1), rgba(25, 135, 84, 0.1))';
            uploadIcon.innerHTML = '<i class="fas fa-check-circle"></i>';
            uploadIcon.style.color = '#28a745';
            uploadText.innerHTML = 'Image Selected Successfully!';
            uploadText.className = 'fw-bold text-success mb-2';
            
            // Show file info
            const fileSize = (file.size / 1024).toFixed(1) + ' KB';
            uploadDesc.innerHTML = `
                <span class="text-success">
                    <i class="fas fa-file-image me-1"></i>
                    <strong>${file.name}</strong> (${fileSize})
                </span><br>
                <small class="text-muted">Click to select a different image</small>
            `;
            
            // Enable save button
            this.currentImageEditor.modal.querySelector('#cmsSaveBtn').disabled = false;
            this.currentImageEditor.selectedFile = file;
            
            // Show success notification
            this.showNotification(`Image "${file.name}" ready to upload!`, 'success');
        };
        reader.readAsDataURL(file);
    }

    async saveImage() {
        const { selectedFile, imageKey, img } = this.currentImageEditor;
        if (!selectedFile) return;

        const saveBtn = this.currentImageEditor.modal.querySelector('#cmsSaveBtn');
        const altText = this.currentImageEditor.modal.querySelector('#cmsAltText').value;

        saveBtn.disabled = true;
        saveBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Uploading...';

        try {
            const formData = new FormData();
            formData.append('image', selectedFile);
            formData.append('image_key', imageKey);
            formData.append('alt_text', altText);

            // Use relative path for upload
            const response = await fetch('cms/functions/upload_image.php', {
                method: 'POST',
                body: formData
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const result = await response.json();

            if (result.success) {
                // Update the image source with timestamp to force refresh
                img.src = result.image_path + '?t=' + Date.now();
                if (altText) img.alt = altText;

                this.showNotification('✅ Team image updated successfully!', 'success');
                
                // Close modal after brief delay to show success
                setTimeout(() => {
                    this.closeImageEditor();
                }, 1000);
            } else {
                throw new Error(result.error || 'Upload failed');
            }
        } catch (error) {
            console.error('Image upload error:', error);
            this.showNotification('❌ Failed to upload image: ' + error.message, 'error');
            saveBtn.disabled = false;
            saveBtn.innerHTML = '<i class="fas fa-upload me-2"></i>Upload & Save';
        }
    }

    removeImage() {
        if (confirm('Are you sure you want to remove this image?')) {
            const { img } = this.currentImageEditor;
            img.src = '/assets/images/placeholder-icon.png'; // Use placeholder
            img.alt = 'Image placeholder';
            
            this.showNotification('Image removed', 'success');
            this.closeImageEditor();
        }
    }

    closeImageEditor() {
        if (this.currentImageEditor && this.currentImageEditor.bootstrapModal) {
            // Hide the Bootstrap modal
            this.currentImageEditor.bootstrapModal.hide();
            // The cleanup will happen automatically in the 'hidden.bs.modal' event
        }
    }

    showNotification(message, type = 'info') {
        // Remove existing notification
        const existing = document.querySelector('.cms-notification');
        if (existing) existing.remove();

        const notification = document.createElement('div');
        notification.className = 'cms-notification';
        notification.style.cssText = `
            position: fixed;
            top: 80px;
            right: 20px;
            background: ${type === 'error' ? '#dc2626' : type === 'success' ? '#059669' : '#2563eb'};
            color: white;
            padding: 12px 20px;
            border-radius: 8px;
            z-index: 10001;
            font-weight: 500;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
            animation: slideInDown 0.3s ease;
            max-width: 300px;
        `;
        notification.textContent = message;

        document.body.appendChild(notification);

        // Auto-remove after 4 seconds
        setTimeout(() => {
            if (notification.parentNode) {
                notification.style.animation = 'fadeOut 0.3s ease';
                setTimeout(() => notification.remove(), 300);
            }
        }, 4000);
    }
}

// Initialize the inline image editor - single initialization
if (typeof window.cmsImageEditor === 'undefined') {
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            if (!window.cmsImageEditor) {
                window.cmsImageEditor = new InlineImageEditor();
            }
        });
    } else {
        window.cmsImageEditor = new InlineImageEditor();
    }
}

} // Close the conditional block for class declaration safety check
