<?php
// Save content from inline editor
header('Content-Type: application/json');
error_reporting(E_ALL);
ini_set('display_errors', 0); // Suppress HTML error output to prevent JSON corruption

// Start output buffering to catch any unexpected output
ob_start();

$response = [
    'success' => false,
    'message' => 'Starting save process',
    'debug' => []
];

try {
    $response['debug'][] = 'Step 1: Started save endpoint';
    
    // Force session configuration (same as working test script)
    ini_set('session.save_handler', 'files');
    ini_set('session.save_path', '/tmp');
    ini_set('session.use_cookies', 1);
    ini_set('session.cookie_lifetime', 0);
    ini_set('session.cookie_path', '/');
    ini_set('session.cookie_domain', '');
    ini_set('session.cookie_secure', 0);
    ini_set('session.cookie_httponly', 1);
    
    $response['debug'][] = 'Step 2: Session configuration set';
    
    // Start session
    session_start();
    $response['debug'][] = 'Step 3: Session started - ID: ' . session_id();

    // Check admin authentication (CMS admin OR main application admin)
    $isAuthorized = false;
    $adminId = null;
    
    if (isset($_SESSION['cms_admin_id']) && !empty($_SESSION['cms_admin_id'])) {
        // CMS admin login (separate CMS system)
        $isAuthorized = true;
        $adminId = $_SESSION['cms_admin_id'];
        $response['debug'][] = 'Step 4: CMS Admin authenticated - ID: ' . $adminId;
    } elseif (isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin' && isset($_SESSION['logged_in']) && $_SESSION['logged_in'] == 1) {
        // Main application admin login (unified system)
        $isAuthorized = true;
        $adminId = $_SESSION['user_id'] ?? 'main_admin';
        $response['debug'][] = 'Step 4: Main App Admin authenticated - ID: ' . $adminId . ', Role: ' . $_SESSION['user_role'];
    }
    
    if (!$isAuthorized) {
        $response['message'] = 'Not authorized - admin not logged in';
        $response['debug'][] = 'Step 4: Admin authentication FAILED - No valid admin session found';
        $response['debug'][] = 'Session data: cms_admin_id=' . ($_SESSION['cms_admin_id'] ?? 'NOT SET') . ', user_role=' . ($_SESSION['user_role'] ?? 'NOT SET') . ', logged_in=' . ($_SESSION['logged_in'] ?? 'NOT SET');
        echo json_encode($response);
        exit();
    }

    // Check POST data
    $response['debug'][] = 'Step 5: POST data received - ' . json_encode($_POST);
    
    if (empty($_POST['page_name']) || empty($_POST['section_key']) || !isset($_POST['content'])) {
        $response['message'] = 'Missing required POST data';
        echo json_encode($response);
        exit();
    }
    
    $response['debug'][] = 'Step 6: POST data validation passed';

    // Include database files
    if (file_exists('../../includes/db.php')) {
        require_once '../../includes/db.php';
        $response['debug'][] = 'Step 7: Database file included';
    } else {
        throw new Exception('Database file not found');
    }
    
    if (file_exists('cms_functions.php')) {
        require_once 'cms_functions.php';
        $response['debug'][] = 'Step 8: CMS functions included';
    } else {
        throw new Exception('CMS functions file not found');
    }

    // Extract POST data
    $page_name = trim($_POST['page_name']);
    $section_key = trim($_POST['section_key']);
    $content = $_POST['content'];
    $admin_id = $_SESSION['cms_admin_id'];
    
    $response['debug'][] = "Step 9: Extracted data - Page: {$page_name}, Section: {$section_key}, Admin: {$admin_id}";
    $response['debug'][] = "Step 10: Content length: " . strlen($content) . " characters";

    // Database operations
    $response['debug'][] = "Step 11: Calling ensureCMSDatabase()";
    if (!ensureCMSDatabase()) {
        throw new Exception('ensureCMSDatabase() returned false');
    }
    $response['debug'][] = "Step 12: Database ensured successfully";

    global $conn;
    if (!$conn) {
        throw new Exception('Global $conn variable is null');
    }
    $response['debug'][] = "Step 13: Database connection confirmed";

    // Test the connection
    if (!$conn->ping()) {
        throw new Exception('Database connection lost');
    }
    $response['debug'][] = "Step 14: Database ping successful";

    // Check if cms_content table exists
    $table_check = $conn->query("SHOW TABLES LIKE 'cms_content'");
    if ($table_check->num_rows == 0) {
        throw new Exception('cms_content table does not exist in database');
    }
    $response['debug'][] = "Step 15: cms_content table exists";
    
    // Check table structure to see what columns exist
    $columns_result = $conn->query("DESCRIBE cms_content");
    $columns = [];
    while ($row = $columns_result->fetch_assoc()) {
        $columns[] = $row['Field'];
    }
    $response['debug'][] = "Step 15b: cms_content columns: " . implode(', ', $columns);

    // Check if content exists
    $check_sql = "SELECT id FROM cms_content WHERE page_name = ? AND section_key = ?";
    $response['debug'][] = "Step 16: Preparing SQL: " . $check_sql;
    
    $check_stmt = $conn->prepare($check_sql);
    if ($check_stmt === false) {
        throw new Exception('SQL prepare failed: ' . $conn->error);
    }
    $response['debug'][] = "Step 17: SQL prepared successfully";
    
    $check_stmt->bind_param("ss", $page_name, $section_key);
    $response['debug'][] = "Step 18: Parameters bound successfully";
    
    $check_stmt->execute();
    $response['debug'][] = "Step 19: Query executed successfully";
    
    $result = $check_stmt->get_result();
    $response['debug'][] = "Step 20: Results retrieved - found " . $result->num_rows . " matching records";

    if ($result->num_rows > 0) {
        // Update existing record
        $response['debug'][] = "Step 21: Record exists, performing UPDATE";
        $update_sql = "UPDATE cms_content SET content_value = ?, content_type = 'html', updated_at = NOW() WHERE page_name = ? AND section_key = ?";
        
        $update_stmt = $conn->prepare($update_sql);
        if ($update_stmt === false) {
            throw new Exception('UPDATE prepare failed: ' . $conn->error);
        }
        $response['debug'][] = "Step 22: UPDATE statement prepared";
        
        $update_stmt->bind_param("sss", $content, $page_name, $section_key);
        $response['debug'][] = "Step 23: UPDATE parameters bound";
        
        if ($update_stmt->execute()) {
            $response['success'] = true;
            $response['message'] = 'Content updated successfully';
            $response['debug'][] = "Step 24: UPDATE executed successfully";
        } else {
            throw new Exception('UPDATE execution failed: ' . $update_stmt->error);
        }
    } else {
        // Insert new record
        $response['debug'][] = "Step 21: Record does not exist, performing INSERT";
        $insert_sql = "INSERT INTO cms_content (page_name, section_key, content_value, content_type, display_order, is_active, created_at, updated_at) VALUES (?, ?, ?, 'html', 0, 1, NOW(), NOW())";
        
        $insert_stmt = $conn->prepare($insert_sql);
        if ($insert_stmt === false) {
            throw new Exception('INSERT prepare failed: ' . $conn->error);
        }
        $response['debug'][] = "Step 22: INSERT statement prepared";
        
        $insert_stmt->bind_param("sss", $page_name, $section_key, $content);
        $response['debug'][] = "Step 23: INSERT parameters bound";
        
        if ($insert_stmt->execute()) {
            $response['success'] = true;
            $response['message'] = 'Content created successfully';
            $response['debug'][] = "Step 24: INSERT executed successfully";
        } else {
            throw new Exception('INSERT execution failed: ' . $insert_stmt->error);
        }
    }

} catch (Exception $e) {
    $response['success'] = false;
    $response['message'] = $e->getMessage();
    $response['debug'][] = 'Exception: ' . $e->getMessage();
} catch (Error $e) {
    $response['success'] = false;
    $response['message'] = 'Fatal Error: ' . $e->getMessage();
    $response['debug'][] = 'Fatal Error: ' . $e->getMessage();
}

// Clean any unwanted output and ensure clean JSON
$unwanted_output = ob_get_clean();
if (!empty($unwanted_output)) {
    $response['debug'][] = 'Unwanted output captured: ' . $unwanted_output;
}

// Output clean JSON only
echo json_encode($response);
exit();
?>
