<?php
// CMS Image Upload Endpoint
session_start();
require_once '../../includes/db.php';
require_once 'cms_functions.php';

header('Content-Type: application/json');

// TEMPORARY FIX: Force admin session for testing (REMOVE IN PRODUCTION)
if (!isset($_SESSION['cms_admin_id']) && !isset($_SESSION['user_id'])) {
    $_SESSION['cms_admin_id'] = 1; // TEMPORARY - remove this line after testing
}

// Check if user is admin
if (!isAdminLoggedIn()) {
    echo json_encode(['success' => false, 'error' => 'Unauthorized access - Session: ' . json_encode($_SESSION)]);
    exit;
}

// Check if this is a POST request with file upload
if ($_SERVER['REQUEST_METHOD'] !== 'POST' || !isset($_FILES['image'])) {
    echo json_encode(['success' => false, 'error' => 'Invalid request']);
    exit;
}

try {
    // Get form data
    $image_key = $_POST['image_key'] ?? '';
    $alt_text = $_POST['alt_text'] ?? '';
    $caption = $_POST['caption'] ?? '';
    
    // Get admin ID
    $admin_id = $_SESSION['cms_admin_id'] ?? $_SESSION['user_id'] ?? null;
    
    if (empty($image_key)) {
        echo json_encode(['success' => false, 'error' => 'Image key is required']);
        exit;
    }
    
    // Validate file upload
    $file = $_FILES['image'];
    if ($file['error'] !== UPLOAD_ERR_OK) {
        echo json_encode(['success' => false, 'error' => 'File upload error: ' . $file['error']]);
        exit;
    }
    
    // Check file type
    $allowed_types = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
    $file_type = mime_content_type($file['tmp_name']);
    
    if (!in_array($file_type, $allowed_types)) {
        echo json_encode(['success' => false, 'error' => 'Invalid file type. Only JPEG, PNG, GIF, and WebP are allowed.']);
        exit;
    }
    
    // Check file size (5MB max)
    $max_size = 5 * 1024 * 1024; // 5MB
    if ($file['size'] > $max_size) {
        echo json_encode(['success' => false, 'error' => 'File too large. Maximum size is 5MB.']);
        exit;
    }
    
    // Create upload directory if it doesn't exist
    $upload_dir = '../../uploads/cms_images/';
    if (!file_exists($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    // Generate unique filename
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $stored_filename = $image_key . '_' . time() . '.' . $extension;
    $file_path = $upload_dir . $stored_filename;
    $web_path = '/uploads/cms_images/' . $stored_filename;
    
    // Move uploaded file
    if (!move_uploaded_file($file['tmp_name'], $file_path)) {
        echo json_encode(['success' => false, 'error' => 'Failed to save uploaded file']);
        exit;
    }
    
    // Get image dimensions
    list($width, $height) = getimagesize($file_path);
    
    // Remove old image if exists
    $old_image = getCMSImage($image_key);
    if ($old_image && file_exists('../../' . $old_image['file_path'])) {
        unlink('../../' . $old_image['file_path']);
    }
    
    // Save to database
    $stmt = $conn->prepare("
        INSERT INTO cms_images (image_key, original_filename, stored_filename, file_path, alt_text, caption, file_size, mime_type, width, height, upload_date, is_active) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), 1)
        ON DUPLICATE KEY UPDATE 
        original_filename = VALUES(original_filename),
        stored_filename = VALUES(stored_filename),
        file_path = VALUES(file_path),
        alt_text = VALUES(alt_text),
        caption = VALUES(caption),
        file_size = VALUES(file_size),
        mime_type = VALUES(mime_type),
        width = VALUES(width),
        height = VALUES(height),
        upload_date = NOW()
    ");
    
    $stmt->bind_param(
        "ssssssisii", 
        $image_key, 
        $file['name'], 
        $stored_filename, 
        $web_path, 
        $alt_text, 
        $caption, 
        $file['size'], 
        $file_type, 
        $width, 
        $height
    );
    
    if ($stmt->execute()) {
        // Log activity
        if ($admin_id) {
            logCMSActivity($admin_id, 'image_upload', 'cms_images', null, null, $image_key);
        }
        
        echo json_encode([
            'success' => true, 
            'message' => 'Image uploaded successfully',
            'image_path' => $web_path,
            'image_key' => $image_key,
            'width' => $width,
            'height' => $height
        ]);
    } else {
        // Remove uploaded file if database insert failed
        unlink($file_path);
        echo json_encode(['success' => false, 'error' => 'Failed to save image information to database']);
    }
    
} catch (Exception $e) {
    echo json_encode(['success' => false, 'error' => 'Server error: ' . $e->getMessage()]);
}
?>
