<?php
/**
 * Email Drafts Service for Panadite Academy
 * Handle saving, loading, and managing email drafts
 */

require_once 'db_connect.php';

/**
 * Create drafts table if it doesn't exist
 */
function createDraftsTable() {
    global $pdo;
    
    try {
        $sql = "CREATE TABLE IF NOT EXISTS email_drafts (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            to_email VARCHAR(500) DEFAULT '',
            cc_email TEXT DEFAULT '',
            bcc_email TEXT DEFAULT '',
            subject VARCHAR(500) DEFAULT '',
            body LONGTEXT DEFAULT '',
            priority ENUM('low', 'normal', 'high') DEFAULT 'normal',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            INDEX idx_user_id (user_id),
            INDEX idx_created_at (created_at)
        )";
        
        $pdo->exec($sql);
        return true;
    } catch (PDOException $e) {
        error_log("Error creating drafts table: " . $e->getMessage());
        return false;
    }
}

/**
 * Save email draft
 */
function saveDraft($userId, $to, $cc, $bcc, $subject, $body, $priority = 'normal', $draftId = null) {
    global $pdo;
    
    // Ensure table exists
    createDraftsTable();
    
    try {
        if ($draftId) {
            // Update existing draft
            $stmt = $pdo->prepare("
                UPDATE email_drafts 
                SET to_email = ?, cc_email = ?, bcc_email = ?, subject = ?, body = ?, priority = ?, updated_at = NOW()
                WHERE id = ? AND user_id = ?
            ");
            $result = $stmt->execute([$to, $cc, $bcc, $subject, $body, $priority, $draftId, $userId]);
            return $result ? $draftId : false;
        } else {
            // Create new draft
            $stmt = $pdo->prepare("
                INSERT INTO email_drafts (user_id, to_email, cc_email, bcc_email, subject, body, priority)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            $result = $stmt->execute([$userId, $to, $cc, $bcc, $subject, $body, $priority]);
            return $result ? $pdo->lastInsertId() : false;
        }
    } catch (PDOException $e) {
        error_log("Error saving draft: " . $e->getMessage());
        return false;
    }
}

/**
 * Get all drafts for a user
 */
function getUserDrafts($userId, $limit = 50, $offset = 0) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT id, to_email, cc_email, bcc_email, subject, body, priority, created_at, updated_at
            FROM email_drafts 
            WHERE user_id = ? 
            ORDER BY updated_at DESC 
            LIMIT ? OFFSET ?
        ");
        $stmt->execute([$userId, $limit, $offset]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error getting user drafts: " . $e->getMessage());
        return [];
    }
}

/**
 * Get single draft by ID
 */
function getDraft($draftId, $userId) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT id, to_email, cc_email, bcc_email, subject, body, priority, created_at, updated_at
            FROM email_drafts 
            WHERE id = ? AND user_id = ?
        ");
        $stmt->execute([$draftId, $userId]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error getting draft: " . $e->getMessage());
        return false;
    }
}

/**
 * Delete draft
 */
function deleteDraft($draftId, $userId) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("DELETE FROM email_drafts WHERE id = ? AND user_id = ?");
        return $stmt->execute([$draftId, $userId]);
    } catch (PDOException $e) {
        error_log("Error deleting draft: " . $e->getMessage());
        return false;
    }
}

/**
 * Get drafts count for user
 */
function getDraftsCount($userId) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM email_drafts WHERE user_id = ?");
        $stmt->execute([$userId]);
        return $stmt->fetchColumn();
    } catch (PDOException $e) {
        error_log("Error getting drafts count: " . $e->getMessage());
        return 0;
    }
}

/**
 * Auto-save draft (for auto-save functionality)
 */
function autoSaveDraft($userId, $to, $cc, $bcc, $subject, $body, $priority = 'normal') {
    // Only save if there's meaningful content
    if (empty(trim($to)) && empty(trim($subject)) && empty(trim($body))) {
        return false;
    }
    
    // Check if there's an existing auto-save draft (subject starts with "[Auto-Save]")
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            SELECT id FROM email_drafts 
            WHERE user_id = ? AND subject LIKE '[Auto-Save]%' 
            ORDER BY updated_at DESC 
            LIMIT 1
        ");
        $stmt->execute([$userId]);
        $existingDraft = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $autoSaveSubject = '[Auto-Save] ' . $subject;
        
        if ($existingDraft) {
            return saveDraft($userId, $to, $cc, $bcc, $autoSaveSubject, $body, $priority, $existingDraft['id']);
        } else {
            return saveDraft($userId, $to, $cc, $bcc, $autoSaveSubject, $body, $priority);
        }
    } catch (PDOException $e) {
        error_log("Error auto-saving draft: " . $e->getMessage());
        return false;
    }
}

/**
 * Format draft preview text
 */
function formatDraftPreview($body, $maxLength = 100) {
    // Strip HTML tags and get plain text
    $plainText = strip_tags($body);
    
    // Remove extra whitespace
    $plainText = preg_replace('/\s+/', ' ', trim($plainText));
    
    // Truncate if needed
    if (strlen($plainText) > $maxLength) {
        $plainText = substr($plainText, 0, $maxLength) . '...';
    }
    
    return $plainText ?: '[No content]';
}
?>
