<?php
/**
 * Email Verification Service for Panadite Academy
 * Handles email verification during registration
 */

require_once 'db_connect.php';
require_once 'fast_email_service.php';

/**
 * Create email verification table if it doesn't exist
 */
function createEmailVerificationTable() {
    global $conn;
    
    if (!$conn) {
        return false;
    }
    
    $sql = "CREATE TABLE IF NOT EXISTS email_verifications (
        verification_id INT AUTO_INCREMENT PRIMARY KEY,
        email VARCHAR(255) NOT NULL,
        username VARCHAR(100) NOT NULL,
        password_hash VARCHAR(255) NOT NULL,
        first_name VARCHAR(100) NOT NULL,
        last_name VARCHAR(100) NOT NULL,
        user_role ENUM('student', 'teacher') NOT NULL,
        verification_token VARCHAR(255) NOT NULL UNIQUE,
        additional_data TEXT NULL,
        expires_at DATETIME NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_token (verification_token),
        INDEX idx_email (email),
        INDEX idx_expires (expires_at)
    ) ENGINE=InnoDB";
    
    try {
        return $conn->query($sql);
    } catch (Exception $e) {
        error_log("Error creating email_verifications table: " . $e->getMessage());
        return false;
    }
}

/**
 * Generate a secure verification token
 */
function generateVerificationToken() {
    return bin2hex(random_bytes(32));
}

/**
 * Store pending registration for email verification
 */
function storePendingRegistration($username, $email, $password, $firstName, $lastName, $role, $additionalData = null, $referralCode = null) {
    global $conn;
    
    // Ensure table exists
    createEmailVerificationTable();
    
    if (!$conn) {
        return ['success' => false, 'message' => 'Database connection not available'];
    }
    
    try {
        // Clean up expired verifications
        cleanupExpiredVerifications();
        
        // Check if email already has pending verification
        $stmt = $conn->prepare("SELECT verification_id FROM email_verifications WHERE email = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            // Delete existing pending verification for this email
            $deleteStmt = $conn->prepare("DELETE FROM email_verifications WHERE email = ?");
            $deleteStmt->bind_param("s", $email);
            $deleteStmt->execute();
        }
        
        // Generate verification token
        $token = generateVerificationToken();
        $passwordHash = password_hash($password, PASSWORD_DEFAULT);
        // Store verification with 10-minute expiry
        $expiresAt = date('Y-m-d H:i:s', strtotime('+10 minutes'));
        
        // Prepare additional data including referral code
        $registrationData = $additionalData ? $additionalData : [];
        if (!empty($referralCode)) {
            $registrationData['referral_code'] = $referralCode;
        }
        $additionalDataJson = !empty($registrationData) ? json_encode($registrationData) : null;
        
        // Store pending registration
        $stmt = $conn->prepare("INSERT INTO email_verifications (email, username, password_hash, first_name, last_name, user_role, verification_token, additional_data, expires_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("sssssssss", $email, $username, $passwordHash, $firstName, $lastName, $role, $token, $additionalDataJson, $expiresAt);
        
        if ($stmt->execute()) {
            return ['success' => true, 'token' => $token, 'verification_id' => $stmt->insert_id];
        } else {
            return ['success' => false, 'message' => 'Failed to store pending registration'];
        }
        
    } catch (Exception $e) {
        error_log("Error storing pending registration: " . $e->getMessage());
        return ['success' => false, 'message' => 'Registration failed due to technical issues'];
    }
}

/**
 * Send verification email - SIMPLIFIED TO MATCH WORKING TEST SCRIPT
 */
function sendVerificationEmail($email, $firstName, $lastName, $token) {
    // Generate verification URL - simplified
    $baseUrl = 'https://panaditeacademy.com'; // Use your working domain directly
    $verificationUrl = $baseUrl . '/verify-email.php?token=' . urlencode($token);
    
    $subject = 'Verify Your Email - Panadite Academy';
    
    // Use the same simple HTML format as the working test script
    $emailBody = '<div style="padding:20px;font-family:Arial;background:#f8f9fa;border-radius:8px;">
        <h2 style="color:#007bff;">🎉 Welcome to Panadite Academy!</h2>
        <p>Hello ' . htmlspecialchars($firstName) . ' ' . htmlspecialchars($lastName) . ',</p>
        <p>Please verify your email address to complete your registration.</p>
        <p><strong>⏰ Important:</strong> This link expires in <strong>10 minutes</strong> for security.</p>
        <div style="text-align:center;margin:20px 0;">
            <a href="' . $verificationUrl . '" 
               style="display:inline-block;background:#28a745;color:white;text-decoration:none;padding:15px 30px;border-radius:8px;font-weight:bold;">
                ✅ Verify Email Address
            </a>
        </div>
        <p><strong>Time sent:</strong> ' . date('Y-m-d H:i:s') . '</p>
        <hr>
        <p style="font-size:12px;color:#666;">
            If you cannot click the button, copy this link: <br>
            ' . $verificationUrl . '
        </p>
    </div>';
    
    // Use the exact same call as the working test script
    $result = sendEmailNow($email, $subject, $emailBody);
    
    // Debug logging
    error_log("VERIFICATION EMAIL: Sending to $email");
    error_log("VERIFICATION EMAIL: Result = " . json_encode($result));
    
    if ($result['success']) {
        error_log("SUCCESS: Verification email sent to $email");
        return ['success' => true, 'message' => 'Verification email sent successfully'];
    } else {
        error_log("ERROR: Failed to send verification email to $email: " . $result['error']);
        return ['success' => false, 'message' => 'Failed to send verification email: ' . $result['error']];
    }
}

/**
 * Verify email token and create user account
 */
function verifyEmailAndCreateAccount($token) {
    global $conn;
    
    if (!$conn) {
        return ['success' => false, 'message' => 'Database connection not available'];
    }
    
    try {
        // Clean up expired verifications first
        cleanupExpiredVerifications();
        
        // Find pending registration
        $stmt = $conn->prepare("SELECT * FROM email_verifications WHERE verification_token = ? AND expires_at > NOW()");
        $stmt->bind_param("s", $token);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            return ['success' => false, 'message' => 'Invalid or expired verification token'];
        }
        
        $verification = $result->fetch_assoc();
        
        // Check if user already exists
        $checkStmt = $conn->prepare("SELECT user_id FROM users WHERE email = ? OR username = ?");
        $checkStmt->bind_param("ss", $verification['email'], $verification['username']);
        $checkStmt->execute();
        $checkResult = $checkStmt->get_result();
        
        if ($checkResult->num_rows > 0) {
            // Clean up the verification record
            $deleteStmt = $conn->prepare("DELETE FROM email_verifications WHERE verification_id = ?");
            $deleteStmt->bind_param("i", $verification['verification_id']);
            $deleteStmt->execute();
            
            return ['success' => false, 'message' => 'Account already exists with this email or username'];
        }
        
        // Parse additional data
        $additionalData = json_decode($verification['additional_data'], true) ?: [];
        
        // Create user account with email verification timestamp
        if ($verification['user_role'] === 'student') {
            // Assign all values to variables (required for bind_param by reference)
            $title = isset($additionalData['title']) ? $additionalData['title'] : '';
            $cell_number = isset($additionalData['cell_number']) ? $additionalData['cell_number'] : '';
            $id_number = isset($additionalData['id_number']) ? $additionalData['id_number'] : '';
            $suburb = isset($additionalData['suburb']) ? $additionalData['suburb'] : '';
            $province = isset($additionalData['province']) ? $additionalData['province'] : '';
            $communication_method = isset($additionalData['communication_method']) ? $additionalData['communication_method'] : '';
            $owns_laptop = isset($additionalData['owns_laptop']) ? $additionalData['owns_laptop'] : '';
            $training_preference = isset($additionalData['training_preference']) ? $additionalData['training_preference'] : '';
            $job_field_interest = isset($additionalData['job_field_interest']) ? $additionalData['job_field_interest'] : '';
            $job_field_other = isset($additionalData['job_field_other']) ? $additionalData['job_field_other'] : '';
            $education_status = isset($additionalData['education_status']) ? $additionalData['education_status'] : '';
            $employment_status = isset($additionalData['employment_status']) ? $additionalData['employment_status'] : '';
            $heard_about_us = isset($additionalData['heard_about_us']) ? $additionalData['heard_about_us'] : '';
            $referral_person_name = isset($additionalData['referral_person_name']) ? $additionalData['referral_person_name'] : '';
            
            // Insert student with all additional fields
            $sql = "INSERT INTO users (username, email, password, first_name, last_name, user_role, email_verified_at, title, cell_number, id_number, suburb, province, communication_method, owns_laptop, training_preference, job_field_interest, job_field_other, education_status, employment_status, heard_about_us, referral_person_name) VALUES (?, ?, ?, ?, ?, ?, NOW(), ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $createStmt = $conn->prepare($sql);
            
            if (!$createStmt) {
                return ['success' => false, 'message' => 'Database prepare error: ' . $conn->error];
            }
            // Fix: 20 parameters total (email_verified_at uses NOW(), not a placeholder)
            $createStmt->bind_param("ssssssssssssssssssss", 
                $verification['username'], 
                $verification['email'], 
                $verification['password_hash'], 
                $verification['first_name'], 
                $verification['last_name'], 
                $verification['user_role'],
                $title,
                $cell_number,
                $id_number,
                $suburb,
                $province,
                $communication_method,
                $owns_laptop,
                $training_preference,
                $job_field_interest,
                $job_field_other,
                $education_status,
                $employment_status,
                $heard_about_us,
                $referral_person_name
            );
        } else {
            // TEACHER REGISTRATION FIX: Use same schema approach as profile fixes
            // Assign all values to variables (required for bind_param by reference)
            $title = isset($additionalData['title']) ? $additionalData['title'] : '';
            $cell_number = isset($additionalData['cell_number']) ? $additionalData['cell_number'] : '';
            $id_number = isset($additionalData['id_number']) ? $additionalData['id_number'] : '';
            $suburb = isset($additionalData['suburb']) ? $additionalData['suburb'] : '';
            $province = isset($additionalData['province']) ? $additionalData['province'] : '';
            // Fix enum validation (same as profile fix)
            $communication_method = in_array($additionalData['communication_method'] ?? '', ['Email', 'WhatsApp']) 
                ? $additionalData['communication_method'] : 'Email';
            $owns_laptop = isset($additionalData['owns_laptop']) ? $additionalData['owns_laptop'] : '';
            // REMOVE training_preference - column doesn't exist!
            $job_field_interest = isset($additionalData['job_field_interest']) ? $additionalData['job_field_interest'] : '';
            // Store teacher experience in job_field_other (same as profile fix)
            $experience = isset($additionalData['experience']) ? $additionalData['experience'] : '';
            $job_field_other = $experience; // Permanent fix: experience goes in job_field_other
            $education_status = isset($additionalData['education_status']) ? $additionalData['education_status'] : '';
            $employment_status = isset($additionalData['employment_status']) ? $additionalData['employment_status'] : '';
            $heard_about_us = isset($additionalData['heard_about_us']) ? $additionalData['heard_about_us'] : '';
            $referral_person_name = isset($additionalData['referral_person_name']) ? $additionalData['referral_person_name'] : '';
            $bio = isset($additionalData['bio']) ? $additionalData['bio'] : '';
            
            // FIXED SQL: Remove training_preference column + correct parameter count
            $sql = "INSERT INTO users (username, email, password, first_name, last_name, user_role, email_verified_at, title, cell_number, id_number, suburb, province, communication_method, owns_laptop, job_field_interest, job_field_other, education_status, employment_status, heard_about_us, referral_person_name, bio) VALUES (?, ?, ?, ?, ?, ?, NOW(), ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $createStmt = $conn->prepare($sql);
            
            if (!$createStmt) {
                return ['success' => false, 'message' => 'Database prepare error: ' . $conn->error];
            }
            
            // FIXED: Correct parameter count (20 parameters)
            $createStmt->bind_param("ssssssssssssssssssss", 
                $verification['username'], 
                $verification['email'], 
                $verification['password_hash'], 
                $verification['first_name'], 
                $verification['last_name'], 
                $verification['user_role'],
                $title,
                $cell_number,
                $id_number,
                $suburb,
                $province,
                $communication_method,
                $owns_laptop,
                $job_field_interest,
                $job_field_other,
                $education_status,
                $employment_status,
                $heard_about_us,
                $referral_person_name,
                $bio
            );
        }
        
        if ($createStmt->execute()) {
            $userId = $createStmt->insert_id;
            
            // Create role-specific profile
            createRoleProfile($userId, $verification['user_role'], $verification['additional_data']);
            
            // Create referral code for the new user
            createReferralCodeForNewUser($userId, $verification['user_role'], $verification['username']);
            
            // Process referral on email verification (AFTER user account is created)
            processReferralOnVerification($userId, $verification['additional_data'], $verification['email']);
            
            // Clean up verification record
            $deleteStmt = $conn->prepare("DELETE FROM email_verifications WHERE verification_id = ?");
            $deleteStmt->bind_param("i", $verification['verification_id']);
            $deleteStmt->execute();
            
            return [
                'success' => true, 
                'message' => 'Email verified successfully! Your account has been created and you can now login.',
                'user_id' => $userId,
                'role' => $verification['user_role']
            ];
        } else {
            $executeError = $createStmt->error ?: $conn->error;
            return ['success' => false, 'message' => 'Failed to create user account: ' . $executeError];
        }
        
    } catch (Exception $e) {
        error_log('Profile creation error: ' . $e->getMessage());
    }
}

/**
 * Create referral code for the new user
 */
function createReferralCodeForNewUser($userId, $role, $username) {
    global $conn;
    
    try {
        // Check if user already has a referral code
        $existingQuery = "SELECT referral_code FROM referrals WHERE referrer_id = ? AND referrer_type = ?";
        $existingStmt = $conn->prepare($existingQuery);
        $existingStmt->bind_param("is", $userId, $role);
        $existingStmt->execute();
        $existingResult = $existingStmt->get_result();
        
        if ($existingResult->num_rows > 0) {
            error_log("User $userId already has a referral code");
            return;
        }
        
        // Generate referral code based on user role and username
        $usernamePrefix = strtoupper(substr($username, 0, 2));
        
        if ($role === 'teacher') {
            // Teacher pattern: TBN + first 2 letters of username + U + user_id
            $referralCode = 'TBN' . $usernamePrefix . 'U' . $userId;
        } else {
            // Student pattern: KABR + first 2 letters of username + user_id
            $referralCode = 'KABR' . $usernamePrefix . $userId;
            
            // Alternative student pattern if code is too long
            if (strlen($referralCode) > 10) {
                $referralCode = 'KABR' . $userId;
            }
        }
        
        // Insert referral code into referrals table
        $insertQuery = "INSERT INTO referrals (referrer_id, referrer_type, referral_code, status, created_at) VALUES (?, ?, ?, 'active', NOW())";
        $insertStmt = $conn->prepare($insertQuery);
        
        if ($insertStmt) {
            $insertStmt->bind_param("iss", $userId, $role, $referralCode);
            
            if ($insertStmt->execute()) {
                error_log("Created referral code '$referralCode' for new $role user ID: $userId ($username)");
            } else {
                error_log("Failed to create referral code for user ID: $userId - " . $insertStmt->error);
            }
        } else {
            error_log("Failed to prepare referral code insert statement: " . $conn->error);
        }
        
    } catch (Exception $e) {
        error_log("Error creating referral code for user $userId: " . $e->getMessage());
    }
}

/**
 * Process referral code when email is verified and account is created
 */
function processReferralOnVerification($userId, $additionalDataJson, $userEmail) {
    global $conn; // Access the global database connection
    
    try {
        $additionalData = $additionalDataJson ? json_decode($additionalDataJson, true) : [];
        
        if (!empty($additionalData['referral_code'])) {
            $referralCode = $additionalData['referral_code'];
            
            // Include referral processor
            if (file_exists(__DIR__ . '/../includes/referral_processor.php')) {
                require_once __DIR__ . '/../includes/referral_processor.php';
                
                $referralProcessor = new ReferralProcessor($conn);
                $result = $referralProcessor->processReferralRegistration($referralCode, $userId, $userEmail);
                
                if ($result['success']) {
                    error_log("Referral processed successfully for user ID: $userId with code: $referralCode");
                } else {
                    error_log("Referral processing failed for user ID: $userId with code: $referralCode - " . $result['message']);
                }
            }
        }
    } catch (Exception $e) {
        error_log('Referral processing error during verification: ' . $e->getMessage());
    }
}

/**
 * Create role-specific profile after account creation
 */
function createRoleProfile($userId, $role, $additionalDataJson) {
    global $conn;
    
    try {
        $additionalData = $additionalDataJson ? json_decode($additionalDataJson, true) : [];
        
        if ($role === 'student') {
            if (tableExists($conn, 'student_profiles')) {
                $profileStmt = $conn->prepare("INSERT INTO student_profiles (student_id) VALUES (?)");
                if ($profileStmt) {
                    $profileStmt->bind_param("i", $userId);
                    $profileStmt->execute();
                }
            }
        } else if ($role === 'teacher') {
            // CRITICAL FIX: Create teachers table FIRST to get teacher_id for teacher_profiles
            if (tableExists($conn, 'teachers')) {
                // Get user data for teachers table
                $userQuery = "SELECT first_name, last_name, bio FROM users WHERE user_id = ?";
                $userStmt = $conn->prepare($userQuery);
                $userStmt->bind_param("i", $userId);
                $userStmt->execute();
                $userResult = $userStmt->get_result();
                $userData = $userResult->fetch_assoc();
                
                $first_name = $userData['first_name'] ?? '';
                $last_name = $userData['last_name'] ?? '';
                $user_bio = $userData['bio'] ?? '';
                $expertise = $additionalData['course_title'] ?? '';  // course_title → teachers.expertise
                $website = $additionalData['portfolio_url'] ?? '';
                $linkedin = $additionalData['linkedin_profile'] ?? '';
                
                // FIXED: Create teachers record first to get teacher_id
                $teacherStmt = $conn->prepare("INSERT INTO teachers (user_id, first_name, last_name, bio, expertise, website, social_linkedin) VALUES (?, ?, ?, ?, ?, ?, ?)");
                if ($teacherStmt) {
                    $teacherStmt->bind_param("issssss", $userId, $first_name, $last_name, $user_bio, $expertise, $website, $linkedin);
                    if ($teacherStmt->execute()) {
                        // Get the teacher_id that was just created
                        $teacher_id = $conn->insert_id;
                        
                        // NOW create teacher_profiles using the correct teacher_id
                        if (tableExists($conn, 'teacher_profiles')) {
                            $bio = $additionalData['course_description'] ?? '';  // course_description → teacher_profiles.bio
                            $qualifications = $additionalData['qualifications'] ?? '';
                            $expertise_profile = $additionalData['course_title'] ?? '';  // course_title → teacher_profiles.expertise
                            
                            $profileStmt = $conn->prepare("INSERT INTO teacher_profiles (teacher_id, bio, qualifications, expertise) VALUES (?, ?, ?, ?)");
                            if ($profileStmt) {
                                $profileStmt->bind_param("isss", $teacher_id, $bio, $qualifications, $expertise_profile);
                                $profileStmt->execute();
                            }
                        }
                    }
                }
            }
        }
    } catch (Exception $e) {
        error_log("Error creating profile for user $userId: " . $e->getMessage());
    }
}

/**
 * Clean up expired verification tokens
 */
function cleanupExpiredVerifications() {
    global $conn;
    
    if (!$conn) {
        return;
    }
    
    try {
        $stmt = $conn->prepare("DELETE FROM email_verifications WHERE expires_at < NOW()");
        $stmt->execute();
    } catch (Exception $e) {
        error_log("Error cleaning up expired verifications: " . $e->getMessage());
    }
}

/**
 * Check if username or email already exists
 */
function checkExistingUser($username, $email) {
    global $conn;
    
    if (!$conn) {
        return ['success' => false, 'message' => 'Database connection not available'];
    }
    
    try {
        // Check username
        $stmt = $conn->prepare("SELECT user_id FROM users WHERE username = ?");
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            return ['success' => false, 'message' => 'Username already exists'];
        }
        
        // Check email
        $stmt = $conn->prepare("SELECT user_id FROM users WHERE email = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            return ['success' => false, 'message' => 'Email already exists'];
        }
        
        return ['success' => true];
        
    } catch (Exception $e) {
        error_log("Error checking existing user: " . $e->getMessage());
        return ['success' => false, 'message' => 'Failed to validate user information'];
    }
}

/**
 * Check if table exists helper function
 */
if (!function_exists('tableExists')) {
    function tableExists($conn, $tableName) {
        if (!$conn) return false;
        $result = $conn->query("SHOW TABLES LIKE '{$tableName}'");
        return ($result && $result->num_rows > 0);
    }
}

/**
 * Resend verification email
 */
function resendVerificationEmail($email) {
    global $conn;
    
    if (!$conn) {
        return ['success' => false, 'message' => 'Database connection not available'];
    }
    
    try {
        $stmt = $conn->prepare("SELECT * FROM email_verifications WHERE email = ? AND expires_at > NOW()");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            return ['success' => false, 'message' => 'No pending verification found for this email'];
        }
        
        $verification = $result->fetch_assoc();
        
        return sendVerificationEmail(
            $verification['email'], 
            $verification['first_name'], 
            $verification['last_name'], 
            $verification['verification_token']
        );
        
    } catch (Exception $e) {
        error_log("Error resending verification email: " . $e->getMessage());
        return ['success' => false, 'message' => 'Failed to resend verification email'];
    }
}
?>
