<?php
/**
 * Fast Email Service for Panadite Academy
 * Lightweight system using native PHP functions with background processing
 * No PHPMailer dependency - Fast and efficient
 */

require_once 'db_connect.php';

/**
 * Get SMTP settings from database
 */
function getEmailSettings() {
    global $pdo;
    
    $settings = [
        'smtp_enabled' => '0',
        'smtp_host' => '',
        'smtp_port' => '587',
        'smtp_encryption' => 'tls',
        'smtp_username' => '',
        'smtp_password' => '',
        'from_email' => '',
        'from_name' => 'Panadite Academy',
        'smtp_auth' => '1'
    ];
    
    if ($pdo) {
        try {
            $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM system_settings WHERE setting_group = 'email'");
            $stmt->execute();
            $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            foreach ($results as $row) {
                $settings[$row['setting_key']] = $row['setting_value'];
            }
        } catch (Exception $e) {
            error_log('Error loading email settings: ' . $e->getMessage());
        }
    }
    
    return $settings;
}

/**
 * Queue email for background processing (INSTANT - no delays)
 */
function queueEmail($to, $subject, $body, $type = 'general', $priority = 'normal') {
    global $pdo;
    
    if (!$pdo) return false;
    
    try {
        // Create email queue table if not exists
        $createQueueSQL = "
        CREATE TABLE IF NOT EXISTS `email_queue` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `to_email` varchar(255) NOT NULL,
            `subject` varchar(500) NOT NULL,
            `body` longtext NOT NULL,
            `type` varchar(50) DEFAULT 'general',
            `priority` enum('low','normal','high') DEFAULT 'normal',
            `status` enum('queued','sending','sent','failed') DEFAULT 'queued',
            `attempts` int(3) DEFAULT 0,
            `max_attempts` int(3) DEFAULT 3,
            `error_message` text,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            `send_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            `sent_at` timestamp NULL,
            PRIMARY KEY (`id`),
            KEY `status` (`status`),
            KEY `priority` (`priority`),
            KEY `send_at` (`send_at`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        
        $pdo->exec($createQueueSQL);
        
        // Insert email into queue
        $stmt = $pdo->prepare("INSERT INTO email_queue (to_email, subject, body, type, priority, send_at) VALUES (?, ?, ?, ?, ?, NOW())");
        $stmt->execute([$to, $subject, $body, $type, $priority]);
        
        return true;
    } catch (Exception $e) {
        error_log('Error queueing email: ' . $e->getMessage());
        return false;
    }
}

/**
 * Send email instantly (for critical emails)
 */
function sendEmailNow($to, $subject, $body) {
    $settings = getEmailSettings();
    
    // Log attempt
    logEmailAttempt($to, $subject, 'sending', 'Attempting to send email');
    
    $result = false;
    $error = '';
    
    if ($settings['smtp_enabled'] !== '1') {
        // SMTP not enabled, try native mail
        $result = sendEmailNative($to, $subject, $body);
        if (!$result) {
            $error = 'Native mail() function failed. SMTP is not configured.';
        }
    } else {
        // SMTP is enabled, try secure methods
        // 1. Try SMTPS (implicit TLS on port 465) - most reliable
        $result = sendEmailSMTPS($to, $subject, $body, $settings);
        
        if (!$result) {
            // 2. Fallback to STARTTLS if SMTPS fails
            $result = sendEmailSMTP($to, $subject, $body, $settings);
            
            if (!$result) {
                // 3. Last resort - try without TLS
                $result = sendEmailSMTPNoTLS($to, $subject, $body, $settings);
                
                if (!$result) {
                    $error = 'All SMTP methods failed. Check SMTP configuration.';
                }
            }
        }
    }
    
    // Log final result
    logEmailAttempt($to, $subject, $result ? 'sent' : 'failed', $error);
    
    // Return format expected by email verification service
    return [
        'success' => $result,
        'error' => $error
    ];
}

/**
 * Native PHP mail() function - Ultra fast
 */
function sendEmailNative($to, $subject, $body) {
    $settings = getEmailSettings();
    
    $headers = [
        "MIME-Version: 1.0",
        "Content-Type: text/html; charset=UTF-8",
        "From: " . $settings['from_name'] . " <" . $settings['from_email'] . ">",
        "Reply-To: " . $settings['from_email'],
        "X-Mailer: Panadite Academy Fast Email System"
    ];
    
    $success = mail($to, $subject, $body, implode("\r\n", $headers));
    
    logEmailAttempt($to, $subject, $success ? 'sent' : 'failed', $success ? null : 'Native mail() failed');
    
    return $success;
}

/**
 * SMTP using native PHP sockets - Fast and lightweight
 */
function sendEmailSMTP($to, $subject, $body, $settings) {
    $socket = null;
    
    try {
        // Create socket connection
        $socket = fsockopen($settings['smtp_host'], (int)$settings['smtp_port'], $errno, $errstr, 10);
        
        if (!$socket) {
            throw new Exception("Socket connection failed: $errstr ($errno)");
        }
        
        // Set timeout
        stream_set_timeout($socket, 10);
        
        // Read complete server greeting (may be multi-line)
        $greeting = '';
        do {
            $line = fgets($socket, 512);
            $greeting .= $line;
        } while (substr($line, 3, 1) == '-');
        
        if (substr($greeting, 0, 3) != '220') {
            throw new Exception("Server greeting failed: $greeting");
        }
        
        // Send EHLO and read complete response
        fputs($socket, "EHLO " . ($_SERVER['HTTP_HOST'] ?? 'localhost') . "\r\n");
        $ehlo_response = '';
        do {
            $line = fgets($socket, 512);
            $ehlo_response .= $line;
        } while (substr($line, 3, 1) == '-');
        
        // Start TLS if required
        if (isset($settings['smtp_encryption']) && $settings['smtp_encryption'] === 'tls') {
            // Check if server supports STARTTLS
            if (stripos($ehlo_response, 'STARTTLS') === false) {
                throw new Exception("Server does not support STARTTLS encryption");
            }
            
            fputs($socket, "STARTTLS\r\n");
            $tls_response = fgets($socket, 512);
            
            if (substr($tls_response, 0, 3) == '220') {
                // Configure SSL context to handle certificate hostname mismatch
                $ssl_context = [
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true,
                    'disable_compression' => true,
                    'SNI_enabled' => false,
                    'cafile' => null,
                    'peer_name' => 'lon106.truehost.cloud'  // Use actual certificate hostname
                ];
                
                // Apply SSL context to socket
                foreach ($ssl_context as $key => $value) {
                    stream_context_set_option($socket, 'ssl', $key, $value);
                }
                
                // Try TLS with hostname verification disabled (secure but handles cert mismatch)
                if (stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT)) {
                    // TLS 1.2 enabled successfully
                } else if (stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLSv1_1_CLIENT)) {
                    // TLS 1.1 fallback successful
                } else if (stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT)) {
                    // Generic TLS fallback successful
                } else {
                    throw new Exception("Failed to establish secure TLS connection - all TLS methods failed");
                }
                
                // Send EHLO again after TLS
                fputs($socket, "EHLO " . ($_SERVER['HTTP_HOST'] ?? 'localhost') . "\r\n");
                $post_tls_response = '';
                do {
                    $line = fgets($socket, 512);
                    $post_tls_response .= $line;
                } while (substr($line, 3, 1) == '-');
                
            } else {
                throw new Exception("STARTTLS failed: $tls_response");
            }
        }
        
        // Authenticate if required
        if ($settings['smtp_auth'] === '1' && !empty($settings['smtp_username'])) {
            fputs($socket, "AUTH LOGIN\r\n");
            $response = fgets($socket, 512);
            
            fputs($socket, base64_encode($settings['smtp_username']) . "\r\n");
            $response = fgets($socket, 512);
            
            fputs($socket, base64_encode($settings['smtp_password']) . "\r\n");
            $response = fgets($socket, 512);
            
            if (substr($response, 0, 3) != '235') {
                throw new Exception("Authentication failed: $response");
            }
        }
        
        // Send mail from
        fputs($socket, "MAIL FROM: <" . $settings['from_email'] . ">\r\n");
        $response = fgets($socket, 512);
        
        // Send mail to
        fputs($socket, "RCPT TO: <$to>\r\n");
        $response = fgets($socket, 512);
        
        // Send data command
        fputs($socket, "DATA\r\n");
        $response = fgets($socket, 512);
        
        // Send headers and body
        $headers = [
            "From: " . $settings['from_name'] . " <" . $settings['from_email'] . ">",
            "To: $to",
            "Subject: $subject",
            "MIME-Version: 1.0",
            "Content-Type: text/html; charset=UTF-8",
            "Date: " . date('r'),
            ""
        ];
        
        fputs($socket, implode("\r\n", $headers) . "\r\n");
        fputs($socket, $body . "\r\n.\r\n");
        
        $response = fgets($socket, 512);
        
        // Quit
        fputs($socket, "QUIT\r\n");
        fclose($socket);
        
        $success = substr($response, 0, 3) == '250';
        logEmailAttempt($to, $subject, $success ? 'sent' : 'failed', $success ? null : $response);
        
        return $success;
        
    } catch (Exception $e) {
        if ($socket) fclose($socket);
        logEmailAttempt($to, $subject, 'failed', $e->getMessage());
        return false;
    }
}

/**
 * SMTPS - Implicit TLS on port 465 (Secure alternative to STARTTLS)
 */
function sendEmailSMTPS($to, $subject, $body, $settings) {
    $socket = null;
    
    try {
        // Force port 465 for SMTPS (implicit TLS) - this is what works
        $smtp_port = 465;
        
        // Create SSL context for implicit TLS
        $context = stream_context_create([
            'ssl' => [
                'verify_peer' => false,
                'verify_peer_name' => false,
                'allow_self_signed' => true,
                'disable_compression' => true,
                'SNI_enabled' => false,
                'crypto_method' => STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT
            ]
        ]);
        
        // Create encrypted socket connection (TLS from start)
        $socket = stream_socket_client(
            "ssl://{$settings['smtp_host']}:$smtp_port",
            $errno,
            $errstr,
            30,
            STREAM_CLIENT_CONNECT,
            $context
        );
        
        if (!$socket) {
            throw new Exception("SMTPS connection failed: $errstr ($errno)");
        }
        
        stream_set_timeout($socket, 30);
        
        // Read complete server greeting (may be multi-line)
        $greeting = '';
        do {
            $line = fgets($socket, 512);
            $greeting .= $line;
        } while (substr($line, 3, 1) == '-');
        
        if (substr($greeting, 0, 3) != '220') {
            throw new Exception("SMTPS server greeting failed: $greeting");
        }
        
        // Send EHLO and read complete response
        fputs($socket, "EHLO " . ($_SERVER['HTTP_HOST'] ?? 'localhost') . "\r\n");
        $ehlo_response = '';
        do {
            $line = fgets($socket, 512);
            $ehlo_response .= $line;
        } while (substr($line, 3, 1) == '-');
        
        // Authenticate if required (connection is already TLS encrypted)
        if ($settings['smtp_auth'] === '1' && !empty($settings['smtp_username'])) {
            fputs($socket, "AUTH LOGIN\r\n");
            $response = fgets($socket, 512);
            
            if (substr($response, 0, 3) != '334') {
                throw new Exception("AUTH LOGIN not supported: $response");
            }
            
            fputs($socket, base64_encode($settings['smtp_username']) . "\r\n");
            $response = fgets($socket, 512);
            
            if (substr($response, 0, 3) != '334') {
                throw new Exception("Username rejected: $response");
            }
            
            fputs($socket, base64_encode($settings['smtp_password']) . "\r\n");
            $response = fgets($socket, 512);
            
            if (substr($response, 0, 3) != '235') {
                throw new Exception("Authentication failed: $response");
            }
        }
        
        // Send mail from
        fputs($socket, "MAIL FROM: <" . $settings['from_email'] . ">\r\n");
        $response = fgets($socket, 512);
        
        if (substr($response, 0, 3) != '250') {
            throw new Exception("MAIL FROM failed: $response");
        }
        
        // Send rcpt to
        fputs($socket, "RCPT TO: <$to>\r\n");
        $response = fgets($socket, 512);
        
        if (substr($response, 0, 3) != '250') {
            throw new Exception("RCPT TO failed: $response");
        }
        
        // Send data
        fputs($socket, "DATA\r\n");
        $response = fgets($socket, 512);
        
        if (substr($response, 0, 3) != '354') {
            throw new Exception("DATA command failed: $response");
        }
        
        // Send headers and body
        $headers = [
            "From: " . $settings['from_name'] . " <" . $settings['from_email'] . ">",
            "To: $to",
            "Subject: $subject",
            "MIME-Version: 1.0",
            "Content-Type: text/html; charset=UTF-8",
            "Date: " . date('r'),
            "Message-ID: <" . time() . "." . uniqid() . "@" . ($_SERVER['HTTP_HOST'] ?? 'localhost') . ">",
            ""
        ];
        
        fputs($socket, implode("\r\n", $headers) . "\r\n");
        fputs($socket, $body . "\r\n.\r\n");
        
        $response = fgets($socket, 512);
        
        // Quit
        fputs($socket, "QUIT\r\n");
        fclose($socket);
        
        $success = substr($response, 0, 3) == '250';
        logEmailAttempt($to, $subject, $success ? 'sent' : 'failed', $success ? null : $response);
        
        return $success;
        
    } catch (Exception $e) {
        if ($socket) fclose($socket);
        logEmailAttempt($to, $subject, 'failed', $e->getMessage());
        return false;
    }
}

/**
 * Alternative SMTP without TLS (fallback method)
 */
function sendEmailSMTPNoTLS($to, $subject, $body, $settings) {
    $socket = null;
    
    try {
        // Try port 25 or 2525 for non-TLS
        $ports = [25, 2525, 587];
        $connected = false;
        
        foreach ($ports as $port) {
            $socket = @fsockopen($settings['smtp_host'], $port, $errno, $errstr, 5);
            if ($socket) {
                $connected = true;
                break;
            }
        }
        
        if (!$connected) {
            throw new Exception("Could not connect to SMTP server on any port");
        }
        
        stream_set_timeout($socket, 10);
        
        // Read server greeting
        $response = fgets($socket, 512);
        if (substr($response, 0, 3) != '220') {
            throw new Exception("Server greeting failed: $response");
        }
        
        // Send EHLO
        fputs($socket, "EHLO " . ($_SERVER['HTTP_HOST'] ?? 'localhost') . "\r\n");
        $response = fgets($socket, 512);
        
        // Skip TLS - authenticate directly if required
        if ($settings['smtp_auth'] === '1' && !empty($settings['smtp_username'])) {
            fputs($socket, "AUTH LOGIN\r\n");
            $response = fgets($socket, 512);
            
            fputs($socket, base64_encode($settings['smtp_username']) . "\r\n");
            $response = fgets($socket, 512);
            
            fputs($socket, base64_encode($settings['smtp_password']) . "\r\n");
            $response = fgets($socket, 512);
            
            if (substr($response, 0, 3) != '235') {
                throw new Exception("Authentication failed: $response");
            }
        }
        
        // Send mail from
        fputs($socket, "MAIL FROM: <" . $settings['from_email'] . ">\r\n");
        $response = fgets($socket, 512);
        
        // Send rcpt to
        fputs($socket, "RCPT TO: <$to>\r\n");
        $response = fgets($socket, 512);
        
        // Send data
        fputs($socket, "DATA\r\n");
        $response = fgets($socket, 512);
        
        // Send headers and body
        $headers = [
            "From: " . $settings['from_name'] . " <" . $settings['from_email'] . ">",
            "To: $to",
            "Subject: $subject",
            "MIME-Version: 1.0",
            "Content-Type: text/html; charset=UTF-8",
            "Date: " . date('r'),
            ""
        ];
        
        fputs($socket, implode("\r\n", $headers) . "\r\n");
        fputs($socket, $body . "\r\n.\r\n");
        
        $response = fgets($socket, 512);
        
        // Quit
        fputs($socket, "QUIT\r\n");
        fclose($socket);
        
        $success = substr($response, 0, 3) == '250';
        logEmailAttempt($to, $subject, $success ? 'sent' : 'failed', $success ? null : $response);
        
        return $success;
        
    } catch (Exception $e) {
        if ($socket) fclose($socket);
        logEmailAttempt($to, $subject, 'failed', $e->getMessage());
        return false;
    }
}

/**
 * Process email queue in background
 */
function processEmailQueue($limit = 10) {
    global $pdo;
    
    if (!$pdo) return 0;
    
    try {
        // Get queued emails ordered by priority and creation time
        $stmt = $pdo->prepare("
            SELECT * FROM email_queue 
            WHERE status = 'queued' AND attempts < max_attempts AND send_at <= NOW() 
            ORDER BY 
                CASE priority WHEN 'high' THEN 1 WHEN 'normal' THEN 2 ELSE 3 END,
                created_at ASC 
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        $emails = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $processed = 0;
        
        foreach ($emails as $email) {
            // Update status to sending
            $updateStmt = $pdo->prepare("UPDATE email_queue SET status = 'sending', attempts = attempts + 1 WHERE id = ?");
            $updateStmt->execute([$email['id']]);
            
            // Send the email
            $success = sendEmailNow($email['to_email'], $email['subject'], $email['body']);
            
            // Update final status
            if ($success) {
                $finalStmt = $pdo->prepare("UPDATE email_queue SET status = 'sent', sent_at = NOW() WHERE id = ?");
                $finalStmt->execute([$email['id']]);
                $processed++;
            } else {
                $status = $email['attempts'] >= $email['max_attempts'] ? 'failed' : 'queued';
                $errorStmt = $pdo->prepare("UPDATE email_queue SET status = ?, error_message = 'Send attempt failed' WHERE id = ?");
                $errorStmt->execute([$status, $email['id']]);
            }
        }
        
        return $processed;
        
    } catch (Exception $e) {
        error_log('Error processing email queue: ' . $e->getMessage());
        return 0;
    }
}

/**
 * Log email attempts
 */
function logEmailAttempt($to, $subject, $status, $error = null) {
    global $pdo;
    
    if (!$pdo) return;
    
    try {
        // Use the existing email_logs table structure (don't try to create it)
        // Map the simple status to the table's enum values
        $mappedStatus = ($status === 'sent') ? 'sent' : 'failed';
        
        // Insert into the actual table structure that exists
        $stmt = $pdo->prepare("INSERT INTO email_logs (to_email, subject, status, error_details, delivery_timestamp, created_at) VALUES (?, ?, ?, ?, NOW(), NOW())");
        $stmt->execute([$to, $subject, $mappedStatus, $error]);
        
        // Also log success to error log for debugging
        error_log("Email logged successfully: $to - $subject - $mappedStatus");
        
    } catch (Exception $e) {
        error_log('Error logging email: ' . $e->getMessage());
        
        // Try alternative insert method in case column names are different
        try {
            $stmt = $pdo->prepare("INSERT INTO email_logs (to_email, subject, status, created_at) VALUES (?, ?, ?, NOW())");
            $stmt->execute([$to, $subject, $mappedStatus]);
            error_log("Email logged with simplified method: $to - $subject - $mappedStatus");
        } catch (Exception $e2) {
            error_log('Alternative email logging also failed: ' . $e2->getMessage());
        }
    }
}

/**
 * Quick email functions for common use cases
 */
function sendWelcomeEmail($email, $name, $role = 'student') {
    $subject = "Welcome to Panadite Academy!";
    $body = getEmailTemplate('welcome', ['name' => $name, 'role' => $role]);
    return queueEmail($email, $subject, $body, 'welcome', 'high');
}

function sendPasswordResetEmail($email, $name, $resetLink) {
    $subject = "Password Reset - Panadite Academy";
    $body = getEmailTemplate('reset', ['name' => $name, 'reset_link' => $resetLink]);
    return queueEmail($email, $subject, $body, 'reset', 'high');
}

function sendEnrollmentEmail($email, $name, $courseName) {
    $subject = "Course Enrollment Confirmed - $courseName";
    $body = getEmailTemplate('enrollment', ['name' => $name, 'course' => $courseName]);
    return queueEmail($email, $subject, $body, 'enrollment', 'normal');
}

function sendCertificateEmail($email, $name, $courseName, $certificateUrl) {
    $subject = "Your Certificate is Ready! - $courseName";
    $body = getEmailTemplate('certificate', ['name' => $name, 'course' => $courseName, 'cert_url' => $certificateUrl]);
    return queueEmail($email, $subject, $body, 'certificate', 'high');
}

/**
 * Get email templates
 */
function getEmailTemplate($type, $vars = []) {
    $templates = [
        'welcome' => '
        <div style="max-width:600px;margin:0 auto;font-family:Arial,sans-serif;">
            <div style="background:#007bff;color:white;padding:20px;text-align:center;">
                <h1>🎓 Welcome to Panadite Academy!</h1>
            </div>
            <div style="padding:20px;background:white;">
                <h2>Hello {name}!</h2>
                <p>Welcome to Panadite Academy! Your account as a <strong>{role}</strong> is now active.</p>
                <div style="background:#f8f9fa;padding:15px;border-radius:5px;margin:20px 0;">
                    <h3>🚀 Get Started</h3>
                    <p>Log in to your account and start exploring our courses!</p>
                </div>
            </div>
        </div>',
        
        'reset' => '
        <div style="max-width:600px;margin:0 auto;font-family:Arial,sans-serif;">
            <div style="background:#dc3545;color:white;padding:20px;text-align:center;">
                <h1>🔐 Password Reset</h1>
            </div>
            <div style="padding:20px;background:white;">
                <h2>Hello {name}!</h2>
                <p>Click the button below to reset your password:</p>
                <div style="text-align:center;margin:20px 0;">
                    <a href="{reset_link}" style="background:#dc3545;color:white;padding:12px 25px;text-decoration:none;border-radius:5px;">Reset Password</a>
                </div>
            </div>
        </div>',
        
        'enrollment' => '
        <div style="max-width:600px;margin:0 auto;font-family:Arial,sans-serif;">
            <div style="background:#28a745;color:white;padding:20px;text-align:center;">
                <h1>🎉 Enrollment Confirmed!</h1>
            </div>
            <div style="padding:20px;background:white;">
                <h2>Congratulations {name}!</h2>
                <p>You are now enrolled in <strong>{course}</strong>!</p>
                <p>Start learning today and achieve your goals.</p>
            </div>
        </div>',
        
        'certificate' => '
        <div style="max-width:600px;margin:0 auto;font-family:Arial,sans-serif;">
            <div style="background:#ffc107;color:#212529;padding:20px;text-align:center;">
                <h1>🏆 Certificate Ready!</h1>
            </div>
            <div style="padding:20px;background:white;">
                <h2>Congratulations {name}!</h2>
                <p>Your certificate for <strong>{course}</strong> is ready!</p>
                <div style="text-align:center;margin:20px 0;">
                    <a href="{cert_url}" style="background:#ffc107;color:#212529;padding:12px 25px;text-decoration:none;border-radius:5px;">Download Certificate</a>
                </div>
            </div>
        </div>'
    ];
    
    $template = $templates[$type] ?? '<p>Template not found</p>';
    
    foreach ($vars as $key => $value) {
        $template = str_replace('{' . $key . '}', $value, $template);
    }
    
    return $template;
}

/**
 * Test email function
 */
function sendTestEmail($email) {
    $subject = "Test Email - Panadite Academy";
    $body = '
    <div style="max-width:600px;margin:0 auto;font-family:Arial,sans-serif;">
        <div style="background:#007bff;color:white;padding:20px;text-align:center;">
            <h1>📧 Email Test Successful!</h1>
        </div>
        <div style="padding:20px;background:white;">
            <h2>Configuration Working!</h2>
            <p>Your fast email system is working correctly.</p>
            <div style="background:#d4edda;padding:15px;border-radius:5px;">
                <p><strong>✅ System Status:</strong> Active</p>
                <p><strong>⚡ Performance:</strong> Fast</p>
                <p><strong>🕒 Sent at:</strong> ' . date('Y-m-d H:i:s') . '</p>
            </div>
        </div>
    </div>';
    
    return sendEmailNow($email, $subject, $body);
}

/**
 * Send Study Reminder Email - Professional Template
 */
function sendStudyReminderEmail($email, $name, $course_title, $reminder_date, $reminder_time, $reminder_type, $description) {
    $subject = "📚 Study Reminder: " . $course_title . " - " . date('F j, Y', strtotime($reminder_date));
    
    // Get reminder type badge
    $type_badges = [
        'assignment' => ['🎯 Assignment Due', '#e67e22'],
        'exam_prep' => ['📝 Exam Preparation', '#dc3545'], 
        'review_session' => ['📖 Review Session', '#17a2b8'],
        'project_deadline' => ['🚀 Project Deadline', '#6f42c1'],
        'study_group' => ['👥 Study Group', '#28a745'],
        'general' => ['📚 Study Time', '#007bff']
    ];
    
    $badge_info = $type_badges[$reminder_type] ?? $type_badges['general'];
    $badge_text = $badge_info[0];
    $badge_color = $badge_info[1];
    
    // Format date and time nicely
    $formatted_date = date('l, F j, Y', strtotime($reminder_date));
    $formatted_time = date('g:i A', strtotime($reminder_time));
    
    // Create email using SAME STRUCTURE as working templates (NO DOCTYPE/HTML/HEAD/BODY)
    $message = '
    <div style="max-width:600px;margin:0 auto;font-family:Arial,sans-serif;">
        <!-- Header -->
        <div style="background: linear-gradient(135deg, #4a90e2 0%, #357abd 100%); color:white; padding:25px; text-align:center;">
            <h1 style="margin:0; font-size:28px;">🎓 Panadite Academy</h1>
            <p style="margin:8px 0 0 0; font-size:16px; opacity:0.9;">Study Reminder System</p>
        </div>
        
        <!-- Content -->
        <div style="padding:25px; background:white;">
            <h2 style="color:#333; margin:0 0 20px 0;">Hello ' . htmlspecialchars($name) . '! 👋</h2>
            
            <p style="font-size:16px; line-height:1.6; color:#555; margin-bottom:20px;">
                Your study reminder has been set successfully. Here are the details:
            </p>
            
            <!-- Reminder Details -->
            <div style="background:#f8f9ff; border-left:5px solid #4a90e2; padding:20px; margin:20px 0; border-radius:8px;">
                <h3 style="color:#4a90e2; margin:0 0 15px 0;">📚 ' . htmlspecialchars($course_title) . '</h3>
                
                <!-- Badge -->
                <div style="margin:15px 0;">
                    <span style="background:' . $badge_color . '; color:white; padding:8px 15px; border-radius:20px; font-size:14px; font-weight:bold; display:inline-block;">
                        ' . $badge_text . '
                    </span>
                </div>
                
                <p style="margin:10px 0; font-size:16px;">
                    <strong style="color:#28a745;">📅 Date:</strong> ' . $formatted_date . '
                </p>
                <p style="margin:10px 0; font-size:16px;">
                    <strong style="color:#e67e22;">⏰ Time:</strong> ' . $formatted_time . '
                </p>
                
                <div style="margin-top:15px;">
                    <p style="margin:8px 0; color:#6c757d;"><strong>📝 Description:</strong></p>
                    <div style="background:white; padding:15px; border-radius:5px; border:1px solid #e9ecef;">
                        <p style="margin:0; line-height:1.5; color:#333;">
                            ' . nl2br(htmlspecialchars($description)) . '
                        </p>
                    </div>
                </div>
            </div>
            
            <!-- Button -->
            <div style="text-align:center; margin:25px 0;">
                <a href="http://panaditeacademy.com/student/dashboard.php" 
                   style="background:#4a90e2; color:white; text-decoration:none; padding:15px 30px; border-radius:8px; font-weight:bold; display:inline-block;">
                    🚀 Go to Dashboard
                </a>
            </div>
            
            <!-- Note -->
            <div style="background:#e8f5e8; border-left:4px solid #28a745; padding:15px; margin:20px 0; border-radius:5px;">
                <p style="margin:0; color:#2d5016;">
                    <strong>📌 Note:</strong> We\'ll send you another reminder email at the scheduled time.
                </p>
            </div>
            
            <!-- Study Tip -->
            <div style="background:#fff3cd; border-left:4px solid #ffc107; padding:15px; margin:20px 0; border-radius:5px;">
                <p style="margin:0; color:#856404;">
                    <strong>💡 Study Tip:</strong> Use 25-minute focused study blocks with 5-minute breaks!
                </p>
            </div>
        </div>
        
        <!-- Footer -->
        <div style="background:#f8f9fa; padding:20px; text-align:center; border-top:1px solid #e9ecef;">
            <p style="margin:0 0 10px 0; color:#6c757d; font-weight:bold;">
                <strong>Panadite Academy</strong> - Empowering minds, shaping futures
            </p>
            <p style="margin:0; color:#adb5bd; font-size:12px;">
                © ' . date('Y') . ' Panadite Academy. Email sent on ' . date('M j, Y \a\t g:i A') . '
            </p>
        </div>
    </div>';

    return sendEmailNow($email, $subject, $message);
}

/**
 * Background email processor (can be called via cron or AJAX)
 */
function runEmailProcessor() {
    $processed = processEmailQueue(20);
    return [
        'success' => true,
        'processed' => $processed,
        'timestamp' => date('Y-m-d H:i:s')
    ];
}

?>
