<?php
/**
 * Background Email Processor for Panadite Academy
 * Processes queued emails without affecting application performance
 * Secured in config folder - Can be run via cron job or AJAX calls
 */

// Prevent direct browser access for security
if (!defined('EMAIL_PROCESSOR_ALLOWED')) {
    // Allow if called from command line or specific conditions
    $allowed = (php_sapi_name() === 'cli' || 
               isset($_GET['process_key']) && $_GET['process_key'] === 'panadite_email_2024' ||
               isset($_POST['ajax_process']) && $_POST['ajax_process'] === 'true');
    
    if (!$allowed) {
        http_response_code(403);
        exit('Access denied - Email processor is secured');
    }
}

// Set time limit and memory for processing
set_time_limit(300); // 5 minutes max
ini_set('memory_limit', '128M');

// Include required files (paths updated for config folder location)
require_once __DIR__ . '/db_connect.php';
require_once __DIR__ . '/fast_email_service.php';

// Start processing
$startTime = microtime(true);
$result = [
    'success' => true,
    'processed' => 0,
    'errors' => 0,
    'start_time' => date('Y-m-d H:i:s'),
    'execution_time' => 0,
    'status' => 'running',
    'location' => 'config/process_emails.php'
];

try {
    // Process emails in batches
    $totalProcessed = 0;
    $batchSize = 20;
    $maxBatches = 10; // Prevent infinite loops
    $currentBatch = 0;
    
    do {
        $processed = processEmailQueue($batchSize);
        $totalProcessed += $processed;
        $currentBatch++;
        
        // Small delay between batches to prevent server overload
        if ($processed > 0) {
            usleep(100000); // 0.1 second delay
        }
        
        // Log progress for large batches
        if ($totalProcessed > 0 && $currentBatch % 3 == 0) {
            error_log("Email processor: Processed $totalProcessed emails so far...");
        }
        
    } while ($processed > 0 && $currentBatch < $maxBatches);
    
    $result['processed'] = $totalProcessed;
    $result['batches'] = $currentBatch;
    $result['status'] = 'completed';
    
    // Log processing activity
    logProcessingActivity($totalProcessed, 0, microtime(true) - $startTime);
    
} catch (Exception $e) {
    $result['success'] = false;
    $result['status'] = 'error';
    $result['error_message'] = $e->getMessage();
    $result['errors'] = 1;
    
    error_log('Email processing error: ' . $e->getMessage());
    logProcessingActivity(0, 1, microtime(true) - $startTime, $e->getMessage());
}

$result['execution_time'] = round(microtime(true) - $startTime, 3);
$result['end_time'] = date('Y-m-d H:i:s');

// Return result based on how it was called
if (isset($_POST['ajax_process']) || isset($_GET['ajax'])) {
    header('Content-Type: application/json');
    echo json_encode($result);
} else {
    // Command line or direct access
    echo "=== Panadite Academy Email Processor ===\n";
    echo "Location: config/process_emails.php (Secured)\n";
    echo "Processed: {$result['processed']} emails\n";
    echo "Batches: {$result['batches']}\n";
    echo "Execution time: {$result['execution_time']} seconds\n";
    echo "Status: {$result['status']}\n";
    if (isset($result['error_message'])) {
        echo "Error: {$result['error_message']}\n";
    }
    echo "========================================\n";
}

/**
 * Log processing activity with enhanced security logging
 */
function logProcessingActivity($processed, $errors, $executionTime, $errorMessage = null) {
    global $pdo;
    
    if (!$pdo) return;
    
    try {
        // Create processing log table with enhanced security fields
        $createLogSQL = "
        CREATE TABLE IF NOT EXISTS `email_processing_log` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `processed_count` int(11) DEFAULT 0,
            `error_count` int(11) DEFAULT 0,
            `execution_time` decimal(8,3) DEFAULT 0.000,
            `error_message` text,
            `processor_location` varchar(100) DEFAULT 'config',
            `ip_address` varchar(45) DEFAULT NULL,
            `user_agent` varchar(500) DEFAULT NULL,
            `processed_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `processed_at` (`processed_at`),
            KEY `processor_location` (`processor_location`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        
        $pdo->exec($createLogSQL);
        
        // Get security info
        $ipAddress = $_SERVER['REMOTE_ADDR'] ?? 'CLI';
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Command Line';
        
        // Insert log entry with security info
        $stmt = $pdo->prepare("
            INSERT INTO email_processing_log 
            (processed_count, error_count, execution_time, error_message, processor_location, ip_address, user_agent) 
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([$processed, $errors, $executionTime, $errorMessage, 'config/process_emails.php', $ipAddress, $userAgent]);
        
    } catch (Exception $e) {
        error_log('Error logging email processing: ' . $e->getMessage());
    }
}

/**
 * Security check function
 */
function validateProcessorAccess() {
    // Additional security checks can be added here
    $validSources = [
        'admin/email_monitor.php',
        'cron',
        'cli'
    ];
    
    $referer = $_SERVER['HTTP_REFERER'] ?? '';
    $isValidReferer = false;
    
    foreach ($validSources as $source) {
        if (strpos($referer, $source) !== false) {
            $isValidReferer = true;
            break;
        }
    }
    
    // Log access attempt
    if (!$isValidReferer && !empty($referer)) {
        error_log("Email processor accessed from unauthorized source: " . $referer);
    }
    
    return true; // Allow for now, but log suspicious activity
}

// Run security validation
validateProcessorAccess();

// Define constant for security
define('EMAIL_PROCESSOR_ALLOWED', true);

?>
