<?php
/**
 * Study Reminders Database Setup Script
 * Creates the required database tables for the email notification system
 * Run this once to set up the tables
 */

// Security check
if (!defined('DB_SETUP_ALLOWED')) {
    $allowed = (php_sapi_name() === 'cli' || 
               isset($_GET['setup_key']) && $_GET['setup_key'] === 'panadite_db_setup_2024' ||
               isset($_POST['admin_setup']) && $_POST['admin_setup'] === 'true');
    
    if (!$allowed) {
        http_response_code(403);
        exit('Access denied - Database setup is secured. Use: ?setup_key=panadite_db_setup_2024');
    }
}

// Include database connection
require_once __DIR__ . '/db_connect.php';

// Setup result tracking
$setup_result = [
    'success' => true,
    'tables_created' => [],
    'tables_exists' => [],
    'errors' => [],
    'warnings' => [],
    'start_time' => date('Y-m-d H:i:s')
];

echo "<h1>Study Reminders Database Setup</h1>\n";
echo "<p>Setting up database tables for email notification system...</p>\n";

try {
    // Check if we're using PDO or mysqli
    $using_pdo = isset($pdo) && $pdo instanceof PDO;
    $using_mysqli = isset($conn) && $conn instanceof mysqli;
    
    if (!$using_pdo && !$using_mysqli) {
        throw new Exception('No database connection found. Check db_connect.php');
    }

    echo "<h2>Database Connection: " . ($using_pdo ? 'PDO' : 'MySQLi') . "</h2>\n";

    // 1. Create study_reminders table
    echo "<h3>Creating study_reminders table...</h3>\n";
    
    $create_reminders_table = "
    CREATE TABLE IF NOT EXISTS `study_reminders` (
      `id` int(11) NOT NULL AUTO_INCREMENT,
      `student_id` int(11) NOT NULL,
      `course_id` int(11) NOT NULL,
      `reminder_date` date NOT NULL,
      `reminder_time` time NOT NULL,
      `reminder_type` varchar(50) NOT NULL DEFAULT 'study_session',
      `description` text NOT NULL,
      `email_sent` tinyint(1) DEFAULT 0 COMMENT 'Confirmation email sent when reminder is created',
      `reminder_sent` tinyint(1) DEFAULT 0 COMMENT 'Actual reminder email sent at scheduled time',
      `sent_at` timestamp NULL DEFAULT NULL COMMENT 'When the reminder email was actually sent',
      `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
      `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      PRIMARY KEY (`id`),
      KEY `student_id` (`student_id`),
      KEY `course_id` (`course_id`),
      KEY `reminder_datetime` (`reminder_date`, `reminder_time`),
      KEY `reminder_sent` (`reminder_sent`),
      KEY `idx_due_reminders` (`reminder_sent`, `reminder_date`, `reminder_time`),
      KEY `idx_student_course` (`student_id`, `course_id`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($using_pdo) {
        $result1 = $pdo->exec($create_reminders_table);
        if ($result1 !== false) {
            $setup_result['tables_created'][] = 'study_reminders';
            echo "<p>✅ study_reminders table created successfully!</p>\n";
        } else {
            $setup_result['tables_exists'][] = 'study_reminders';
            echo "<p>ℹ️ study_reminders table already exists</p>\n";
        }
    } else {
        $result1 = $conn->query($create_reminders_table);
        if ($result1) {
            $setup_result['tables_created'][] = 'study_reminders';
            echo "<p>✅ study_reminders table created successfully!</p>\n";
        } else {
            $setup_result['errors'][] = "study_reminders table: " . $conn->error;
            echo "<p>❌ Error creating study_reminders table: " . $conn->error . "</p>\n";
        }
    }

    // 2. Create reminder_logs table
    echo "<h3>Creating reminder_logs table...</h3>\n";
    
    $create_logs_table = "
    CREATE TABLE IF NOT EXISTS `reminder_logs` (
      `id` int(11) NOT NULL AUTO_INCREMENT,
      `reminder_id` int(11) NOT NULL,
      `status` varchar(20) NOT NULL,
      `message` text,
      `logged_at` timestamp DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (`id`),
      KEY `reminder_id` (`reminder_id`),
      KEY `logged_at` (`logged_at`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($using_pdo) {
        $result2 = $pdo->exec($create_logs_table);
        if ($result2 !== false) {
            $setup_result['tables_created'][] = 'reminder_logs';
            echo "<p>✅ reminder_logs table created successfully!</p>\n";
        } else {
            $setup_result['tables_exists'][] = 'reminder_logs';
            echo "<p>ℹ️ reminder_logs table already exists</p>\n";
        }
    } else {
        $result2 = $conn->query($create_logs_table);
        if ($result2) {
            $setup_result['tables_created'][] = 'reminder_logs';
            echo "<p>✅ reminder_logs table created successfully!</p>\n";
        } else {
            $setup_result['errors'][] = "reminder_logs table: " . $conn->error;
            echo "<p>❌ Error creating reminder_logs table: " . $conn->error . "</p>\n";
        }
    }

    // 3. Check if system_settings table exists and add reminder settings
    echo "<h3>Adding system settings...</h3>\n";
    
    $check_settings_table = "SHOW TABLES LIKE 'system_settings'";
    
    if ($using_pdo) {
        $settings_exists = $pdo->query($check_settings_table)->rowCount() > 0;
    } else {
        $settings_result = $conn->query($check_settings_table);
        $settings_exists = $settings_result && $settings_result->num_rows > 0;
    }

    if ($settings_exists) {
        // Insert reminder settings
        $reminder_settings = [
            ['study_reminders', 'enabled', '1', 'Enable study reminder system'],
            ['study_reminders', 'default_advance_minutes', '15', 'Default minutes before scheduled time to send reminder'],
            ['study_reminders', 'max_reminders_per_day', '10', 'Maximum reminders a student can set per day'],
            ['study_reminders', 'cleanup_old_reminders_days', '30', 'Days after which old sent reminders are cleaned up'],
            ['reminder_types', 'next_lesson', 'Next Lesson', 'Reminder for upcoming lesson'],
            ['reminder_types', 'review_session', 'Review Session', 'Reminder for review session'],
            ['reminder_types', 'assignment', 'Assignment Due', 'Reminder for assignment deadline'],
            ['reminder_types', 'quiz_prep', 'Quiz Preparation', 'Reminder to prepare for quiz'],
            ['reminder_types', 'study_session', 'Study Session', 'General study session reminder'],
            ['reminder_types', 'custom', 'Custom Reminder', 'Custom study reminder']
        ];

        foreach ($reminder_settings as $setting) {
            $insert_setting = "
                INSERT IGNORE INTO `system_settings` 
                (`setting_group`, `setting_key`, `setting_value`, `description`) 
                VALUES (?, ?, ?, ?)
            ";
            
            if ($using_pdo) {
                $stmt = $pdo->prepare($insert_setting);
                $stmt->execute($setting);
            } else {
                $stmt = $conn->prepare($insert_setting);
                $stmt->bind_param("ssss", $setting[0], $setting[1], $setting[2], $setting[3]);
                $stmt->execute();
            }
        }
        
        echo "<p>✅ System settings added successfully!</p>\n";
        $setup_result['tables_created'][] = 'system_settings (updated)';
    } else {
        echo "<p>⚠️ system_settings table doesn't exist. Reminder settings not added.</p>\n";
        $setup_result['warnings'][] = 'system_settings table not found';
    }

    // 4. Test database tables
    echo "<h3>Testing created tables...</h3>\n";
    
    $test_queries = [
        'study_reminders' => "SELECT COUNT(*) as count FROM study_reminders",
        'reminder_logs' => "SELECT COUNT(*) as count FROM reminder_logs"
    ];

    foreach ($test_queries as $table => $query) {
        try {
            if ($using_pdo) {
                $test_result = $pdo->query($query)->fetch(PDO::FETCH_ASSOC);
                echo "<p>✅ {$table} table working: {$test_result['count']} records</p>\n";
            } else {
                $test_result = $conn->query($query);
                $row = $test_result->fetch_assoc();
                echo "<p>✅ {$table} table working: {$row['count']} records</p>\n";
            }
        } catch (Exception $e) {
            echo "<p>❌ Error testing {$table}: " . $e->getMessage() . "</p>\n";
            $setup_result['errors'][] = "Testing {$table}: " . $e->getMessage();
        }
    }

    // 5. Setup completion summary
    echo "<h2>🎉 Database Setup Complete!</h2>\n";
    echo "<div style='background: #d4edda; padding: 15px; border-radius: 5px; margin: 10px 0;'>\n";
    echo "<h3>📊 Setup Summary:</h3>\n";
    echo "<ul>\n";
    echo "<li><strong>Tables Created:</strong> " . implode(', ', $setup_result['tables_created']) . "</li>\n";
    if (!empty($setup_result['tables_exists'])) {
        echo "<li><strong>Tables Already Existed:</strong> " . implode(', ', $setup_result['tables_exists']) . "</li>\n";
    }
    if (!empty($setup_result['warnings'])) {
        echo "<li><strong>Warnings:</strong> " . implode(', ', $setup_result['warnings']) . "</li>\n";
    }
    echo "</ul>\n";
    echo "</div>\n";

    // 6. Next steps instructions
    echo "<h3>🚀 Next Steps:</h3>\n";
    echo "<ol>\n";
    echo "<li><strong>Set up the scheduler:</strong> Follow instructions in setup_reminder_cron.txt</li>\n";
    echo "<li><strong>Test email notifications:</strong> Create a test reminder in your course</li>\n";
    echo "<li><strong>Monitor logs:</strong> Check reminder_logs table for activity</li>\n";
    echo "<li><strong>Verify SMTP:</strong> Ensure your email service is configured</li>\n";
    echo "</ol>\n";

} catch (Exception $e) {
    $setup_result['success'] = false;
    $setup_result['errors'][] = $e->getMessage();
    
    echo "<h2>❌ Setup Failed</h2>\n";
    echo "<p style='color: red;'><strong>Error:</strong> " . $e->getMessage() . "</p>\n";
    echo "<p>Please check your database connection and try again.</p>\n";
    
    error_log("Study Reminders DB Setup Error: " . $e->getMessage());
}

$setup_result['end_time'] = date('Y-m-d H:i:s');
$setup_result['total_errors'] = count($setup_result['errors']);
$setup_result['total_tables'] = count($setup_result['tables_created']) + count($setup_result['tables_exists']);

// Output JSON result for API calls
if (isset($_GET['format']) && $_GET['format'] === 'json') {
    header('Content-Type: application/json');
    echo json_encode($setup_result, JSON_PRETTY_PRINT);
    exit;
}

echo "<hr>\n";
echo "<p><small>Setup completed at: " . $setup_result['end_time'] . "</small></p>\n";
echo "<p><small>For JSON output, add: ?format=json&setup_key=panadite_db_setup_2024</small></p>\n";

// Log the setup activity
error_log("Study Reminders Database Setup: " . 
          count($setup_result['tables_created']) . " tables created, " . 
          count($setup_result['errors']) . " errors");
?>
