<?php
/**
 * Study Reminder Scheduler for Panadite Academy
 * Automatically sends reminder emails at scheduled times
 * This should be run via cron job every 5-10 minutes
 */

// Security check - prevent direct browser access
if (!defined('REMINDER_SCHEDULER_ALLOWED')) {
    // Allow if called from command line or with security key
    $allowed = (php_sapi_name() === 'cli' || 
               isset($_GET['scheduler_key']) && $_GET['scheduler_key'] === 'panadite_scheduler_2024' ||
               isset($_POST['ajax_scheduler']) && $_POST['ajax_scheduler'] === 'true');
    
    if (!$allowed) {
        http_response_code(403);
        exit('Access denied - Reminder scheduler is secured');
    }
}

// Set time limit and memory
set_time_limit(300); // 5 minutes max
ini_set('memory_limit', '128M');

// Include required files
require_once __DIR__ . '/db_connect.php';
require_once __DIR__ . '/fast_email_service.php';

/**
 * Main scheduler function - checks for due reminders and sends emails
 */
function processStudyReminders() {
    global $pdo;
    
    $result = [
        'success' => true,
        'processed' => 0,
        'errors' => 0,
        'start_time' => date('Y-m-d H:i:s'),
        'reminders_sent' => [],
        'failed_reminders' => []
    ];
    
    try {
        // Get current datetime with small buffer (send reminders up to 5 minutes after scheduled time)
        $now = new DateTime();
        $buffer_time = new DateTime();
        $buffer_time->modify('-5 minutes');
        
        // Find due reminders that haven't been sent yet
        $stmt = $pdo->prepare("
            SELECT sr.*, u.email, u.first_name, u.last_name, c.title as course_title 
            FROM study_reminders sr 
            JOIN users u ON sr.student_id = u.user_id 
            JOIN courses c ON sr.course_id = c.course_id 
            WHERE sr.reminder_sent = 0 
            AND CONCAT(sr.reminder_date, ' ', sr.reminder_time) BETWEEN ? AND ? 
            AND u.role = 'student'
            ORDER BY sr.reminder_date, sr.reminder_time
        ");
        
        $stmt->execute([
            $buffer_time->format('Y-m-d H:i:s'),
            $now->format('Y-m-d H:i:s')
        ]);
        
        $due_reminders = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($due_reminders as $reminder) {
            try {
                $sent = sendScheduledReminderEmail($reminder);
                
                if ($sent) {
                    // Mark reminder as sent
                    $update_stmt = $pdo->prepare("
                        UPDATE study_reminders 
                        SET reminder_sent = 1, sent_at = NOW() 
                        WHERE id = ?
                    ");
                    $update_stmt->execute([$reminder['id']]);
                    
                    $result['processed']++;
                    $result['reminders_sent'][] = [
                        'id' => $reminder['id'],
                        'student' => $reminder['first_name'] . ' ' . $reminder['last_name'],
                        'course' => $reminder['course_title'],
                        'time' => $reminder['reminder_date'] . ' ' . $reminder['reminder_time']
                    ];
                    
                    logReminderActivity($reminder['id'], 'sent', 'Reminder email sent successfully');
                } else {
                    $result['errors']++;
                    $result['failed_reminders'][] = [
                        'id' => $reminder['id'],
                        'student' => $reminder['first_name'] . ' ' . $reminder['last_name'],
                        'error' => 'Email sending failed'
                    ];
                    
                    logReminderActivity($reminder['id'], 'failed', 'Failed to send reminder email');
                }
                
            } catch (Exception $e) {
                $result['errors']++;
                $result['failed_reminders'][] = [
                    'id' => $reminder['id'],
                    'student' => $reminder['first_name'] . ' ' . $reminder['last_name'],
                    'error' => $e->getMessage()
                ];
                
                logReminderActivity($reminder['id'], 'error', $e->getMessage());
                error_log("Study Reminder Scheduler Error: " . $e->getMessage());
            }
        }
        
        $result['execution_time'] = round(microtime(true) - $_SERVER['REQUEST_TIME_FLOAT'], 2);
        
    } catch (Exception $e) {
        $result['success'] = false;
        $result['error'] = $e->getMessage();
        error_log("Study Reminder Scheduler Critical Error: " . $e->getMessage());
    }
    
    return $result;
}

/**
 * Send the actual scheduled reminder email
 */
function sendScheduledReminderEmail($reminder) {
    // Format reminder details
    $reminder_date = date('l, F j, Y', strtotime($reminder['reminder_date']));
    $reminder_time = date('g:i A', strtotime($reminder['reminder_time']));
    
    // Prepare email content
    $subject = "⏰ Study Reminder Alert - " . $reminder['course_title'];
    
    $html_body = generateScheduledReminderHtml($reminder, $reminder_date, $reminder_time);
    $text_body = generateScheduledReminderText($reminder, $reminder_date, $reminder_time);
    
    // Send email using your SMTP service with high priority
    try {
        $result = sendEmailNow($reminder['email'], $subject, $html_body);
        
        if ($result) {
            logEmailAttempt($reminder['email'], $subject, 'sent', null);
            return true;
        } else {
            // If immediate send fails, queue with high priority
            $queue_result = queueEmail($reminder['email'], $subject, $html_body, 'study_reminder_alert', 'urgent');
            logEmailAttempt($reminder['email'], $subject, 'queued', 'Immediate send failed, queued with urgent priority');
            return $queue_result;
        }
    } catch (Exception $e) {
        logEmailAttempt($reminder['email'], $subject, 'failed', $e->getMessage());
        return false;
    }
}

/**
 * Generate HTML email for scheduled reminder
 */
function generateScheduledReminderHtml($reminder, $reminder_date, $reminder_time) {
    $type_names = [
        'next_lesson' => 'Next Lesson',
        'review_session' => 'Review Session',
        'assignment' => 'Assignment Due',
        'quiz_prep' => 'Quiz Preparation',
        'study_session' => 'Study Session',
        'custom' => 'Custom Reminder'
    ];
    
    $type_display = $type_names[$reminder['reminder_type']] ?? 'Study Reminder';
    $type_icons = [
        'next_lesson' => '📖',
        'review_session' => '🔄', 
        'assignment' => '📝',
        'quiz_prep' => '🧠',
        'study_session' => '📚',
        'custom' => '⏰'
    ];
    $type_icon = $type_icons[$reminder['reminder_type']] ?? '📚';
    
    return "
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset='utf-8'>
        <meta name='viewport' content='width=device-width, initial-scale=1.0'>
        <title>Study Reminder - Time to Study!</title>
        <style>
            body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; margin: 0; padding: 0; background-color: #f8f9fa; }
            .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 10px; overflow: hidden; box-shadow: 0 4px 6px rgba(0,0,0,0.1); }
            .header { background: linear-gradient(135deg, #ff7b7b 0%, #667eea 100%); color: white; padding: 30px; text-align: center; }
            .header h1 { margin: 0; font-size: 28px; font-weight: 600; }
            .content { padding: 40px 30px; }
            .urgent-banner { background: linear-gradient(135deg, #ff6b6b, #ee5a24); color: white; padding: 20px; text-align: center; font-size: 18px; font-weight: 600; margin: -40px -30px 30px -30px; }
            .reminder-card { background-color: #f8f9fa; border-left: 5px solid #ff6b6b; padding: 25px; margin: 20px 0; border-radius: 8px; }
            .time-display { background-color: #fff3cd; border: 2px solid #ffeaa7; padding: 15px; border-radius: 8px; text-align: center; margin: 20px 0; }
            .time-display h3 { margin: 0; color: #856404; font-size: 24px; }
            .action-buttons { text-align: center; margin: 30px 0; }
            .btn { display: inline-block; padding: 14px 28px; margin: 8px; border-radius: 6px; text-decoration: none; font-weight: 600; font-size: 16px; }
            .btn-primary { background-color: #007bff; color: white; }
            .btn-secondary { background-color: #6c757d; color: white; }
            .study-tips { background-color: #e7f3ff; border: 1px solid #b3d9ff; padding: 20px; border-radius: 8px; margin: 20px 0; }
            .footer { background-color: #f8f9fa; padding: 20px; text-align: center; font-size: 14px; color: #666; }
            .alert-icon { color: #ff6b6b; font-size: 48px; margin-bottom: 15px; }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <div class='alert-icon'>{$type_icon}</div>
                <h1>Study Reminder Alert!</h1>
                <p>It's time for your scheduled study session</p>
            </div>
            
            <div class='urgent-banner'>
                ⏰ SCHEDULED REMINDER - {$reminder_date} at {$reminder_time}
            </div>
            
            <div class='content'>
                <p>Hi <strong>{$reminder['first_name']} {$reminder['last_name']}</strong>,</p>
                
                <div class='reminder-card'>
                    <h3>{$type_icon} Time for: <strong>{$type_display}</strong></h3>
                    <h4>📚 Course: {$reminder['course_title']}</h4>
                    <p><strong>Description:</strong> {$reminder['description']}</p>
                </div>
                
                <div class='time-display'>
                    <h3>🕐 {$reminder_time} - {$reminder_date}</h3>
                    <p>Your scheduled study time is now!</p>
                </div>
                
                <div class='study-tips'>
                    <h4>💡 Quick Study Tips:</h4>
                    <ul>
                        <li><strong>Focus:</strong> Find a quiet space free from distractions</li>
                        <li><strong>Materials:</strong> Have all your course materials ready</li>
                        <li><strong>Goals:</strong> Set clear objectives for this study session</li>
                        <li><strong>Timer:</strong> Use the Pomodoro technique (25 min study, 5 min break)</li>
                    </ul>
                </div>
                
                <div class='action-buttons'>
                    <a href='#' class='btn btn-primary'>📖 Start Studying Now</a>
                    <a href='#' class='btn btn-secondary'>⏰ Snooze (15 min)</a>
                </div>
                
                <p>Ready to make progress on your learning journey? Click the button above to access your course materials and get started!</p>
            </div>
            
            <div class='footer'>
                <p><strong>Panadite Academy</strong> - Empowering Your Learning Journey</p>
                <p>This is your scheduled study reminder. Stay consistent for the best results!</p>
                <p>Need help? Contact our support team.</p>
            </div>
        </div>
    </body>
    </html>
    ";
}

/**
 * Generate text email for scheduled reminder  
 */
function generateScheduledReminderText($reminder, $reminder_date, $reminder_time) {
    $type_names = [
        'next_lesson' => 'Next Lesson',
        'review_session' => 'Review Session',
        'assignment' => 'Assignment Due',
        'quiz_prep' => 'Quiz Preparation',
        'study_session' => 'Study Session',
        'custom' => 'Custom Reminder'
    ];
    
    $type_display = $type_names[$reminder['reminder_type']] ?? 'Study Reminder';
    $type_icons = [
        'next_lesson' => '📖',
        'review_session' => '🔄',
        'assignment' => '📝',
        'quiz_prep' => '🧠',
        'study_session' => '📚',
        'custom' => '⏰'
    ];
    $type_icon = $type_icons[$reminder['reminder_type']] ?? '📚';
    
    return "
⏰ STUDY REMINDER ALERT! ⏰

Hi {$reminder['first_name']} {$reminder['last_name']},

🚨 IT'S TIME FOR YOUR SCHEDULED STUDY SESSION! 🚨

REMINDER DETAILS:
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
{$type_icon} Type: {$type_display}
📚 Course: {$reminder['course_title']}
📅 Date: {$reminder_date}
🕐 Time: {$reminder_time} (NOW!)
📝 Description: {$reminder['description']}
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

💡 QUICK STUDY TIPS:
• Find a quiet space free from distractions
• Have all your course materials ready
• Set clear objectives for this study session  
• Use the Pomodoro technique (25 min study, 5 min break)

🎯 Ready to make progress on your learning journey? 
   Access your course materials and start studying now!

⏰ Don't have time right now? Set another reminder for later!

PANADITE ACADEMY
Empowering Your Learning Journey

This is your scheduled study reminder. Stay consistent for the best results!
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
    ";
}

/**
 * Log reminder activity
 */
function logReminderActivity($reminder_id, $status, $message = null) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO reminder_logs (reminder_id, status, message, logged_at) 
            VALUES (?, ?, ?, NOW())
        ");
        $stmt->execute([$reminder_id, $status, $message]);
    } catch (Exception $e) {
        error_log("Failed to log reminder activity: " . $e->getMessage());
    }
}

// Execute scheduler if run directly
if (!defined('REMINDER_SCHEDULER_INCLUDED')) {
    define('REMINDER_SCHEDULER_ALLOWED', true);
    
    $result = processStudyReminders();
    
    // Output result as JSON for monitoring
    header('Content-Type: application/json');
    echo json_encode($result, JSON_PRETTY_PRINT);
    
    // Log summary
    error_log("Study Reminder Scheduler: Processed {$result['processed']} reminders, {$result['errors']} errors");
}
?>
