<?php
/**
 * Automated Study Reminder Checker
 * This script should be run every 5 minutes via cron job to check for due reminders
 * 
 * Cron job setup example:
 * */5 * * * * /usr/bin/php /path/to/your/project/cron/check_reminders.php
 */

// Prevent direct browser access
if (php_sapi_name() !== 'cli' && !isset($_GET['debug'])) {
    http_response_code(403);
    die('This script can only be run from command line or debug mode');
}

// Include database connection
require_once dirname(__DIR__) . '/config/database.php';

// Configure error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Log file for debugging
$log_file = dirname(__DIR__) . '/logs/reminder_check.log';
$log_dir = dirname($log_file);
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}

function logMessage($message, $type = 'INFO') {
    global $log_file;
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[{$timestamp}] [{$type}] {$message}" . PHP_EOL;
    file_put_contents($log_file, $log_entry, FILE_APPEND | LOCK_EX);
    
    // Also output to console if running in CLI
    if (php_sapi_name() === 'cli' || isset($_GET['debug'])) {
        echo $log_entry;
    }
}

function getSystemConfig($conn, $key, $default = null) {
    $stmt = $conn->prepare("SELECT config_value FROM reminder_system_config WHERE config_key = ?");
    $stmt->bind_param("s", $key);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($row = $result->fetch_assoc()) {
        return $row['config_value'];
    }
    
    return $default;
}

function getDueReminders($conn) {
    $current_datetime = date('Y-m-d H:i:s');
    $advance_minutes = getSystemConfig($conn, 'reminder_advance_minutes', 0);
    
    // Calculate the reminder time considering advance minutes
    $check_datetime = date('Y-m-d H:i:s', strtotime("-{$advance_minutes} minutes"));
    
    logMessage("Checking for reminders due before: {$current_datetime} (advance: {$advance_minutes} minutes)");
    
    $sql = "
        SELECT 
            sr.reminder_id,
            sr.student_id,
            sr.course_id,
            sr.reminder_date,
            sr.reminder_time,
            sr.reminder_type,
            sr.description,
            sr.email_sent as initial_email_sent,
            u.email,
            u.first_name,
            u.last_name,
            c.title as course_title,
            CONCAT(sr.reminder_date, ' ', sr.reminder_time) as reminder_datetime
        FROM study_reminders sr
        JOIN users u ON sr.student_id = u.user_id
        JOIN courses c ON sr.course_id = c.course_id
        WHERE 
            sr.reminder_sent = 0 
            AND sr.is_completed = 0
            AND CONCAT(sr.reminder_date, ' ', sr.reminder_time) <= ?
            AND u.role = 'student'
        ORDER BY sr.reminder_date ASC, sr.reminder_time ASC
        LIMIT 50
    ";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $current_datetime);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $reminders = [];
    while ($row = $result->fetch_assoc()) {
        $reminders[] = $row;
    }
    
    logMessage("Found " . count($reminders) . " due reminders");
    return $reminders;
}

function sendReminderEmail($reminder) {
    $subject = "⏰ Study Reminder - " . $reminder['course_title'];
    
    $html_body = generateReminderNotificationEmail($reminder);
    $text_body = generateReminderNotificationText($reminder);
    
    // Use the same email function from the API
    $headers = [
        'MIME-Version: 1.0',
        'Content-Type: text/html; charset=UTF-8',
        'From: ' . getSystemConfig($GLOBALS['conn'], 'email_from_name', 'Panadite Academy') . ' <' . getSystemConfig($GLOBALS['conn'], 'email_from_address', 'noreply@panaditeacademy.com') . '>',
        'Reply-To: support@panaditeacademy.com',
        'X-Mailer: PHP/' . phpversion(),
        'X-Priority: 2'
    ];
    
    $success = mail($reminder['email'], $subject, $html_body, implode("\r\n", $headers));
    
    if ($success) {
        logMessage("✅ Reminder email sent to: " . $reminder['email'] . " for reminder ID: " . $reminder['reminder_id']);
    } else {
        logMessage("❌ Failed to send reminder email to: " . $reminder['email'] . " for reminder ID: " . $reminder['reminder_id'], 'ERROR');
    }
    
    return $success;
}

function generateReminderNotificationEmail($reminder) {
    $reminder_time = date('g:i A', strtotime($reminder['reminder_time']));
    $reminder_date = date('l, F j, Y', strtotime($reminder['reminder_date']));
    
    $type_icons = [
        'next_lesson' => '📚',
        'review_session' => '📖',
        'assignment' => '📝',
        'quiz_prep' => '🧠',
        'study_session' => '💡',
        'custom' => '⏰'
    ];
    
    $type_names = [
        'next_lesson' => 'Next Lesson',
        'review_session' => 'Review Session',
        'assignment' => 'Assignment Due',
        'quiz_prep' => 'Quiz Preparation',
        'study_session' => 'Study Session',
        'custom' => 'Study Reminder'
    ];
    
    $icon = $type_icons[$reminder['reminder_type']] ?? '⏰';
    $type_name = $type_names[$reminder['reminder_type']] ?? 'Study Reminder';
    
    return "
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset='UTF-8'>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; margin: 0; padding: 0; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #FF6B6B 0%, #4ECDC4 100%); color: white; padding: 30px; text-align: center; border-radius: 15px 15px 0 0; }
            .header h1 { margin: 0; font-size: 1.8em; }
            .content { background: #f8f9fa; padding: 30px; border-radius: 0 0 15px 15px; }
            .reminder-alert { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 25px; border-radius: 12px; margin: 20px 0; text-align: center; }
            .reminder-alert h2 { margin: 0 0 10px 0; font-size: 1.5em; }
            .reminder-details { background: white; padding: 25px; border-radius: 10px; border-left: 5px solid #FF6B6B; margin: 20px 0; }
            .detail-row { display: flex; justify-content: space-between; margin-bottom: 10px; padding: 8px 0; border-bottom: 1px solid #eee; }
            .detail-label { font-weight: bold; color: #666; }
            .detail-value { color: #333; }
            .action-button { display: inline-block; background: #4ECDC4; color: white; padding: 15px 30px; text-decoration: none; border-radius: 8px; margin: 20px auto; text-align: center; font-weight: bold; }
            .footer { text-align: center; margin-top: 30px; color: #666; font-size: 0.9em; }
            .urgent { animation: pulse 2s infinite; }
            @keyframes pulse { 0% { transform: scale(1); } 50% { transform: scale(1.02); } 100% { transform: scale(1); } }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <h1>{$icon} Study Reminder Alert!</h1>
                <p>It's time for your scheduled study activity</p>
            </div>
            
            <div class='content'>
                <div class='reminder-alert urgent'>
                    <h2>🚨 Your reminder is due now!</h2>
                    <p style='font-size: 1.1em; margin: 0;'>{$reminder['description']}</p>
                </div>
                
                <p>Hi {$reminder['first_name']},</p>
                
                <p>This is your scheduled study reminder. It's time to focus on your learning!</p>
                
                <div class='reminder-details'>
                    <h4>📋 Reminder Details</h4>
                    <div class='detail-row'>
                        <span class='detail-label'>Course:</span>
                        <span class='detail-value'>{$reminder['course_title']}</span>
                    </div>
                    <div class='detail-row'>
                        <span class='detail-label'>Type:</span>
                        <span class='detail-value'>{$type_name}</span>
                    </div>
                    <div class='detail-row'>
                        <span class='detail-label'>Scheduled Date:</span>
                        <span class='detail-value'>{$reminder_date}</span>
                    </div>
                    <div class='detail-row'>
                        <span class='detail-label'>Scheduled Time:</span>
                        <span class='detail-value'>{$reminder_time}</span>
                    </div>
                    <div class='detail-row'>
                        <span class='detail-label'>Description:</span>
                        <span class='detail-value'>{$reminder['description']}</span>
                    </div>
                </div>
                
                <div style='text-align: center; margin: 30px 0;'>
                    <a href='https://panaditeacademy.com/student/course-detail.php?id={$reminder['course_id']}' class='action-button'>
                        🚀 Go to Course Now
                    </a>
                </div>
                
                <div style='background: #e8f5e8; padding: 20px; border-radius: 10px; margin: 20px 0;'>
                    <h4 style='color: #2e7d32; margin-top: 0;'>💡 Study Tips:</h4>
                    <ul style='color: #2e7d32; margin-bottom: 0;'>
                        <li>Find a quiet, comfortable study space</li>
                        <li>Turn off distractions like social media</li>
                        <li>Take breaks every 25-30 minutes</li>
                        <li>Stay hydrated and keep healthy snacks nearby</li>
                    </ul>
                </div>
                
                <div class='footer'>
                    <p>This reminder was automatically sent based on your study schedule.</p>
                    <p>Keep up the excellent work! 🎓</p>
                    <p style='font-size: 0.8em; color: #999;'>Sent at " . date('F j, Y \a\t g:i A') . "</p>
                </div>
            </div>
        </div>
    </body>
    </html>
    ";
}

function generateReminderNotificationText($reminder) {
    $reminder_time = date('g:i A', strtotime($reminder['reminder_time']));
    $reminder_date = date('l, F j, Y', strtotime($reminder['reminder_date']));
    
    return "
🚨 STUDY REMINDER ALERT! 🚨

Hi {$reminder['first_name']},

Your scheduled study reminder is due now!

REMINDER DETAILS:
📚 Course: {$reminder['course_title']}
📅 Date: {$reminder_date}
🕒 Time: {$reminder_time}
📝 Description: {$reminder['description']}

It's time to focus on your learning! Don't let this opportunity slip away.

💡 Quick Study Tips:
- Find a quiet, distraction-free environment
- Have all your materials ready
- Set a specific goal for this session
- Take notes as you learn

🚀 Access your course: https://panaditeacademy.com/student/course-detail.php?id={$reminder['course_id']}

This reminder was automatically sent based on your study schedule.
Keep up the excellent work! 🎓

---
Panadite Academy
Sent at " . date('F j, Y \a\t g:i A') . "
    ";
}

function markReminderAsSent($conn, $reminder_id, $email_status, $error_message = null) {
    // Update the reminder as sent
    $update_stmt = $conn->prepare("UPDATE study_reminders SET reminder_sent = 1, updated_at = NOW() WHERE reminder_id = ?");
    $update_stmt->bind_param("i", $reminder_id);
    $update_success = $update_stmt->execute();
    
    // Log the email attempt
    $log_stmt = $conn->prepare("INSERT INTO reminder_log (reminder_id, student_id, email_address, email_status, error_message) SELECT reminder_id, student_id, (SELECT email FROM users WHERE user_id = study_reminders.student_id), ?, ? FROM study_reminders WHERE reminder_id = ?");
    $log_stmt->bind_param("ssi", $email_status, $error_message, $reminder_id);
    $log_success = $log_stmt->execute();
    
    return $update_success && $log_success;
}

function processReminders($conn) {
    $system_enabled = getSystemConfig($conn, 'system_enabled', '1');
    
    if ($system_enabled !== '1') {
        logMessage("⏸️ Reminder system is disabled. Skipping check.");
        return;
    }
    
    logMessage("🔍 Starting reminder check process...");
    
    $due_reminders = getDueReminders($conn);
    
    if (empty($due_reminders)) {
        logMessage("✅ No due reminders found.");
        return;
    }
    
    $sent_count = 0;
    $failed_count = 0;
    
    foreach ($due_reminders as $reminder) {
        logMessage("📧 Processing reminder ID: {$reminder['reminder_id']} for {$reminder['email']}");
        
        try {
            $email_sent = sendReminderEmail($reminder);
            
            if ($email_sent) {
                markReminderAsSent($conn, $reminder['reminder_id'], 'sent');
                $sent_count++;
                logMessage("✅ Reminder {$reminder['reminder_id']} processed successfully");
            } else {
                markReminderAsSent($conn, $reminder['reminder_id'], 'failed', 'Mail function returned false');
                $failed_count++;
                logMessage("❌ Failed to send reminder {$reminder['reminder_id']}", 'ERROR');
            }
            
            // Small delay to prevent overwhelming the mail server
            usleep(500000); // 0.5 seconds
            
        } catch (Exception $e) {
            markReminderAsSent($conn, $reminder['reminder_id'], 'failed', $e->getMessage());
            $failed_count++;
            logMessage("❌ Exception processing reminder {$reminder['reminder_id']}: " . $e->getMessage(), 'ERROR');
        }
    }
    
    logMessage("📊 Reminder processing complete. Sent: {$sent_count}, Failed: {$failed_count}");
}

// Main execution
try {
    logMessage("🚀 Study Reminder Checker Started - " . date('Y-m-d H:i:s'));
    
    if (!$conn) {
        throw new Exception("Database connection failed");
    }
    
    // Check if required tables exist
    $tables_check = $conn->query("SHOW TABLES LIKE 'study_reminders'");
    if ($tables_check->num_rows === 0) {
        throw new Exception("study_reminders table does not exist. Please run the setup script first.");
    }
    
    processReminders($conn);
    
    logMessage("✅ Study Reminder Checker Completed Successfully");
    
} catch (Exception $e) {
    logMessage("❌ Fatal error: " . $e->getMessage(), 'ERROR');
    exit(1);
} finally {
    if ($conn) {
        $conn->close();
    }
}

// Clean up old log entries (keep last 30 days)
$log_retention_days = 30;
$old_logs = date('Y-m-d', strtotime("-{$log_retention_days} days"));
if ($conn) {
    $conn->query("DELETE FROM reminder_log WHERE sent_at < '{$old_logs} 00:00:00'");
}

logMessage("🏁 Script execution finished");
?>
