<?php
/**
 * Database Migration Script: Add Student Registration Fields
 * Run this script ONCE to add all the new student registration fields to the users table
 * 
 * Access via browser: /database/migrate_student_fields.php
 */

// Include database connection
require_once __DIR__ . '/../includes/db_connection.php';

// Security: Only allow migration in development or with proper authentication
// Uncomment one of these lines for security:
// if ($_SERVER['SERVER_NAME'] !== 'localhost') die('Migration only allowed on localhost');
// if (!isset($_GET['confirm']) || $_GET['confirm'] !== 'yes') die('Add ?confirm=yes to run migration');

echo "<h1>🔄 Student Registration Fields Migration</h1>";
echo "<p><strong>Date:</strong> " . date('Y-m-d H:i:s') . "</p>";
echo "<p><strong>Purpose:</strong> Add comprehensive student registration fields to users table</p>";

if (!$conn) {
    die("❌ <strong>Error:</strong> Database connection failed!");
}

echo "<h2>📋 Migration Progress</h2>";

// Function to check if column exists
function columnExists($conn, $table, $column) {
    $result = $conn->query("SHOW COLUMNS FROM `$table` LIKE '$column'");
    return $result && $result->num_rows > 0;
}

// Function to safely add column
function addColumn($conn, $table, $columnDef, $description) {
    // Extract column name from definition
    preg_match('/ADD COLUMN (\w+)/', $columnDef, $matches);
    $columnName = $matches[1] ?? '';
    
    if (columnExists($conn, $table, $columnName)) {
        echo "⚠️ <strong>$description:</strong> Column '$columnName' already exists - skipping<br>";
        return true;
    }
    
    $sql = "ALTER TABLE `$table` $columnDef";
    if ($conn->query($sql) === TRUE) {
        echo "✅ <strong>$description:</strong> Successfully added column '$columnName'<br>";
        return true;
    } else {
        echo "❌ <strong>$description:</strong> Error - " . $conn->error . "<br>";
        return false;
    }
}

// Array of columns to add
$columns = [
    [
        'def' => "ADD COLUMN title VARCHAR(10) AFTER last_name",
        'desc' => "Title Field"
    ],
    [
        'def' => "ADD COLUMN cell_number VARCHAR(15) AFTER email",
        'desc' => "Cell Number Field"
    ],
    [
        'def' => "ADD COLUMN id_number VARCHAR(20) AFTER cell_number",
        'desc' => "ID Number Field"
    ],
    [
        'def' => "ADD COLUMN suburb VARCHAR(100) AFTER id_number",
        'desc' => "Suburb Field"
    ],
    [
        'def' => "ADD COLUMN province VARCHAR(50) AFTER suburb",
        'desc' => "Province Field"
    ],
    [
        'def' => "ADD COLUMN communication_method ENUM('WhatsApp', 'Email') AFTER province",
        'desc' => "Communication Method Field"
    ],
    [
        'def' => "ADD COLUMN owns_laptop ENUM('Yes', 'No') AFTER communication_method",
        'desc' => "Laptop Ownership Field"
    ],
    [
        'def' => "ADD COLUMN training_preference ENUM('Onsite', 'Online') AFTER owns_laptop",
        'desc' => "Training Preference Field"
    ],
    [
        'def' => "ADD COLUMN job_field_interest VARCHAR(100) AFTER training_preference",
        'desc' => "Job Field Interest"
    ],
    [
        'def' => "ADD COLUMN job_field_other VARCHAR(100) AFTER job_field_interest",
        'desc' => "Job Field Other Specification"
    ],
    [
        'def' => "ADD COLUMN education_status VARCHAR(100) AFTER job_field_other",
        'desc' => "Education Status Field"
    ],
    [
        'def' => "ADD COLUMN employment_status TEXT AFTER education_status",
        'desc' => "Employment Status Field"
    ],
    [
        'def' => "ADD COLUMN heard_about_us VARCHAR(50) AFTER employment_status",
        'desc' => "Marketing Source Field"
    ],
    [
        'def' => "ADD COLUMN referral_person_name VARCHAR(100) AFTER heard_about_us",
        'desc' => "Referral Person Name"
    ]
];

$successCount = 0;
$totalColumns = count($columns);

echo "<div style='background: #f8f9fa; padding: 15px; border-left: 4px solid #007bff; margin: 10px 0;'>";
echo "<h3>Adding Columns to 'users' Table:</h3>";

// Add all columns
foreach ($columns as $column) {
    if (addColumn($conn, 'users', $column['def'], $column['desc'])) {
        $successCount++;
    }
}

echo "</div>";

// Add indexes for performance
echo "<h3>🔍 Adding Database Indexes:</h3>";
$indexes = [
    "CREATE INDEX idx_users_province ON users(province)" => "Province Index",
    "CREATE INDEX idx_users_job_field ON users(job_field_interest)" => "Job Field Index",
    "CREATE INDEX idx_users_education ON users(education_status)" => "Education Index",
    "CREATE INDEX idx_users_heard_about ON users(heard_about_us)" => "Marketing Source Index"
];

foreach ($indexes as $indexSql => $indexDesc) {
    if ($conn->query($indexSql) === TRUE) {
        echo "✅ <strong>$indexDesc:</strong> Successfully created<br>";
    } else {
        // Index might already exist, check if it's a duplicate key error
        if (strpos($conn->error, 'Duplicate key name') !== false) {
            echo "⚠️ <strong>$indexDesc:</strong> Already exists - skipping<br>";
        } else {
            echo "❌ <strong>$indexDesc:</strong> Error - " . $conn->error . "<br>";
        }
    }
}

// Add column comments for documentation
echo "<h3>📝 Adding Column Documentation:</h3>";
$comments = [
    "ALTER TABLE users MODIFY COLUMN title VARCHAR(10) COMMENT 'Student title: Mr, Mrs, Miss, Other'" => "Title Documentation",
    "ALTER TABLE users MODIFY COLUMN cell_number VARCHAR(15) COMMENT 'Student cell phone number'" => "Cell Number Documentation",
    "ALTER TABLE users MODIFY COLUMN id_number VARCHAR(20) COMMENT 'South African ID number'" => "ID Number Documentation",
    "ALTER TABLE users MODIFY COLUMN suburb VARCHAR(100) COMMENT 'Student suburb/area'" => "Suburb Documentation",
    "ALTER TABLE users MODIFY COLUMN province VARCHAR(50) COMMENT 'South African province'" => "Province Documentation"
];

foreach ($comments as $commentSql => $commentDesc) {
    if ($conn->query($commentSql) === TRUE) {
        echo "✅ <strong>$commentDesc:</strong> Added<br>";
    } else {
        echo "⚠️ <strong>$commentDesc:</strong> " . $conn->error . "<br>";
    }
}

// Final summary
echo "<div style='background: #d4edda; border: 1px solid #c3e6cb; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
echo "<h2>🎉 Migration Summary</h2>";
echo "<p><strong>Columns Added:</strong> $successCount / $totalColumns</p>";
echo "<p><strong>Status:</strong> " . ($successCount === $totalColumns ? "✅ All columns added successfully!" : "⚠️ Some columns may have already existed") . "</p>";
echo "<p><strong>Next Steps:</strong> You can now test the student registration form with all the new fields!</p>";
echo "</div>";

// Show current table structure
echo "<h3>📊 Current Users Table Structure:</h3>";
$result = $conn->query("DESCRIBE users");
if ($result) {
    echo "<div style='max-height: 400px; overflow-y: auto; border: 1px solid #ddd; padding: 10px;'>";
    echo "<table style='width: 100%; border-collapse: collapse;'>";
    echo "<tr style='background: #f1f1f1;'><th style='border: 1px solid #ddd; padding: 8px;'>Field</th><th style='border: 1px solid #ddd; padding: 8px;'>Type</th><th style='border: 1px solid #ddd; padding: 8px;'>Null</th><th style='border: 1px solid #ddd; padding: 8px;'>Key</th></tr>";
    
    while ($row = $result->fetch_assoc()) {
        $highlight = in_array($row['Field'], ['title', 'cell_number', 'id_number', 'suburb', 'province', 'communication_method', 'owns_laptop', 'training_preference', 'job_field_interest', 'job_field_other', 'education_status', 'employment_status', 'heard_about_us', 'referral_person_name']) ? 'background-color: #fff3cd;' : '';
        echo "<tr style='$highlight'>";
        echo "<td style='border: 1px solid #ddd; padding: 8px;'>" . htmlspecialchars($row['Field']) . "</td>";
        echo "<td style='border: 1px solid #ddd; padding: 8px;'>" . htmlspecialchars($row['Type']) . "</td>";
        echo "<td style='border: 1px solid #ddd; padding: 8px;'>" . htmlspecialchars($row['Null']) . "</td>";
        echo "<td style='border: 1px solid #ddd; padding: 8px;'>" . htmlspecialchars($row['Key']) . "</td>";
        echo "</tr>";
    }
    echo "</table>";
    echo "</div>";
    echo "<p><em>Highlighted rows are the newly added student registration fields.</em></p>";
}

$conn->close();

echo "<div style='background: #d1ecf1; border: 1px solid #b8daff; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
echo "<h3>🔒 Security Notice</h3>";
echo "<p><strong>Important:</strong> For security reasons, consider deleting this migration script after running it, or add proper authentication.</p>";
echo "<p>You can now test the enhanced student registration form at: <a href='/auth/register.php'>/auth/register.php</a></p>";
echo "</div>";
?>

<style>
body { font-family: Arial, sans-serif; max-width: 1000px; margin: 0 auto; padding: 20px; }
h1 { color: #007bff; }
h2 { color: #28a745; margin-top: 30px; }
h3 { color: #17a2b8; margin-top: 25px; }
</style>
