<?php
/**
 * LIGHTWEIGHT Student Registration Fields Migration
 * Fast migration script for live server - minimal output, maximum speed
 */

// Include database connection
require_once __DIR__ . '/../includes/db_connection.php';

// Set time limit and disable output buffering for faster execution
set_time_limit(300); // 5 minutes max
ini_set('max_execution_time', 300);

// Start output
echo "<h1>🚀 Quick Migration: Student Fields</h1>";
echo "<p>Adding student registration fields...</p>";
flush();

if (!$conn) {
    die("❌ Database connection failed!");
}

// Single SQL statement to add all columns at once (much faster)
$addColumnsSql = "ALTER TABLE users 
    ADD COLUMN IF NOT EXISTS title VARCHAR(10) AFTER last_name,
    ADD COLUMN IF NOT EXISTS cell_number VARCHAR(15) AFTER email,
    ADD COLUMN IF NOT EXISTS id_number VARCHAR(20) AFTER cell_number,
    ADD COLUMN IF NOT EXISTS suburb VARCHAR(100) AFTER id_number,
    ADD COLUMN IF NOT EXISTS province VARCHAR(50) AFTER suburb,
    ADD COLUMN IF NOT EXISTS communication_method ENUM('WhatsApp', 'Email') AFTER province,
    ADD COLUMN IF NOT EXISTS owns_laptop ENUM('Yes', 'No') AFTER communication_method,
    ADD COLUMN IF NOT EXISTS training_preference ENUM('Onsite', 'Online') AFTER owns_laptop,
    ADD COLUMN IF NOT EXISTS job_field_interest VARCHAR(100) AFTER training_preference,
    ADD COLUMN IF NOT EXISTS job_field_other VARCHAR(100) AFTER job_field_interest,
    ADD COLUMN IF NOT EXISTS education_status VARCHAR(100) AFTER job_field_other,
    ADD COLUMN IF NOT EXISTS employment_status TEXT AFTER education_status,
    ADD COLUMN IF NOT EXISTS heard_about_us VARCHAR(50) AFTER employment_status,
    ADD COLUMN IF NOT EXISTS referral_person_name VARCHAR(100) AFTER heard_about_us";

echo "⏳ Adding all 14 columns in one operation...<br>";
flush();

if ($conn->query($addColumnsSql) === TRUE) {
    echo "✅ <strong>SUCCESS!</strong> All student registration fields added successfully!<br>";
} else {
    // If the single statement fails, try individual columns for better error reporting
    echo "⚠️ Bulk operation failed, trying individual columns...<br>";
    
    $individualColumns = [
        "ALTER TABLE users ADD COLUMN IF NOT EXISTS title VARCHAR(10) AFTER last_name",
        "ALTER TABLE users ADD COLUMN IF NOT EXISTS cell_number VARCHAR(15) AFTER email",
        "ALTER TABLE users ADD COLUMN IF NOT EXISTS id_number VARCHAR(20) AFTER cell_number",
        "ALTER TABLE users ADD COLUMN IF NOT EXISTS suburb VARCHAR(100) AFTER id_number",
        "ALTER TABLE users ADD COLUMN IF NOT EXISTS province VARCHAR(50) AFTER suburb",
        "ALTER TABLE users ADD COLUMN IF NOT EXISTS communication_method ENUM('WhatsApp', 'Email') AFTER province",
        "ALTER TABLE users ADD COLUMN IF NOT EXISTS owns_laptop ENUM('Yes', 'No') AFTER communication_method",
        "ALTER TABLE users ADD COLUMN IF NOT EXISTS training_preference ENUM('Onsite', 'Online') AFTER owns_laptop",
        "ALTER TABLE users ADD COLUMN IF NOT EXISTS job_field_interest VARCHAR(100) AFTER training_preference",
        "ALTER TABLE users ADD COLUMN IF NOT EXISTS job_field_other VARCHAR(100) AFTER job_field_interest",
        "ALTER TABLE users ADD COLUMN IF NOT EXISTS education_status VARCHAR(100) AFTER job_field_other",
        "ALTER TABLE users ADD COLUMN IF NOT EXISTS employment_status TEXT AFTER education_status",
        "ALTER TABLE users ADD COLUMN IF NOT EXISTS heard_about_us VARCHAR(50) AFTER employment_status",
        "ALTER TABLE users ADD COLUMN IF NOT EXISTS referral_person_name VARCHAR(100) AFTER heard_about_us"
    ];
    
    $successCount = 0;
    foreach ($individualColumns as $i => $sql) {
        if ($conn->query($sql) === TRUE) {
            $successCount++;
        }
    }
    
    echo "✅ Added $successCount / " . count($individualColumns) . " columns<br>";
}

echo "⏳ Adding performance indexes...<br>";
flush();

// Essential indexes only
$indexes = [
    "CREATE INDEX IF NOT EXISTS idx_users_province ON users(province)",
    "CREATE INDEX IF NOT EXISTS idx_users_job_field ON users(job_field_interest)"
];

foreach ($indexes as $indexSql) {
    $conn->query($indexSql); // Execute silently
}

echo "✅ Indexes added<br>";

// Quick verification - count columns
$result = $conn->query("SHOW COLUMNS FROM users");
$columnCount = $result ? $result->num_rows : 0;

echo "<div style='background: #d4edda; padding: 15px; margin: 20px 0; border-radius: 5px;'>";
echo "<h2>🎉 Migration Complete!</h2>";
echo "<p><strong>Total columns in users table:</strong> $columnCount</p>";
echo "<p><strong>Status:</strong> ✅ Student registration fields are ready!</p>";
echo "<p><strong>Time:</strong> " . date('Y-m-d H:i:s') . "</p>";
echo "</div>";

// Test the registration form
echo "<p><strong>Next step:</strong> <a href='/auth/register.php' target='_blank'>Test the student registration form →</a></p>";

$conn->close();

echo "<p><em>Migration completed in " . (microtime(true) - $_SERVER['REQUEST_TIME_FLOAT']) . " seconds</em></p>";
?>

<style>
body { font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px; }
h1 { color: #28a745; }
</style>
