-- Panadite Academy Database Schema

-- Drop database if it exists and recreate it
DROP DATABASE IF EXISTS panadite_academy;
CREATE DATABASE panadite_academy;
USE panadite_academy;

-- Set the default character set and collation
SET NAMES utf8mb4;
SET foreign_key_checks = 0;

-- Users table (common for all user types)
CREATE TABLE users (
    user_id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    email VARCHAR(100) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    first_name VARCHAR(50) NOT NULL,
    last_name VARCHAR(50) NOT NULL,
    profile_picture VARCHAR(255),
    user_role ENUM('student', 'teacher', 'admin') NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Student profiles
CREATE TABLE student_profiles (
    student_id INT PRIMARY KEY,
    bio TEXT,
    FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Teacher profiles
CREATE TABLE teacher_profiles (
    teacher_id INT PRIMARY KEY,
    bio TEXT,
    qualifications TEXT,
    expertise VARCHAR(255),
    FOREIGN KEY (teacher_id) REFERENCES users(user_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Admin profiles
CREATE TABLE admin_profiles (
    admin_id INT PRIMARY KEY,
    permissions TEXT,
    FOREIGN KEY (admin_id) REFERENCES users(user_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Course categories
CREATE TABLE categories (
    category_id INT AUTO_INCREMENT PRIMARY KEY,
    category_name VARCHAR(100) NOT NULL,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Courses
CREATE TABLE courses (
    course_id INT AUTO_INCREMENT PRIMARY KEY,
    teacher_id INT NOT NULL,
    category_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    thumbnail VARCHAR(255),
    price DECIMAL(10, 2) DEFAULT 0.00,
    is_published BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (teacher_id) REFERENCES users(user_id) ON DELETE CASCADE,
    FOREIGN KEY (category_id) REFERENCES categories(category_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Videos
CREATE TABLE videos (
    video_id INT AUTO_INCREMENT PRIMARY KEY,
    course_id INT NOT NULL,
    teacher_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    video_url VARCHAR(255) NOT NULL,
    thumbnail VARCHAR(255),
    duration VARCHAR(10),
    price DECIMAL(10, 2) DEFAULT 0.00,
    views INT DEFAULT 0,
    likes INT DEFAULT 0,
    category_name VARCHAR(100),
    is_free BOOLEAN DEFAULT FALSE,
    is_published BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (course_id) REFERENCES courses(course_id) ON DELETE CASCADE,
    FOREIGN KEY (teacher_id) REFERENCES users(user_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Enrollments (students enrolled in courses)
CREATE TABLE enrollments (
    enrollment_id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    course_id INT NOT NULL,
    enrolled_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    progress INT DEFAULT 0,
    FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE,
    FOREIGN KEY (course_id) REFERENCES courses(course_id) ON DELETE CASCADE,
    UNIQUE KEY (student_id, course_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Video purchases
CREATE TABLE video_purchases (
    purchase_id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    video_id INT NOT NULL,
    amount DECIMAL(10, 2) NOT NULL,
    payment_id VARCHAR(255) NOT NULL,
    payment_status ENUM('pending', 'completed', 'failed') DEFAULT 'pending',
    purchase_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE,
    FOREIGN KEY (video_id) REFERENCES videos(video_id) ON DELETE CASCADE,
    UNIQUE KEY (user_id, video_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Course purchases
CREATE TABLE course_purchases (
    purchase_id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    course_id INT NOT NULL,
    amount DECIMAL(10, 2) NOT NULL,
    payment_id VARCHAR(255) NOT NULL,
    payment_status ENUM('pending', 'completed', 'failed') DEFAULT 'pending',
    purchase_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE,
    FOREIGN KEY (course_id) REFERENCES courses(course_id) ON DELETE CASCADE,
    UNIQUE KEY (user_id, course_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Comments
CREATE TABLE comments (
    comment_id INT AUTO_INCREMENT PRIMARY KEY,
    video_id INT NOT NULL,
    user_id INT NOT NULL,
    comment_text TEXT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (video_id) REFERENCES videos(video_id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Ratings
CREATE TABLE ratings (
    rating_id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    course_id INT,
    video_id INT,
    rating DECIMAL(2, 1) NOT NULL,
    review TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE,
    FOREIGN KEY (course_id) REFERENCES courses(course_id) ON DELETE CASCADE,
    FOREIGN KEY (video_id) REFERENCES videos(video_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Assignments
CREATE TABLE assignments (
    assignment_id INT AUTO_INCREMENT PRIMARY KEY,
    course_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    due_date DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (course_id) REFERENCES courses(course_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Assignment submissions
CREATE TABLE assignment_submissions (
    submission_id INT AUTO_INCREMENT PRIMARY KEY,
    assignment_id INT NOT NULL,
    student_id INT NOT NULL,
    submission_text TEXT,
    file_url VARCHAR(255),
    submission_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    grade DECIMAL(5, 2),
    feedback TEXT,
    FOREIGN KEY (assignment_id) REFERENCES assignments(assignment_id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE,
    UNIQUE KEY (student_id, assignment_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Messages
CREATE TABLE messages (
    message_id INT AUTO_INCREMENT PRIMARY KEY,
    sender_id INT NOT NULL,
    recipient_id INT NOT NULL,
    message_text TEXT NOT NULL,
    is_read BOOLEAN DEFAULT FALSE,
    sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (sender_id) REFERENCES users(user_id) ON DELETE CASCADE,
    FOREIGN KEY (recipient_id) REFERENCES users(user_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Forums
CREATE TABLE forums (
    forum_id INT AUTO_INCREMENT PRIMARY KEY,
    course_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (course_id) REFERENCES courses(course_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Forum topics
CREATE TABLE forum_topics (
    topic_id INT AUTO_INCREMENT PRIMARY KEY,
    forum_id INT NOT NULL,
    user_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    content TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (forum_id) REFERENCES forums(forum_id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Forum replies
CREATE TABLE forum_replies (
    reply_id INT AUTO_INCREMENT PRIMARY KEY,
    topic_id INT NOT NULL,
    user_id INT NOT NULL,
    content TEXT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (topic_id) REFERENCES forum_topics(topic_id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Calendar events
CREATE TABLE calendar_events (
    event_id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    course_id INT,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    start_time DATETIME NOT NULL,
    end_time DATETIME NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE,
    FOREIGN KEY (course_id) REFERENCES courses(course_id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- PayFast settings
CREATE TABLE payfast_settings (
    setting_id INT AUTO_INCREMENT PRIMARY KEY,
    merchant_id VARCHAR(50) NOT NULL,
    merchant_key VARCHAR(50) NOT NULL,
    passphrase VARCHAR(255),
    sandbox_mode BOOLEAN DEFAULT TRUE,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Enable foreign key checks
SET foreign_key_checks = 1;

-- Sample data
-- Insert default admin
INSERT INTO users (user_id, username, email, password, first_name, last_name, user_role) 
VALUES (1, 'admin', 'admin@panadite.com', '$2y$10$6Y5dkf5929ZqX.Y4zOEFgOKKVwOUtSRaRLugdSWwWgDqKZ80GNlGi', 'Admin', 'User', 'admin');

-- Insert sample teachers
INSERT INTO users (user_id, username, email, password, first_name, last_name, user_role)
VALUES 
(2, 'teacher1', 'teacher1@panadite.com', '$2y$10$6Y5dkf5929ZqX.Y4zOEFgOKKVwOUtSRaRLugdSWwWgDqKZ80GNlGi', 'John', 'Doe', 'teacher'),
(3, 'teacher2', 'teacher2@panadite.com', '$2y$10$6Y5dkf5929ZqX.Y4zOEFgOKKVwOUtSRaRLugdSWwWgDqKZ80GNlGi', 'Jane', 'Smith', 'teacher');

-- Insert sample students
INSERT INTO users (user_id, username, email, password, first_name, last_name, user_role)
VALUES 
(4, 'student1', 'student1@panadite.com', '$2y$10$6Y5dkf5929ZqX.Y4zOEFgOKKVwOUtSRaRLugdSWwWgDqKZ80GNlGi', 'Alex', 'Johnson', 'student'),
(5, 'student2', 'student2@panadite.com', '$2y$10$6Y5dkf5929ZqX.Y4zOEFgOKKVwOUtSRaRLugdSWwWgDqKZ80GNlGi', 'Sarah', 'Brown', 'student');

-- Insert admin profile
INSERT INTO admin_profiles (admin_id, permissions)
VALUES (1, 'all');

-- Insert teacher profiles
INSERT INTO teacher_profiles (teacher_id, bio, qualifications, expertise)
VALUES 
(2, 'Experienced programming instructor with 10 years of teaching experience', 'MSc Computer Science, Certified Software Developer', 'Python, JavaScript, Web Development'),
(3, 'Math and Science teacher with passion for making difficult concepts easy to understand', 'PhD Mathematics, Science Education Certification', 'Mathematics, Physics, Chemistry');

-- Insert student profiles
INSERT INTO student_profiles (student_id, bio)
VALUES 
(4, 'Studying computer science and eager to learn new programming languages'),
(5, 'Business major with interest in entrepreneurship and marketing');

-- Insert default PayFast settings
INSERT INTO payfast_settings (merchant_id, merchant_key, sandbox_mode)
VALUES ('10038898', 'jkeryt8c9i8sy', TRUE);

-- Insert default categories
INSERT INTO categories (category_id, category_name, description)
VALUES 
(1, 'Programming', 'Learn programming languages and software development'),
(2, 'Mathematics', 'Mathematics courses from basic to advanced'),
(3, 'Business', 'Business, management, and entrepreneurship'),
(4, 'Science', 'Physics, chemistry, biology and more'),
(5, 'Languages', 'Learn new languages'),
(6, 'Technology', 'Modern tech trends and applications'),
(7, 'Arts', 'Creative arts and design principles');

-- Insert sample courses
INSERT INTO courses (course_id, teacher_id, category_id, title, description, thumbnail, price, is_published)
VALUES
(1, 2, 1, 'Introduction to Python Programming', 'Learn Python programming from scratch with practical examples', '/assets/images/courses/python.jpg', 49.99, TRUE),
(2, 3, 2, 'Advanced Mathematics', 'Take your math skills to the next level with advanced topics', '/assets/images/courses/math.jpg', 39.99, TRUE),
(3, 2, 1, 'Web Development Bootcamp', 'Comprehensive web development course covering HTML, CSS, and JavaScript', '/assets/images/courses/webdev.jpg', 59.99, TRUE),
(4, 3, 4, 'Introduction to Physics', 'Understand the fundamental principles of physics', '/assets/images/courses/physics.jpg', 44.99, TRUE);

-- Insert sample videos
INSERT INTO videos (video_id, course_id, teacher_id, title, description, video_url, thumbnail, duration, price, views, likes, category_name, is_free, is_published)
VALUES
(1, 1, 2, 'Getting Started with Python', 'Learn the basics of Python programming language', 'https://youtu.be/example1', '/assets/images/videos/python1.jpg', '15:30', 0.00, 1250, 320, 'Programming', TRUE, TRUE),
(2, 1, 2, 'Python Data Structures', 'Understanding lists, tuples, dictionaries and sets in Python', 'https://youtu.be/example2', '/assets/images/videos/python2.jpg', '22:15', 0.00, 980, 256, 'Programming', FALSE, TRUE),
(3, 2, 3, 'Introduction to Calculus', 'Learn the fundamentals of calculus', 'https://youtu.be/example3', '/assets/images/videos/math1.jpg', '18:45', 0.00, 850, 192, 'Mathematics', TRUE, TRUE),
(4, 3, 2, 'HTML Fundamentals', 'Getting started with HTML for web development', 'https://youtu.be/example4', '/assets/images/videos/webdev1.jpg', '20:10', 0.00, 1450, 385, 'Programming', TRUE, TRUE),
(5, 3, 2, 'CSS Styling Basics', 'Learn how to style web pages with CSS', 'https://youtu.be/example5', '/assets/images/videos/webdev2.jpg', '25:40', 0.00, 1280, 340, 'Programming', FALSE, TRUE),
(6, 4, 3, 'Understanding Newton''s Laws', 'Comprehensive explanation of Newton''s laws of motion', 'https://youtu.be/example6', '/assets/images/videos/physics1.jpg', '19:20', 0.00, 720, 185, 'Science', TRUE, TRUE);
